﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/os.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/updater/updater.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

namespace
{
NN_ALIGNAS(4096) char s_Buffer[64 << 10]; // 256KiB

// see: ns_SystemUpdateApplyManager.cpp
nn::updater::BootImageUpdateType GetBootImageUpdateType() NN_NOEXCEPT
{
    int bootImageUpdateType;
    auto size = nn::settings::fwdbg::GetSettingsItemValue(&bootImageUpdateType, sizeof(bootImageUpdateType), "systeminitializer", "boot_image_update_type");
    // 設定項目の取得に失敗した場合は旧来の挙動にする
    if (size != sizeof(bootImageUpdateType))
    {
        return nn::updater::BootImageUpdateType::Original;
    }
    return nn::updater::GetBootImageUpdateType(bootImageUpdateType);
}

class ScopeTimer
{
    NN_DISALLOW_COPY(ScopeTimer);
public:
    explicit ScopeTimer(const char* name) NN_NOEXCEPT: m_Name(name)
    {
        m_T1 = nn::os::GetSystemTick();
        NN_LOG("%s Start\n", m_Name);
    }
    ~ScopeTimer() NN_NOEXCEPT
    {
        m_T2 = nn::os::GetSystemTick();
        NN_LOG("%s: %d msec\n", m_Name, nn::os::ConvertToTimeSpan(m_T2 - m_T1).GetMilliSeconds());
    }

private:
    const char* m_Name;
    nn::os::Tick m_T1, m_T2;
};

bool GetFlag(const nn::updater::VerifyingRequiredFlags& flags, nn::updater::TargetBootMode mode) NN_NOEXCEPT
{
    switch(mode)
    {
    case nn::updater::TargetBootMode::Normal:
        return flags.normalRequired;
    case nn::updater::TargetBootMode::Safe:
        return flags.safeRequired;
    default:
        NN_LOG("Invalid mode: %d\n", mode);
    }
    return true;
}

}

void UpdateBootImages(nn::ncm::SystemDataId id, nn::updater::TargetBootMode mode)
{
    auto updateType = GetBootImageUpdateType();
    {
        ScopeTimer t("----Verify");
        auto result = nn::updater::VerifyBootImages(id, mode, s_Buffer, sizeof(s_Buffer), updateType);

        if (result.IsFailure())
        {
            NN_LOG("Error: %08x\n", result.GetInnerValueForDebug());
            NN_LOG("Incorrect state, update necessary\n");
        }
    }
    {
        ScopeTimer t("----Update");
        auto result = nn::updater::UpdateBootImagesFromPackage(id, mode, s_Buffer, sizeof(s_Buffer), updateType);
        if (result.IsFailure())
        {
            NN_LOG("Error: %08x\n", result.GetInnerValueForDebug());
            return;
        }
    }
    {
        ScopeTimer t("----Verify2");
        auto result = nn::updater::VerifyBootImages(id, mode, s_Buffer, sizeof(s_Buffer), updateType);

        if (result.IsFailure())
        {
            NN_LOG("Error: %08x\n", result.GetInnerValueForDebug());
            return;
        }
    }
    NN_LOG("Success!\n");
}
void UpdateBootImagesWithVerifyingMark(nn::ncm::SystemDataId id,  nn::updater::TargetBootMode mode)
{
    auto updateType = GetBootImageUpdateType();
    nn::updater::VerifyingRequiredFlags flags;

    {
        ScopeTimer t("----Get verify flag");
        auto result = nn::updater::CheckVerifyingRequired(&flags, s_Buffer, sizeof(s_Buffer));
        if (result.IsFailure())
        {
            NN_LOG("Error: %08x\n", result.GetInnerValueForDebug());
            return;
        }
    }
    auto flag = GetFlag(flags, mode);
    if (flag)
    {
        nn::Result result;
        {
            ScopeTimer t("----Verify");
            result = nn::updater::VerifyBootImages(id, mode, s_Buffer, sizeof(s_Buffer), updateType);
        }
        if (result.IsSuccess())
        {
            NN_LOG("Correct state\n");
            {
                ScopeTimer t("----Mark verified");
                result = nn::updater::MarkVerified(mode, s_Buffer, sizeof(s_Buffer));
                if (result.IsFailure())
                {
                    NN_LOG("Error: %08x\n", result.GetInnerValueForDebug());
                    return;
                }
            }
            return;
        }
        else if (nn::updater::ResultRequiresUpdateBootImages::Includes(result))
        {
            NN_LOG("Do update boot images\n");
            {
                ScopeTimer t("----Update");
                result = nn::updater::UpdateBootImagesFromPackage(id, mode, s_Buffer, sizeof(s_Buffer), updateType);
                if (result.IsFailure())
                {
                    NN_LOG("Error: %08x\n", result.GetInnerValueForDebug());
                    return;
                }
            }
            NN_LOG("Update success, re-verify\n");
            {
                ScopeTimer t("----Verify");
                result = nn::updater::VerifyBootImages(id, mode, s_Buffer, sizeof(s_Buffer), updateType);
                if (result.IsFailure())
                {
                    NN_LOG("Error: %08x\n", result.GetInnerValueForDebug());
                    return;
                }
            }
            {
                ScopeTimer t("----Mark verified");
                result = nn::updater::MarkVerified(mode, s_Buffer, sizeof(s_Buffer));
                if (result.IsFailure())
                {
                    NN_LOG("Error: %08x\n", result.GetInnerValueForDebug());
                    return;
                }
            }
        }
        else
        {
            NN_LOG("Error: %08x\n", result.GetInnerValueForDebug());
            return;
        }
    }
    NN_LOG("Success!\n");
}
void NormalCheck()
{
    nn::ncm::SystemDataId normalId, safeId;
    normalId.value = 0x0100000000000819LL;
    safeId.value = 0x010000000000081ALL;

    {
        ScopeTimer t("-- Update normal");
        UpdateBootImages(normalId, nn::updater::TargetBootMode::Normal);
    }
    {
        ScopeTimer t("-- Update safe");
        UpdateBootImages(safeId, nn::updater::TargetBootMode::Safe);
    }
    {
        ScopeTimer t("-- Write safe into normal");
        UpdateBootImages(safeId, nn::updater::TargetBootMode::Normal);
    }
    {
        ScopeTimer t("-- Update normal with verify flag");
        UpdateBootImagesWithVerifyingMark(normalId, nn::updater::TargetBootMode::Normal);
    }
    {
        ScopeTimer t("-- Disable mark");
        nn::updater::MarkVerifyingRequired(nn::updater::TargetBootMode::Normal, s_Buffer, sizeof(s_Buffer));
    }
    {
        ScopeTimer t("-- Update normal with verify flag2");
        UpdateBootImagesWithVerifyingMark(normalId, nn::updater::TargetBootMode::Normal);
    }
    {
        ScopeTimer t("-- Disable mark");
        nn::updater::MarkVerifyingRequired(nn::updater::TargetBootMode::Normal, s_Buffer, sizeof(s_Buffer));
    }
    {
        ScopeTimer t("-- Update normal with verify flag3");
        UpdateBootImagesWithVerifyingMark(normalId, nn::updater::TargetBootMode::Normal);
    }
    {
        ScopeTimer t("-- Update normal with verify flag4");
        UpdateBootImagesWithVerifyingMark(normalId, nn::updater::TargetBootMode::Normal);
    }
}
void SafeCheck()
{
    nn::ncm::SystemDataId normalId, safeId;
    normalId.value = 0x0100000000000819LL;
    safeId.value = 0x010000000000081ALL;

    // reverse
    {
        ScopeTimer t("-- Write normal into safe");
        UpdateBootImages(normalId, nn::updater::TargetBootMode::Safe);
    }
    {
        ScopeTimer t("-- Update normal with verify flag");
        UpdateBootImagesWithVerifyingMark(safeId, nn::updater::TargetBootMode::Safe);
    }
    {
        ScopeTimer t("-- Disable mark");
        nn::updater::MarkVerifyingRequired(nn::updater::TargetBootMode::Safe, s_Buffer, sizeof(s_Buffer));
    }
    {
        ScopeTimer t("-- Update normal with verify flag2");
        UpdateBootImagesWithVerifyingMark(safeId, nn::updater::TargetBootMode::Safe);
    }
    {
        ScopeTimer t("-- Disable mark");
        nn::updater::MarkVerifyingRequired(nn::updater::TargetBootMode::Safe, s_Buffer, sizeof(s_Buffer));
    }
    {
        ScopeTimer t("-- Update normal with verify flag3");
        UpdateBootImagesWithVerifyingMark(safeId, nn::updater::TargetBootMode::Safe);
    }
    {
        ScopeTimer t("-- Update normal with verify flag4");
        UpdateBootImagesWithVerifyingMark(safeId, nn::updater::TargetBootMode::Safe);
    }
}
void FlagCheck()
{
    nn::updater::VerifyingRequiredFlags flags;
    nn::updater::CheckVerifyingRequired(&flags, s_Buffer, sizeof(s_Buffer));
    NN_LOG("Expect false: %d\n", flags.normalRequired);

    nn::updater::MarkVerifyingRequired(nn::updater::TargetBootMode::Normal, s_Buffer, sizeof(s_Buffer));
    nn::updater::CheckVerifyingRequired(&flags, s_Buffer, sizeof(s_Buffer));
    NN_LOG("Expect true: %d\n", flags.normalRequired);
    NN_LOG("Expect false: %d\n", flags.safeRequired);

    nn::updater::MarkVerified(nn::updater::TargetBootMode::Normal, s_Buffer, sizeof(s_Buffer));
    nn::updater::CheckVerifyingRequired(&flags, s_Buffer, sizeof(s_Buffer));
    NN_LOG("Expect false: %d\n", flags.normalRequired);
    NN_LOG("Expect false: %d\n", flags.safeRequired);

    nn::updater::MarkVerifyingRequired(nn::updater::TargetBootMode::Safe, s_Buffer, sizeof(s_Buffer));
    nn::updater::CheckVerifyingRequired(&flags, s_Buffer, sizeof(s_Buffer));
    NN_LOG("Expect false: %d\n", flags.normalRequired);
    NN_LOG("Expect true: %d\n", flags.safeRequired);

    nn::updater::MarkVerified(nn::updater::TargetBootMode::Safe, s_Buffer, sizeof(s_Buffer));
    nn::updater::CheckVerifyingRequired(&flags, s_Buffer, sizeof(s_Buffer));
    NN_LOG("Expect false: %d\n", flags.normalRequired);
    NN_LOG("Expect false: %d\n", flags.safeRequired);
}

extern "C" void nnMain()
{
    nn::ncm::SystemDataId normalId, safeId, id;
    normalId.value = 0x0100000000000819LL;
    safeId.value = 0x010000000000081ALL;

    nn::ncm::Initialize();

    {
        ScopeTimer t("-- Get normal id");
        auto result = nn::updater::GetBootImagePackageId(&id, nn::updater::TargetBootMode::Normal, s_Buffer, sizeof(s_Buffer));
        if (result.IsSuccess())
        {
            NN_LOG("Get id: %016llx\n", id.value);
        }
        else
        {
            NN_LOG("Get id failed: %08x\n", result.GetInnerValueForDebug());
        }
    }
    {
        ScopeTimer t("-- Get safe id");
        auto result = nn::updater::GetBootImagePackageId(&id, nn::updater::TargetBootMode::Safe, s_Buffer, sizeof(s_Buffer));
        if (result.IsSuccess())
        {
            NN_LOG("Get id: %016llx\n", id.value);
        }
        else
        {
            NN_LOG("Get id failed: %08x\n", result.GetInnerValueForDebug());
        }
    }
    {
        ScopeTimer t("-- Just read flag");
        nn::updater::VerifyingRequiredFlags flags;
        nn::updater::CheckVerifyingRequired(&flags, s_Buffer, sizeof(s_Buffer));
        NN_LOG("flag: %d\n", flags.normalRequired);
    }
    NormalCheck();
    SafeCheck();
    FlagCheck();
}

