﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/uart/uart.h>

#include <nn/gpio/gpio.h>

#include <nnt/gtest/gtest.h>

#include "../Common/testUart_Helper.h"
#include "testUart_Bcm4356_details.h"

namespace nnt {
namespace uart {
namespace bcm4356 {
namespace details {

namespace
{

// テストに使用するデータ一式
struct TestDataInfo
{
    nn::os::SemaphoreType   bufferSemaphore;    // 同時に送信するパケット数を制限するセマフォ
    nn::os::SystemEventType receiveReadyEvent;  // 受信イベント
    nn::uart::BaudRate      baudRate;           // テスト時のボーレート
    nn::uart::PortSession*  pSession;           // UART ポート
    nn::TimeSpan            pollingInterval;    // データ受信時のポーリング間隔
    size_t                  payloadSize;        // ACL パケットのペイロード長
    size_t                  packetSize;         // ACL パケット全体のサイズ
    int                     testCount;          // テストで送受信する ACL パケットの数
    char*                   pTestData;          // ACL パケットのペイロードに格納するデータ
};

// テストの想定時間 [秒]
const int ExpectedTestSeconds = 10;

// 受信タイムアウト
const auto ReceiveTimeout = nn::TimeSpan::FromSeconds(5);

// ループバックモード時に発生するコネクション数 (BCM4356 は 2 個)
const int LoopbackConnectionCount = 2;

// ループバック用の Bluetooth 接続ハンドル
uint16_t g_ConnectionHandles[LoopbackConnectionCount];

// 受信スレッド用スタック
NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[nn::os::ThreadStackAlignment * 4];

/*
 * Local loopback モードの有効/無効を切り替えます。
 */
void SetLocalLoopbackModeEnabled(nn::uart::PortSession* pSession, bool isEnabled) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pSession);

    NN_LOG("## Test [HCI_Write_Loopback_Mode: %s]\n",
           isEnabled ? "Enable" : "Disable");

    const char WriteLoopbackModeCommand[] =
        {
            0x01, 0x02, 0x18, 0x01,                     // OGF=0x06, OCF=0x0002, ParamLength=1
            static_cast<char>(isEnabled ? 0x01 : 0x00)  // 0x00: No loopback  0x01: Local loopback
        };
    const char ConnectResult[] =
        {
            HciPacketType_Event,
            0x03,   // Connection Complete Event
            0x0B,   // 後続のデータ長
            0x00    // Success
        };
    const char DisconnectResult[] =
        {
            HciPacketType_Event,
            0x05,   // Disconnection Complete Event
            0x04,   // 後続のデータ長
            0x00    // Success
        };

    // モード切替コマンドを送信
    size_t doneBytes;
    nn::uart::Send(&doneBytes,
                   pSession,
                   WriteLoopbackModeCommand,
                   sizeof(WriteLoopbackModeCommand));
    ASSERT_EQ(sizeof(WriteLoopbackModeCommand), doneBytes);

    const char *pCompleteEvent;
    size_t completeEventSize;
    size_t compareSize;
    if (isEnabled)
    {
        pCompleteEvent    = ConnectResult;
        completeEventSize = 14;
        compareSize       = sizeof(ConnectResult);
    }
    else
    {
        pCompleteEvent    = DisconnectResult;
        completeEventSize = 7;
        compareSize       = sizeof(DisconnectResult);
    }

    // 接続/切断イベントが規定回数発生する仕様なので、チェックする
    for (int i = 0; i < LoopbackConnectionCount; ++i)
    {
        size_t receivedBytes;
        char receiveBuffer[completeEventSize];
        ASSERT_TRUE(nnt::uart::ReceiveDataSync(&receivedBytes,
                                               receiveBuffer,
                                               sizeof(receiveBuffer),
                                               *pSession,
                                               ReceiveTimeout));

        // 受信したデータのサイズと先頭部分を確認
        // (後半はハンドル値やアドレス情報なので無視)
        ASSERT_EQ(receivedBytes, completeEventSize);
        ASSERT_TRUE(nnt::uart::CompareTestData(pCompleteEvent,
                                               receiveBuffer,
                                               compareSize));

        if (isEnabled)
        {
            // ハンドルを保存 (Little endian で返ってくる)
            g_ConnectionHandles[i] = (receiveBuffer[5] << 8) | receiveBuffer[4];
            NN_LOG("Connection handle %d is 0x%03X\n", i, g_ConnectionHandles[i]);
        }
    }

    const char ExpectedResult[] =
        {
            HciPacketType_Event,
            0x0E,        // Command complete event
            0x04,        // 後続のデータ長
            0x01,        // 次に受け付けるコマンドの数
            0x02, 0x18,  // コマンド 0x1802 に対する応答であることを表す
            0x00,        // success
        };

    size_t receivedBytes;
    char receiveBuffer[sizeof(ExpectedResult)];
    ASSERT_TRUE(nnt::uart::ReceiveDataSync(&receivedBytes,
                                           receiveBuffer,
                                           sizeof(receiveBuffer),
                                           *pSession,
                                           ReceiveTimeout));

    // 受信したデータ全体とサイズを確認
    ASSERT_EQ(receivedBytes, sizeof(ExpectedResult));
    ASSERT_TRUE(nnt::uart::CompareTestData(ExpectedResult,
                                           receiveBuffer,
                                           sizeof(ExpectedResult)));
}

// 現在はイベントパケットのループバックを実施しないのでコメントアウト
/*
void TransferForLoopback_Command(nn::uart::PortSession* pSession,
                                 const char* pCommand,
                                 size_t commandBytes) NN_NOEXCEPT
{
    // HCI コマンドに対しては Loopback Command Event が発生する。
    // Loopback Command Event は最大 255 バイト。
    // 先頭 3 バイトがヘッダ、残りは送信したコマンド (先頭 1 バイトの Type を除く) が入る。
    const size_t LoopbackEventSizeMax = 255;
    const size_t ExpectedSize = commandBytes + 2;
    NN_ABORT_UNLESS(ExpectedSize <= LoopbackEventSizeMax);

    char expectedData[LoopbackEventSizeMax] =
        {
            HciPacketType_Event, 0x19     // Loopback command event
        };
    expectedData[2] = commandBytes - 1;  // 3 バイト目は後続データサイズ
    std::memcpy(expectedData + 3, pCommand + 1, commandBytes - 1);

    size_t receivedBytes;
    char receiveBuffer[LoopbackEventSizeMax];
    TransferCommand(&receivedBytes,
                    receiveBuffer,
                    ExpectedSize,
                    pSession,
                    pCommand,
                    commandBytes,
                    false);
    if (::testing::Test::HasFatalFailure())
    {
        return;
    }

    // 受信データの確認
    ASSERT_EQ(receivedBytes, ExpectedSize);
    ASSERT_TRUE(nnt::uart::CompareTestData(expectedData,
                                receiveBuffer,
                                ExpectedSize));
}
*/

/*
 * 受信しているパケットの種類を取得します。
 */
bool ReceivePacketType(char* pOutType, nn::uart::PortSession* pSession) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutType);
    NN_ASSERT_NOT_NULL(pSession);

#ifdef NNT_UART_TEST_INCLUDE_RECEIVE_INTERVAL
    auto result = nnt::uart::ReceiveData(*pSession,
                                        pOutType,
                                        1,
                                        UartBufferSize,
                                        nn::TimeSpan::FromMilliSeconds(1));
    return result;
#else
    size_t doneBytes;
    auto result = nn::uart::Receive(&doneBytes,
                                    pOutType,
                                    1,
                                    pSession);
    return result.IsSuccess() && doneBytes == 1;
#endif
}

/*
 * ACL パケットのレスポンスを確認します。
 */
void CheckAclDataResponse(size_t* pOutReceivedBytes,
                          int* pOutReceivedPacketCount,
                          int* pOutCompletedPacketCount,
                          TestDataInfo* pTestData) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutReceivedBytes);
    NN_ASSERT_NOT_NULL(pOutReceivedPacketCount);
    NN_ASSERT_NOT_NULL(pOutCompletedPacketCount);
    NN_ASSERT_NOT_NULL(pTestData);

    *pOutReceivedBytes        = 0;
    *pOutReceivedPacketCount  = 0;
    *pOutCompletedPacketCount = 0;

    static char s_ReceiveBuffer[UartBufferSize];

    // ACL Data パケットの送信が完了すると Number Of Completed Packets Event
    // が発生する。
    // ACL Data パケットの応答は、送信したものと同じデータが返ってくる。

    char type;
    ASSERT_TRUE(ReceivePacketType(&type, pTestData->pSession));
    (*pOutReceivedBytes)++;

    switch (type)
    {
    case HciPacketType_AclData:
        {
            // ACL データの応答内容を確認し、セマフォを返却
            size_t receiveBytes = pTestData->packetSize - 1;
            ASSERT_TRUE(nnt::uart::ReceiveData(
                *pTestData->pSession,
                s_ReceiveBuffer,
                receiveBytes,
                sizeof(s_ReceiveBuffer),
                pTestData->pollingInterval));
            ASSERT_TRUE(nnt::uart::CompareTestData(pTestData->pTestData + 1,
                                                   s_ReceiveBuffer,
                                                   receiveBytes));
            *pOutReceivedBytes += receiveBytes;
            (*pOutReceivedPacketCount)++;
#ifndef NNT_UART_TEST_INCLUDE_RECEIVE_INTERVAL
            nn::os::ReleaseSemaphore(&pTestData->bufferSemaphore);
#endif
        }
        break;

    case HciPacketType_Event:
        {
            // ペイロード長を確認
            size_t receiveBytes = 2;  // イベントコード + ペイロード長
            ASSERT_TRUE(nnt::uart::ReceiveData(
                *pTestData->pSession,
                s_ReceiveBuffer,
                receiveBytes,
                sizeof(s_ReceiveBuffer),
                pTestData->pollingInterval));
            *pOutReceivedBytes += receiveBytes;

            char   eventCode   = s_ReceiveBuffer[0];
            size_t payloadSize = s_ReceiveBuffer[1];

            // パケットの残りを受信
            ASSERT_TRUE(nnt::uart::ReceiveData(
                *pTestData->pSession,
                s_ReceiveBuffer,
                payloadSize,
                sizeof(s_ReceiveBuffer),
                pTestData->pollingInterval));
            *pOutReceivedBytes += payloadSize;

            // eventCode 0x13 は Number Of Completed Packets Event
            const char CompleteEventCode = 0x13;
            if (eventCode == CompleteEventCode)
            {
                int handles = s_ReceiveBuffer[0];  // ハンドル数

                // 完了済みパケット数を取得 (Little endian)
                // パラメータの (ハンドル数) * 2 + 1 バイト目以降に格納されている
                *pOutCompletedPacketCount =
                    (s_ReceiveBuffer[handles * 2 + 2] << 8) |
                     s_ReceiveBuffer[handles * 2 + 1];
            }
        }
        break;

    default:
        {
            NN_LOG("%s: Unexpected packet type (%d)\n",
                   NN_CURRENT_FUNCTION_NAME,
                   type);
            GTEST_FAIL();
        }
        break;
    }

    // 受信バッファが空になっていたら受信待ちする
    if (nn::uart::GetReadableLength(pTestData->pSession) == 0)
    {
// Interval を入れると本テストはタイミングによってイベントを取り逃すことがあるので、 wait はしない。
#ifndef NNT_UART_TEST_INCLUDE_RECEIVE_INTERVAL
        nn::os::ClearSystemEvent(&pTestData->receiveReadyEvent);
#endif
    }
}

/*
 * 受信スレッド用関数です。
 */
void ReceiveThreadFunc(void* pArg) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pArg);

    auto* pTestData = reinterpret_cast<TestDataInfo*>(pArg);

    nn::os::ClearSystemEvent(&pTestData->receiveReadyEvent);

    uint64_t totalReceivedBytes = 0;
    int totalReceivedPackets  = 0;
    int totalCompletedPackets = 0;
    auto startTick = nn::os::GetSystemTick();

    // 規定数のレスポンスと完了通知を受け取ったら終了
    while (totalReceivedPackets  < pTestData->testCount ||
           totalCompletedPackets < pTestData->testCount)
    {
        nn::os::WaitSystemEvent(&pTestData->receiveReadyEvent);

        // データの受信確認
        size_t receivedBytes;
        int receivedPacketCount;
        int completedPacketCount;

#ifdef NNT_UART_TEST_INCLUDE_RECEIVE_INTERVAL
        // 受信にあえて間隔をあける
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
#endif

        CheckAclDataResponse(&receivedBytes, &receivedPacketCount, &completedPacketCount, pTestData);
        if (::testing::Test::HasFatalFailure())
        {
            return;
        }

        totalReceivedBytes    += receivedBytes;
        totalReceivedPackets  += receivedPacketCount;
        totalCompletedPackets += completedPacketCount;
        /*
        NN_LOG("Total: %d bytes  ACL loopback: %d packets  Completed: %d packets\n",
               totalReceivedBytes,
               totalReceivedPackets,
               totalCompletedPackets);
        */
    }
    nn::os::ClearSystemEvent(&pTestData->receiveReadyEvent);

    auto diffTick = nn::os::GetSystemTick() - startTick;

    NN_LOG("Done!\n");

    // 経過時間とスループットを計測
    auto passedMs = diffTick.ToTimeSpan().GetMilliSeconds();
    if (passedMs > 0)
    {
        uint64_t bps = totalReceivedBytes * 8000 / passedMs;
        NN_LOG("[Result]\n");
        NN_LOG("  Total received: %u bytes\n", totalReceivedBytes);
        NN_LOG("  Test time     : %ld msec\n", passedMs);
        NN_LOG("  Throughput    : %d bps\n", bps);
    }
}

/*
 * テスト用データを初期化します。
 */
void InitializeTestData(TestDataInfo* pTestData,
                        nn::uart::PortSession* pSession,
                        const HciBufferInfo bufferInfo,
                        nn::uart::BaudRate baudRate) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pTestData);

    const size_t AclPayloadSizeMax   = 2048;
    const size_t AclPacketHeaderSize =    5;

    pTestData->payloadSize = std::min(bufferInfo.maxAclPacketSize, AclPayloadSizeMax);
    pTestData->packetSize  = pTestData->payloadSize + AclPacketHeaderSize;
    pTestData->baudRate    = baudRate;
    pTestData->pSession    = pSession;

    // 通信に規定時間かかる程度のパケット数を算出
    pTestData->testCount = static_cast<int>(baudRate) * ExpectedTestSeconds / pTestData->payloadSize / 10;
    NN_LOG("Test packet count: %d\n", pTestData->testCount);

    // 受信ポーリング間隔を、1 パケット受信するのにかかる時間の半分程度にする
    pTestData->pollingInterval = nn::TimeSpan::FromMilliSeconds(
        pTestData->packetSize * 4000 / static_cast<int>(pTestData->baudRate));

    // テスト用 ACL データパケットを作成 (ハンドルとサイズ情報は Little Endian で格納)
    const nn::Bit8 PacketFlag = 0x20;  // 先頭 or 単一パケットは 0x20、他は 0x10
    static char s_TestAclData[AclPacketHeaderSize + AclPayloadSizeMax] =
        {
            HciPacketType_AclData,
            static_cast<char>(g_ConnectionHandles[0] & 0xFF),                        // ハンドル下位バイト
            static_cast<char>(PacketFlag | ((g_ConnectionHandles[0] >> 8) & 0x0F)),  // ハンドル上位バイト + フラグ
            static_cast<char>(pTestData->payloadSize & 0xFF),                        // サイズ下位バイト
            static_cast<char>((pTestData->payloadSize >> 8) & 0xFF)                  // サイズ上位バイト
        };
    nnt::uart::GenerateTestData(s_TestAclData + AclPacketHeaderSize,
                                pTestData->payloadSize);
    pTestData->pTestData = s_TestAclData;

    // バッファ数に基づくフロー制御用のセマフォ
    // ※ ACL バッファ数を超えるデータを送ると HCI イベントが発生するので、
    //    テスト簡略化のために同時送信するパケット数に上限を設ける。
    const int SemaphoreCount =
        std::min<int>(bufferInfo.aclBufferCount,
                      UartBufferSize / pTestData->payloadSize - 1);
    NN_LOG("Concurrent packet count: %d\n", SemaphoreCount);

    NN_ABORT_UNLESS_GREATER_EQUAL(SemaphoreCount, 1);
    nn::os::InitializeSemaphore(&pTestData->bufferSemaphore,
                                SemaphoreCount,
                                SemaphoreCount);

    NN_ABORT_UNLESS(nn::uart::BindPortEvent(
        &pTestData->receiveReadyEvent,
        pTestData->pSession,
        nn::uart::PortEventType_ReceiveBufferReady,
        1));
}

/*
 * テスト用データに対する終了処理を行います。
 */
void FinalizeTestData(TestDataInfo* pTestData) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pTestData);

    nn::uart::UnbindPortEvent(&pTestData->receiveReadyEvent, pTestData->pSession);
    nn::os::DestroySystemEvent(&pTestData->receiveReadyEvent);

    nn::os::FinalizeSemaphore(&pTestData->bufferSemaphore);
}

/*
 * HCI ループバックモードでの送受信テストを行います。
 */
void TestHciLoopback(nn::uart::PortSession* pSession,
                     const HciBufferInfo& hciBufferInfo,
                     nn::uart::BaudRate baudRate) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pSession);

    NN_LOG("## Starting HCI Loopback test\n");

    TestDataInfo testInfo;
    InitializeTestData(&testInfo, pSession, hciBufferInfo, baudRate);

    // 送信バッファの空きを通知するイベント
    nn::os::SystemEventType sendReadyEvent;
    ASSERT_TRUE(nn::uart::BindPortEvent(
        &sendReadyEvent,
        pSession,
        nn::uart::PortEventType_SendBufferReady,
        testInfo.packetSize));

    // 受信用のスレッドを作成
    nn::os::ThreadType receiveThread;
    nn::os::CreateThread(&receiveThread,
                         ReceiveThreadFunc,
                         &testInfo,
                         g_ThreadStack,
                         sizeof(g_ThreadStack),
                         nn::os::DefaultThreadPriority);
    nn::os::StartThread(&receiveThread);

    NN_LOG("Test in progress...\n");

    // 規定回数のデータを送信
    for (int i = 0; i < testInfo.testCount; ++i)
    {
        nn::os::WaitSystemEvent(&sendReadyEvent);
        nn::os::ClearSystemEvent(&sendReadyEvent);
#ifndef NNT_UART_TEST_INCLUDE_RECEIVE_INTERVAL
        nn::os::AcquireSemaphore(&testInfo.bufferSemaphore);
#endif

        size_t doneBytes;
        nn::uart::Send(&doneBytes,
                       pSession,
                       testInfo.pTestData,
                       testInfo.packetSize);
        ASSERT_EQ(doneBytes, testInfo.packetSize);
    }

    // スレッドの停止を待つ
    nn::os::WaitThread(&receiveThread);
    nn::os::DestroyThread(&receiveThread);

    // ゴミが残っていないことを確認
    EXPECT_EQ(nn::uart::GetReadableLength(pSession), 0);
    if (nn::uart::GetReadableLength(pSession) > 0)
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        static char s_ReceiveBuffer[4096];
        size_t doneBytes;
        auto result = nn::uart::Receive(&doneBytes,
                          s_ReceiveBuffer,
                          sizeof(s_ReceiveBuffer),
                          pSession);
        if (result.IsSuccess())
        {
            NN_LOG("[Remain data: %d bytes]\n", doneBytes);
            nnt::uart::DumpBinary(s_ReceiveBuffer, doneBytes);
        }
    }

    // 後始末
    nn::uart::UnbindPortEvent(&sendReadyEvent, pSession);
    nn::os::DestroySystemEvent(&sendReadyEvent);
    FinalizeTestData(&testInfo);
}

}

void TEST_LoopbackBody(nn::uart::PortSession* pSession,
                       const HciBufferInfo& hciBufferInfo,
                       nn::uart::BaudRate baudRate) NN_NOEXCEPT
{
    // Local loopback mode に設定
    SetLocalLoopbackModeEnabled(pSession, true);
    if (::testing::Test::HasFatalFailure())
    {
        return;
    }

    TestHciLoopback(pSession, hciBufferInfo, baudRate);
    if (::testing::Test::HasFatalFailure())
    {
        return;
    }

    // Local loopback mode を解除
    SetLocalLoopbackModeEnabled(pSession, false);
}

}}}}
