﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Includes.h"

//////////////////////////////////////////////////////////////////////////////
void Mix::Initialize(Input *pInput, uint32_t inputEntries)
{
    NN_SDK_ASSERT(pInput);
    NN_SDK_ASSERT(inputEntries);

    m_pInput        = pInput;
    m_InputEntries  = inputEntries;
    m_FrameCounter  = 0;
}


//////////////////////////////////////////////////////////////////////////////
void Mix::Finalize()
{
}


//////////////////////////////////////////////////////////////////////////////
uint8_t * Mix::GetOutputFrame()
{
    uint8_t *pBuffer = m_Buffer[m_FrameCounter++ % BuffersCount];

    // Zero s32 accumulater
    memset(pBuffer, 0, SamplesPerFrameCount * OutputChannelsCount * sizeof(int32_t));

    Input *pInput = m_pInput;

    // Mix each input's s16 to s32 accumulater
    for (uint32_t i = 0; i < m_InputEntries; i++)
    {
        int16_t *pInputData = pInput->GetInputFrame();

        if (pInputData)
        {
            int32_t *pAccumulater = reinterpret_cast<int32_t*>(pBuffer);

            // Sum the input into the output buffer
            // - input buffer assumed to be mono, same sample rate, same frame size
            // - output buffer assumed to be stereo
            // - source always mixed equally on both sides (center)
            // - this loop can also be optimized by unrolling... don't care for this test
            for (uint32_t j = 0; j < SamplesPerFrameCount; j++)
            {
                *pAccumulater++ += *pInputData;     // L = source
                *pAccumulater++ += *pInputData++;   // R = source
            }
        }

        pInput++;
    }

    // Reduce the accumulater from s32 to s16 for final output buffer in place
    int16_t *pDst = reinterpret_cast<int16_t*>(pBuffer);
    int32_t *pSrc = reinterpret_cast<int32_t*>(pBuffer);

    for (uint32_t j = 0; j < SamplesPerFrameCount; j++)
    {
        // The source samples are divided by 2 because we distributed the samples
        // into both channles of the accumulator at full volume.
        *pDst++ = *pSrc++ / 2;  // L
        *pDst++ = *pSrc++ / 2;  // R
    }

    return pBuffer;
}

