﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/cduac/cduac_Spec.h>
#include <nn/cduac/cduac_Api.h>

#include "./UacCommon.h"

//////////////////////////////////////////////////////////////////////////////
static void PrintTerminalType(uint16_t type)
{
    switch (type)
    {
    case nn::cduac::UsbTerminalType_Undefined:                      NN_SDK_LOG("UsbTerminalType_Undefined\n");                      break;
    case nn::cduac::UsbTerminalType_Streaming:                      NN_SDK_LOG("UsbTerminalType_Streaming\n");                      break;
    case nn::cduac::UsbTerminalType_Vendor:                         NN_SDK_LOG("UsbTerminalType_Vendor\n");                         break;

    case nn::cduac::InputTerminalType_Undefined:                    NN_SDK_LOG("InputTerminalType_Undefined\n");                    break;
    case nn::cduac::InputTerminalType_Microphone:                   NN_SDK_LOG("InputTerminalType_Microphone\n");                   break;
    case nn::cduac::InputTerminalType_DesktopMicrophone:            NN_SDK_LOG("InputTerminalType_DesktopMicrophone\n");            break;
    case nn::cduac::InputTerminalType_PersonalMicrophone:           NN_SDK_LOG("InputTerminalType_PersonalMicrophone\n");           break;
    case nn::cduac::InputTerminalType_OmnidirectionalMicrophone:    NN_SDK_LOG("InputTerminalType_OmnidirectionalMicrophone\n");    break;
    case nn::cduac::InputTerminalType_MicrophoneArray:              NN_SDK_LOG("InputTerminalType_MicrophoneArray\n");              break;
    case nn::cduac::InputTerminalType_ProcessingMicArray:           NN_SDK_LOG("InputTerminalType_ProcessingMicArray\n");           break;

    case nn::cduac::OutputTermainalType_Undefined:                  NN_SDK_LOG("OutputTermainalType_Undefined\n");                  break;
    case nn::cduac::OutputTermainalType_Speaker:                    NN_SDK_LOG("OutputTermainalType_Speaker\n");                    break;
    case nn::cduac::OutputTermainalType_Headphones:                 NN_SDK_LOG("OutputTermainalType_Headphones\n");                 break;
    case nn::cduac::OutputTermainalType_HeadMountedDisplay:         NN_SDK_LOG("OutputTermainalType_HeadMountedDisplay\n");         break;
    case nn::cduac::OutputTermainalType_DesktopSpeaker:             NN_SDK_LOG("OutputTermainalType_DesktopSpeaker\n");             break;
    case nn::cduac::OutputTermainalType_RoomSpeaker:                NN_SDK_LOG("OutputTermainalType_RoomSpeaker\n");                break;
    case nn::cduac::OutputTermainalType_Communication:              NN_SDK_LOG("OutputTermainalType_Communication\n");              break;
    case nn::cduac::OutputTermainalType_LowFrequecyEffects:         NN_SDK_LOG("OutputTermainalType_LowFrequecyEffects\n");         break;

    case nn::cduac::BidirectionalTerminalType_Undefined:            NN_SDK_LOG("BidirectionalTerminalType_Undefined\n");            break;
    case nn::cduac::BidirectionalTerminalType_Handset:              NN_SDK_LOG("BidirectionalTerminalType_Handset\n");              break;
    case nn::cduac::BidirectionalTerminalType_Headset:              NN_SDK_LOG("BidirectionalTerminalType_Headset\n");              break;
    case nn::cduac::BidirectionalTerminalType_Speakerphone:         NN_SDK_LOG("BidirectionalTerminalType_Speakerphone\n");         break;
    case nn::cduac::BidirectionalTerminalType_EchoSuppressing:      NN_SDK_LOG("BidirectionalTerminalType_EchoSuppressing\n");      break;
    case nn::cduac::BidirectionalTerminalType_EchoCancelling:       NN_SDK_LOG("BidirectionalTerminalType_EchoCancelling\n");       break;

    case nn::cduac::TelephonyTerminalType_Undefined:                NN_SDK_LOG("TelephonyTerminalType_Undefined\n");                break;
    case nn::cduac::TelephonyTerminalType_PhoneLine:                NN_SDK_LOG("TelephonyTerminalType_PhoneLine\n");                break;
    case nn::cduac::TelephonyTerminalType_Telephone:                NN_SDK_LOG("TelephonyTerminalType_Telephone\n");                break;
    case nn::cduac::TelephonyTerminalType_DownLinePhone:            NN_SDK_LOG("TelephonyTerminalType_DownLinePhone\n");            break;

    case nn::cduac::ExternalTerminalType_Undefined:                 NN_SDK_LOG("ExternalTerminalType_Undefined\n");                 break;
    case nn::cduac::ExternalTerminalType_AnalogConnector:           NN_SDK_LOG("ExternalTerminalType_AnalogConnector\n");           break;
    case nn::cduac::ExternalTerminalType_DigitalAudioInterface:     NN_SDK_LOG("ExternalTerminalType_DigitalAudioInterface\n");     break;
    case nn::cduac::ExternalTerminalType_LineConnector:             NN_SDK_LOG("ExternalTerminalType_LineConnector\n");             break;
    case nn::cduac::ExternalTerminalType_LegacyAudioConnector:      NN_SDK_LOG("ExternalTerminalType_LegacyAudioConnector\n");      break;
    case nn::cduac::ExternalTerminalType_SPDIFInterface:            NN_SDK_LOG("ExternalTerminalType_SPDIFInterface\n");            break;
    case nn::cduac::ExternalTerminalType_1395DAStream:              NN_SDK_LOG("ExternalTerminalType_1395DAStream\n");              break;
    case nn::cduac::ExternalTerminalType_1394StreamSoundtrack:      NN_SDK_LOG("ExternalTerminalType_1394StreamSoundtrack\n");      break;

    case nn::cduac::EmbeddedTerminalType_Undefined:                 NN_SDK_LOG("EmbeddedTerminalType_Undefined\n");                 break;
    case nn::cduac::EmbeddedTerminalType_LevelCalibrationSource:     NN_SDK_LOG("EmbeddedTerminalType_LevelCalibrationSource\n");    break;
    case nn::cduac::EmbeddedTerminalType_EqualizationNoise:         NN_SDK_LOG("EmbeddedTerminalType_EqualizationNoise\n");         break;
    case nn::cduac::EmbeddedTerminalType_CDPlayer:                  NN_SDK_LOG("EmbeddedTerminalType_CDPlayer\n");                  break;
    case nn::cduac::EmbeddedTerminalType_DAT:                       NN_SDK_LOG("EmbeddedTerminalType_DAT\n");                       break;
    case nn::cduac::EmbeddedTerminalType_DCC:                       NN_SDK_LOG("EmbeddedTerminalType_DCC\n");                       break;
    case nn::cduac::EmbeddedTerminalType_MiniDisk:                  NN_SDK_LOG("EmbeddedTerminalType_MiniDisk\n");                  break;
    case nn::cduac::EmbeddedTerminalType_AnalogTape:                NN_SDK_LOG("EmbeddedTerminalType_AnalogTape\n");                break;
    case nn::cduac::EmbeddedTerminalType_Phonograph:                NN_SDK_LOG("EmbeddedTerminalType_Phonograph\n");                break;
    case nn::cduac::EmbeddedTerminalType_VCRAudio:                  NN_SDK_LOG("EmbeddedTerminalType_VCRAudio\n");                  break;
    case nn::cduac::EmbeddedTerminalType_VideoDiscAudio:            NN_SDK_LOG("EmbeddedTerminalType_VideoDiscAudio\n");            break;
    case nn::cduac::EmbeddedTerminalType_DVDAudio:                  NN_SDK_LOG("EmbeddedTerminalType_DVDAudio\n");                  break;
    case nn::cduac::EmbeddedTerminalType_TVTunerAudio:              NN_SDK_LOG("EmbeddedTerminalType_TVTunerAudio\n");              break;
    case nn::cduac::EmbeddedTerminalType_SatelliteReceiverAudio:    NN_SDK_LOG("EmbeddedTerminalType_SatelliteReceiverAudio\n");    break;
    case nn::cduac::EmbeddedTerminalType_CableTunerAudio:           NN_SDK_LOG("EmbeddedTerminalType_CableTunerAudio\n");           break;
    case nn::cduac::EmbeddedTerminalType_DSSAudio:                  NN_SDK_LOG("EmbeddedTerminalType_DSSAudio\n");                  break;
    case nn::cduac::EmbeddedTerminalType_RadioReceiver:             NN_SDK_LOG("EmbeddedTerminalType_RadioReceiver\n");             break;
    case nn::cduac::EmbeddedTerminalType_RadioTransmitter:          NN_SDK_LOG("EmbeddedTerminalType_RadioTransmitter\n");          break;
    case nn::cduac::EmbeddedTerminalType_MultitrackRecorder:        NN_SDK_LOG("EmbeddedTerminalType_MultitrackRecorder\n");        break;
    case nn::cduac::EmbeddedTerminalType_Sunthesizer:               NN_SDK_LOG("EmbeddedTerminalType_Sunthesizer\n");               break;

    default:                                                        NN_SDK_LOG("Unknown\n");                                        break;
    }
}


//////////////////////////////////////////////////////////////////////////////
static void PrintChannels(uint16_t wChannelConfig)
{
    if (wChannelConfig & nn::cduac::ChannelCluster_LeftFront)               {NN_SDK_LOG("L ");};
    if (wChannelConfig & nn::cduac::ChannelCluster_RightFront)              {NN_SDK_LOG("R ");};
    if (wChannelConfig & nn::cduac::ChannelCluster_CenterFront)             {NN_SDK_LOG("C ");};
    if (wChannelConfig & nn::cduac::ChannelCluster_LowFrequencyEnhancement) {NN_SDK_LOG("LFE ");};
    if (wChannelConfig & nn::cduac::ChannelCluster_LeftSurround)            {NN_SDK_LOG("Ls ");};
    if (wChannelConfig & nn::cduac::ChannelCluster_RightSurround)           {NN_SDK_LOG("Rs ");};
    if (wChannelConfig & nn::cduac::ChannelCluster_LeftCenter)              {NN_SDK_LOG("Lc ");};
    if (wChannelConfig & nn::cduac::ChannelCluster_RightCenter)             {NN_SDK_LOG("Rc ");};
    if (wChannelConfig & nn::cduac::ChannelCluster_Surround)                {NN_SDK_LOG("S ");};
    if (wChannelConfig & nn::cduac::ChannelCluster_SideLeft)                {NN_SDK_LOG("sL ");};
    if (wChannelConfig & nn::cduac::ChannelCluster_SideRight)               {NN_SDK_LOG("sR ");};
    if (wChannelConfig & nn::cduac::ChannelCluster_Top)                     {NN_SDK_LOG("T ");};
}


//////////////////////////////////////////////////////////////////////////////
static void PrintAudioControls(uint16_t controls)
{
    if (controls & nn::cduac::AudioControl_Mute)                            {NN_SDK_LOG("Mute ");};
    if (controls & nn::cduac::AudioControl_Volume)                          {NN_SDK_LOG("Volume ");};
    if (controls & nn::cduac::AudioControl_Bass)                            {NN_SDK_LOG("Bass ");};
    if (controls & nn::cduac::AudioControl_Mid)                             {NN_SDK_LOG("Mid ");};
    if (controls & nn::cduac::AudioControl_Treble)                          {NN_SDK_LOG("Treble ");};
    if (controls & nn::cduac::AudioControl_GraphicEqualizer)                {NN_SDK_LOG("GraphicEqualizer ");};
    if (controls & nn::cduac::AudioControl_AutomaticGain)                   {NN_SDK_LOG("AutomaticGain ");};
    if (controls & nn::cduac::AudioControl_Delay)                           {NN_SDK_LOG("Delay ");};
    if (controls & nn::cduac::AudioControl_BassBoost)                       {NN_SDK_LOG("BassBoost ");};
    if (controls & nn::cduac::AudioControl_Loudness)                        {NN_SDK_LOG("Loudness ");};
}


//////////////////////////////////////////////////////////////////////////////
static void PrintAudioDataFormat(uint16_t format)
{
    switch (format)
    {
    case nn::cduac::AudioDataFormat_TypeIUndefined:         NN_SDK_LOG("Format I Undefined\n");         break;
    case nn::cduac::AudioDataFormat_Pcm:                    NN_SDK_LOG("PCM\n");                        break;
    case nn::cduac::AudioDataFormat_Pcm8:                   NN_SDK_LOG("PCM8\n");                       break;
    case nn::cduac::AudioDataFormat_IeeeFloat:              NN_SDK_LOG("IEEE_FLOAT\n");                 break;
    case nn::cduac::AudioDataFormat_Alaw:                   NN_SDK_LOG("ALAW\n");                       break;
    case nn::cduac::AudioDataFormat_Mulaw:                  NN_SDK_LOG("MULAW\n");                      break;

    case nn::cduac::AudioDataFormat_TypeIIUndefined:        NN_SDK_LOG("Format II Undefined\n");        break;
    case nn::cduac::AudioDataFormat_TypeIIMpeg:             NN_SDK_LOG("MPEG\n");                       break;
    case nn::cduac::AudioDataFormat_TypeIIAc3:              NN_SDK_LOG("AC-3\n");                       break;

    case nn::cduac::AudioDataFormat_TypeIIIUndefined:       NN_SDK_LOG("Format III Undefined\n");       break;
    case nn::cduac::AudioDataFormat_Iec1937Ac3:             NN_SDK_LOG("IEC1937_AC-3\n");               break;
    case nn::cduac::AudioDataFormat_Iec1937Mpeg1Layer1:     NN_SDK_LOG("IEC1937_MPEG-1_Layer1\n");      break;
    case nn::cduac::AudioDataFormat_Iec1937Mpeg1Layer23:    NN_SDK_LOG("IEC1937_MPEG-1_Layer2/3\n");    break;
    case nn::cduac::AudioDataFormat_Iec1937Mpeg2Ext:        NN_SDK_LOG("IEC1937_MPEG-2_EXT\n");         break;
    case nn::cduac::AudioDataFormat_Iec1937Mpeg2Layer1Ls:   NN_SDK_LOG("IEC1938_MPEG-2_Layer1_LS\n");   break;
    case nn::cduac::AudioDataFormat_Iec1937Mpeg2Layer23Ls:  NN_SDK_LOG("IEC1937_MPEG-2_Layer2/3_LS\n"); break;

    default:                                                NN_SDK_LOG("Unknown\n");                    break;
    }
}

//////////////////////////////////////////////////////////////////////////////
static void PrintInputTerminalParams(nn::cduac::Parser *pParser, nn::cduac::InterfaceProfile *pInterfaceProfile)
{
    nn::cduac::AudioControlInputTerminal *pInputTerminal = pParser->GetInputTerminal(pInterfaceProfile);

    if (pInputTerminal)
    {
        NN_SDK_LOG("AudioControlInputTerminal:      %08x\n",    pInputTerminal);
        NN_SDK_LOG("    bTerminalId:                %02x\n",    pInputTerminal->bTerminalId);
        NN_SDK_LOG("    wTermainalType:             %04x : ",   pInputTerminal->wTermainalType);

        PrintTerminalType(pInputTerminal->wTermainalType);

        NN_SDK_LOG("    bAssocTerminal:             %02x\n",    pInputTerminal->bAssocTerminal);
        NN_SDK_LOG("    bNrChannels:                %02x\n",    pInputTerminal->bNrChannels);
        NN_SDK_LOG("    wChannelConfig:             %02x : ",   pInputTerminal->wChannelConfig);

        PrintChannels(pInputTerminal->wChannelConfig);
        NN_SDK_LOG("\n");

        NN_SDK_LOG("    iChannelNames:              %02x\n",    pInputTerminal->iChannelNames);
        NN_SDK_LOG("    iTermainal:                 %02x\n\n",  pInputTerminal->iTermainal);
    }
}


//////////////////////////////////////////////////////////////////////////////
static void PrintFeatureUnitParams(nn::cduac::Parser *pParser, nn::cduac::InterfaceProfile *pInterfaceProfile)
{
    nn::cduac::AudioControlInputTerminal  *pInputTerminal = pParser->GetInputTerminal(pInterfaceProfile);
    nn::cduac::AudioControlFeatureUnit    *pFeatureUnit   = pParser->GetFeatureUnit(pInterfaceProfile);

    if (pFeatureUnit && pInputTerminal)
    {
        NN_SDK_LOG("AudioControlFeatureUnit:        %08x\n",    pFeatureUnit);
        NN_SDK_LOG("    bUnitId:                    %02x\n",    pFeatureUnit->bUnitId);
        NN_SDK_LOG("    bSourceId:                  %02x\n",    pFeatureUnit->bSourceId);
        NN_SDK_LOG("    bControlSize:               %02x\n",    pFeatureUnit->bControlSize);

        uint8_t *pBmaControls = pFeatureUnit->bmaControls;

        for (uint8_t i = 0; i <= pInputTerminal->bNrChannels; i++)
        {
            uint16_t controls = 0;

            NN_SDK_LOG("    bmaControls[%d]:            ", i);

            for (uint8_t j = 0; j < pFeatureUnit->bControlSize; j++)
            {
                controls <<= 8;
                controls |= *pBmaControls;

                NN_SDK_LOG(" %02x", *pBmaControls);

                pBmaControls++;
            }

            NN_SDK_LOG(" : ");

            PrintAudioControls(controls);

            NN_SDK_LOG("\n");
        }

        NN_SDK_LOG("    iFeature:                   %02x\n",    *pBmaControls);
        NN_SDK_LOG("\n");
    }
}


//////////////////////////////////////////////////////////////////////////////
static void PrintOutputTerminalParams(nn::cduac::Parser *pParser, nn::cduac::InterfaceProfile *pInterfaceProfile)
{
    nn::cduac::AudioControlOutputTerminal *pOutputTerminal = pParser->GetOutputTerminal(pInterfaceProfile);

    if (pOutputTerminal)
    {
        NN_SDK_LOG("AudioControlOutputTerminal:     %08x:\n",   pOutputTerminal);
        NN_SDK_LOG("    bTerminalId:                %02x\n",    pOutputTerminal->bTerminalId);
        NN_SDK_LOG("    wTermainalType:             %04x : ",   pOutputTerminal->wTermainalType);

        PrintTerminalType(pOutputTerminal->wTermainalType);

        NN_SDK_LOG("    bAssocTerminal:             %02x\n",    pOutputTerminal->bAssocTerminal);
        NN_SDK_LOG("    bSourceId:                  %02x\n",    pOutputTerminal->bSourceId);
        NN_SDK_LOG("    iTermainal:                 %02x\n",    pOutputTerminal->iTermainal);
        NN_SDK_LOG("\n");
    }
}


//////////////////////////////////////////////////////////////////////////////
static void PrintAudioStreamingGeneral(nn::cduac::AudioStreamingGeneral *p)
{
    if (p)
    {
        NN_SDK_LOG("AudioStreamingGeneral:          %08x:\n",   p);
        NN_SDK_LOG("    bTerminalLink:              %02x\n",    p->bTerminalLink);
        NN_SDK_LOG("    bDelay:                     %02x\n",    p->bDelay);
        NN_SDK_LOG("    wFormatTag:                 %04x : ",   p->wFormatTag);

        PrintAudioDataFormat(p->wFormatTag);

        NN_SDK_LOG("\n");
    }
}


//////////////////////////////////////////////////////////////////////////////
static void PrintAudioStreamingFormatType(nn::cduac::AudioStreamingFormatType *p)
{
    if (p)
    {
        NN_SDK_LOG("AudioStreamingFormatType:       %08x:\n",   p);
        NN_SDK_LOG("    bFormatType:                %02x\n",    p->bFormatType);
        NN_SDK_LOG("    bNrChannles:                %02x\n",    p->bNrChannles);
        NN_SDK_LOG("    bSubFrameSize:              %02x\n",    p->bSubFrameSize);
        NN_SDK_LOG("    bBitResolution:             %02x\n",    p->bBitResolution);
        NN_SDK_LOG("    bSamFreqType:               %02x\n",    p->bSamFreqType);

        for (uint8_t i = 0; i < p->bSamFreqType; i++)
        {
            NN_SDK_LOG("    tSamFreq[%d]:                %d Hz\n",    i, p->tSamFreq[i].n1 + (p->tSamFreq[i].n256 * 256) + (p->tSamFreq[i].n65536 * 65536));
        }

        NN_SDK_LOG("\n");
    }
}


//////////////////////////////////////////////////////////////////////////////
static void PrintEndpointDescriptor(nn::usb::UsbEndpointDescriptor *p)
{
    if (p)
    {
        NN_SDK_LOG("UsbEndpointDescriptor:          %08x:\n",   p);
        NN_SDK_LOG("    bEndpointAddress:           %02x\n",    p->bEndpointAddress);
        NN_SDK_LOG("    bmAttributes:               %02x\n",    p->bmAttributes);
        NN_SDK_LOG("    wMaxPacketSize:             %02x\n",    p->wMaxPacketSize);
        NN_SDK_LOG("    bInterval:                  %02x\n",    p->bInterval);
        NN_SDK_LOG("\n");
    }
}


//////////////////////////////////////////////////////////////////////////////
static void PrintAudioStreamingEndpointGeneral(nn::cduac::AudioStreamingEndpointGeneral *p)
{
    if (p)
    {
        NN_SDK_LOG("AudioStreamingEndpointGeneral:  %08x:\n",   p);
        NN_SDK_LOG("    bmAttributes:               %02x\n",    p->bmAttributes);
        NN_SDK_LOG("    bLockDelayUnits:            %02x\n",    p->bLockDelayUnits);
        NN_SDK_LOG("    wLockDelay:                 %04x\n",    p->wLockDelay);
        NN_SDK_LOG("\n");
    }
}


//////////////////////////////////////////////////////////////////////////////
static void PrintInterface(nn::cduac::Parser *pParser, nn::cduac::InterfaceProfile *pInterfaceProfile)
{
    for (uint8_t bAlternateSetting = 1; bAlternateSetting < 0xff; bAlternateSetting++)
    {
        nn::usb::UsbInterfaceDescriptor *pInterfaceDescriptor = pParser->GetInterface(pInterfaceProfile, bAlternateSetting);

        if (pInterfaceDescriptor)
        {
            NN_SDK_LOG("UsbInterfaceDescriptor:         %08x:\n",   pInterfaceDescriptor);
            NN_SDK_LOG("    bInterfaceNumber:           %02x\n",    pInterfaceDescriptor->bInterfaceNumber);
            NN_SDK_LOG("    bAlternateSetting:          %02x\n",    pInterfaceDescriptor->bAlternateSetting);
            NN_SDK_LOG("    bNumEndpoints:              %02x\n",    pInterfaceDescriptor->bNumEndpoints);
            NN_SDK_LOG("    bInterfaceClass:            %02x\n",    pInterfaceDescriptor->bInterfaceClass);
            NN_SDK_LOG("    bInterfaceSubClass:         %02x\n",    pInterfaceDescriptor->bInterfaceSubClass);
            NN_SDK_LOG("    bInterfaceProtocol:         %02x\n",    pInterfaceDescriptor->bInterfaceProtocol);
            NN_SDK_LOG("    iInterface:                 %02x\n",    pInterfaceDescriptor->iInterface);
            NN_SDK_LOG("\n");

            PrintAudioStreamingGeneral(pParser->GetAudioStreamingGeneral(pInterfaceProfile, bAlternateSetting));
            PrintAudioStreamingFormatType(pParser->GetAudioStreamingFormatType(pInterfaceProfile, bAlternateSetting));

            for (uint8_t endpointIndex = 0; endpointIndex < pInterfaceDescriptor->bNumEndpoints; endpointIndex++)
            {
                PrintEndpointDescriptor(pParser->GetEndpoint(pInterfaceProfile, bAlternateSetting, endpointIndex));
                PrintAudioStreamingEndpointGeneral(pParser->GetAudioStreamingEndpointGeneral(pInterfaceProfile, bAlternateSetting, endpointIndex));
            }
        }
        else
        {
            break;
        }
    }
}


//////////////////////////////////////////////////////////////////////////////
void UacPrint(nn::cduac::Parser *pParser, nn::cduac::Interface *pInterface)
{
    PrintInputTerminalParams(pParser, pInterface->GetInterfaceProfile());
    PrintFeatureUnitParams(pParser, pInterface->GetInterfaceProfile());
    PrintOutputTerminalParams(pParser, pInterface->GetInterfaceProfile());
    PrintInterface(pParser, pInterface->GetInterfaceProfile());
}

