﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace Nintendo.AudioToolKit.Xml
{
    using Microsoft.VisualStudio.TestTools.UnitTesting;
    using System.IO;
    using System.Linq;
    using System.Reflection;

    [TestClass]
    public class WaveSoundResourceDataXmlSerializerTest
    {
        [TestMethod]
        public void Serialize()
        {
            var xmlModel = CreateXmlModel();
            var filePath = CreateTestFilePath("WaveSoundResourceDataXmlSerializerTestData.xml");

            using (var file = File.Open(filePath, FileMode.Create))
            {
                WaveSoundResourceDataXmlSerializer.Default.Serialize(file, xmlModel);
            }

            string resultXml;

            using (var file = File.Open(filePath, FileMode.Open))
            {
                using (var reader = new StreamReader(file))
                {
                    resultXml = reader.ReadToEnd();
                }
            }

            Assert.AreEqual(WaveSoundResourceDataXmlSerializerTestResources.AtkWaveSoundResource, resultXml);
        }

        [TestMethod]
        public void Deserialize()
        {
            var xmlModel = CreateXmlModel();

            WaveSoundResourceDataXml result;

            using (var reader = new StringReader(WaveSoundResourceDataXmlSerializerTestResources.AtkWaveSoundResource))
            {
                result = WaveSoundResourceDataXmlSerializer.Default.Deserialize(reader);
            }

            Assert.AreEqual(result.TrackResources.Count, xmlModel.TrackResources.Count);
            Assert.AreEqual(result.TrackResources.Count, xmlModel.TrackResources.Count);
        }

        private static string CreateTestFilePath(string fileName)
        {
            return Path.Combine(
                Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location),
                fileName);
        }

        private static void ValidateWaveSoundResourceDataXml(
            WaveSoundResourceDataXml expected,
            WaveSoundResourceDataXml actual)
        {
            Assert.IsNotNull(expected);
            Assert.IsNotNull(actual);

            Assert.AreEqual(expected.TrackResources.Count, actual.TrackResources.Count);

            foreach (var index in Enumerable.Range(0, expected.TrackResources.Count))
            {
                ValidateWaveSoundTrackResourceDataXml(expected.TrackResources[index], actual.TrackResources[index]);
            }
        }

        private static void ValidateWaveSoundTrackResourceDataXml(
            WaveSoundTrackResourceDataXml expected,
            WaveSoundTrackResourceDataXml actual)
        {
            Assert.IsNotNull(expected);
            Assert.IsNotNull(actual);

            Assert.AreEqual(expected.IsEnabled, actual.IsEnabled);
            Assert.AreEqual(expected.ClipResources.Count, actual.ClipResources.Count);

            foreach (var index in Enumerable.Range(0, expected.ClipResources.Count))
            {
                ValidateWaveSoundClipResourceDataXml(expected.ClipResources[index], actual.ClipResources[index]);
            }
        }

        private static void ValidateWaveSoundClipResourceDataXml(
            WaveSoundClipResourceDataXml expected,
            WaveSoundClipResourceDataXml actual)
        {
            Assert.IsNotNull(expected);
            Assert.IsNotNull(actual);

            Assert.AreEqual(expected.IsEnabled, actual.IsEnabled);
            Assert.AreEqual(expected.Row, actual.Row);
            Assert.AreEqual(expected.Path, actual.Path);
            Assert.AreEqual(expected.Position, actual.Position);
            Assert.AreEqual(expected.StartOffset, actual.StartOffset);
            Assert.AreEqual(expected.Duration, actual.Duration);
            Assert.AreEqual(expected.Volume, actual.Volume);
            Assert.AreEqual(expected.Pitch, actual.Pitch);
            Assert.AreEqual(expected.Pan, actual.Pan);
        }

        private WaveSoundResourceDataXml CreateXmlModel()
        {
            var track1 = new WaveSoundTrackResourceDataXml()
            {
                IsEnabled = true,
            };

            track1.ClipResources.Add(
                new WaveSoundClipResourceDataXml()
                {
                    IsEnabled = true,
                    Row = 0,
                    Path = "track1/path",
                    Position = 2,
                    StartOffset = 0,
                    Duration = 10,
                    Volume = 1.0F,
                    Pan = 0.0F,
                    Pitch = 1.2F,
                });
            track1.ClipResources.Add(
                new WaveSoundClipResourceDataXml()
                {
                    IsEnabled = false,
                    Row = 1,
                    Path = "track2/path",
                });

            var track2 = new WaveSoundTrackResourceDataXml()
            {
                IsEnabled = false,
            };

            var xmlModel = new WaveSoundResourceDataXml()
            {
                Name = "SE_YOSHI",
            };
            xmlModel.TrackResources.Add(track1);
            xmlModel.TrackResources.Add(track2);

            return xmlModel;
        }
    }
}
