﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;

using VsSolutionLibrary;
using System.Collections.Generic;

namespace VsSolutionLibraryTest
{
    /// <summary>
    /// VsSolutionLibrary の正常系テストです。
    /// </summary>
    [TestClass]
    public class NormalSequenceTest
    {
        private const string TextDefault =
@"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 2012
Global
EndGlobal
";

        /// <summary>
        /// 空の VsSolution オブジェクトを作成
        /// </summary>
        [TestMethod]
        public void TestConstructor1()
        {
            VsSolution solution = new VsSolution();
            Assert.AreEqual(solution.ToString(), TextDefault);
        }

        /// <summary>
        /// 空の VsSolution オブジェクトをパース
        /// </summary>
        [TestMethod]
        public void TestConstructor2()
        {
            VsSolution solution = new VsSolution(TextDefault);
            Assert.AreEqual(solution.ToString(), TextDefault);
        }

        private const string TextChangeVersion2013 = @"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 2013
VisualStudioVersion = 12.0.40629.0
MinimumVisualStudioVersion = 10.0.40219.1
Global
EndGlobal
";

        /// <summary>
        /// バージョンの変更 (2013)
        /// </summary>
        [TestMethod]
        public void TestChangeVersion2013()
        {
            VsSolution solution = new VsSolution();

            Assert.AreEqual(VsSolutionVersion.VisualStudio2012, solution.Version);

            solution.Version = VsSolutionVersion.VisualStudio2013;

            Assert.AreEqual(TextChangeVersion2013, solution.ToString());
            Assert.AreEqual(VsSolutionVersion.VisualStudio2013, solution.Version);

            solution.Version = VsSolutionVersion.VisualStudio2012;

            Assert.AreEqual(TextDefault, solution.ToString());
            Assert.AreEqual(VsSolutionVersion.VisualStudio2012, solution.Version);
        }

        private const string TextChangeVersion2015 = @"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 14
VisualStudioVersion = 14.0.25123.0
MinimumVisualStudioVersion = 10.0.40219.1
Global
EndGlobal
";

        /// <summary>
        /// バージョンの変更 (2015)
        /// </summary>
        [TestMethod]
        public void TestChangeVersion2015()
        {
            VsSolution solution = new VsSolution();

            Assert.AreEqual(VsSolutionVersion.VisualStudio2012, solution.Version);

            solution.Version = VsSolutionVersion.VisualStudio2015;

            Assert.AreEqual(TextChangeVersion2015, solution.ToString());
            Assert.AreEqual(VsSolutionVersion.VisualStudio2015, solution.Version);

            solution.Version = VsSolutionVersion.VisualStudio2012;

            Assert.AreEqual(TextDefault, solution.ToString());
            Assert.AreEqual(VsSolutionVersion.VisualStudio2012, solution.Version);
        }

        private const string TextChangeVersion2017 = @"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 15
VisualStudioVersion = 15.0.26228.10
MinimumVisualStudioVersion = 10.0.40219.1
Global
EndGlobal
";

        /// <summary>
        /// バージョンの変更 (2017)
        /// </summary>
        [TestMethod]
        public void TestChangeVersion2017()
        {
            VsSolution solution = new VsSolution();

            Assert.AreEqual(VsSolutionVersion.VisualStudio2012, solution.Version);

            solution.Version = VsSolutionVersion.VisualStudio2017;

            Assert.AreEqual(TextChangeVersion2017, solution.ToString());
            Assert.AreEqual(VsSolutionVersion.VisualStudio2017, solution.Version);

            solution.Version = VsSolutionVersion.VisualStudio2012;

            Assert.AreEqual(TextDefault, solution.ToString());
            Assert.AreEqual(VsSolutionVersion.VisualStudio2012, solution.Version);
        }

        private const string TextChangeSolutionProperty = @"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 2012
Global
    GlobalSection(SolutionProperties) = preSolution
        HideSolutionNode = FALSE
    EndGlobalSection
EndGlobal
";

        /// <summary>
        /// ソリューション・プロパティの取得と変更
        /// </summary>
        [TestMethod]
        public void TestChangeSolutionProperty()
        {
            VsSolution solution = new VsSolution();

            IDictionary<string, string> solutionProperties = solution.SolutionProperties;
            solutionProperties.Add("HideSolutionNode", "FALSE");
            solution.SolutionProperties = solutionProperties;
            Assert.AreEqual(solutionProperties, solution.SolutionProperties);

            Assert.IsTrue(TestUtility.CompareSolutionString(TextChangeSolutionProperty, solution.ToString()));
        }

        private const string TextAddProject = @"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 2012
Project(""{8BC9CEB8-8B4A-11D0-8D11-00A0C91BC942}"") = ""Project0"", ""Project0\Project0.vcxproj"", ""{11111111-1111-1111-1111-111111111111}""
EndProject
Project(""{FAE04EC0-301F-11D3-BF4B-00C04F79EFBC}"") = ""Project 1"", ""全角\Project 1.csproj"", ""{22222222-2222-2222-2222-222222222222}""
    ProjectSection(ProjectDependencies) = postProject
        {11111111-1111-1111-1111-111111111111} = {11111111-1111-1111-1111-111111111111}
    EndProjectSection
EndProject
Global
EndGlobal
";

        /// <summary>
        /// プロジェクトの追加
        /// </summary>
        [TestMethod]
        public void TestAddProject()
        {
            VsSolution solution = new VsSolution();

            List<Guid> projects = new List<Guid>();
            projects.Add(new Guid("{11111111-1111-1111-1111-111111111111}"));
            projects.Add(new Guid("{22222222-2222-2222-2222-222222222222}"));

            VsSolutionProjectProperty[] properties = new VsSolutionProjectProperty[] {
                new VsSolutionProjectProperty(VsProjectType.VisualC, "Project0", "Project0\\Project0.vcxproj", null),
                new VsSolutionProjectProperty(VsProjectType.CSharp, "Project 1", "全角\\Project 1.csproj", new Guid[] { projects[0] })
            };

            solution.AddProject(projects[0], properties[0]);
            solution.AddProject(projects[1], properties[1]);

            Assert.IsTrue(Enumerable.SequenceEqual(projects, solution.Projects));
            Assert.IsTrue(TestUtility.CompareSolutionString(TextAddProject, solution.ToString()));
        }

        private const string TextRemoveProject = @"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 2012
Project(""{8BC9CEB8-8B4A-11D0-8D11-00A0C91BC942}"") = ""Project0"", ""Project0\Project0.vcxproj"", ""{11111111-1111-1111-1111-111111111111}""
EndProject
Global
EndGlobal
";

        /// <summary>
        /// プロジェクトの削除
        /// </summary>
        [TestMethod]
        public void TestRemoveProject()
        {
            VsSolution solution = new VsSolution(TextAddProject);

            solution.RemoveProject(new Guid("{22222222-2222-2222-2222-222222222222}"));
            VsSolutionProjectProperty property = new VsSolutionProjectProperty(VsProjectType.VisualC, "Project0", "Project0\\Project0.vcxproj", null);

            Guid[] projects = new Guid[] { new Guid("{11111111-1111-1111-1111-111111111111}") };

            Assert.IsTrue(Enumerable.SequenceEqual(projects, solution.Projects));
            Assert.IsTrue(TestUtility.CompareSolutionString(TextRemoveProject, solution.ToString()));
        }

        private const string TextAddSolutionConfiguration = @"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 2012
Global
    GlobalSection(SolutionConfigurationPlatforms) = preSolution
        Debug|Siglo 0 = Debug|Siglo 0
        Debug|Siglo 1 = Debug|Siglo 1
        Develop|Siglo 0 = Develop|Siglo 0
        Develop|Siglo 1 = Develop|Siglo 1
        Release|Siglo 0 = Release|Siglo 0
        Release|Siglo 1 = Release|Siglo 1
    EndGlobalSection
EndGlobal
";

        /// <summary>
        /// ソリューション構成の追加
        /// </summary>
        [TestMethod]
        public void TestAddSolutionConfiguration()
        {
            VsSolution solution = new VsSolution();

            List<VsConfigurationPair> pairs = new List<VsConfigurationPair>();

            // ソリューション構成の追加
            foreach (string configuration in new string[] { "Debug", "Develop", "Release" })
            {
                foreach (string platform in new string[] { "Siglo 0", "Siglo 1" })
                {
                    VsConfigurationPair pair = new VsConfigurationPair(configuration, platform);

                    solution.AddSolutionConfiguration(pair);
                    pairs.Add(pair);
                }
            }

            Assert.IsTrue(Enumerable.SequenceEqual(pairs, solution.Configurations));
            Assert.IsTrue(TestUtility.CompareSolutionString(TextAddSolutionConfiguration, solution.ToString()));
        }

        private const string TextRemoveSolutionConfiguration = @"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 2012
Global
    GlobalSection(SolutionConfigurationPlatforms) = preSolution
        Debug|Siglo 0 = Debug|Siglo 0
        Debug|Siglo 1 = Debug|Siglo 1
        Release|Siglo 0 = Release|Siglo 0
        Release|Siglo 1 = Release|Siglo 1
    EndGlobalSection
EndGlobal
";
        /// <summary>
        /// ソリューション構成の削除
        /// </summary>
        [TestMethod]
        public void TestRemoveSolutionConfiguration()
        {
            VsSolution solution = new VsSolution(TextAddSolutionConfiguration);

            foreach (string platform in new string[] { "Siglo 0", "Siglo 1" })
            {
                solution.RemoveSolutionConfiguration(new VsConfigurationPair("Develop", platform));
            }

            List<VsConfigurationPair> pairs = new List<VsConfigurationPair>();
            foreach (string configuration in new string[] { "Debug", "Release" })
            {
                foreach (string platform in new string[] { "Siglo 0", "Siglo 1" })
                {
                    pairs.Add(new VsConfigurationPair(configuration, platform));
                }
            }

            Assert.IsTrue(Enumerable.SequenceEqual(pairs, solution.Configurations));
            Assert.IsTrue(TestUtility.CompareSolutionString(TextRemoveSolutionConfiguration, solution.ToString()));
        }

        private const string TextGetProjectProperty = TextAddProject;

        /// <summary>
        /// プロジェクト・プロパティの取得
        /// </summary>
        [TestMethod]
        public void TestGetProjectProperty()
        {
            VsSolution solution = new VsSolution(TextGetProjectProperty);

            Guid project = new Guid("{22222222-2222-2222-2222-222222222222}");

            VsSolutionProjectProperty property = solution.GetProjectProperty(project);

            Assert.AreEqual(VsProjectType.CSharp, property.ProjectType);
            Assert.AreEqual("Project 1", property.Name);
            Assert.AreEqual("全角\\Project 1.csproj", property.Path);

            Guid[] expectedDependencies = new Guid[] { new Guid("{11111111-1111-1111-1111-111111111111}") };
            Assert.IsTrue(Enumerable.SequenceEqual(expectedDependencies, property.Dependencies));
        }

        private const string TextSetProjectProperty = @"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 2012
Project(""{8BC9CEB8-8B4A-11D0-8D11-00A0C91BC942}"") = ""Project0"", ""Project0\Project0.vcxproj"", ""{11111111-1111-1111-1111-111111111111}""
EndProject
Project(""{FAE04EC0-301F-11D3-BF4B-00C04F79EFBC}"") = ""NewProject"", ""NewProject\NewProject.csproj"", ""{22222222-2222-2222-2222-222222222222}""
EndProject
Global
EndGLobal
";
        /// <summary>
        /// プロジェクト・プロパティの変更
        /// </summary>
        [TestMethod]
        public void TestSetProjectProperty()
        {
            VsSolution solution = new VsSolution(TextGetProjectProperty);

            Guid project = new Guid("{22222222-2222-2222-2222-222222222222}");

            VsSolutionProjectProperty oldProperty = solution.GetProjectProperty(project);
            VsSolutionProjectProperty newProperty = new VsSolutionProjectProperty(oldProperty.ProjectType, "NewProject", "NewProject\\NewProject.csproj", null);
            solution.SetProjectProperty(project, newProperty);

            Assert.AreEqual(newProperty, solution.GetProjectProperty(project));
            Assert.IsTrue(TestUtility.CompareSolutionString(TextSetProjectProperty, solution.ToString()));
        }

        private const string TextGetProjectConfiguration = @"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 2012
Project(""{8BC9CEB8-8B4A-11D0-8D11-00A0C91BC942}"") = ""Project0"", ""Project0\Project0.vcxproj"", ""{11111111-1111-1111-1111-111111111111}""
EndProject
Global
    GlobalSection(SolutionConfigurationPlatforms) = preSolution
        Debug|Siglo 0 = Debug|Siglo 0
        Debug|Siglo 1 = Debug|Siglo 1
        Release|Siglo 0 = Release|Siglo 0
        Release|Siglo 1 = Release|Siglo 1
    EndGlobalSection
    GlobalSection(ProjectConfigurationPlatforms) = postSolution
        {11111111-1111-1111-1111-111111111111}.Debug|Siglo 0.ActiveCfg = Debug|Siglo 0
        {11111111-1111-1111-1111-111111111111}.Debug|Siglo 0.Build.0 = Debug|Siglo 0
        {11111111-1111-1111-1111-111111111111}.Debug|Siglo 1.ActiveCfg = Debug|Siglo 1
        {11111111-1111-1111-1111-111111111111}.Debug|Siglo 1.Build.0 = Debug|Siglo 1
        {11111111-1111-1111-1111-111111111111}.Release|Siglo 0.ActiveCfg = Debug|Siglo 0
        {11111111-1111-1111-1111-111111111111}.Release|Siglo 1.ActiveCfg = Release|Siglo 1
        {11111111-1111-1111-1111-111111111111}.Release|Siglo 1.Build.0 = Release|Siglo 1
    EndGlobalSection
EndGlobal
";

        /// <summary>
        /// プロジェクト構成の取得
        /// </summary>
        [TestMethod]
        public void TestGetProjectConfiguration()
        {
            VsSolution solution = new VsSolution(TextGetProjectConfiguration);

            Guid project = new Guid("{11111111-1111-1111-1111-111111111111}");

            VsConfigurationPair solutionConfiguration = new VsConfigurationPair("Release", "Siglo 0");

            VsProjectConfiguration projectConfiguration = solution.GetProjectConfiguration(solutionConfiguration, project);

            Assert.AreEqual("Debug", projectConfiguration.ConfigurationPair.Configuration);
            Assert.AreEqual("Siglo 0", projectConfiguration.ConfigurationPair.Platform);
            Assert.AreEqual(false, projectConfiguration.EnableBuild);
        }

        private const string TextSetProjectConfiguration = @"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 2012
Project(""{8BC9CEB8-8B4A-11D0-8D11-00A0C91BC942}"") = ""Project0"", ""Project0\Project0.vcxproj"", ""{11111111-1111-1111-1111-111111111111}""
EndProject
Global
    GlobalSection(SolutionConfigurationPlatforms) = preSolution
        Debug|Siglo 0 = Debug|Siglo 0
        Debug|Siglo 1 = Debug|Siglo 1
        Release|Siglo 0 = Release|Siglo 0
        Release|Siglo 1 = Release|Siglo 1
    EndGlobalSection
    GlobalSection(ProjectConfigurationPlatforms) = postSolution
        {11111111-1111-1111-1111-111111111111}.Debug|Siglo 0.ActiveCfg = Debug|Siglo 0
        {11111111-1111-1111-1111-111111111111}.Debug|Siglo 0.Build.0 = Debug|Siglo 0
        {11111111-1111-1111-1111-111111111111}.Debug|Siglo 1.ActiveCfg = Debug|Siglo 9
        {11111111-1111-1111-1111-111111111111}.Release|Siglo 0.ActiveCfg = Debug|Siglo 0
        {11111111-1111-1111-1111-111111111111}.Release|Siglo 1.ActiveCfg = Release|Siglo 1
        {11111111-1111-1111-1111-111111111111}.Release|Siglo 1.Build.0 = Release|Siglo 1
    EndGlobalSection
EndGlobal
";

        /// <summary>
        /// プロジェクト構成の変更
        /// </summary>
        [TestMethod]
        public void TestSetProjectConfiguration()
        {
            VsSolution solution = new VsSolution(TextGetProjectConfiguration);

            Guid project = new Guid("{11111111-1111-1111-1111-111111111111}");

            VsConfigurationPair solutionConfiguration = new VsConfigurationPair("Debug", "Siglo 1");

            VsProjectConfiguration projectConfiguration = new VsProjectConfiguration(new VsConfigurationPair("Debug", "Siglo 9"), false);
            solution.SetProjectConfiguration(solutionConfiguration, project, projectConfiguration);

            Assert.AreEqual(projectConfiguration, solution.GetProjectConfiguration(solutionConfiguration, project));
            Assert.IsTrue(TestUtility.CompareSolutionString(TextSetProjectConfiguration, solution.ToString()));
        }

        /// <summary>
        /// ソリューションフォルダ・プロジェクト間のネスト関係を追加
        /// </summary>
        [TestMethod]
        public void TestAddNestRelation()
        {
            VsSolution solution = new VsSolution();

            Guid guid0 = new Guid("{00000000-0000-0000-0000-000000000000}");
            Guid guid1 = new Guid("{11111111-1111-1111-1111-111111111111}");
            Guid guid2 = new Guid("{22222222-2222-2222-2222-222222222222}");

            VsNestRelation relation01 = new VsNestRelation(guid0, guid1);
            VsNestRelation relation12 = new VsNestRelation(guid1, guid2);

            VsSolutionProjectProperty property0 = new VsSolutionProjectProperty(VsProjectType.SolutionFolder, "Name0", "Path0", null);
            VsSolutionProjectProperty property1 = new VsSolutionProjectProperty(VsProjectType.SolutionFolder, "Name1", "Path1", null);
            VsSolutionProjectProperty property2 = new VsSolutionProjectProperty(VsProjectType.VisualC, "Name2", "Path2", null);

            solution.AddProject(guid0, property0);
            solution.AddProject(guid1, property1);
            solution.AddProject(guid2, property2);

            solution.AddNestRelation(relation01);
            solution.AddNestRelation(relation12);

            var expected = new List<VsNestRelation>();
            expected.Add(relation01);
            expected.Add(relation12);

            Assert.IsTrue(Enumerable.SequenceEqual(expected, solution.NestRelations));
            Assert.IsTrue(TestUtility.CompareSolutionString(TextAddNestRelation, solution.ToString()));
        }

        private const string TextAddNestRelation = @"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 2012
Project(""{2150E333-8FDC-42A3-9474-1A3956D46DE8}"") = ""Name0"", ""Path0"", ""{00000000-0000-0000-0000-000000000000}""
EndProject
Project(""{2150E333-8FDC-42A3-9474-1A3956D46DE8}"") = ""Name1"", ""Path1"", ""{11111111-1111-1111-1111-111111111111}""
EndProject
Project(""{8BC9CEB8-8B4A-11D0-8D11-00A0C91BC942}"") = ""Name2"", ""Path2"", ""{22222222-2222-2222-2222-222222222222}""
EndProject
Global
    GlobalSection(NestedProjects) = preSolution
        {11111111-1111-1111-1111-111111111111} = {00000000-0000-0000-0000-000000000000}
        {22222222-2222-2222-2222-222222222222} = {11111111-1111-1111-1111-111111111111}
    EndGlobalSection
EndGlobal
";

        /// <summary>
        /// ソリューションフォルダ・プロジェクト間のネスト関係を削除
        /// </summary>
        [TestMethod]
        public void TestRemoveNestRelation()
        {
            VsSolution solution = new VsSolution(TextAddNestRelation);

            Guid guid0 = new Guid("{00000000-0000-0000-0000-000000000000}");
            Guid guid1 = new Guid("{11111111-1111-1111-1111-111111111111}");
            Guid guid2 = new Guid("{22222222-2222-2222-2222-222222222222}");

            VsNestRelation relation01 = new VsNestRelation(guid0, guid1);
            VsNestRelation relation12 = new VsNestRelation(guid1, guid2);

            solution.RemoveNestRelation(relation12);

            var expected = new List<VsNestRelation>();
            expected.Add(relation01);

            Assert.IsTrue(Enumerable.SequenceEqual(expected, solution.NestRelations));
            Assert.IsTrue(TestUtility.CompareSolutionString(TextRemoveNestRelation, solution.ToString()));
        }

        private const string TextRemoveNestRelation = @"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 2012
Project(""{2150E333-8FDC-42A3-9474-1A3956D46DE8}"") = ""Name0"", ""Path0"", ""{00000000-0000-0000-0000-000000000000}""
EndProject
Project(""{2150E333-8FDC-42A3-9474-1A3956D46DE8}"") = ""Name1"", ""Path1"", ""{11111111-1111-1111-1111-111111111111}""
EndProject
Project(""{8BC9CEB8-8B4A-11D0-8D11-00A0C91BC942}"") = ""Name2"", ""Path2"", ""{22222222-2222-2222-2222-222222222222}""
EndProject
Global
    GlobalSection(NestedProjects) = preSolution
        {11111111-1111-1111-1111-111111111111} = {00000000-0000-0000-0000-000000000000}
    EndGlobalSection
EndGlobal
";

        /// <summary>
        /// ソリューションフォルダ配下にプロジェクトを配置したときのネスト関係
        /// </summary>
        [TestMethod]
        public void TestNestRelationWithProject()
        {
            var solution = new VsSolution();
            var solutionConfiguration = new VsConfigurationPair("Debug", "Win32");

            solution.AddSolutionConfiguration(solutionConfiguration);

            var rootGuid       = new Guid("{00000000-0000-0000-0000-000000000000}");
            var childrenGuid   = new Guid("{11111111-1111-1111-1111-111111111111}");
            var descendantGuid = new Guid("{22222222-2222-2222-2222-222222222222}");

            // ソリューションフォルダと VC プロジェクト間で親子関係を構築
            solution.AddNestRelation(new VsNestRelation(rootGuid, childrenGuid));
            solution.AddNestRelation(new VsNestRelation(childrenGuid, descendantGuid));

            // ソリューションにフォルダと VC プロジェクトを追加
            solution.AddProject(rootGuid, CreateSolutionFolder("Root"));
            solution.AddProject(childrenGuid, CreateSolutionFolder("Children"));
            solution.AddProject(descendantGuid, CreateVcProject("DescendantProject.vcxproj"));

            // VC プロジェクトにビルド構成を追加
            solution.SetProjectConfiguration(
                 solutionConfiguration,
                 descendantGuid,
                 new VsProjectConfiguration(solutionConfiguration, true));

            Assert.IsTrue(TestUtility.CompareSolutionString(TextNestRelationWithProject, solution.ToString()));
        }

        // VC プロジェクトにだけプロジェクト構成が存在することをテスト
        private const string TextNestRelationWithProject = @"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 2012
Project(""{2150E333-8FDC-42A3-9474-1A3956D46DE8}"") = ""Root"", ""Root"", ""{00000000-0000-0000-0000-000000000000}""
EndProject
Project(""{2150E333-8FDC-42A3-9474-1A3956D46DE8}"") = ""Children"", ""Children"", ""{11111111-1111-1111-1111-111111111111}""
EndProject
Project(""{8BC9CEB8-8B4A-11D0-8D11-00A0C91BC942}"") = ""DescendantProject"", ""DescendantProject.vcxproj"", ""{22222222-2222-2222-2222-222222222222}""
EndProject
Global
    GlobalSection(SolutionConfigurationPlatforms) = preSolution
        Debug|Win32 = Debug|Win32
    EndGlobalSection

    GlobalSection(ProjectConfigurationPlatforms) = postSolution
        {22222222-2222-2222-2222-222222222222}.Debug|Win32.ActiveCfg = Debug|Win32
        {22222222-2222-2222-2222-222222222222}.Debug|Win32.Build.0 = Debug|Win32
    EndGlobalSection

    GlobalSection(NestedProjects) = preSolution
        {11111111-1111-1111-1111-111111111111} = {00000000-0000-0000-0000-000000000000}
        {22222222-2222-2222-2222-222222222222} = {11111111-1111-1111-1111-111111111111}
    EndGlobalSection
EndGlobal
";

        private static VsSolutionProjectProperty CreateSolutionFolder(string name)
        {
            return new VsSolutionProjectProperty(VsProjectType.SolutionFolder, name, name, null);
        }

        private static VsSolutionProjectProperty CreateVcProject(string projectPath)
        {
            return new VsSolutionProjectProperty(
                VsProjectType.VisualC,
                System.IO.Path.GetFileNameWithoutExtension(projectPath),
                projectPath,
                null);
        }
    }
}
