﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace UartTestHost
{
    /// <summary>
    /// テスト本体の実装クラス
    /// </summary>
    public class TestBody
    {
        /// <summary>
        /// もっとも基本的な送受信が機能していることのテスト。
        /// テストの成否はターゲット側で判定します。
        /// テスト手順についてはターゲット側の対応するテストプログラムを参照してください。
        /// </summary>
        /// <param name="testParam">テストパラメータ</param>
        /// <returns>テスト結果</returns>
        public static TestResult RunTestShortLoopback(TestParam testParam)
        {
            TestResult result;
            const int TestDataCount = 256;
            byte[] buffer;

            SerialStream stream = new SerialStream();
            result = stream.Open(testParam);
            if (result != TestResult.Success)
            {
                return result;
            }

            // テスト開始前の同期
            WaitForSyncTestParam(testParam);

            // Receive
            Console.WriteLine("Receiving test data");
            result = stream.ReceiveSync(out buffer, TestDataCount, 2000);
            if (result != TestResult.Success)
            {
                stream.Close();
                return result;
            }

            // Send
            Console.WriteLine("Sending test data");
            result = stream.Send(buffer, TestDataCount);
            if (result != TestResult.Success)
            {
                stream.Close();
                return result;
            }

            // 転送完了を待機
            System.Threading.Thread.Sleep(100);

            result = stream.Close();
            if (result != TestResult.Success)
            {
                return result;
            }
            return TestResult.Success;
        }

        /// <summary>
        /// フロー制御オンでターゲットが送信側になり、オーバーランが発生しないことのテスト。
        /// テスト手順についてはターゲット側の対応するテストプログラムを参照してください。
        /// </summary>
        /// <param name="testParam">テストパラメータ</param>
        /// <returns>テスト結果</returns>
        public static TestResult RunTestSendWithFlowControl(TestParam testParam)
        {
            TestResult result;
            const int ReadBufferSize = 1024;
            const int TestDataMaxCount = 8 * 1024 * 1024;

            // 最大 4 秒分ほどのテストデータを扱う
            // 注意：データサイズの値はターゲット側でも同じ式で計算しているので、変更時にはターゲット側も合わせること
            int TestDataCount = ((SerialTypesUtil.ToIntBaudRate(testParam.BaudRate) / 2) < TestDataMaxCount) ?
                                 (SerialTypesUtil.ToIntBaudRate(testParam.BaudRate) / 2) : TestDataMaxCount;
            byte[] buffer;

            // 受信バッファを小さくしておく
            SerialStream stream = new SerialStream();
            SerialConfig config = new SerialConfig(testParam);
            config.ReadBufferSize = ReadBufferSize;
            result = stream.Open(config);
            if (result != TestResult.Success)
            {
                return result;
            }

            // テスト開始前の同期
            WaitForSyncTestParam(testParam);

            // 受信バッファが一杯になるのを時限つきで待つ
            Console.WriteLine("Waiting receive buffer getting full..");
            const int TimeoutMsec = 3000;
            if (!stream.WaitReadBufferFill(ReadBufferSize, TimeoutMsec))
            {
                stream.Close();
                return result;
            }

            // 4. 対向デバイス側は、受信バッファがいっぱいになった状態をテストしてから 3 秒待機し、
            //    データをすべて受信する
            Console.WriteLine("Receive buffer seems full now");
            System.Threading.Thread.Sleep(3000);

            // 送信データ量とボーレート (bps とだいたい一致するものとする) から大まかに転送時間を見積もって
            // タイムアウト時間を決める。
            // 式中の * 2 はマージン 100% として補正する係数。
            Console.WriteLine("Receiving test data");
            int timeoutMsec = (TestDataCount * 1000 * 2) / (SerialTypesUtil.ToIntBaudRate(testParam.BaudRate) / 8);
            result = stream.ReceiveSync(out buffer, TestDataCount, timeoutMsec);
            if (result != TestResult.Success)
            {
                stream.Close();
                return result;
            }

            // Send
            Console.WriteLine("Sending test data");
            result = stream.Send(buffer, TestDataCount);
            if (result != TestResult.Success)
            {
                stream.Close();
                return result;
            }

            // 転送完了を待機
            System.Threading.Thread.Sleep(100);

            result = stream.Close();
            if (result != TestResult.Success)
            {
                return result;
            }
            return TestResult.Success;
        }

        /// <summary>
        /// フロー制御オンでターゲットが受信側になり、オーバーランが発生しないことのテスト。
        /// テスト手順についてはターゲット側の対応するテストプログラムを参照してください。
        /// </summary>
        /// <param name="testParam">テストパラメータ</param>
        /// <returns>テスト結果</returns>
        public static TestResult RunTestLazyReceiveWithFlowControl(TestParam testParam)
        {
            TestResult result;

            // 注意：データサイズの値はターゲット側でも同じ式で計算しているので、変更時にはターゲット側も合わせること
            int TestDataCount = 10 * 1024 + 19;
            byte[] buffer;

            SerialStream stream = new SerialStream();
            result = stream.Open(testParam);
            if (result != TestResult.Success)
            {
                return result;
            }

            // テスト開始前の同期
            WaitForSyncTestParam(testParam);

            // 2. 対向デバイス側はデータをすべて受信したらすぐに全体を送信し返す（ループバック）

            Console.WriteLine("Receiving test data");
            int timeoutMsec = CalcTransportTimeoutMsec(testParam, TestDataCount);
            result = stream.ReceiveSync(out buffer, TestDataCount, timeoutMsec);
            if (result != TestResult.Success)
            {
                stream.Close();
                return result;
            }

            // Send
            Console.WriteLine("Sending test data");
            result = stream.Send(buffer, TestDataCount);
            if (result != TestResult.Success)
            {
                stream.Close();
                return result;
            }

            // 転送完了を待機
            System.Threading.Thread.Sleep(100);

            result = stream.Close();
            if (result != TestResult.Success)
            {
                return result;
            }
            return TestResult.Success;
        }

        /// <summary>
        /// フロー制御オンでターゲットが受信側になり、ポートイベントを使用してデータ受信を行うテスト。
        /// テスト手順についてはターゲット側の対応するテストプログラムを参照してください。
        /// </summary>
        /// <param name="testParam">テストパラメータ</param>
        /// <returns>テスト結果</returns>
        public static TestResult RunTestReceiveUsingPortEvent(TestParam testParam)
        {
            TestResult result;

            SerialStream stream = new SerialStream();
            result = stream.Open(testParam);
            if (result != TestResult.Success)
            {
                return result;
            }

            // テスト開始前の同期
            WaitForSyncTestParam(testParam);

            int testDataCount;
            do
            {
                byte[] buffer;

                // テストデータサイズの取得
                Console.WriteLine("Receiving test data size");
                result = stream.ReceiveSync(out buffer, 4, 2000);
                if (result != TestResult.Success)
                {
                    stream.Close();
                    return result;
                }

                testDataCount = BitConverter.ToInt32(buffer, 0);
                Console.WriteLine(string.Format("Test data size is {0}", testDataCount));

                if (testDataCount > 0)
                {
                    // 取得したサイズ分のテストデータを作成
                    buffer = CreateTestData(testDataCount);

                    // Send
                    Console.WriteLine("Sending test data");
                    result = stream.Send(buffer, testDataCount);
                    if (result != TestResult.Success)
                    {
                        stream.Close();
                        return result;
                    }

                    // 転送完了を待機
                    System.Threading.Thread.Sleep(100);
                }
            } while (testDataCount > 0);

            result = stream.Close();
            if (result != TestResult.Success)
            {
                return result;
            }
            return TestResult.Success;
        }

        /// <summary>
        /// フロー制御オンでターゲットが送信側になり、ポートイベントを使用してデータ送信完了を確認するテスト。
        /// テスト手順についてはターゲット側の対応するテストプログラムを参照してください。
        /// </summary>
        /// <param name="testParam">テストパラメータ</param>
        /// <returns>テスト結果</returns>
        public static TestResult RunTestSendUsingPortEvent(TestParam testParam)
        {
            TestResult result;

            SerialStream stream = new SerialStream();
            result = stream.Open(testParam);
            if (result != TestResult.Success)
            {
                return result;
            }

            // テスト開始前の同期
            WaitForSyncTestParam(testParam);

            int testDataCount = 0;
            do
            {
                byte[] receiveBuffer;

                // テストデータサイズの取得
                Console.WriteLine("Receiving test data size...");
                result = stream.ReceiveSync(out receiveBuffer, 4, 2000);
                if (result != TestResult.Success)
                {
                    stream.Close();
                    return result;
                }

                testDataCount = BitConverter.ToInt32(receiveBuffer, 0);
                Console.WriteLine(string.Format("Test data size is {0}", testDataCount));

                if (testDataCount > 0)
                {
                    // 取得したサイズ分のテストデータを受信
                    Console.WriteLine("Receiving test data...");

                    int timeoutMsec = CalcTransportTimeoutMsec(testParam, testDataCount);
                    result = stream.ReceiveSync(out receiveBuffer, testDataCount, timeoutMsec);
                    if (result != TestResult.Success)
                    {
                        stream.Close();
                        return result;
                    }

                    Console.WriteLine("Receive OK");
                }
            } while (testDataCount > 0);

            result = stream.Close();
            if (result != TestResult.Success)
            {
                return result;
            }
            return TestResult.Success;
        }

        /// <summary>
        /// フロー制御オンでターゲットが受信側になり、受信終了イベントを使用してデータ受信を行うテスト。
        /// テスト手順についてはターゲット側の対応するテストプログラムを参照してください。
        /// </summary>
        /// <param name="testParam">テストパラメータ</param>
        /// <returns>テスト結果</returns>
        public static TestResult RunTestReceiveUsingReceiveEndEvent(TestParam testParam)
        {
            // デバイス側が行うことは ReceiveUsingPortEvent と同じ
            return RunTestReceiveUsingPortEvent(testParam);
        }

        /// <summary>
        /// テスト開始前のパラメータ同期待ちです。
        /// </summary>
        /// <param name="testParam">テストパラメータ</param>
        private static void WaitForSyncTestParam(TestParam testParam)
        {
            // 同期のための待ち時間
            const int WaitTimeForSyncMsec = 500;

            System.Threading.Thread.Sleep(WaitTimeForSyncMsec);

            Console.WriteLine("Running test param - Scenario {0:X} / BaudRate {1:X} / FlowControlMode {2:X}\n",
                                testParam.Scenario, testParam.BaudRate, testParam.FlowControlMode);
        }

        /// <summary>
        /// テストデータを生成します。
        /// </summary>
        /// <param name="dataSize">テストデータのサイズ</param>
        /// <returns>生成されたテストデータ</returns>
        private static byte[] CreateTestData(int dataSize)
        {
            var testData = new byte[dataSize];

            // 0 ～ 255 の連番の繰り返し
            for (int i = 0; i < dataSize; ++i)
            {
                testData[i] = (byte)(i & 0xFF);
            }

            return testData;
        }

        /// <summary>
        /// 転送データ量とボーレート (bps とだいたい一致するものとする) から大まかに転送時間を見積もってタイムアウト時間を決める。
        /// 式中の * 2 はマージン 100% として補正する係数。
        /// </summary>
        /// <param name="testParam">テストパラメータ</param>
        /// <param name="dataCount">転送データサイズ</param>
        /// <returns></returns>
        private static int CalcTransportTimeoutMsec(TestParam testParam, int dataCount)
        {
            return (dataCount * 1000 * 2) / (SerialTypesUtil.ToIntBaudRate(testParam.BaudRate) / 8);
        }

        /// <summary>
        /// 一つのテストセットを回します。
        /// つまり、ターゲットとハンドシェイクしてテストパラメータを受信し、そのパラメータでテストを実行する一連の処理を一度実行します。
        /// </summary>
        /// <param name="toBeContinuedOnSuccess">この呼び出しが成功した場合に、引き続き本メソッドを呼び出す必要があるかを返します。</param>
        /// <returns>テスト結果</returns>
        public static TestResult RunTest(out bool toBeContinuedOnSuccess)
        {
            toBeContinuedOnSuccess = true;
            TestParam testParam;
            TestResult result = TestHandshake.HandshakeWithTarget(out testParam);
            if (result != TestResult.Success)
            {
                Console.WriteLine("Failed to handshake with target to run test");
                return result;
            }

            switch (testParam.Scenario)
            {
                case TestScenario.ShortLoopback:
                    {
                        Console.WriteLine("--- RunTestShortLoopback: Begin");
                        result = RunTestShortLoopback(testParam);
                        Console.WriteLine("--- RunTestShortLoopback: End");
                    }
                    break;
                case TestScenario.OverRunDetection:
                    {
                        Console.WriteLine("--- RunTestOverRunDetection: Begin");
                        // TODO: Implement me
                        // result = RunTestOverRunDetection(testParam);
                        Console.WriteLine("--- RunTestOverRunDetection: End");
                    }
                    break;
                case TestScenario.OutOfBufferDetection:
                    {
                        Console.WriteLine("--- RunTestOutOfBufferDetection: Begin");
                        // TODO: Implement me
                        // result = RunTestOutOfBufferDetection(testParam);
                        Console.WriteLine("--- RunTestOutOfBufferDetection: End");
                    }
                    break;
                case TestScenario.SendWithFlowControl:
                    {
                        Console.WriteLine("--- RunTestSendWithFlowControl: Begin");
                        result = RunTestSendWithFlowControl(testParam);
                        Console.WriteLine("--- RunTestSendWithFlowControl: End");
                    }
                    break;
                case TestScenario.LazyReceiveWithFlowControl:
                    {
                        Console.WriteLine("--- RunTestLazyReceiveWithFlowControl: Begin");
                        result = RunTestLazyReceiveWithFlowControl(testParam);
                        Console.WriteLine("--- RunTestLazyReceiveWithFlowControl: End");
                    }
                    break;
                case TestScenario.ReceiveUsingPortEvent:
                    {
                        Console.WriteLine("--- RunTestReceiveUsingPortEvent: Begin");
                        result = RunTestReceiveUsingPortEvent(testParam);
                        Console.WriteLine("--- RunTestReceiveUsingPortEvent: End");
                    }
                    break;
                case TestScenario.SendUsingPortEvent:
                    {
                        Console.WriteLine("--- RunTestSendUsingPortEvent: Begin");
                        result = RunTestSendUsingPortEvent(testParam);
                        Console.WriteLine("--- RunTestSendUsingPortEvent: End");
                    }
                    break;
                case TestScenario.ReceiveUsingReceiveEndEvent:
                    {
                        Console.WriteLine("--- RunTestReceiveUsingReceiveEndEvent: Begin");
                        result = RunTestReceiveUsingReceiveEndEvent(testParam);
                        Console.WriteLine("--- RunTestReceiveUsingReceiveEndEvent: End");
                    }
                    break;
                case TestScenario.TestTermination:
                    {
                        Console.WriteLine("**** All Tests PASSED! ****");
                        result = TestResult.Success;
                        toBeContinuedOnSuccess = false; // TEST 終了
                    }
                    break;
                default:
                    {
                        Console.WriteLine("Test scneario is not implemented");
                        result = TestResult.InvalidTestScenario;
                    }
                    break;
            }

            return result;
        }
    }
}
