﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO.Ports;
using System.Diagnostics;

namespace UartTestHost
{
    /// <summary>
    /// シリアル通信モード設定。
    /// 利便性のため、テストパラメータなどを引数に取るコンストラクタを持ちます。
    /// </summary>
    public struct SerialConfig
    {
        public TestBaudRateType BaudRate;
        public TestFlowControlMode FlowControlMode;

        public bool IsReadBufferSizeSpecified { get; private set; }
        public bool IsWriteBufferSizeSpecified { get; private set; }

        private const int UnspecifiedBufferSize = -1;
        private int readBufferSize;
        public int ReadBufferSize
        {
            get
            {
                return this.readBufferSize;
            }
            set
            {
                this.readBufferSize = value;
                if (value == UnspecifiedBufferSize)
                {
                    this.IsReadBufferSizeSpecified = false;
                }
                else
                {
                    this.IsReadBufferSizeSpecified = true;
                }
            }
        }
        private int writeBufferSize;
        public int WriteBufferSize
        {
            get
            {
                return this.writeBufferSize;
            }
            set
            {
                this.writeBufferSize = value;
                if (value == UnspecifiedBufferSize)
                {
                    this.IsWriteBufferSizeSpecified = false;
                }
                else
                {
                    this.IsWriteBufferSizeSpecified = true;
                }
            }
        }

        public SerialConfig(TestBaudRateType baudRate, TestFlowControlMode flowControlMode)
            : this()
        {
            BaudRate = baudRate;
            FlowControlMode = flowControlMode;

            IsReadBufferSizeSpecified = false;
            IsWriteBufferSizeSpecified = false;
            readBufferSize = UnspecifiedBufferSize;
            writeBufferSize = UnspecifiedBufferSize;
        }
        public SerialConfig(TestParam testParam)
            : this()
        {
            BaudRate = testParam.BaudRate;
            FlowControlMode = testParam.FlowControlMode;

            IsReadBufferSizeSpecified = false;
            IsWriteBufferSizeSpecified = false;
            readBufferSize = UnspecifiedBufferSize;
            writeBufferSize = UnspecifiedBufferSize;
        }
    }

    /// <summary>
    /// システムのシリアル通信のラッパークラス
    /// </summary>
    public class SerialStream
    {
        private SerialPort serialPort;
        public bool IsOpen { get; private set; }
        public int BytesToRead
        {
            get { return serialPort.BytesToRead; }
        }
        public int BytesToWrite
        {
            get { return serialPort.BytesToWrite; }
        }

        public SerialStream()
        {
            IsOpen = false;
            serialPort = new SerialPort();
        }
        ~SerialStream()
        {
            Close();
        }

        /// <summary>
        /// 使用するシリアルポートを決定します。
        /// コマンドラインオプションでポート名が指定されていた場合はそれを優先して探索します。
        /// </summary>
        /// <returns>ポート名</returns>
        private string FindSerialPort()
        {
            string[] portNameList = SerialPort.GetPortNames();

            if (!portNameList.Any())
            {
                Console.WriteLine("No port available");
                return null;
            }

#if DEBUG
            Console.WriteLine("--- Port List ---");
            foreach (string portName in portNameList)
            {
                Console.WriteLine("  " + portName);
            }
            Console.WriteLine("-----------------");
#endif

            // コマンドラインで指定されていればそれを探す
            if (CmdlineOption.PortName != null &&
                portNameList.Any(portName => portName.Equals(CmdlineOption.PortName)))
            {
                Console.WriteLine("Using " + CmdlineOption.PortName);
                return CmdlineOption.PortName;
            }

            // 未指定か、見つからなかったら
            // 利用可能なシリアルポートから先頭のものを使ってみる
            Console.WriteLine("Using " + portNameList[0]);
            return portNameList[0];
        }

        /// <summary>
        /// シリアルポートをオープンします。
        /// </summary>
        /// <param name="serialConfig">シリアル通信モード設定</param>
        /// <returns>処理結果</returns>
        public TestResult Open(SerialConfig serialConfig)
        {
            if (IsOpen)
            {
                TestResult result = Close();
                if (result != TestResult.Success)
                {
                    return result;
                }
            }

            serialPort.PortName = FindSerialPort();
            if (serialPort.PortName == null)
            {
                return TestResult.PortNotFound;
            }

            try
            {
                serialPort.BaudRate = SerialTypesUtil.ToIntBaudRate(serialConfig.BaudRate);
                serialPort.DataBits = 8;
                serialPort.Parity = Parity.None;
                serialPort.StopBits = StopBits.One;
                serialPort.Handshake = SerialTypesUtil.ToHandshake(serialConfig.FlowControlMode);
                serialPort.Encoding = Encoding.UTF8;
                if (serialConfig.IsReadBufferSizeSpecified)
                {
                    serialPort.ReadBufferSize = serialConfig.ReadBufferSize;
                }
                if (serialConfig.IsWriteBufferSizeSpecified)
                {
                    serialPort.WriteBufferSize = serialConfig.WriteBufferSize;
                }
                serialPort.Open();
                serialPort.DiscardInBuffer();
                serialPort.DiscardOutBuffer();
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                return TestResult.PortOpenFailed;
            }

            IsOpen = true;
            return TestResult.Success;
        }

        /// <summary>
        /// シリアルポートをオープンします。
        /// </summary>
        /// <param name="serialConfig">テストパラメータ</param>
        /// <returns>処理結果</returns>
        public TestResult Open(TestParam testParam)
        {
            SerialConfig config = new SerialConfig(testParam);
            return Open(config);
        }

        /// <summary>
        /// シリアルポートにバイト列を送信します。
        /// </summary>
        /// <param name="buffer">データバッファ</param>
        /// <param name="count">データサイズ [bytes]</param>
        /// <returns>処理結果</returns>
        public TestResult Send(byte[] buffer, int count)
        {
            Trace.Assert(IsOpen);
            try
            {
                serialPort.Write(buffer, 0, count);
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                return TestResult.PortSendFailed;
            }
            return TestResult.Success;
        }

        /// <summary>
        /// シリアルポートからバイト列を受信します。
        /// </summary>
        /// <param name="outCount">受信できたデータサイズ [bytes]</param>
        /// <param name="buffer">データバッファ</param>
        /// <param name="count">受信可能なデータサイズ [bytes]</param>
        /// <returns>処理結果</returns>
        public TestResult Receive(out int outCount, out byte[] buffer, int count)
        {
            Trace.Assert(IsOpen);
            buffer = new byte[count];

            try
            {
                serialPort.Read(buffer, 0, count);
            }
            catch (TimeoutException)
            {
                outCount = 0;
                return TestResult.Success;
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                outCount = 0;
                return TestResult.PortReceiveFailed;
            }
            outCount = count;
            return TestResult.Success;
        }

        /// <summary>
        /// 受信バッファが指定したサイズに達するのを時限つきで待機します。
        /// </summary>
        /// <param name="count"></param>
        /// <param name="timeoutMsec">タイムアウト時間。0 以下の場合1度だけチェックしてすぐに返ります。</param>
        /// <returns></returns>
        public bool WaitReadBufferFill(int count, int timeoutMsec)
        {
            Trace.Assert(count > 0);
            Trace.Assert(count <= serialPort.ReadBufferSize); // 永久に待ち続けるような count になっていないか

            const int PollIntervalMsec = 50;
            int waitedTimeMsec = 0;
            while (serialPort.BytesToRead < count)
            {
                if (waitedTimeMsec >= timeoutMsec)
                {
                    Console.WriteLine("Read buffer was not filled in time");
                    return false;
                }
                System.Threading.Thread.Sleep(PollIntervalMsec);
                waitedTimeMsec += PollIntervalMsec;
            }
            return true;
        }

        /// <summary>
        /// シリアルポートからバイト列を同期的に受信します。
        /// 渡したバッファがすべて埋められるか、タイムアウトなどのエラーが発生するまで返りません。
        /// </summary>
        /// <param name="buffer">データバッファ</param>
        /// <param name="count">受信可能なデータサイズ [bytes]</param>
        /// <param name="timeoutMsec">タイムアウト時間</param>
        /// <returns>処理結果</returns>
        public TestResult ReceiveSync(out byte[] buffer, int count, int timeoutMsec)
        {
            const int PollIntervalMsec = 50;
            int waitedTimeMsec = 0;
            int totalReceivedCount = 0;
            System.IO.MemoryStream memstream = new System.IO.MemoryStream(count);
            while (totalReceivedCount < count)
            {
                int bytesInBuffer = serialPort.BytesToRead;
                if (bytesInBuffer == 0)
                {
                    if (waitedTimeMsec >= timeoutMsec)
                    {
                        Console.WriteLine("Received " + totalReceivedCount + " byte in " + timeoutMsec + " [msec]" +
                                            " expecting " + count + " byte");
                        buffer = null;
                        return TestResult.DataInsufficient;
                    }
                    System.Threading.Thread.Sleep(PollIntervalMsec);
                    waitedTimeMsec += PollIntervalMsec;
                    continue;
                }

                int receivedCount;
                byte[] tmpBuffer;
                int countToRead = (bytesInBuffer < (count - totalReceivedCount)) ? bytesInBuffer : (count - totalReceivedCount);
                TestResult result = Receive(out receivedCount, out tmpBuffer, countToRead);
                if (result != TestResult.Success)
                {
                    buffer = null;
                    return result;
                }
                memstream.Write(tmpBuffer, 0, receivedCount);
                totalReceivedCount += receivedCount;
            }
            buffer = memstream.GetBuffer();
            return TestResult.Success;
        }

        /// <summary>
        /// シリアルポートの受信バッファを空にします。
        /// </summary>
        public void DiscardInBuffer()
        {
            Trace.Assert(IsOpen);
            serialPort.DiscardInBuffer();
        }

        /// <summary>
        /// シリアルポートをクローズします。
        /// </summary>
        /// <returns>処理結果</returns>
        public TestResult Close()
        {
            // デストラクタからも呼ぶため、例外を外に投げないこと
            if (IsOpen)
            {
                try
                {
                    serialPort.Close();
                }
                catch (Exception ex)
                {
                    Console.WriteLine(ex.Message);
                    return TestResult.PortCloseFailed;
                }
                IsOpen = false;
            }
            return TestResult.Success;
        }
    }
}
