﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunnerTest.TestNodeInfoTest
{
    using System;
    using System.Collections.Generic;
    using Microsoft.VisualStudio.TestTools.UnitTesting;
    using TestRunner;

    /// <summary>
    /// [TestRunner.TestNodeInfo]::IsWrongYaml用のテスト関数を定義します。
    /// </summary>
    [TestClass]
    public class IsWrongYamlTest
    {
        /// <summary>
        /// 境界値分析: ノードの初期化に失敗
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest001()
        {
            var msg = string.Empty;

            Assert.IsTrue(new TestNodeInfo(null).IsWrongYaml(out msg));
            Assert.AreEqual("Invalid node type for a test entry", msg);

            Assert.IsTrue(new TestNodeInfo(new object()).IsWrongYaml(out msg));
            Assert.AreEqual("Invalid node type for a test entry", msg);

            Assert.IsFalse(new TestNodeInfo(new Dictionary<string, object>()).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);
        }

        /// <summary>
        /// 境界値分析: ノード中の Units 要素中に不正なノードが存在
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest002()
        {
            var msg = string.Empty;

            var node = new Dictionary<string, object>();

            node["Units"] = new List<object>(new object[] { new object() });

            Assert.IsTrue(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual("Invalid node type for a test unit", msg);

            node["Units"] = new List<object>(new object[] { null });

            Assert.IsFalse(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);

            node["Units"] = new List<object>(new object[] { new Dictionary<string, object>() });

            Assert.IsFalse(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);
        }

        /// <summary>
        /// 境界値分析: ノード中の Path 要素の型が文字列でない
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest003()
        {
            HasExpectedTypeOrNotTest(new Dictionary<string, object>(), "Path", "value");
        }

        /// <summary>
        /// 境界値分析: ノード中の Option 要素の型が文字列でない
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest004()
        {
            var mappingNode = new Dictionary<string, object>();

            mappingNode["Path"] = "value1";

            HasExpectedTypeOrNotTest(mappingNode, "Option", "value2");
        }

        /// <summary>
        /// 境界値分析: ノード中の Timeout 要素が正の整数や0を表す文字列でない
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest005()
        {
            var key = "Timeout";

            HasExpectedTypeOrNotTest(new Dictionary<string, object>(), key, "0");

            HasExpectedValueOrNotTest(new Dictionary<string, object>(), key, new[] { "1" }, new[] { "-1", "value" });
        }

        /// <summary>
        /// 境界値分析: ノード中の BuildType 要素が規定のビルドタイプを表す文字列でない
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest006()
        {
            var key = "BuildType";

            HasExpectedTypeOrNotTest(new Dictionary<string, object>(), key, "Debug");

            HasExpectedValueOrNotTest(new Dictionary<string, object>(), key, new[] { "Debug", "Develop", "Release" }, new[] { "Invalid" });
        }

        /// <summary>
        /// 境界値分析: ノード中の Units 要素の型がシーケンスノードでない
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest007()
        {
            HasExpectedTypeOrNotTest(new Dictionary<string, object>(), "Units", new List<object>());
        }

        /// <summary>
        /// 境界値分析: ノード中で Path 要素と Units 要素が衝突
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest008()
        {
            var msg = string.Empty;

            var node = new Dictionary<string, object>();

            node["Path"] = "value";

            node["Units"] = new List<object>();

            Assert.IsTrue(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual("'Units' conflicting with 'Path'", msg);

            node = new Dictionary<string, object>();

            node["Path"] = "value";

            Assert.IsFalse(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);

            node = new Dictionary<string, object>();

            node["Units"] = new List<object>();

            Assert.IsFalse(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);
        }

        /// <summary>
        /// 境界値分析: ノード中に Option 要素が対応する Path 要素や ProgramName 要素を持たずに存在
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest009()
        {
            var msg = string.Empty;

            var node = new Dictionary<string, object>();

            node["Option"] = "value1";

            Assert.IsTrue(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual("'Option' missing the corresponding 'Path'", msg);

            node["Path"] = "value2";

            Assert.IsFalse(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);

            Assert.IsTrue(node.Remove("Path"));

            node["ProgramName"] = "value3";

            Assert.IsFalse(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);

            Assert.IsTrue(node.Remove("ProgramName"));

            node["ProgramId"] = "0x0000000000000000";

            Assert.IsFalse(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);
        }

        /// <summary>
        /// 境界値分析: ノード中の LocalRepeat 要素が正の整数を表す文字列でない
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest010()
        {
            var key = "LocalRepeat";

            HasExpectedTypeOrNotTest(new Dictionary<string, object>(), key, "1");

            HasExpectedValueOrNotTest(new Dictionary<string, object>(), key, new[] { "2" }, new[] { "0", "-1", "value" });
        }

        /// <summary>
        /// 境界値分析: ノード中の Parameter 要素の型が文字列でない
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest011()
        {
            var mappingNode = new Dictionary<string, object>();

            mappingNode["Path"] = "value1";

            HasExpectedTypeOrNotTest(mappingNode, "Parameter", "value2");
        }

        /// <summary>
        /// 境界値分析: ノード中に Parameter 要素が対応する Path 要素や ProgramName 要素を持たずに存在
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest012()
        {
            var msg = string.Empty;

            var node = new Dictionary<string, object>();

            node["Parameter"] = "value1";

            Assert.IsTrue(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual("'Parameter' missing the corresponding 'Path'", msg);

            node["Path"] = "value2";

            Assert.IsFalse(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);

            Assert.IsTrue(node.Remove("Path"));

            node["ProgramName"] = "value3";

            Assert.IsFalse(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);

            Assert.IsTrue(node.Remove("ProgramName"));

            node["ProgramId"] = "0x0000000000000000";

            Assert.IsFalse(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);
        }

        /// <summary>
        /// 境界値分析: ノード中の Path 要素の値が不正なパス文字列
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest013()
        {
            var node = new Dictionary<string, object>();

            HasPathStringOrNotTest(node, "Path");
        }

        /// <summary>
        /// 境界値分析: ノード中の ReportPath 要素の型が文字列でない
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest014()
        {
            var mappingNode = new Dictionary<string, object>();

            mappingNode["Path"] = "value1";

            HasExpectedTypeOrNotTest(mappingNode, "ReportPath", "value2");
        }

        /// <summary>
        /// 境界値分析: ノード中に ReportPath 要素が対応する Path 要素や ProgramName 要素を持たずに存在
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest015()
        {
            var msg = string.Empty;

            var node = new Dictionary<string, object>();

            node["ReportPath"] = "value1";

            Assert.IsTrue(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual("'ReportPath' missing the corresponding 'Path'", msg);

            node["Path"] = "value2";

            Assert.IsFalse(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);

            Assert.IsTrue(node.Remove("Path"));

            node["ProgramName"] = "value3";

            Assert.IsFalse(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);

            Assert.IsTrue(node.Remove("ProgramName"));

            node["ProgramId"] = "0x0000000000000000";

            Assert.IsFalse(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);
        }

        /// <summary>
        /// 境界値分析: ノード中の ReportPath 要素の値が不正なパス文字列
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest016()
        {
            var node = new Dictionary<string, object>();

            node["Path"] = "value";

            HasPathStringOrNotTest(node, "ReportPath");
        }

        /// <summary>
        /// 境界値分析: ノード中の ProgramName 要素の型が文字列でない
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest017()
        {
            HasExpectedTypeOrNotTest(new Dictionary<string, object>(), "ProgramName", "value");
        }

        /// <summary>
        /// 境界値分析: ノード中で ProgramName 要素と Units 要素が衝突
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest018()
        {
            var msg = string.Empty;

            var node = new Dictionary<string, object>();

            node["ProgramName"] = "value";

            node["Units"] = new List<object>();

            Assert.IsTrue(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual("'Units' conflicting with 'ProgramName'", msg);

            node = new Dictionary<string, object>();

            node["ProgramName"] = "value";

            Assert.IsFalse(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);

            node = new Dictionary<string, object>();

            node["Units"] = new List<object>();

            Assert.IsFalse(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);
        }

        /// <summary>
        /// 境界値分析: ノード中の Platform 要素が規定のビルドタイプを表す文字列でない
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest019()
        {
            var key = "Platform";

            HasExpectedTypeOrNotTest(new Dictionary<string, object>(), key, "Win32");

            HasExpectedValueOrNotTest(
                new Dictionary<string, object>(),
                key,
                new[] {
                    "Win32",
                    "Win32_VS2013",
                    "Win32_VS2015",
                    "x86",
                    "Win64",
                    "Win64_VS2013",
                    "Win64_VS2015",
                    "x64",
                    "bdsl-imx6",
                    "jetson-tk1",
                    "jetson-tk2",
                    "jetson-tk2-a64",
                    "NXOnJetsonTk2",
                    "NXOnWin32",
                    "NX-Win32",
                    "NX-Win32_VS2013",
                    "NX-Win32_VS2015",
                    "NXOnWin64",
                    "NX-Win64",
                    "NX-Win64_VS2013",
                    "NX-Win64_VS2015",
                    "NX32",
                    "NX64",
                    "NXFP2",
                    "NXFP2-a32",
                    "NXFP2-a64",
                },
                new[] { "Invalid" });
        }

        /// <summary>
        /// 境界値分析: ノード中の TargetName 要素の型が文字列でない
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest020()
        {
            var mappingNode = new Dictionary<string, object>();

            mappingNode["Path"] = "value1";

            HasExpectedTypeOrNotTest(mappingNode, "TargetName", "value2");
        }

        /// <summary>
        /// 境界値分析: ノード中の TargetName 要素の値が不正な正規表現文字列
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest021()
        {
            var node = new Dictionary<string, object>();

            node["Path"] = "value";

            HasRegexStringOrNotTest(node, "TargetName");
        }

        /// <summary>
        /// 境界値分析: ノード中の TargetInterface 要素の型が文字列でない
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest022()
        {
            var mappingNode = new Dictionary<string, object>();

            mappingNode["Path"] = "value1";

            HasExpectedTypeOrNotTest(mappingNode, "TargetInterface", "value2");
        }

        /// <summary>
        /// 境界値分析: ノード中の TargetInterface 要素の値が不正な正規表現文字列
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest023()
        {
            var node = new Dictionary<string, object>();

            node["Path"] = "value";

            HasRegexStringOrNotTest(node, "TargetInterface");
        }

        /// <summary>
        /// 境界値分析: ノード中の TestName 要素の型が文字列でない
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest024()
        {
            var mappingNode = new Dictionary<string, object>();

            mappingNode["Path"] = "value1";

            HasExpectedTypeOrNotTest(mappingNode, "TestName", "value2");
        }

        /// <summary>
        /// 境界値分析: ノード中に TestName 要素が対応する Path 要素や ProgramName 要素を持たずに存在
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest025()
        {
            var msg = string.Empty;

            var node = new Dictionary<string, object>();

            node["TestName"] = "value1";

            Assert.IsTrue(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual("'TestName' missing the corresponding 'Path'", msg);

            node["Path"] = "value2";

            Assert.IsFalse(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);

            Assert.IsTrue(node.Remove("Path"));

            node["ProgramName"] = "value3";

            Assert.IsFalse(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);

            Assert.IsTrue(node.Remove("ProgramName"));

            node["ProgramId"] = "0x0000000000000000";

            Assert.IsFalse(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);
        }

        /// <summary>
        /// 境界値分析: ノード中の TestName 要素の値が不正なテスト名文字列
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest026()
        {
            var node = new Dictionary<string, object>();

            node["Path"] = "value";

            HasTestNameStringOrNotTest(node, "TestName");
        }

        /// <summary>
        /// 境界値分析: ノード中の ProgramId 要素の型が文字列でない
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest027()
        {
            var mappingNode = new Dictionary<string, object>();

            var key = "ProgramId";

            var value = string.Empty;

            var msg = string.Empty;

            mappingNode[key] = value;

            Assert.IsTrue(new TestNodeInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual($"Invalid value '{value}' for '{key}'", msg);

            value = "value";

            mappingNode[key] = value;

            Assert.IsTrue(new TestNodeInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual($"Invalid value '{value}' for '{key}'", msg);

            value = "0x000000000000000";

            mappingNode[key] = value;

            Assert.IsTrue(new TestNodeInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual($"Invalid value '{value}' for '{key}'", msg);

            value = "gamecard";

            mappingNode[key] = value;

            Assert.IsFalse(new TestNodeInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);

            value = "0x0123456789ABCDEF";

            mappingNode[key] = value;

            Assert.IsFalse(new TestNodeInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);
        }

        /// <summary>
        /// 境界値分析: ノード中で ProgramId 要素と Units 要素が衝突
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest028()
        {
            var msg = string.Empty;

            var node = new Dictionary<string, object>();

            node["ProgramId"] = "gamecard";

            node["Units"] = new List<object>();

            Assert.IsTrue(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual("'Units' conflicting with 'ProgramId'", msg);

            node = new Dictionary<string, object>();

            node["ProgramId"] = "gamecard";

            Assert.IsFalse(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);

            node = new Dictionary<string, object>();

            node["Units"] = new List<object>();

            Assert.IsFalse(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);
        }

        /// <summary>
        /// 不変条件: Units 要素中のある時点での結果が true ならば結果は true
        /// </summary>
        [TestMethod]
        public void IsWrongYamlInvariantTest001()
        {
            var msg = string.Empty;

            var validNode1 = new Dictionary<string, object>();

            var validNode2 = new Dictionary<string, object>();

            var invalidNode = new List<object>();

            var node = new Dictionary<string, object>();

            node["Units"] = new List<object>(new object[] { validNode1, validNode2 });

            Assert.IsFalse(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);

            node["Units"] = new List<object>(new object[] { validNode1, invalidNode });

            Assert.IsTrue(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual("Invalid node type for a test unit", msg);

            node["Units"] = new List<object>(new object[] { validNode1, invalidNode, validNode2 });

            Assert.IsTrue(new TestNodeInfo(node).IsWrongYaml(out msg));
            Assert.AreEqual("Invalid node type for a test unit", msg);
        }

        private static void HasExpectedTypeOrNotTest<T>(Dictionary<string, object> mappingNode, string key, T value)
        {
            var msg = string.Empty;

            mappingNode[key] = new object();

            Assert.IsTrue(new TestNodeInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual("Invalid node type for '" + key + "'", msg);

            mappingNode[key] = value;

            Assert.IsFalse(new TestNodeInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);
        }

        private static void HasExpectedValueOrNotTest<T>(Dictionary<string, object> mappingNode, string key, IEnumerable<T> validValues, IEnumerable<T> invalidValues)
        {
            var msg = string.Empty;

            foreach (var invalidValue in invalidValues)
            {
                mappingNode[key] = invalidValue;

                Assert.IsTrue(new TestNodeInfo(mappingNode).IsWrongYaml(out msg));
                Assert.AreEqual("Invalid value '" + invalidValue + "' for '" + key + "'", msg);
            }

            foreach (var validValue in validValues)
            {
                mappingNode[key] = validValue;

                Assert.IsFalse(new TestNodeInfo(mappingNode).IsWrongYaml(out msg));
                Assert.AreEqual(string.Empty, msg);
            }
        }

        private static void HasPathStringOrNotTest(Dictionary<string, object> mappingNode, string key)
        {
            var msg = string.Empty;

            mappingNode[key] = string.Empty;

            Assert.IsTrue(new TestNodeInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual("Empty string for '" + key + "'", msg);

            mappingNode[key] = "|";

            Assert.IsTrue(new TestNodeInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual("Invalid path string '|' for '" + key + "'", msg);

            mappingNode[key] = "value";

            Assert.IsFalse(new TestNodeInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);
        }

        private static void HasRegexStringOrNotTest(Dictionary<string, object> mappingNode, string key)
        {
            var msg = string.Empty;

            mappingNode[key] = string.Empty;

            Assert.IsFalse(new TestNodeInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);

            mappingNode[key] = "[";

            Assert.IsTrue(new TestNodeInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual("Invalid regex string '[' for '" + key + "'", msg);

            mappingNode[key] = "value";

            Assert.IsFalse(new TestNodeInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);
        }

        private static void HasTestNameStringOrNotTest(
            Dictionary<string, object> mappingNode, string key)
        {
            var msg = string.Empty;

            mappingNode[key] = string.Empty;

            Assert.IsTrue(new TestNodeInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual("Empty string for '" + key + "'", msg);

            mappingNode[key] = "<";

            Assert.IsTrue(new TestNodeInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual(
                "Invalid test name string '<' for '" + key + "'", msg);

            mappingNode[key] = @"\";

            Assert.IsTrue(new TestNodeInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual(
                "Invalid test name string '\\' for '" + key + "'", msg);

            mappingNode[key] = "value";

            Assert.IsFalse(new TestNodeInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);
        }
    }
}
