﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Runtime.InteropServices;
using Microsoft.VisualStudio.TestTools.UnitTesting;

using Nintendo.ServiceFramework;
using Nintendo.ServiceFramework.Validation;

namespace Nintendo.ServiceFrameworkTest.Validation
{
    [TestClass]
    public class StructValidatorTest
    {
        /// <summary>
        /// AssertInvalidStruct の格納用属性
        /// </summary>
        [AttributeUsage(AttributeTargets.All)]
        private class AssertInvalidStructAttribute : Attribute
        {
            public string ErrorCode { get; private set; }
            public string[] AdditionalStrings { get; private set; }

            public AssertInvalidStructAttribute(string errorCode, params object[] additionalStrings)
            {
                this.ErrorCode = errorCode;
                this.AdditionalStrings = additionalStrings.Select(x => x.ToString()).ToArray();
            }
        }

        /// <summary>
        /// 指定した構造体の validate において、指定した例外が起きて失敗しその例外内容が想定通りであることを assert する
        /// </summary>
        /// <typeparam name="ExpectedException">期待する例外型</typeparam>
        /// <param name="t">validate 対象の型(AssertInvalidStructAttribute による修飾が必要)</param>
        private void AssertInvalidStruct<ExpectedException>(Type t)
            where ExpectedException : UserErrorException
        {
            var a = t.GetCustomAttribute<AssertInvalidStructAttribute>();
            AssertionUtility.AssertException<ExpectedException>(
                () => new StructValidator().Validate(t),
                e => AssertionUtility.AssertUserErrorException(e, a.ErrorCode, a.AdditionalStrings));
        }

        /// <summary>
        /// 指定した構造体の validate に成功することを assert する
        /// </summary>
        /// <param name="t">validate 対象の型</param>
        private void AssertValidStruct(Type t)
        {
            AssertionUtility.AssertNoException(() => new StructValidator().Validate(t));
        }

        #region 正当な構造体に関するテスト

        public struct ValidStruct
        {
            public int A;
        }

        public struct NestedValidStruct
        {
            public ValidStruct Nested;
        }

        public struct FixedArrayStruct
        {
            [FixedArray(10)]
            public int[] FixedArray;
        }

        [ExternalStruct(16, 4)]
        public struct ExternalStruct
        {
        }

        private static Type[] ValidStructTypes = new[]
        {
            typeof(ValidStruct),
            typeof(NestedValidStruct),
            typeof(FixedArrayStruct),
            typeof(ExternalStruct),
        };

        [TestMethod]
        public void TestValidStructs()
        {
            foreach (var type in ValidStructTypes)
            {
                AssertValidStruct(type);
            }
        }

        #endregion

        #region 不正な構造体に関するテスト

        [AssertInvalidStruct(@"SF-ST-InvalidAutoLayout", typeof(StructLayoutAttribute), LayoutKind.Auto)]
        [StructLayout(LayoutKind.Auto)]
        public struct AutoStructLayout
        {
        }

        [AssertInvalidStruct(@"SF-ST-InvalidExternalStruct:HasFields", typeof(ExternalStruct))]
        [ExternalStruct(16, 4)]
        public struct ExternalStructWithField
        {
            public int A;
        }

        private static Type[] InvalidStructTypes = new[]
        {
            typeof(AutoStructLayout),
            typeof(ExternalStructWithField),
        };

        [TestMethod]
        public void TestInvalidStructs()
        {
            foreach (var type in InvalidStructTypes)
            {
                AssertInvalidStruct<InvalidStructException>(type);
            }
        }

        #endregion

        #region フィールドに関するテスト

        [AssertInvalidStruct("SF-ST-NotPublicField", "public")]
        public struct InternalField
        {
            internal int A;

            // 非 public メンバへの警告抑制
            public int AProperty
            {
                get
                {
                    return A;
                }
                set
                {
                    this.A = value;
                }
            }
        }

        [AssertInvalidStruct("SF-ST-NotPublicField", "public")]
        public struct PrivateField
        {
            private int A;

            // 非 public メンバへの警告抑制
            public int AProperty
            {
                get
                {
                    return A;
                }
                set
                {
                    this.A = value;
                }
            }
        }

        [AssertInvalidStruct("SF-ST-NotPublicField", "public")]
        public struct AutoPropertyField
        {
            // 自動生成プロパティフィールド
            // 若干エラーメッセージが難解(「フィールドは public である必要がある」)だが、
            // 必要に応じてドキュメントで対応予定。
            public int A { get; set; }
        }

        [AssertInvalidStruct(@"SF-ST-NotSupported:DynamicArray", typeof(FixedArrayAttribute))]
        public struct DynamicArrayStruct
        {
            public int[] A;
        }

        [AssertInvalidStruct(@"SF-ST-NotSupported:MultiRankArray")]
        public struct MultiRankArrayStruct
        {
            [FixedArray(10)]
            public int[][] A;
        }

        [AssertInvalidStruct(@"SF-ST-InvalidField:FixedArrayedScalar", typeof(FixedArrayAttribute), "[]")]
        public struct FixedArrayedScalarStruct
        {
            [FixedArray(10)]
            public int A;
        }

        private static Type[] InvalidFieldTypes = new[]
        {
            typeof(InternalField),
            typeof(PrivateField),
            typeof(AutoPropertyField),
            typeof(DynamicArrayStruct),
            typeof(MultiRankArrayStruct),
            typeof(FixedArrayedScalarStruct),
        };

        [TestMethod]
        public void TestInvalidFields()
        {
            foreach (var type in InvalidFieldTypes)
            {
                AssertInvalidStruct<InvalidFieldException>(type);
            }
        }

        #endregion
    }
}
