﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Reflection;
using Microsoft.VisualStudio.TestTools.UnitTesting;

using Nintendo.ServiceFramework;

namespace Nintendo.ServiceFrameworkTest
{
    [TestClass]
    public class AttributesTest
    {
        #region InterfaceIdAttribute

        private interface INoInterfaceIdInterface
        {
        }

        [InterfaceId(100)]
        private interface IInterfaceIdInterface
        {
        }

        [TestMethod]
        public void TestInterfaceIdAttribute()
        {
            Assert.IsNull(InterfaceIdAttribute.GetId(typeof(INoInterfaceIdInterface)));
            Assert.AreEqual(100, InterfaceIdAttribute.GetId(typeof(IInterfaceIdInterface)));
        }

        #endregion

        #region MethodIdAttribute

        private interface IMethodIdInterface
        {
            void A();

            [MethodId(100)]
            void B();
        }

        [TestMethod]
        public void TestMethodIdAttribute()
        {
            Assert.IsNull(MethodIdAttribute.GetId(typeof(IMethodIdInterface).GetMethod("A")));
            Assert.AreEqual(100, MethodIdAttribute.GetId(typeof(IMethodIdInterface).GetMethod("B")));
        }

        #endregion

        #region LargeDataAttribute

        public struct UnspecifiedSizeData
        {
            public int A;
        }

        [LargeData]
        public struct DefaultSizeData
        {
            public int A;
        }

        [LargeData(false)]
        public struct NotLargeSizeData
        {
            public int A;
        }

        [LargeData(true)]
        public struct LargeSizeData
        {
            public int A;
        }

        [AttributeUsage(AttributeTargets.Method)]
        internal class ExpectedTestLargeDataAttribute : Attribute
        {
            public bool Unspecified { get; private set; }
            public bool Default { get; private set; }
            public bool NotLarge { get; private set; }
            public bool Large { get; private set; }

            public ExpectedTestLargeDataAttribute(bool u, bool d, bool f, bool t)
            {
                this.Unspecified = u;
                this.Default = d;
                this.NotLarge = f;
                this.Large = t;
            }
        }

        private interface ITestLargeDataInterface
        {
            [ExpectedTestLargeData(false, true, false, true)]
            void Unspecified(
                UnspecifiedSizeData u,
                [LargeData] UnspecifiedSizeData d,
                [LargeData(false)] UnspecifiedSizeData f,
                [LargeData(true)] UnspecifiedSizeData t);

            [ExpectedTestLargeData(true, true, false, true)]
            void Default(
                DefaultSizeData u,
                [LargeData] DefaultSizeData d,
                [LargeData(false)] DefaultSizeData f,
                [LargeData(true)] DefaultSizeData t);

            [ExpectedTestLargeData(false, true, false, true)]
            void NotLarge(
                NotLargeSizeData u,
                [LargeData] NotLargeSizeData d,
                [LargeData(false)] NotLargeSizeData f,
                [LargeData(true)] NotLargeSizeData t);

            [ExpectedTestLargeData(true, true, false, true)]
            void Large(
                LargeSizeData u,
                [LargeData] LargeSizeData d,
                [LargeData(false)] LargeSizeData f,
                [LargeData(true)] LargeSizeData t);

            [ExpectedTestLargeData(false, true, false, true)]
            void UnspecifiedOut(
                nn.sf.Out<UnspecifiedSizeData> u,
                [LargeData] nn.sf.Out<UnspecifiedSizeData> d,
                [LargeData(false)] nn.sf.Out<UnspecifiedSizeData> f,
                [LargeData(true)] nn.sf.Out<UnspecifiedSizeData> t);

            [ExpectedTestLargeData(true, true, false, true)]
            void DefaultOut(
                nn.sf.Out<DefaultSizeData> u,
                [LargeData] nn.sf.Out<DefaultSizeData> d,
                [LargeData(false)] nn.sf.Out<DefaultSizeData> f,
                [LargeData(true)] nn.sf.Out<DefaultSizeData> t);

            [ExpectedTestLargeData(false, true, false, true)]
            void NotLargeOut(
                nn.sf.Out<NotLargeSizeData> u,
                [LargeData] nn.sf.Out<NotLargeSizeData> d,
                [LargeData(false)] nn.sf.Out<NotLargeSizeData> f,
                [LargeData(true)] nn.sf.Out<NotLargeSizeData> t);

            [ExpectedTestLargeData(true, true, false, true)]
            void LargeOut(
                nn.sf.Out<LargeSizeData> u,
                [LargeData] nn.sf.Out<LargeSizeData> d,
                [LargeData(false)] nn.sf.Out<LargeSizeData> f,
                [LargeData(true)] nn.sf.Out<LargeSizeData> t);
        }

        [TestMethod]
        public void TestLargeDataAttribute()
        {
            foreach (var m in typeof(ITestLargeDataInterface).GetMethods())
            {
                var u = m.GetParameters().Single(p => p.Name == "u");
                var d = m.GetParameters().Single(p => p.Name == "d");
                var f = m.GetParameters().Single(p => p.Name == "f");
                var t = m.GetParameters().Single(p => p.Name == "t");
                if (u.ParameterType.GetSfParameterElementType() == typeof(UnspecifiedSizeData))
                {
                    Assert.IsTrue(!LargeDataAttribute.IsSpecifiedLargeDataParameter(u));
                }
                else
                {
                    Assert.IsTrue(LargeDataAttribute.IsSpecifiedLargeDataParameter(u));
                }
                Assert.IsTrue(LargeDataAttribute.IsSpecifiedLargeDataParameter(d));
                Assert.IsTrue(LargeDataAttribute.IsSpecifiedLargeDataParameter(f));
                Assert.IsTrue(LargeDataAttribute.IsSpecifiedLargeDataParameter(t));
                var expected = m.GetCustomAttribute<ExpectedTestLargeDataAttribute>();
                Assert.IsTrue(LargeDataAttribute.IsLargeDataParameter(u) == expected.Unspecified);
                Assert.IsTrue(LargeDataAttribute.IsLargeDataParameter(d) == expected.Default);
                Assert.IsTrue(LargeDataAttribute.IsLargeDataParameter(f) == expected.NotLarge);
                Assert.IsTrue(LargeDataAttribute.IsLargeDataParameter(t) == expected.Large);
            }
        }

        #endregion
    }
}
