﻿using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System.Net;
using System.Net.Sockets;
using Nintendo.ControlTarget;
using Nintendo.ControlTarget.ConsoleShell;
using System.IO;
using System.Text;
using System.Linq;
using System.Collections.Generic;
using System.Threading.Tasks;
using System.Threading.Tasks.Dataflow;
using System.Diagnostics;

namespace RunnerToolsTest
{
    [TestClass]
    public class ConsoleShellTest
    {
        private byte[] ApplicationLaunchRequest = new byte[] {
                0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                0x01, 0x00, 0x00, 0x00, 0x1C, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF,
                0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x61, 0x62, 0x20, 0x63
            };

        [TestMethod]
        public void TestConcreteConsoleShell()
        {
            using (var tma = new TargetManagerAccessor())
            {
                using (var messenger = new ConsoleShellMessenger(tma.FindService("iywys@$csForRunnerTools")))
                using (var cs = new ConsoleShellAccessor(messenger))
                {

                    var applicationWaiter = cs.LaunchApplication(
                        new TargetProgramInfo("D:/home/siglo/sdk/Programs/Chris/Outputs/NX-NXFP2-a32/TargetTools/ExportPartition/Develop/ExportPartition.nca", true),
                        new string[] { "--output", "aaa", "--help" });

                    applicationWaiter.Wait(3000);

                    Assert.IsTrue(applicationWaiter.IsCompleted);
                    Assert.AreEqual(0, applicationWaiter.Result);
                }
            }
        }

        [TestMethod]
        public void TestConsoleShell()
        {
            var messenger = new DummyConsoleShellMessenger();
            var cs = new ConsoleShellAccessor(messenger);
            var dummyServer = new ActionBlock<ConsoleShellRequest>(request =>
            {
                messenger.DummyReceivePort.Post(new SuccessResponse(request.RequestId));
                messenger.DummyReceivePort.Post(new ApplicationExitResponse(request.RequestId, 123));
            });

            messenger.DummySendPort.LinkTo(dummyServer);

            Assert.AreEqual(0, messenger.DummySendPort.Count);

            var applicationWaiter = cs.LaunchApplication(new TargetProgramInfo("test.nca", true), new string[] { "--help" });

            applicationWaiter.Wait();

            Assert.AreEqual(123, applicationWaiter.Result);

            messenger.DummyReceivePort.Complete();
            messenger.DummyReceivePort.Completion.Wait();
            dummyServer.Complete();
            dummyServer.Completion.Wait();
            cs.Dispose();
        }

        [TestMethod]
        public void TestLaunchApplicationWithReversedResponse()
        {
            var messenger = new DummyConsoleShellMessenger();
            var cs = new ConsoleShellAccessor(messenger);
            var dummyServer = new ActionBlock<ConsoleShellRequest>(request =>
            {
                messenger.DummyReceivePort.Post(new ApplicationExitResponse(request.RequestId, 123));
                messenger.DummyReceivePort.Post(new SuccessResponse(request.RequestId));
            });

            messenger.DummySendPort.LinkTo(dummyServer);

            Assert.AreEqual(0, messenger.DummySendPort.Count);

            var applicationWaiter = cs.LaunchApplication(new TargetProgramInfo(PathUtility.FindSigloRoot() + "/NintendoSdkRootMark", true), new string[] { "--help" });

            applicationWaiter.Wait();

            Assert.AreEqual(123, applicationWaiter.Result);

            messenger.DummyReceivePort.Complete();
            messenger.DummyReceivePort.Completion.Wait();
            dummyServer.Complete();
            dummyServer.Completion.Wait();
            cs.Dispose();
        }

        [TestMethod]
        public void TestLaunchApplicationCommandData()
        {
            var command = new Nintendo.ControlTarget.ConsoleShell.LaunchApplicationCommand("a", new string[] {"b", "c"}, new LaunchApplicationOption());
            var commandBinary = command.GetCommand(0xffffffffffffffffUL);

            foreach (var b in commandBinary)
            {
                Console.Write(string.Format("0x{0,0:X2}, ", b));
            }
            Console.WriteLine();

            Assert.AreEqual(ApplicationLaunchRequest.Count(), commandBinary.Count());

            CollectionAssert.AreEqual(ApplicationLaunchRequest, commandBinary);
        }

        [TestMethod]
        public void TestTerminateApplicationCommandData()
        {
            var command = new TerminateAppicationCommand();
            var commandBinary = command.GetCommand(0xffffffffffffffffUL);

            foreach (var b in commandBinary)
            {
                Console.Write(string.Format("0x{0,0:X2}, ", b));
            }
            Console.WriteLine();

            var expected = new byte[] {
                0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            };

            Assert.AreEqual(expected.Count(), commandBinary.Count());

            CollectionAssert.AreEqual(expected, commandBinary);
        }

        [TestMethod]
        public void TestCommandResponse()
        {
            var responseBinary = new byte[] {
                0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
            };

            var header = new CommandResponseHeader(responseBinary);

            Assert.AreEqual(0xFFFFFFFFFFFFFFFF, header.RequestId);
            Assert.AreEqual(1, header.CommandId);
            Assert.AreEqual(4U, header.BodySize);
        }

        private IConsoleShellResponse MakeDummyResponse(ulong requestId)
        {
            var responseBinary = new byte[] {
                0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
            };

            var header = new CommandResponseHeader(requestId, 1, 8);

            return new FailureResponse(header, new byte[] { 0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00 });
        }

        private IConsoleShellResponse MakeDummyResponse()
        {
            var responseBinary = new byte[] {
                0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
            };

            var header = new CommandResponseHeader(responseBinary);

            return new FailureResponse(header, new byte[] { 0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00 });
        }

        [TestMethod]
        public void TestConsoleShellMessenger()
        {
            var messenger = new DummyConsoleShellMessenger();

            bool received = false;

            var receiver = new ActionBlock<IConsoleShellResponse>(response =>
            {
                received = true;
            });
            messenger.ReceivePort.LinkTo(receiver);

            messenger.DummyReceivePort.Post(MakeDummyResponse());

            messenger.DummyReceivePort.Complete();
            messenger.DummyReceivePort.Completion.Wait();
            receiver.Complete();
            receiver.Completion.Wait();

            Assert.IsTrue(received);
        }
    }

    public class DummyConsoleShellMessenger : IConsoleShellMessenger
    {
        public DummyConsoleShellMessenger()
        {
            DummyReceivePort = new BufferBlock<IConsoleShellResponse>();
            DummySendPort = new BufferBlock<ConsoleShellRequest>();
            ReceivePort = DummyReceivePort;
            SendPort = DummySendPort;
        }

        public BufferBlock<IConsoleShellResponse> DummyReceivePort { get; private set; }
        public BufferBlock<ConsoleShellRequest> DummySendPort { get; private set; }
    }
}
