﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.ApplicationControlProperty;
using System;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Xml;
using System.Xml.XPath;
using System.Xml.Linq;
using System.Xml.Serialization;

namespace NacpTest
{
    [TestClass]
    public class ApplicationControlPropertyModelTest
    {
        public TestContext TestContext { get; set; }

        private static UInt64 GetProgramId(XDocument xml)
        {
            var xmlCoreList = xml.Descendants("Core");
            Assert.AreEqual(1, xmlCoreList.Count());
            var xmlCore = xmlCoreList.First();

            var xmlId =
                from pid in xmlCore.Descendants("ProgramId").
                     Concat(xmlCore.Descendants("SystemProgramId")).
                     Concat(xmlCore.Descendants("ApplicationId"))
                select (string)pid;
            return Convert.ToUInt64(Enumerable.First(xmlId), 16);
        }

        private static UInt64 GetContentMetaId(XDocument xml)
        {
            var programId = GetProgramId(xml);
            var appNode = xml.XPathSelectElement("//Application");
            if (appNode != null)
            {
                var appIdNode = appNode.XPathSelectElement("ApplicationId");
                return (appIdNode != null) ? Convert.ToUInt64(appIdNode.Value, 16) : programId;
            }
            else
            {
                return programId;
            }
        }

        private static UInt64 GetDefaultAddOnContentBaseId(UInt64 programId)
        {
            return programId + 0x1000;
        }

        private ApplicationControlPropertyModel ReadXml(string filePath)
        {
            var document = new XmlDocument();
            document.Load(filePath);
            var node = document.SelectSingleNode("//Application");
            var reader = XmlReader.Create(new StringReader(node.OuterXml));

            XmlSerializer serializer = new System.Xml.Serialization.XmlSerializer(typeof(ApplicationControlPropertyModel));
            serializer.UnknownElement += serializer_UnknownElement;
            return (ApplicationControlPropertyModel)serializer.Deserialize(reader);
        }

        static void serializer_UnknownElement(object sender, XmlElementEventArgs e)
        {
            throw new ArgumentException(string.Format("未定義の要素が使われているか、要素の定義が重複しています: '{0}'", e.Element.LocalName));
        }

        private void CheckDefaultMeta(string metaFilePath)
        {
            var inModel = ReadXml(metaFilePath);
            var xDocument = XDocument.Load(metaFilePath);
            var programId = GetProgramId(xDocument);
            var metaId = GetContentMetaId(xDocument);
            var bytes = inModel.MakePropertyBytes(programId, GetDefaultAddOnContentBaseId(metaId));
            var outModel = ApplicationControlPropertyModel.PropertyBytesToModel(bytes);
            Assert.IsTrue(outModel.Title.Count == 0);
            Assert.IsTrue(outModel.Isbn == string.Empty);
            Assert.IsTrue(outModel.StartupUserAccount == "None");
            Assert.IsTrue(outModel.Attribute.Count == 0);
            Assert.IsTrue(outModel.ParentalControl.Count == 0);
            Assert.IsTrue(outModel.SupportedLanguage.Count == 0);
            Assert.IsTrue(outModel.Screenshot == "Allow");
            Assert.IsTrue(outModel.VideoCapture == "Enable");
            Assert.IsTrue(outModel.PresenceGroupId == "0x0005000c10000001");
            Assert.IsTrue(outModel.DisplayVersion == "");
            Assert.IsTrue(outModel.Rating.Count == 0);
            Assert.IsTrue(outModel.DataLossConfirmation == "None");
            Assert.IsTrue(outModel.PlayLogPolicy == "All");
            Assert.IsTrue(outModel.SaveDataOwnerId == "0x0005000c10000001");
            Assert.IsTrue(outModel.UserAccountSaveDataSize == "0x0000000000000000");
            Assert.IsTrue(outModel.UserAccountSaveDataJournalSize == "0x0000000000000000");
            Assert.IsTrue(outModel.DeviceSaveDataSize == "0x0000000000000000");
            Assert.IsTrue(outModel.DeviceSaveDataJournalSize == "0x0000000000000000");
            Assert.IsTrue(outModel.BcatDeliveryCacheStorageSize == "0x0000000000000000");
            Assert.IsTrue(outModel.ApplicationErrorCodeCategory == string.Empty);
            Assert.IsTrue(outModel.AddOnContentBaseId == "0x0005000c10001001");
            Assert.IsTrue(outModel.LogoType == "LicensedByNintendo");
            Assert.IsTrue(outModel.LocalCommunicationId.Count == 1);
            Assert.AreEqual(outModel.LocalCommunicationId[0], string.Format("0x{0:x16}", programId));
            Assert.IsTrue(outModel.AddOnContentRegistrationType == "OnDemand");
            Assert.IsTrue(outModel.UserAccountSaveDataSizeMax == "0x0000000000000000");
            Assert.IsTrue(outModel.UserAccountSaveDataJournalSizeMax == "0x0000000000000000");
            Assert.IsTrue(outModel.DeviceSaveDataSizeMax == "0x0000000000000000");
            Assert.IsTrue(outModel.DeviceSaveDataJournalSizeMax == "0x0000000000000000");
            Assert.IsTrue(outModel.TemporaryStorageSize == "0x0000000000000000");
            Assert.IsTrue(outModel.CacheStorageSize == "0x0000000000000000");
            Assert.IsTrue(outModel.CacheStorageJournalSize == "0x0000000000000000");
        }

        private void CheckDescribeAllMeta(string metaFilePath)
        {
            var inModel = ReadXml(metaFilePath);
            var xDocument = XDocument.Load(metaFilePath);
            var programId = GetProgramId(xDocument);
            var metaId = GetContentMetaId(xDocument);
            var bytes = inModel.MakePropertyBytes(programId, GetDefaultAddOnContentBaseId(metaId));
            var outModel = ApplicationControlPropertyModel.PropertyBytesToModel(bytes);
            Assert.IsTrue(outModel.Title.Count == 2);
            Assert.IsTrue(outModel.Title[0].Language == "AmericanEnglish");
            Assert.IsTrue(outModel.Title[0].Name == "Title");
            Assert.IsTrue(outModel.Title[0].Publisher == "Publisher");
            Assert.IsTrue(outModel.Title[1].Language == "Japanese");
            Assert.IsTrue(outModel.Title[1].Name == "タイトル");
            Assert.IsTrue(outModel.Title[1].Publisher == "パブリッシャー");
            Assert.IsTrue(outModel.Isbn == "012-3-45-678901-2");
            Assert.IsTrue(outModel.StartupUserAccount == "RequiredWithNetworkServiceAccountAvailable");
            Assert.IsTrue(outModel.Attribute.Count == 1);
            Assert.IsTrue(outModel.Attribute[0] == "Demo");
            Assert.IsTrue(outModel.ParentalControl.Count == 1);
            Assert.IsTrue(outModel.ParentalControl[0] == "FreeCommunication");
            Assert.IsTrue(outModel.SupportedLanguage.Count == 2);
            Assert.IsTrue(outModel.SupportedLanguage[0] == "AmericanEnglish");
            Assert.IsTrue(outModel.SupportedLanguage[1] == "Japanese");
            Assert.IsTrue(outModel.Screenshot == "Deny");
            Assert.IsTrue(outModel.VideoCapture == "Disable");
            Assert.IsTrue(outModel.PresenceGroupId == "0x0005000c10000002");
            Assert.IsTrue(outModel.DisplayVersion == "1.0.0");
            Assert.IsTrue(outModel.Rating.Count == 2);
            Assert.IsTrue(outModel.Rating[0].Organization == "CERO");
            Assert.IsTrue(outModel.Rating[0].Age == 18);
            Assert.IsTrue(outModel.Rating[1].Organization == "ESRB");
            Assert.IsTrue(outModel.Rating[1].Age == 17);
            Assert.IsTrue(outModel.DataLossConfirmation == "Required");
            Assert.IsTrue(outModel.PlayLogPolicy == "All");
            Assert.IsTrue(outModel.SaveDataOwnerId == "0x0005000c10000001");
            Assert.IsTrue(outModel.UserAccountSaveDataSize == "0x0000000000400000");
            Assert.IsTrue(outModel.UserAccountSaveDataJournalSize == "0x0000000000100000");
            Assert.IsTrue(outModel.DeviceSaveDataSize == "0x0000000001000000");
            Assert.IsTrue(outModel.DeviceSaveDataJournalSize == "0x0000000000300000");
            Assert.IsTrue(outModel.BcatDeliveryCacheStorageSize == "0x0000000000800000");
            Assert.IsTrue(outModel.ApplicationErrorCodeCategory == "ABC12");
            Assert.IsTrue(outModel.AddOnContentBaseId == "0x0005000c10001000");
            Assert.IsTrue(outModel.LogoType == "LicensedByNintendo");
            Assert.IsTrue(outModel.LocalCommunicationId.Count == 3);
            Assert.IsTrue(outModel.LocalCommunicationId.Contains("0x0100000c10000000"));
            Assert.IsTrue(outModel.LocalCommunicationId.Contains("0x0100000c10000001"));
            Assert.IsTrue(outModel.LocalCommunicationId.Contains("0x0005000c10000001"));
            Assert.IsTrue(outModel.AddOnContentRegistrationType == "AllOnLaunch");
            Assert.IsTrue(outModel.UserAccountSaveDataSizeMax == "0x0000000014000000");
            Assert.IsTrue(outModel.UserAccountSaveDataJournalSizeMax == "0x000000000f000000");
            Assert.IsTrue(outModel.DeviceSaveDataSizeMax == "0x000000000a000000");
            Assert.IsTrue(outModel.DeviceSaveDataJournalSizeMax == "0x0000000005000000");
            Assert.IsTrue(outModel.TemporaryStorageSize == "0x0000000001200000");
            Assert.IsTrue(outModel.CacheStorageSize == "0x0000000001400000");
            Assert.IsTrue(outModel.CacheStorageJournalSize == "0x0000000000500000");

        }

        private void CheckApplicationIdMeta(string metaFilePath)
        {
            var inModel = ReadXml(metaFilePath);
            var xDocument = XDocument.Load(metaFilePath);
            var programId = GetProgramId(xDocument);
            var metaId = GetContentMetaId(xDocument);
            var bytes = inModel.MakePropertyBytes(programId, GetDefaultAddOnContentBaseId(metaId));
            var outModel = ApplicationControlPropertyModel.PropertyBytesToModel(bytes);
            Assert.IsTrue(outModel.ProgramIndex == "1");
            Assert.IsTrue(outModel.PresenceGroupId == "0x0005000c10000000");
            Assert.IsTrue(outModel.SaveDataOwnerId == "0x0005000c10000000");
            Assert.IsTrue(outModel.AddOnContentBaseId == "0x0005000c10001000");
            Assert.IsTrue(outModel.LocalCommunicationId.Count == 1);
            Assert.IsTrue(outModel.LocalCommunicationId.Contains("0x0005000c10000000"));
        }

        [TestMethod]
        public void MakePropertyTest()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var describeAllMetaFile = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\NacpTest\\Meta\\describe_all.nmeta";
            var defaultMetaFile = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\NacpTest\\Meta\\use_default.nmeta";
            var applicationIdTestMetaFile = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\NacpTest\\Meta\\applicationIdTest.nmeta";
            CheckDescribeAllMeta(describeAllMetaFile);
            CheckDefaultMeta(defaultMetaFile);
            CheckApplicationIdMeta(applicationIdTestMetaFile);
        }
    }
}
