﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.MakeVisualStudioProject.Converter;

namespace Nintendo.MakeVisualStudioProjectTest.Converter
{
    [TestClass]
    public class ItemMetadataConverterTest
    {
        [TestMethod]
        public void TestConvertToItemMetadata()
        {
            var converter = new TestItemMetadataConverter();
            var testInstances = new Tuple<IEnumerable<OptionValue>, IEnumerable<ItemMetadata>, IEnumerable<OptionValue>>[]
            {
                new Tuple<IEnumerable<OptionValue>, IEnumerable<ItemMetadata>, IEnumerable<OptionValue>>(
                    new[] { new OptionValue("/a") },
                    new[] { new ItemMetadata("aName", "aValue") },
                    new OptionValue[0]),
                new Tuple<IEnumerable<OptionValue>, IEnumerable<ItemMetadata>, IEnumerable<OptionValue>>(
                    new[] { new OptionValue("/b1", "b1Arg"), new OptionValue("/b2", "b2Arg") },
                    new[] { new ItemMetadata("b1Name", "b1Arg"), new ItemMetadata("b2Name", "b2Arg") },
                    new OptionValue[0]),
                new Tuple<IEnumerable<OptionValue>, IEnumerable<ItemMetadata>, IEnumerable<OptionValue>>(
                    new[] { new OptionValue("/b1"), new OptionValue("/b2") },
                    new[] { new ItemMetadata("b1Name", string.Empty) },
                    new[] { new OptionValue("/b2") }),
                new Tuple<IEnumerable<OptionValue>, IEnumerable<ItemMetadata>, IEnumerable<OptionValue>>(
                    new[] { new OptionValue("/c1", "c1Arg"), new OptionValue("/c2", "c2Arg") },
                    new[] { new ItemMetadata("c1Name", "c1Value"), new ItemMetadata("c1ArgName", "c1Arg"),
                            new ItemMetadata("c2Name", "c2Value"), new ItemMetadata("c2ArgName", "c2Arg") },
                    new OptionValue[0]),
                new Tuple<IEnumerable<OptionValue>, IEnumerable<ItemMetadata>, IEnumerable<OptionValue>>(
                    new[] { new OptionValue("/c2"), new OptionValue("/c1") },
                    new[] { new ItemMetadata("c1Name", "c1Value") },
                    new[] { new OptionValue("/c2") }),
                new Tuple<IEnumerable<OptionValue>, IEnumerable<ItemMetadata>, IEnumerable<OptionValue>>(
                    new[] { new OptionValue("/d1", "arg1;arg2;arg3"), new OptionValue("/d2", "a r g"), new OptionValue("/d1", "arg4") },
                    new[] { new ItemMetadata("d1Name", "arg1;arg2;arg3;arg4;%(d1Name)"), new ItemMetadata("d2Name", "a r g;%(d2Name)") },
                    new OptionValue[0]),
                new Tuple<IEnumerable<OptionValue>, IEnumerable<ItemMetadata>, IEnumerable<OptionValue>>(
                    new[] { new OptionValue("/d2"), new OptionValue("/d1"), new OptionValue("/unknownSwitch"), new OptionValue("/d1", "d1arg") },
                    new[] { new ItemMetadata("d1Name", "d1arg;%(d1Name)") },
                    new[] { new OptionValue("/d2"), new OptionValue("/unknownSwitch") }),
                new Tuple<IEnumerable<OptionValue>, IEnumerable<ItemMetadata>, IEnumerable<OptionValue>>(
                    new[] { new OptionValue("/e1", ";;arg1;;arg2;;arg3;;"), new OptionValue("/e2", "arg1  arg2") },
                    new[] { new ItemMetadata("e1Name", "e1Value"), new ItemMetadata("e1ArgName", ";;arg1;;arg2;;arg3;;;%(e1ArgName)"),
                            new ItemMetadata("e2Name", "e2Value"), new ItemMetadata("e2ArgName", "arg1  arg2;%(e2ArgName)") },
                    new OptionValue[0]),
                new Tuple<IEnumerable<OptionValue>, IEnumerable<ItemMetadata>, IEnumerable<OptionValue>>(
                    new[] { new OptionValue("/e1"), new OptionValue("/e2"), new OptionValue("/e2", "e2Arg"), new OptionValue("/e2"), new OptionValue("/e3") },
                    new[] { new ItemMetadata("e1Name", "e1Value"),
                            new ItemMetadata("e2Name", "e2Value"), new ItemMetadata("e2ArgName", "e2Arg;%(e2ArgName)") },
                    new[] { new OptionValue("/e2"), new OptionValue("/e2"), new OptionValue("/e3") }),
                new Tuple<IEnumerable<OptionValue>, IEnumerable<ItemMetadata>, IEnumerable<OptionValue>>(
                    new[] { new OptionValue("/g", "arg") },
                    new[] { new ItemMetadata("gName", "gValue"), new ItemMetadata("gArgName", "arg") },
                    new OptionValue[0]),
            };

            foreach (var testInstance in testInstances)
            {
                var input = testInstance.Item1;
                var expected = testInstance.Item2;
                var expectedUnknowns = testInstance.Item3;

                IEnumerable<OptionValue> actualUnknowns;
                var actual = converter.ConvertToItemMetadata(input, out actualUnknowns);

                Assert.IsTrue(
                    actual.SequenceEqual(expected),
                    "expected = {0}, actual = {1}",
                    Utility.ToArrayString(expected),
                    Utility.ToArrayString(actual));
                Assert.IsTrue(
                    actualUnknowns.SequenceEqual(expectedUnknowns),
                    "expected = {0}, actual = {1}",
                    Utility.ToArrayString(expectedUnknowns),
                    Utility.ToArrayString(actualUnknowns));
            }
        }

        [TestMethod]
        public void TestConvertToOptionValue()
        {
            var converter = new TestItemMetadataConverter();
            var testInstances = new Tuple<IEnumerable<ItemMetadata>, IEnumerable<OptionValue>, IEnumerable<ItemMetadata>>[]
            {
                new Tuple<IEnumerable<ItemMetadata>, IEnumerable<OptionValue>, IEnumerable<ItemMetadata>>(
                    new[] { new ItemMetadata("aName", "aValue") },
                    new[] { new OptionValue("/a") },
                    new ItemMetadata[0]),
                new Tuple<IEnumerable<ItemMetadata>, IEnumerable<OptionValue>, IEnumerable<ItemMetadata>>(
                    new[] { new ItemMetadata("b1Name", "b1Arg"), new ItemMetadata("b2Name", "b2Arg") },
                    new[] { new OptionValue("/b1", "b1Arg"), new OptionValue("/b2", "b2Arg") },
                    new ItemMetadata[0]),
                new Tuple<IEnumerable<ItemMetadata>, IEnumerable<OptionValue>, IEnumerable<ItemMetadata>>(
                    new[] { new ItemMetadata("b1Name", string.Empty), new ItemMetadata("b2Name", string.Empty) },
                    new[] { new OptionValue("/b1") },
                    new[] { new ItemMetadata("b2Name", string.Empty) }),
                new Tuple<IEnumerable<ItemMetadata>, IEnumerable<OptionValue>, IEnumerable<ItemMetadata>>(
                    new[] { new ItemMetadata("c1Name", "c1Value"), new ItemMetadata("c1ArgName", "c1Arg"),
                            new ItemMetadata("c2Name", "c2Value"), new ItemMetadata("c2ArgName", "c2Arg") },
                    new[] { new OptionValue("/c1", "c1Arg"), new OptionValue("/c2", "c2Arg") },
                    new ItemMetadata[0]),
                new Tuple<IEnumerable<ItemMetadata>, IEnumerable<OptionValue>, IEnumerable<ItemMetadata>>(
                    new[] { new ItemMetadata("c1Name", "c1Value"), new ItemMetadata("c2Name", "c2Value") },
                    new[] { new OptionValue("/c1") },
                    new[] { new ItemMetadata("c2Name", "c2Value") }),
                new Tuple<IEnumerable<ItemMetadata>, IEnumerable<OptionValue>, IEnumerable<ItemMetadata>>(
                    new[] { new ItemMetadata("d1Name", "arg1;arg2;arg3;arg4;%(d1Name)"), new ItemMetadata("d2Name", "a r g;%(d2Name)") },
                    new[] { new OptionValue("/d1", "arg1"), new OptionValue("/d1", "arg2"), new OptionValue("/d1", "arg3"), new OptionValue("/d1", "arg4"),
                            new OptionValue("/d2", "a r g") },
                    new ItemMetadata[0]),
                new Tuple<IEnumerable<ItemMetadata>, IEnumerable<OptionValue>, IEnumerable<ItemMetadata>>(
                    new[] { new ItemMetadata("d1Name", string.Empty), new ItemMetadata("unknownName", string.Empty), new ItemMetadata("d2Name", ";") },
                    new[] { new OptionValue("/d1") },
                    new[] { new ItemMetadata("unknownName", string.Empty), new ItemMetadata("d2Name", ";") }),
                new Tuple<IEnumerable<ItemMetadata>, IEnumerable<OptionValue>, IEnumerable<ItemMetadata>>(
                    new[] { new ItemMetadata("e1Name", "e1Value"), new ItemMetadata("e1ArgName", ";;arg1;;arg2;;arg3;;;%(e1ArgName)"),
                            new ItemMetadata("e2Name", "e2Value"), new ItemMetadata("e2ArgName", "%(e2ArgName)") },
                    new[] { new OptionValue("/e1"), new OptionValue("/e1"), new OptionValue("/e1", "arg1"),
                            new OptionValue("/e1"), new OptionValue("/e1", "arg2"),
                            new OptionValue("/e1"), new OptionValue("/e1", "arg3"),
                            new OptionValue("/e1"), new OptionValue("/e1") },
                    new ItemMetadata[0]),
                new Tuple<IEnumerable<ItemMetadata>, IEnumerable<OptionValue>, IEnumerable<ItemMetadata>>(
                    new[] { new ItemMetadata("e1Name", "e1Value"), new ItemMetadata("e1ArgName", "%(e1ArgName)"),
                            new ItemMetadata("e2Name", "e2Value"), new ItemMetadata("e2ArgName", "%(e2ArgName)") },
                    new OptionValue[0],
                    new ItemMetadata[0]),
                new Tuple<IEnumerable<ItemMetadata>, IEnumerable<OptionValue>, IEnumerable<ItemMetadata>>(
                    new[] { new ItemMetadata("gName", "gValue"), new ItemMetadata("gArgName", "arg") },
                    new[] { new OptionValue("/g", "arg") },
                    new ItemMetadata[0]),
            };

            foreach (var testInstance in testInstances)
            {
                var input = testInstance.Item1;
                var expected = testInstance.Item2;
                var expectedUnknowns = testInstance.Item3;

                IEnumerable<ItemMetadata> actualUnknowns;
                var actual = converter.ConvertToOptionValue(input, out actualUnknowns);

                Assert.IsTrue(
                    actual.SequenceEqual(expected),
                    "expected = {0}, actual = {1}",
                    Utility.ToArrayString(expected),
                    Utility.ToArrayString(actual));
                Assert.IsTrue(
                    actualUnknowns.SequenceEqual(expectedUnknowns),
                    "expected = {0}, actual = {1}",
                    Utility.ToArrayString(expectedUnknowns),
                    Utility.ToArrayString(actualUnknowns));
            }
        }
    }
}
