﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using TestUtility;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace MakeMetaTest
{
    [TestClass]
    public class TestOption : TestBase
    {
        private XmlFile metaFile;
        private XmlFile descFile;
        private XmlElement acidElement;

        public TestOption()
        {
            {
                metaFile = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement programId = new XmlElement("ProgramId");
                programId.InnerText = "0x0005000C10000000";
                core.AddChild(programId);

                metaFile.AddLine(elem.GetXml());
                metaFile.WriteData();
            }

            {
                acidElement = new XmlElement("Acid");
                // 読み込みのみのテストなので、Acid のバイナリは何でもよい
                acidElement.InnerText = "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABBQ0lEgAEAAAAAAAAAAAAAAQAAAAAAAAD//wAAAAAAAEACAAAAAAAAQAIAAA0AAABQAgAAKAAAAAAAAAAAAAAAhVNlcnZlcgRDbGllKgAAAPdjAAHPAAAAPwgAgD8BAIA/EAAAvwEAAP8XgAD/vwkA/38AAv//AgAAAAAAAAAAAA==";
            }

            {
                descFile = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);

                descFile.AddLine(elem.GetXml());
                descFile.WriteData();
            }
        }

        [TestMethod]
        public void TestKernelVersion()
        {
            // 正常値
            // 10 進数のメジャーバージョンを受け取れる
            // 10 進数のマイナーバージョンを受け取れる
            {
                SuccessExecute(MakeKernelVersionDesc("1.3"), metaFile, "1.3");
            }

            // 16 進数のメジャーバージョンを受け取れない
            {
                FailExecute(null, MakeKernelVersionDesc("1.3"), metaFile, "0x1.3");
            }

            // メジャーバージョン最大値
            {
                SuccessExecute(MakeKernelVersionDesc("8191.1"), metaFile, "8191.1");
            }

            // メジャーバージョン最小値
            {
                SuccessExecute(MakeKernelVersionDesc("0.1"), metaFile, "0.1");
            }

            // メジャーバージョン最大値 + 1
            {
                FailExecute(
                    string.Format(MakeMeta.Properties.Resources.Message_ValueShouldBeUnsigned, "KernelVersion/MajorVersion", 13),
                    MakeKernelVersionDesc("8191.1"), metaFile, "8192.1");
            }

            // メジャーバージョン最小値 - 1
            {
                FailExecute(null, MakeKernelVersionDesc("1.0"), metaFile, "-1.1");
            }

            // 16 進数のマイナーバージョンを受け取れない
            {
                FailExecute(null, MakeKernelVersionDesc("1.3"), metaFile, "1.0x3");
            }

            // マイナーバージョン最大値
            {
                SuccessExecute(MakeKernelVersionDesc("1.15"), metaFile, "1.15");
            }

            // マイナーバージョン最小値
            {
                SuccessExecute(MakeKernelVersionDesc("1.0"), metaFile, "1.0");
            }

            // マイナーバージョン最大値 + 1
            {
                FailExecute(
                    string.Format(MakeMeta.Properties.Resources.Message_ValueShouldBeUnsigned, "KernelVersion/MinorVersion", 4),
                    MakeKernelVersionDesc("1.15"), metaFile, "1.16");
            }

            // マイナーバージョン最小値 - 1
            {
                FailExecute(null, MakeKernelVersionDesc("1.0"), metaFile, "1.-1");
            }
        }

        [TestMethod]
        public void TestDefaultCoreVersion()
        {
            // 指定しなければ 0
            {
                var npdmFile = SuccessExecute(MakeKernelVersionDesc("1.0"), metaFile, "1.0");
                var metaHeader = npdmFile.ReadFile(MetaHeaderSize, MetaHeaderOffset);
                byte[] target = new byte[UInt32Size];
                Array.Copy(metaHeader, MetaVersionOffset, target, 0, UInt32Size);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == 0);
            }

            // 正常値
            {
                var npdmFile = SuccessExecute(MakeKernelVersionDesc("1.0"), metaFile, "1.0", "--default_core_version 1");
                var metaHeader = npdmFile.ReadFile(MetaHeaderSize, MetaHeaderOffset);
                byte[] target = new byte[UInt32Size];
                Array.Copy(metaHeader, MetaVersionOffset, target, 0, UInt32Size);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == 1);
            }

            // 最大値
            {
                var npdmFile = SuccessExecute(MakeKernelVersionDesc("1.0"), metaFile, "1.0", "--default_core_version " + uint.MaxValue.ToString());
                var metaHeader = npdmFile.ReadFile(MetaHeaderSize, MetaHeaderOffset);
                byte[] target = new byte[UInt32Size];
                Array.Copy(metaHeader, MetaVersionOffset, target, 0, UInt32Size);
                Assert.IsTrue(BitConverter.ToUInt32(target, 0) == uint.MaxValue);
            }

            // マイナスは受け取らない
            {
                FailExecute(null, MakeKernelVersionDesc("1.0"), metaFile, "1.0", "--default_core_version -1");
            }
        }

        [TestMethod]
        public void TestOptionVariable()
        {
            FileManager kernelVersionDesc = MakeKernelVersionDesc("2.0");

            {
                XmlFile varFile = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement programId = new XmlElement("ProgramId");
                programId.InnerText = "${id}";
                core.AddChild(programId);

                varFile.AddLine(elem.GetXml());
                varFile.WriteData();

                // 正常値
                // -d 変数名=値 で meta の宣言を上書きできる
                {
                    SuccessExecute(kernelVersionDesc, varFile, "2.0", "-d id=0x1");
                }

                // 異常系
                // Meta に変数は宣言されているが、オプションに指定がないと失敗する
                {
                    FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidHexadecimalNumberString, "Core/ProgramId"), kernelVersionDesc, varFile, "2.0");
                }

                // オプションのフォーマットが異なると失敗する。
                {
                    FailExecute(null, kernelVersionDesc, varFile, "2.0", "-did=0x1");
                    FailExecute(null, kernelVersionDesc, varFile, "2.0", "-d id 0x1");
                }
            }

            // 正常系
            // -d で複数の宣言ができる
            {
                XmlFile varFile = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement programId = new XmlElement("ProgramId");
                programId.InnerText = "${id}";
                core.AddChild(programId);

                XmlElement instruction = new XmlElement("Is64BitInstruction");
                instruction.InnerText = "${inst}";
                core.AddChild(instruction);

                varFile.AddLine(elem.GetXml());
                varFile.WriteData();


                // 正常値
                // -d 変数名=値 で meta の宣言を上書きできる
                {
                    SuccessExecute(kernelVersionDesc, varFile, "2.0", "-d id=0x1 -d inst=False");
                }

                // 異常系
                // Meta に変数は宣言されているが、オプションに指定がないと失敗する
                {
                    FailExecute(
                        string.Format(DescModel.Properties.Resources.Message_InvalidBoolString, "Core/Is64BitInstruction"),
                        kernelVersionDesc, varFile, "2.0", "-d id=0x1");
                }
            }
        }

        FileManager MakeKernelVersionDesc(string version)
        {
            FileManager kernelVersionDesc = new FileManager();
            MakeDesc(ref kernelVersionDesc, version);
            return kernelVersionDesc;
        }


    }
}
