﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using TestUtility;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace MakeDescTest
{
    public class TestBase
    {
        public TestContext TestContext { get; set; }

        public const string DefaultIs64BitInstruction = "false";
        public const string DefaultProcessAddressSpace = "AddressSpace32Bit";
        public const string DefaultMainThreadPriority = "25";
        public const string DefaultMainThreadCoreNumber = "1";
        public const string DefaultMainThreadStackSize = "0x4000";

        public const string BaseIdHeader = "svc_BaseId.autogen.h";
        public const string ServerIdHeader = "svc_ServerId.autogen.h";
        public const string DdIdHeader = "svc_DdId.autogen.h";
        public const string DmntIdHeader = "svc_DmntId.autogen.h";
        public const string TcbIdHeader = "svc_TcbId.autogen.h";

        public string ExecuteProgram(FileManager inputFile, FileManager outputFile, string[] svcHeaders, string kernelVersion, string additionalParam = "")
        {
            System.Diagnostics.Process process = new System.Diagnostics.Process();
            TestUtility.TestPath testPath = new TestUtility.TestPath(this.TestContext);

            string svcHeaderParam = string.Empty;
            if (svcHeaders != null)
            {
                foreach (var headerName in svcHeaders)
                {
                    svcHeaderParam = svcHeaderParam + " --svc_header " + headerName;
                }
            }

            string kernelVersionParam = string.Empty;
            if (kernelVersion != null && kernelVersion.Length > 0)
            {
                kernelVersionParam = "--kernel_version " + kernelVersion;
            }

            process.StartInfo.FileName = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\MakeDesc\\MakeDesc.exe";
            process.StartInfo.Arguments = string.Format(
                "-o {0} --desc {1} {2} {3} {4}",
                outputFile.FilePath, inputFile.FilePath, svcHeaderParam, kernelVersionParam, additionalParam);
            process.StartInfo.CreateNoWindow = true;
            process.StartInfo.UseShellExecute = false;
            process.StartInfo.RedirectStandardError = true;
            process.Start();

            string errorMsg = process.StandardError.ReadToEnd();
            process.WaitForExit();

            Assert.IsTrue(System.IO.File.Exists(inputFile.FilePath));

            return errorMsg;
        }

        public FileManager SuccessExecute(FileManager inputFile, string[] svcHeaders, string kernelVersion, string additionalParam = "")
        {
            FileManager outputDesc = new FileManager();
            string err = ExecuteProgram(inputFile, outputDesc, svcHeaders, kernelVersion, additionalParam);
            Assert.IsTrue(err.Length == 0, err);
            Assert.IsTrue(System.IO.File.Exists(outputDesc.FilePath));
            return outputDesc;
        }

        public void FailExecute(string errMsg, FileManager inputFile, string[] svcHeaders, string kernelVersion, string additionalParam = "")
        {
            FileManager tmpFile = new FileManager();
            string err = ExecuteProgram(inputFile, tmpFile, svcHeaders, kernelVersion);
            Assert.IsTrue(err.Length > 0);
            // デフォルトの例外のメッセージはテストしない
            if (errMsg != null)
            {
                Assert.IsTrue(err.IndexOf(errMsg) >= 0);
            }
            Assert.IsFalse(System.IO.File.Exists(tmpFile.FilePath));
        }

        public XmlElement CreateDefaultParam()
        {
            XmlElement defaultElem = new XmlElement("Default");

            XmlElement is64Bit = new XmlElement("Is64BitInstruction");
            is64Bit.InnerText = DefaultIs64BitInstruction;
            defaultElem.AddChild(is64Bit);

            XmlElement processAddressSpace = new XmlElement("ProcessAddressSpace");
            processAddressSpace.InnerText = DefaultProcessAddressSpace;
            defaultElem.AddChild(processAddressSpace);

            XmlElement mainThreadPriority = new XmlElement("MainThreadPriority");
            mainThreadPriority.InnerText = DefaultMainThreadPriority;
            defaultElem.AddChild(mainThreadPriority);

            XmlElement mainThreadCoreNumber = new XmlElement("MainThreadCoreNumber");
            mainThreadCoreNumber.InnerText = DefaultMainThreadCoreNumber;
            defaultElem.AddChild(mainThreadCoreNumber);

            XmlElement mainThreadStackSize = new XmlElement("MainThreadStackSize");
            mainThreadStackSize.InnerText = DefaultMainThreadStackSize;
            defaultElem.AddChild(mainThreadStackSize);

            return defaultElem;
        }

        public string GetKeyFile()
        {
            TestUtility.TestPath testPath = new TestUtility.TestPath(this.TestContext);
            return testPath.GetTestRoot() + "\\Tools\\Sources\\Tests\\MakeDescTest\\TestAcid\\test.key";
        }

        public string GetSvcHeaderDirPath()
        {
            TestUtility.TestPath testPath = new TestUtility.TestPath(this.TestContext);
            return testPath.GetSigloRoot() + "\\Programs\\Chris\\Include\\nn\\svc\\";
        }
    }
}
