﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Text;
using System.Collections.Generic;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System.Diagnostics;
using System.Reflection;
using LECore.Structures;
using LECore.Manipulator;

namespace LECoreTest.SpecialTests
{
    /// <summary>
    /// フォントに関するテスト
    /// </summary>
    [TestClass]
    public class FontTest
    {
        /// <summary>
        /// スケールを維持したままフォントを差し替える
        /// </summary>
        [TestMethod]
        public void FontTest_KeepScale()
        {
            // フォントパス
            string fontPathBefore = System.IO.Path.Combine(LECore.AppConstants.NwUnitTestDataRoot, "Font\\ComplexFont.fcpx");
            string fontPathAfter = System.IO.Path.Combine(LECore.AppConstants.NwUnitTestDataRoot, "Font\\ComplexFont2.fcpx");

            // 初期化
            TestUtility.InitializeTest(MethodBase.GetCurrentMethod());

            // プロジェクト設定ファイルの KeepingSizeScaleEnabled を有効にする
            bool keepingSizeScaleEnabled = LECore.LayoutEditorCore.KeepingSizeScaleEnabled;
            LECore.LayoutEditorCore.KeepingSizeScaleEnabled = true;

            // レイアウトを新規作成
            ISubScene newSubScene = TestUtility.CreateLayout();

            // テキストボックスを追加
            ILEFont fontBefore = TestUtility.LoadFontFullPath(newSubScene, fontPathBefore);
            IPane pane = TestUtility.AddTextBoxPane(newSubScene, "TextBoxPane", fontBefore, "abcde");
            ITextBox textBox = (ITextBox)pane.ITextBox;

            // フォントのサイズとスケールの取得
            {
                FVec2 size = textBox.FontSize;
                FVec2 scale = textBox.FontScale;
                FVec2 sizeOriginal = textBox.FontSizeOriginal;
                Assert.AreEqual(size.X, 40.0f);
                Assert.AreEqual(size.Y, 60.0f);
                Assert.AreEqual(scale.X, 1.0f);
                Assert.AreEqual(scale.Y, 1.0f);
                Assert.AreEqual(sizeOriginal.X, 40.0f);
                Assert.AreEqual(sizeOriginal.Y, 60.0f);
            }

            // フォントの差し替え
            TestUtility.ModifyTextBoxFont(pane, fontPathAfter);

            // フォントのサイズとスケールの取得
            {
                FVec2 size = textBox.FontSize;
                FVec2 scale = textBox.FontScale;
                FVec2 sizeOriginal = textBox.FontSizeOriginal;
                Assert.AreEqual(size.X, 30.0f);
                Assert.AreEqual(size.Y, 36.0f);
                Assert.AreEqual(scale.X, 1.0f);
                Assert.AreEqual(scale.Y, 1.0f);
                Assert.AreEqual(sizeOriginal.X, 40.0f);
                Assert.AreEqual(sizeOriginal.Y, 60.0f);
            }

            // レイアウトを閉じる
            TestUtility.CloseLayout(newSubScene);

            // プロジェクト設定ファイルの KeepingSizeScaleEnabled を戻す
            LECore.LayoutEditorCore.KeepingSizeScaleEnabled = keepingSizeScaleEnabled;

            // 終了
            TestUtility.FinalizeTest(MethodBase.GetCurrentMethod());
        }
    }
}
