﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Linq;
using System.Reflection;
using LECore;
using LECore.Manipulator;
using LECore.Save_Load;
using LECore.Structures;
using LECore.Structures.Core;
using LECore.Structures.Core.Command;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace LECoreTest
{
    /// <summary>
    ///
    /// * 部品ペインを含むレイアウトファイルの読み込み（関連：SIGLO-69226）
    ///
    /// </summary>
    [TestClass]
    public class BaseValueTest
    {
        // ----------------------------------------------------------------
        // 部品ペインの追加
        // ----------------------------------------------------------------
        [TestMethod]
        public void BaseValueTest_LoadLayoutTest()
        {
            ISubScene partsSubScene = null;
            ISubScene emptySubScene = null;

            // 準備
            {
                // 部品ルートの設定
                SceneManipulator mnp = new SceneManipulator();
                mnp.BindTarget(LayoutEditorCore.Scene);
                mnp.RefreshPartsSubScenes(Path.Combine(AppConstants.NwUnitTestDataRoot, "parts"));
            }

            // テスト
            {
                // 部品ペインを開く
                LayoutEditorCore.Scene.LoadAndCachePartsSubSceneIfNeeded("animation.flyt");
                partsSubScene = LayoutEditorCore.Scene.FindPartsSubSceneByFileName("animation.flyt").SubScene;
                IPane partsWindowPane = partsSubScene.FindPaneByName("W_window_00");
                FVec3 srcTrans = partsWindowPane.Trans; // 時間0の評価値を保持
                FVec3 srcScale = partsWindowPane.Scale;
                FVec3 srcRotAng = partsWindowPane.RotAng;

                // 時間を進める
                GlobalTime.Inst.SetTimeForcibly(25);

                // 空のレイアウトを読み込む
                string filePath = Path.Combine(AppConstants.NwUnitTestDataRoot, "empty.flyt");
                emptySubScene = LayoutEditorCore.LoadLayoutFile(filePath, LayoutEditorCore.LoadOption.TryToOpenRlan).ISubScene;

                // 空のレイアウトに部品ペインを追加する
                SubSceneManipulator subSceneMnp = new SubSceneManipulator();
                subSceneMnp.BindTarget(emptySubScene);
                subSceneMnp.AddPartsPane("newPaneName", "animation.flyt", partsSubScene);

                // 追加した部品ペインが、元になるレイアウトを時間0で評価した結果と同じになっていることを確認する
                IPane newPartsWindowPane = emptySubScene.FindPaneByName("newPaneName").IPartsLayout.PartsSubScene.FindPaneByName("W_window_00");
                FVec3 newTrans = newPartsWindowPane.Trans;
                FVec3 newScale = newPartsWindowPane.Scale;
                FVec3 newRotAng = newPartsWindowPane.RotAng;

                Assert.AreEqual(srcTrans, newTrans);
                Assert.AreEqual(srcScale, newScale);
                Assert.AreEqual(srcRotAng, newRotAng);
            }

            // 後処理
            {
                // 時間を戻す
                GlobalTime.Inst.SetTimeForcibly(0);

                // レイアウトファイルをアンロード
                CloseCubScene(partsSubScene);
                CloseCubScene(emptySubScene);
            }
        }

        // ----------------------------------------------------------------
        // ペイン情報のコピー＆ペースト
        // ----------------------------------------------------------------
        [TestMethod]
        public void BaseValueTest_PaneDupulicateTest()
        {
            ISubScene subScene = null;

            // 準備
            {
                // 部品ルートの設定
                SceneManipulator mnp = new SceneManipulator();
                mnp.BindTarget(LayoutEditorCore.Scene);
                mnp.RefreshPartsSubScenes(Path.Combine(AppConstants.NwUnitTestDataRoot, "parts"));
            }

            // テスト
            {
                // レイアウトファイルを開く
                string filePath = Path.Combine(AppConstants.NwUnitTestDataRoot, "parts_copy.flyt");
                subScene = LayoutEditorCore.LoadLayoutFile(filePath, LayoutEditorCore.LoadOption.TryToOpenRlan).ISubScene;

                // 時間を進める
                GlobalTime.Inst.SetTimeForcibly(25);

                // ペインを複製
                ClipBoardPane clipBoardPane = new ClipBoardPane(subScene.IPaneArray.ToArray(), PaneSetDupulicator.Option.CopyAll);
                IPane[] srcPaneSet =
                    PaneSetDupulicator.ClonePaneSet(
                    clipBoardPane.PaneSet,
                    PaneSetDupulicator.Option.CopyAll);
                IPane srcPane = srcPaneSet[0];

                // 貼り付け
                subScene.BeginMassiveModify();

                IPane paneB = subScene.IPaneArray.ElementAt(1);
                IPane beforePaneB = paneB.Clone();

                PaneManipulator paneMnp = new PaneManipulator();
                paneMnp.BindTarget(paneB);

                PaneParamaterCopyOption option = new PaneParamaterCopyOption();
                option.SetOption(ParamaterKind.All, true);
                option.CopyBaseValue = true;
                paneMnp.Copy(srcPane, option);

                subScene.EndMassiveModify();

                // baseValueを確認する
                IPane paneA = subScene.IPaneArray.ElementAt(0);
                AreEqualBaseValue(paneA, paneB);

                // Undoする
                SubSceneCmdQueue cmdQueue = (LayoutEditorCore.Scene.CurrentISubScene.ISubSceneCmdQueue as SubSceneCmdQueue);
                cmdQueue.Undo();

                // Undo後のbaseValueを比較
                AreEqualBaseValue(beforePaneB, paneB);

                // Redoする
                cmdQueue.Redo();

                // Redo後のbaseValueを比較
                AreEqualBaseValue(paneA, paneB);
            }

            // 後処理
            {
                // 時間を戻す
                GlobalTime.Inst.SetTimeForcibly(0);

                // レイアウトファイルをアンロード
                CloseCubScene(subScene);
            }
        }

        // ----------------------------------------------------------------
        // baseValue書き換え
        // ----------------------------------------------------------------
        [TestMethod]
        public void BaseValueTest_BaseValueWriteTest()
        {
            ISubScene subScene = null;

            // 準備
            {
                // 部品ルートの設定
                SceneManipulator mnp = new SceneManipulator();
                mnp.BindTarget(LayoutEditorCore.Scene);
                mnp.RefreshPartsSubScenes(Path.Combine(AppConstants.NwUnitTestDataRoot, "parts"));
            }

            // テスト
            {
                // レイアウトファイルを開く
                string filePath = Path.Combine(AppConstants.NwUnitTestDataRoot, "baseValue.flyt");
                subScene = LayoutEditorCore.LoadLayoutFile(filePath, LayoutEditorCore.LoadOption.TryToOpenRlan).ISubScene;

                // ペインを取得
                IPane defaultNullPane = subScene.FindPaneByName("N_null_00");
                IPane defaultPictPane = subScene.FindPaneByName("P_pict_00");
                IPane defaultTextPane = subScene.FindPaneByName("T_text_00");
                IPane defaultWindowPane = subScene.FindPaneByName("W_window_00");

                // 区間タグを作成する
                AnimFrameSectionSetManipulator animSectionMnp = new AnimFrameSectionSetManipulator();
                animSectionMnp.BindTarget(subScene.IAnimFrameSectionSet);
                var newSection = TestUtility.AddAnimFrameSection(subScene, "区間00", 0, 1, "コメント");

                // PaneManipulatorを生成
                PaneManipulator paneMnp = new PaneManipulator();

                // [1] ----------------------------------------------------------------------
                // 結合モード時、キーが無い場合
                {
                    SubSceneHelper.SetAnimEditSeparateForTest(subScene, false);// 結合モードにする
                    animSectionMnp.SetTargetAnimFrameSection(null); // 区間タグを選択解除

                    // UIからペインを操作した場合
                    {
                        IPane nullPane = defaultNullPane.Clone(); (nullPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane pictPane = defaultPictPane.Clone(); (pictPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane textPane = defaultTextPane.Clone(); (textPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane windowPane = defaultWindowPane.Clone(); (windowPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;

                        paneMnp.BindTarget(nullPane); paneMnp.Trans = new FVec3(1.1f, 1.1f, 1.1f);
                        paneMnp.BindTarget(pictPane); paneMnp.Trans = new FVec3(1.1f, 1.1f, 1.1f);
                        paneMnp.BindTarget(textPane); paneMnp.Trans = new FVec3(1.1f, 1.1f, 1.1f);
                        paneMnp.BindTarget(windowPane); paneMnp.Trans = new FVec3(1.1f, 1.1f, 1.1f);

                        AreNotEqualsTransBaseValue(nullPane, defaultNullPane); // baseValueが書き換えられ、違う値になるはず
                        AreNotEqualsTransBaseValue(pictPane, defaultPictPane);
                        AreNotEqualsTransBaseValue(textPane, defaultTextPane);
                        AreNotEqualsTransBaseValue(windowPane, defaultWindowPane);
                    }

                    // プログラムから値を直接変更した場合
                    {
                        IPane nullPane = defaultNullPane.Clone(); (nullPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane pictPane = defaultPictPane.Clone(); (pictPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane textPane = defaultTextPane.Clone(); (textPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane windowPane = defaultWindowPane.Clone(); (windowPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;

                        SetTransValue(nullPane, 1.1f);
                        SetTransValue(pictPane, 1.1f);
                        SetTransValue(textPane, 1.1f);
                        SetTransValue(windowPane, 1.1f);

                        AreEqualsTransBaseValue(nullPane, defaultNullPane); // 同じ値になるはず
                        AreEqualsTransBaseValue(pictPane, defaultPictPane);
                        AreEqualsTransBaseValue(textPane, defaultTextPane);
                        AreEqualsTransBaseValue(windowPane, defaultWindowPane);
                    }
                }

                // [2] ----------------------------------------------------------------------
                // 結合モード時、キーが有る場合
                {
                    SubSceneHelper.SetAnimEditSeparateForTest(subScene, false); // 結合モードにする
                    animSectionMnp.SetTargetAnimFrameSection(null); // 区間タグを選択解除

                    // UIからペインを操作した場合
                    {
                        IPane nullPane = defaultNullPane.Clone(); (nullPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane pictPane = defaultPictPane.Clone(); (pictPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane textPane = defaultTextPane.Clone(); (textPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane windowPane = defaultWindowPane.Clone(); (windowPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;

                        AddKey(nullPane);
                        AddKey(pictPane);
                        AddKey(textPane);
                        AddKey(windowPane);

                        paneMnp.BindTarget(nullPane); paneMnp.Trans = new FVec3(1.1f, 1.1f, 1.1f);
                        paneMnp.BindTarget(pictPane); paneMnp.Trans = new FVec3(1.1f, 1.1f, 1.1f);
                        paneMnp.BindTarget(textPane); paneMnp.Trans = new FVec3(1.1f, 1.1f, 1.1f);
                        paneMnp.BindTarget(windowPane); paneMnp.Trans = new FVec3(1.1f, 1.1f, 1.1f);

                        AreNotEqualsTransBaseValue(nullPane, defaultNullPane); // baseValueが書き換えられ、違う値になるはず
                        AreNotEqualsTransBaseValue(pictPane, defaultPictPane);
                        AreNotEqualsTransBaseValue(textPane, defaultTextPane);
                        AreNotEqualsTransBaseValue(windowPane, defaultWindowPane);
                    }

                    // プログラムから値を直接変更した場合
                    {
                        IPane nullPane = defaultNullPane.Clone(); (nullPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane pictPane = defaultPictPane.Clone(); (pictPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane textPane = defaultTextPane.Clone(); (textPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane windowPane = defaultWindowPane.Clone(); (windowPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;

                        AddKey(nullPane);
                        AddKey(pictPane);
                        AddKey(textPane);
                        AddKey(windowPane);

                        SetTransValue(nullPane, 1.1f);
                        SetTransValue(pictPane, 1.1f);
                        SetTransValue(textPane, 1.1f);
                        SetTransValue(windowPane, 1.1f);

                        AreEqualsTransBaseValue(nullPane, defaultNullPane); // 同じ値になるはず
                        AreEqualsTransBaseValue(pictPane, defaultPictPane);
                        AreEqualsTransBaseValue(textPane, defaultTextPane);
                        AreEqualsTransBaseValue(windowPane, defaultWindowPane);
                    }
                }

                // [3] ----------------------------------------------------------------------
                // 分割モード時、キーが無い場合、区間タグが選択されていない場合
                {
                    SubSceneHelper.SetAnimEditSeparateForTest(subScene, true); //分割モードにする
                    animSectionMnp.SetTargetAnimFrameSection(null); // 区間タグを選択解除

                    // UIからペインを操作した場合
                    {
                        IPane nullPane = defaultNullPane.Clone(); (nullPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane pictPane = defaultPictPane.Clone(); (pictPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane textPane = defaultTextPane.Clone(); (textPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane windowPane = defaultWindowPane.Clone(); (windowPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;

                        paneMnp.BindTarget(nullPane); paneMnp.Trans = new FVec3(1.1f, 1.1f, 1.1f);
                        paneMnp.BindTarget(pictPane); paneMnp.Trans = new FVec3(1.1f, 1.1f, 1.1f);
                        paneMnp.BindTarget(textPane); paneMnp.Trans = new FVec3(1.1f, 1.1f, 1.1f);
                        paneMnp.BindTarget(windowPane); paneMnp.Trans = new FVec3(1.1f, 1.1f, 1.1f);

                        AreNotEqualsTransBaseValue(nullPane, defaultNullPane); // baseValueが書き換えられ、違う値になるはず
                        AreNotEqualsTransBaseValue(pictPane, defaultPictPane);
                        AreNotEqualsTransBaseValue(textPane, defaultTextPane);
                        AreNotEqualsTransBaseValue(windowPane, defaultWindowPane);
                    }

                    // プログラムから値を直接変更した場合
                    {
                        IPane nullPane = defaultNullPane.Clone(); (nullPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane pictPane = defaultPictPane.Clone(); (pictPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane textPane = defaultTextPane.Clone(); (textPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane windowPane = defaultWindowPane.Clone(); (windowPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;

                        SetTransValue(nullPane, 1.1f);
                        SetTransValue(pictPane, 1.1f);
                        SetTransValue(textPane, 1.1f);
                        SetTransValue(windowPane, 1.1f);

                        AreEqualsTransBaseValue(nullPane, defaultNullPane); // 同じ値になるはず
                        AreEqualsTransBaseValue(pictPane, defaultPictPane);
                        AreEqualsTransBaseValue(textPane, defaultTextPane);
                        AreEqualsTransBaseValue(windowPane, defaultWindowPane);
                    }
                }

                // [4] ----------------------------------------------------------------------
                // 分割モード時、キーが無い場合、区間タグが選択されている場合
                {
                    SubSceneHelper.SetAnimEditSeparateForTest(subScene, true); //分割モードにする
                    animSectionMnp.SetTargetAnimFrameSection(newSection); // 区間タグを選択

                    // UIからペインを操作した場合
                    {
                        IPane nullPane = defaultNullPane.Clone(); (nullPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane pictPane = defaultPictPane.Clone(); (pictPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane textPane = defaultTextPane.Clone(); (textPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane windowPane = defaultWindowPane.Clone(); (windowPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;

                        paneMnp.BindTarget(nullPane); paneMnp.Trans = new FVec3(1.1f, 1.1f, 1.1f);
                        paneMnp.BindTarget(pictPane); paneMnp.Trans = new FVec3(1.1f, 1.1f, 1.1f);
                        paneMnp.BindTarget(textPane); paneMnp.Trans = new FVec3(1.1f, 1.1f, 1.1f);
                        paneMnp.BindTarget(windowPane); paneMnp.Trans = new FVec3(1.1f, 1.1f, 1.1f);

                        AreEqualsTransBaseValue(nullPane, defaultNullPane); // 同じ値になるはず
                        AreEqualsTransBaseValue(pictPane, defaultPictPane);
                        AreEqualsTransBaseValue(textPane, defaultTextPane);
                        AreEqualsTransBaseValue(windowPane, defaultWindowPane);
                    }

                    // プログラムから値を直接変更した場合
                    {
                        IPane nullPane = defaultNullPane.Clone(); (nullPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane pictPane = defaultPictPane.Clone(); (pictPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane textPane = defaultTextPane.Clone(); (textPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane windowPane = defaultWindowPane.Clone(); (windowPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;

                        SetTransValue(nullPane, 1.1f);
                        SetTransValue(pictPane, 1.1f);
                        SetTransValue(textPane, 1.1f);
                        SetTransValue(windowPane, 1.1f);

                        AreEqualsTransBaseValue(nullPane, defaultNullPane); // 同じ値になるはず
                        AreEqualsTransBaseValue(pictPane, defaultPictPane);
                        AreEqualsTransBaseValue(textPane, defaultTextPane);
                        AreEqualsTransBaseValue(windowPane, defaultWindowPane);
                    }
                }

                // [5] ----------------------------------------------------------------------
                // 分割モード時、キーが有る場合、区間タグが選択されている場合
                {
                    SubSceneHelper.SetAnimEditSeparateForTest(subScene, true); //分割モードにする
                    animSectionMnp.SetTargetAnimFrameSection(newSection); // 区間タグを選択

                    // UIからペインを操作した場合
                    {
                        IPane nullPane = defaultNullPane.Clone(); (nullPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane pictPane = defaultPictPane.Clone(); (pictPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane textPane = defaultTextPane.Clone(); (textPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane windowPane = defaultWindowPane.Clone(); (windowPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;

                        AddKey(nullPane);
                        AddKey(pictPane);
                        AddKey(textPane);
                        AddKey(windowPane);

                        paneMnp.BindTarget(nullPane); paneMnp.Trans = new FVec3(1.1f, 1.1f, 1.1f);
                        paneMnp.BindTarget(pictPane); paneMnp.Trans = new FVec3(1.1f, 1.1f, 1.1f);
                        paneMnp.BindTarget(textPane); paneMnp.Trans = new FVec3(1.1f, 1.1f, 1.1f);
                        paneMnp.BindTarget(windowPane); paneMnp.Trans = new FVec3(1.1f, 1.1f, 1.1f);

                        AreEqualsTransBaseValue(nullPane, defaultNullPane); // 同じ値になるはず
                        AreEqualsTransBaseValue(pictPane, defaultPictPane);
                        AreEqualsTransBaseValue(textPane, defaultTextPane);
                        AreEqualsTransBaseValue(windowPane, defaultWindowPane);
                    }

                    // プログラムから値を直接変更した場合
                    {
                        IPane nullPane = defaultNullPane.Clone(); (nullPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane pictPane = defaultPictPane.Clone(); (pictPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane textPane = defaultTextPane.Clone(); (textPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;
                        IPane windowPane = defaultWindowPane.Clone(); (windowPane as Pane).OwnerSubScene = defaultNullPane.OwnerSubScene;

                        AddKey(nullPane);
                        AddKey(pictPane);
                        AddKey(textPane);
                        AddKey(windowPane);

                        SetTransValue(nullPane, 1.1f);
                        SetTransValue(pictPane, 1.1f);
                        SetTransValue(textPane, 1.1f);
                        SetTransValue(windowPane, 1.1f);

                        AreEqualsTransBaseValue(nullPane, defaultNullPane); // 同じ値になるはず
                        AreEqualsTransBaseValue(pictPane, defaultPictPane);
                        AreEqualsTransBaseValue(textPane, defaultTextPane);
                        AreEqualsTransBaseValue(windowPane, defaultWindowPane);
                    }
                }
            }

            // 後処理
            {
                // レイアウトファイルをアンロード
                CloseCubScene(subScene);
            }
        }

        // ----------------------------------------------------------------
        // ファイル書き込み、読み込み
        // ----------------------------------------------------------------
        [TestMethod]
        public void BaseValueTest_SaveAndLoadTest()
        {
            string filePath;
            string outPath;
            string paneName = "newPaneName";

            // 準備
            {
                // 部品ルートの設定
                SceneManipulator mnp = new SceneManipulator();
                mnp.BindTarget(LayoutEditorCore.Scene);
                mnp.RefreshPartsSubScenes(Path.Combine(AppConstants.NwUnitTestDataRoot, "parts"));

                // テンポラリルートの初期化
                LECore.Util.TemporaryFileUtil.Initialize();

                // ファイルパス
                outPath = Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), "Output");
                filePath = Path.Combine(outPath, "BaseValueSaveAndLoadTest.flyt");

                // 一時フォルダが残っていれば消す
                if (Directory.Exists(outPath))
                {
                    Directory.Delete(outPath, true);
                }
                Directory.CreateDirectory(outPath);
            }

            // テスト
            {
                // [1] ----------------------------------------------------------------------
                // 結合モード、ペインにキーを追加して保存、読み込み
                {
                    // サブシーンを作成してペインを１つ追加する
                    ISubScene subScene = CreateSubSceneAndAddNewPane(paneName);

                    // ペインにキーを追加
                    IPane pane = subScene.FindPaneByName(paneName);
                    AddKey(pane, 100, 1.0f);
                    AddKey(pane, 200, 2.0f);

                    // 時間を進める
                    GlobalTime.Inst.SetTimeForcibly(200);

                    // 保存して閉じる
                    SaveAndClose(subScene, filePath);

                    // 時間を戻す
                    GlobalTime.Inst.SetTimeForcibly(0);

                    // 再読み込み
                    LayoutEditorCore.LoadLayoutFile(filePath, LayoutEditorCore.LoadOption.TryToOpenRlan);
                    subScene = LayoutEditorCore.Scene.CurrentISubScene as SubScene;

                    // 再読み込みしたペインのvalueとbaseValueをチェックする
                    pane = subScene.FindPaneByName(paneName);
                    IAnmAttribute attr = GetTransAttribute(pane);

                    AreEqualsBaseValue(attr.FindSubAttributeByIdx(0), 1.0f); // time=0で評価した値になるはず
                    AreEqualsBaseValue(attr.FindSubAttributeByIdx(1), 1.0f);
                    AreEqualsBaseValue(attr.FindSubAttributeByIdx(2), 1.0f);

                    AreEqualsValue(attr.FindSubAttributeByIdx(0), 1.0f); // time=0で評価した値になるはず
                    AreEqualsValue(attr.FindSubAttributeByIdx(1), 1.0f);
                    AreEqualsValue(attr.FindSubAttributeByIdx(2), 1.0f);

                    // サブシーンを閉じる
                    CloseCubScene(subScene);
                }

                // [2] ----------------------------------------------------------------------
                // 結合モード、ペインにキーを追加し、削除してからして保存、読み込み
                {
                    // サブシーンを作成してペインを１つ追加する
                    ISubScene subScene = CreateSubSceneAndAddNewPane(paneName);

                    // ペインにキーを追加
                    IPane pane = subScene.FindPaneByName(paneName);
                    AddKey(pane, 100, 1.0f);
                    AddKey(pane, 200, 2.0f);

                    // 時間を進める
                    GlobalTime.Inst.SetTimeForcibly(200);

                    // キーを削除する
                    RemoveAllKey(pane);

                    // 保存して閉じる
                    SaveAndClose(subScene, filePath);

                    // 時間を戻す
                    GlobalTime.Inst.SetTimeForcibly(0);

                    // 再読み込み
                    LayoutEditorCore.LoadLayoutFile(filePath, LayoutEditorCore.LoadOption.TryToOpenRlan);
                    subScene = LayoutEditorCore.Scene.CurrentISubScene as SubScene;

                    // 再読み込みしたペインのvalueとbaseValueをチェックする
                    pane = subScene.FindPaneByName(paneName);
                    IAnmAttribute attr = GetTransAttribute(pane);

                    AreEqualsBaseValue(attr.FindSubAttributeByIdx(0), 2.0f); // 保存した瞬間の値になるはず
                    AreEqualsBaseValue(attr.FindSubAttributeByIdx(1), 2.0f);
                    AreEqualsBaseValue(attr.FindSubAttributeByIdx(2), 2.0f);

                    AreEqualsValue(attr.FindSubAttributeByIdx(0), 2.0f); // 保存した瞬間の値になるはず
                    AreEqualsValue(attr.FindSubAttributeByIdx(1), 2.0f);
                    AreEqualsValue(attr.FindSubAttributeByIdx(2), 2.0f);

                    // サブシーンを閉じる
                    CloseCubScene(subScene);
                }

                // [3] ----------------------------------------------------------------------
                // 分割モード、ペインにキーを追加して保存、読み込み
                {
                    // サブシーンを作成してペインを１つ追加する
                    ISubScene subScene = CreateSubSceneAndAddNewPane(paneName);

                    // 分割モードにする
                    SubSceneHelper.SetAnimEditSeparateForTest(subScene, true);

                    // 区間を作成し、選択する
                    AnimFrameSectionSetManipulator animSectionMnp = new AnimFrameSectionSetManipulator();
                    animSectionMnp.BindTarget(subScene.IAnimFrameSectionSet);
                    var newSection = TestUtility.AddAnimFrameSection(subScene, "区間00", 0, 1, "コメント");
                    animSectionMnp.SetTargetAnimFrameSection(newSection);

                    // ペインにキーを追加
                    IPane pane = subScene.FindPaneByName(paneName);
                    AddKey(pane, 100, 1.0f);
                    AddKey(pane, 200, 2.0f);

                    // 時間を進める
                    GlobalTime.Inst.SetTimeForcibly(200);

                    // 保存して閉じる
                    SaveAndClose(subScene, filePath);

                    // 時間を戻す
                    GlobalTime.Inst.SetTimeForcibly(0);

                    // 再読み込み
                    LayoutEditorCore.LoadLayoutFile(filePath, LayoutEditorCore.LoadOption.TryToOpenRlan);
                    subScene = LayoutEditorCore.Scene.CurrentISubScene as SubScene;

                    // 再読み込みしたペインのvalueとbaseValueをチェックする
                    pane = subScene.FindPaneByName(paneName);
                    IAnmAttribute attr = GetTransAttribute(pane);

                    AreEqualsBaseValue(attr.FindSubAttributeByIdx(0), 0.0f); // ペイン追加時のデフォルト値になるはず
                    AreEqualsBaseValue(attr.FindSubAttributeByIdx(1), 0.0f);
                    AreEqualsBaseValue(attr.FindSubAttributeByIdx(2), 0.0f);

                    AreEqualsValue(attr.FindSubAttributeByIdx(0), 0.0f); // ペイン追加時のデフォルト値になるはず
                    AreEqualsValue(attr.FindSubAttributeByIdx(1), 0.0f);
                    AreEqualsValue(attr.FindSubAttributeByIdx(2), 0.0f);

                    // サブシーンを閉じる
                    CloseCubScene(subScene);
                }

                // [4] ----------------------------------------------------------------------
                // 分割モード、ペインにキーを追加し、削除してからして保存、読み込み
                {
                    // サブシーンを作成してペインを１つ追加する
                    ISubScene subScene = CreateSubSceneAndAddNewPane(paneName);

                    // 分割モードにする
                    SubSceneHelper.SetAnimEditSeparateForTest(subScene, true);

                    // 区間を作成し、選択する
                    AnimFrameSectionSetManipulator animSectionMnp = new AnimFrameSectionSetManipulator();
                    animSectionMnp.BindTarget(subScene.IAnimFrameSectionSet);
                    var newSection = TestUtility.AddAnimFrameSection(subScene, "区間00", 0, 1, "コメント");
                    animSectionMnp.SetTargetAnimFrameSection(newSection);

                    // ペインにキーを追加
                    IPane pane = subScene.FindPaneByName(paneName);
                    AddKey(pane, 100, 1.0f);
                    AddKey(pane, 200, 2.0f);

                    // 時間を進める
                    GlobalTime.Inst.SetTimeForcibly(200);

                    // キーを削除する
                    RemoveAllKey(pane);

                    // 保存して閉じる
                    SaveAndClose(subScene, filePath);

                    // 時間を戻す
                    GlobalTime.Inst.SetTimeForcibly(0);

                    // 再読み込み
                    LayoutEditorCore.LoadLayoutFile(filePath, LayoutEditorCore.LoadOption.TryToOpenRlan);
                    subScene = LayoutEditorCore.Scene.CurrentISubScene as SubScene;

                    // 再読み込みしたペインのvalueとbaseValueをチェックする
                    pane = subScene.FindPaneByName(paneName);
                    IAnmAttribute attr = GetTransAttribute(pane);

                    AreEqualsBaseValue(attr.FindSubAttributeByIdx(0), 0.0f); // ペイン追加時のデフォルト値になるはず
                    AreEqualsBaseValue(attr.FindSubAttributeByIdx(1), 0.0f);
                    AreEqualsBaseValue(attr.FindSubAttributeByIdx(2), 0.0f);

                    AreEqualsValue(attr.FindSubAttributeByIdx(0), 0.0f); // ペイン追加時のデフォルト値になるはず
                    AreEqualsValue(attr.FindSubAttributeByIdx(1), 0.0f);
                    AreEqualsValue(attr.FindSubAttributeByIdx(2), 0.0f);

                    // サブシーンを閉じる
                    CloseCubScene(subScene);
                }
            }

            // 後処理
            {
                // 一時フォルダを消す
                if (Directory.Exists(outPath))
                {
                    Directory.Delete(outPath, true);
                }
                Directory.CreateDirectory(outPath);
            }
        }

        // ----------------------------------------------------------------
        // キー追加のUndo/RedoによるbaseValueの確認
        // ----------------------------------------------------------------
        [TestMethod]
        public void BaseValueTest_AddKeyUndoAndRedo()
        {
            ISubScene subScene = null;

            // 準備
            {
                // 部品ルートの設定
                SceneManipulator mnp = new SceneManipulator();
                mnp.BindTarget(LayoutEditorCore.Scene);
                mnp.RefreshPartsSubScenes(Path.Combine(AppConstants.NwUnitTestDataRoot, "parts"));
            }

            // テスト
            {
                // 空のレイアウトを読み込む
                string filePath = Path.Combine(AppConstants.NwUnitTestDataRoot, "empty.flyt");
                subScene = LayoutEditorCore.LoadLayoutFile(filePath, LayoutEditorCore.LoadOption.TryToOpenRlan).ISubScene;

                // ペインを追加する
                SubSceneManipulator subSceneMnp = new SubSceneManipulator();
                subSceneMnp.BindTarget(subScene);
                IPane pane = subSceneMnp.AddNullPane("newPaneName");
                IPane defaultPane = pane.Clone();
                IAnmAttribute attr = GetTransAttribute(pane);

                // キーを追加する
                AddKey(pane, 100, 1.0f);

                // baseValueをチェック
                AreEqualsTransBaseValue(pane, defaultPane); // キー追加ではbaseValueは変わらない

                // 現在値をチェック
                AreEqualsValue(attr.FindSubAttributeByIdx(0), 1.0f); // 追加したキーの値になるはず
                AreEqualsValue(attr.FindSubAttributeByIdx(1), 1.0f);
                AreEqualsValue(attr.FindSubAttributeByIdx(2), 1.0f);

                // キーを削除する
                RemoveAllKey(pane);

                // baseValueをチェック
                AreEqualsTransBaseValue(pane, defaultPane); // キー削除ではbaseValueは変わらない

                // 現在値をチェック
                AreEqualsValue(attr.FindSubAttributeByIdx(0), 1.0f); // キー削除で現在値は変わらないはず
                AreEqualsValue(attr.FindSubAttributeByIdx(1), 1.0f);
                AreEqualsValue(attr.FindSubAttributeByIdx(2), 1.0f);

                // キー削除をUndoする
                SubSceneCmdQueue cmdQueue = (LayoutEditorCore.Scene.CurrentISubScene.ISubSceneCmdQueue as SubSceneCmdQueue);
                cmdQueue.Undo();

                // baseValueをチェック
                AreEqualsTransBaseValue(pane, defaultPane); // キー削除のUndoではbaseValueは変わらない

                // 現在値をチェック
                AreEqualsValue(attr.FindSubAttributeByIdx(0), 1.0f); // 追加したキーの値になるはず
                AreEqualsValue(attr.FindSubAttributeByIdx(1), 1.0f);
                AreEqualsValue(attr.FindSubAttributeByIdx(2), 1.0f);

                // キー追加をUndoする
                cmdQueue.Undo();

                // 現在値をチェック
                AreEqualsValue(attr.FindSubAttributeByIdx(0), 1.0f); // キー削除で現在値は変わらないはず
                AreEqualsValue(attr.FindSubAttributeByIdx(1), 1.0f);
                AreEqualsValue(attr.FindSubAttributeByIdx(2), 1.0f);

                // baseValueをチェック
                AreEqualsTransBaseValue(pane, defaultPane); // キー追加のUndoではbaseValueは変わらない
            }

            // 後処理
            {
                // レイアウトファイルをアンロード
                CloseCubScene(subScene);
            }
        }

        // ----------------------------------------------------------------
        // (SIGLO-75791) 部品ペインに含まれるペインのマテリアルの確認
        // ----------------------------------------------------------------
        [TestMethod]
        public void BaseValueTest_PartsPaneMaterialTest()
        {
            // 準備
            {
                TestUtility.InitializeTest(MethodBase.GetCurrentMethod());
            }

            // テスト
            {
                // テストデータを読み込む
                ISubScene layout = TestUtility.LoadLayout("allPane_separated.flyt");

                // 部品ペインを取得する
                IPane partsPane = layout.FindPaneByName("L_parts_00");

                // 部品ペインの中のピクチャペインを取得する
                IPane pictPane = partsPane.IPartsLayout.PartsSubScene.FindPaneByName("Nt_White_00");

                // マテリアルのbaseValueを確認する
                foreach (IMaterial mat in pictPane.IMaterial)
                {
                    foreach (ParamaterKind kind in ParamaterKindHelper.AnimationKindSet)
                    {
                        IAnmAttribute[] attrs = mat.GetAnimationTargetAttributeSet(kind);
                        foreach (IAnmAttribute attr in attrs)
                        {
                            AreEqualInitializedBaseValueRecursive_(attr);
                        }
                    }
                }

                // レイアウトを閉じる
                TestUtility.CloseLayout(layout);
            }

            // 後処理
            {
                TestUtility.FinalizeTest(MethodBase.GetCurrentMethod());
            }
        }

        /// <summary>
        /// ペインを追加したときに基本値が値と一緒になることを確認
        /// </summary>
        [TestMethod]
        public void BaseValueTest_AddPane()
        {
            TestUtility.InitializeTest(MethodBase.GetCurrentMethod());

            // レイアウトを新規作成
            var subScene = TestUtility.CreateLayout();

            // 事前のファイル読み込み
            TestUtility.RegistTexture(subScene, @"Common\f.tga");
            ILEFont font = TestUtility.LoadFont(subScene, "sample2.ffnt");
            string partsName = "animation.flyt";
            ISubScene partsSubScene = TestUtility.LoadPartsLayout(partsName);

            // ペインの追加
            TestUtility.AddNullPane(subScene, "newNullPane");
            TestUtility.AddPicturePane(subScene, "newPicturePane", "f");
            TestUtility.AddTextBoxPane(subScene, "newTextPane", font, "abc");
            TestUtility.AddWindowPane(subScene, "newWindowPane");
            TestUtility.AddBoudingPane(subScene, "newBoundingPane");
            TestUtility.AddCapturePane(subScene, "newCapturePane");
            TestUtility.AddPartsPane(subScene, "newPartsPane", partsName, partsSubScene);

            // 末端を列挙
            var attributes = TestUtility.EnumerateAttributes(subScene)
                .Where(x => x.NumSubAttribute == 0)
                .OfType<AnmAttribute>();

            // 比較
            foreach (var attribute in attributes)
            {
                object value;
                object baseValue;
                attribute.GetValue(out value);
                attribute.GetBaseValue(out baseValue);
                Assert.IsNotNull(value);
                Assert.AreEqual(value, baseValue);
            }

            TestUtility.FinalizeTest(MethodBase.GetCurrentMethod());
        }


        // ----------------------------------------------------------------
        #region private method

        /// <summary>
        /// 位置についてbaseValueが同じ値かチェックします
        /// </summary>
        void AreEqualsTransBaseValue(IPane paneA, IPane paneB)
        {
            AreEqualsTransBaseValue_(paneA, paneB, true);
        }

        /// <summary>
        /// 位置についてbaseValueが違う値かチェックします
        /// </summary>
        void AreNotEqualsTransBaseValue(IPane paneA, IPane paneB)
        {
            AreEqualsTransBaseValue_(paneA, paneB, false);
        }

        /// <summary>
        /// 位置についてbaseValueの比較を行います
        /// </summary>
        void AreEqualsTransBaseValue_(IPane paneA, IPane paneB, bool equal)
        {
            for (int i = 0; i < paneA.IPaneAttribute.TransIAttr.NumSubAttribute; i++)
            {
                if (equal)
                {
                    AreEqualsBaseValue(
                        paneA.IPaneAttribute.TransIAttr.FindSubAttributeByIdx(i),
                        paneB.IPaneAttribute.TransIAttr.FindSubAttributeByIdx(i));
                }
                else
                {
                    AreNotEqualsTransBaseValue(
                        paneA.IPaneAttribute.TransIAttr.FindSubAttributeByIdx(i),
                        paneB.IPaneAttribute.TransIAttr.FindSubAttributeByIdx(i));
                }
            }
        }

        /// <summary>
        /// baseValueが同じ値かチェックします
        /// </summary>
        void AreEqualsBaseValue(IAnmAttribute iAttrA, IAnmAttribute iAttrB)
        {
            object valA;
            object valB;

            AnmAttribute attrA = iAttrA as AnmAttribute;
            AnmAttribute attrB = iAttrB as AnmAttribute;

            attrA.GetBaseValue(out valA);
            attrB.GetBaseValue(out valB);

            Assert.AreEqual(valA, valB);
        }

        /// <summary>
        /// baseValueが同じ値かチェックします
        /// </summary>
        void AreEqualsBaseValue(IAnmAttribute iAttrA, object valB)
        {
            object valA;

            AnmAttribute attrA = iAttrA as AnmAttribute;

            attrA.GetBaseValue(out valA);

            Assert.AreEqual(valA, valB);
        }

        /// <summary>
        /// baseValueが同じ値かチェックします
        /// </summary>
        void AreEqualsValue(IAnmAttribute iAttrA, object valB)
        {
            object valA;

            AnmAttribute attrA = iAttrA as AnmAttribute;

            attrA.GetValue(out valA);

            Assert.AreEqual(valA, valB);
        }

        /// <summary>
        /// baseValueが違う値かチェックします
        /// </summary>
        void AreNotEqualsTransBaseValue(IAnmAttribute iAttrA, IAnmAttribute iAttrB)
        {
            object valA;
            object valB;

            AnmAttribute attrA = iAttrA as AnmAttribute;
            AnmAttribute attrB = iAttrB as AnmAttribute;

            attrA.GetBaseValue(out valA);
            attrB.GetBaseValue(out valB);

            Assert.AreNotEqual(valA, valB);
        }

        /// <summary>
        /// baseValueが同じ値かチェックします
        /// </summary>
        void AreNotEqualsBaseValue(IAnmAttribute iAttrA, object valB)
        {
            object valA;

            AnmAttribute attrA = iAttrA as AnmAttribute;

            attrA.GetBaseValue(out valA);

            Assert.AreNotEqual(valA, valB);
        }

        /// <summary>
        /// baseValueが同じ値かチェックします
        /// </summary>
        void AreNotEqualsValue(IAnmAttribute iAttrA, object valB)
        {
            object valA;

            AnmAttribute attrA = iAttrA as AnmAttribute;

            attrA.GetValue(out valA);

            Assert.AreNotEqual(valA, valB);
        }

        /// <summary>
        /// ペインの持つ複数のbaseValueがすべて同じ値かチェックします
        /// </summary>
        void AreEqualBaseValue(IPane paneA, IPane paneB)
        {
            AreEqualBaseValue_(paneA, paneB, true);
        }

        /// <summary>
        /// ペインの持つ複数のbaseValueがすべて違う値かチェックします
        /// </summary>
        void AreNotEqualBaseValue(IPane paneA, IPane paneB)
        {
            AreEqualBaseValue_(paneA, paneB, false);
        }

        /// <summary>
        /// ペインの持つ複数のbaseValueを比較します
        /// アニメーションカーブを持つアトリビュートに限定されます
        /// </summary>
        void AreEqualBaseValue_(IPane paneA, IPane paneB, bool equal)
        {
            foreach (ParamaterKind kind in ParamaterKindHelper.AnimationKindSet)
            {
                IAnmAttribute[] attrSetA = paneA.GetAnimationTargetAttributeSet(kind);
                IAnmAttribute[] attrSetB = paneB.GetAnimationTargetAttributeSet(kind);
                for (int i = 0; i < attrSetA.Length; i++)
                {
                    AnmAttribute attrA = attrSetA[i] as AnmAttribute;
                    AnmAttribute attrB = attrSetB[i] as AnmAttribute;

                    object baseValueA, baseValueB;
                    if (attrA.CanGetOrSetValue() && attrB.CanGetOrSetValue())
                    {
                        attrA.GetBaseValue(out baseValueA);
                        attrB.GetBaseValue(out baseValueB);

                        if (equal)
                        {
                            Assert.AreEqual(baseValueA, baseValueB);
                        }
                        else
                        {
                            Assert.AreNotEqual(baseValueA, baseValueB);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 対象のアトリビュートのbaseValueが初期値になっているか再帰的に調べます
        /// </summary>
        void AreEqualInitializedBaseValueRecursive_(IAnmAttribute attr)
        {
            if (attr.NumSubAttribute == 0 && (attr as AnmAttribute).CanGetOrSetValue())
            {
                object val;
                object baseVal;
                (attr as AnmAttribute).GetValue(out val);
                (attr as AnmAttribute).GetBaseValue(out baseVal);

                Assert.AreEqual(Convert.ToSingle(val), Convert.ToSingle(baseVal));
            }

            for (int i = 0; i < attr.NumSubAttribute; i++)
            {
                AreEqualInitializedBaseValueRecursive_(attr.FindSubAttributeByIdx(i));
            }
        }

        /// <summary>
        /// 位置の値を変更します
        /// </summary>
        void SetTransValue(IPane srcPane, object val)
        {
            for (int i = 0; i < srcPane.IPaneAttribute.TransIAttr.NumSubAttribute; i++)
            {
                var attr = srcPane.IPaneAttribute.TransIAttr.FindSubAttributeByIdx(i);
                (attr as AnmAttribute).SetValue(val);
            }
        }

        /// <summary>
        /// ペインの持つ複数のアトリビュートにキーを設定します
        /// </summary>
        void AddKey(IPane pane, float time = 100, float val = 0.0f)
        {
            foreach (ParamaterKind kind in ParamaterKindHelper.AnimationKindSet)
            {
                IAnmAttribute[] attrSet = pane.GetAnimationTargetAttributeSet(kind);
                for (int i = 0; i < attrSet.Length; i++)
                {
                    AddKey_(attrSet[i], time, val);
                }
            }
        }

        /// <summary>
        /// アトリビュートにキーを設定します
        /// </summary>
        void AddKey_(IAnmAttribute iattr, float time, float val)
        {
            AnmAttribute attr = iattr as AnmAttribute;

            if (attr.NumSubAttribute == 0 &&
                attr.CanGetOrSetValue())
            {
                AnmCurveManipulator anmCurveMnp = new AnmCurveManipulator();
                anmCurveMnp.BindTarget(attr.ICurrentAnimationCurve);
                anmCurveMnp.MakeKeyFrame((int)time, val);
                anmCurveMnp.Update(true);
            }

            for (int i = 0; i < attr.NumSubAttribute; i++)
            {
                AddKey_(attr.SubAttrArray.ElementAt(i), time, val);
            }
        }

        /// <summary>
        /// ペインの持つ複数のアトリビュートのキーをすべて削除します
        /// </summary>
        void RemoveAllKey(IPane pane)
        {
            foreach (ParamaterKind kind in ParamaterKindHelper.AnimationKindSet)
            {
                IAnmAttribute[] attrSet = pane.GetAnimationTargetAttributeSet(kind);
                for (int i = 0; i < attrSet.Length; i++)
                {
                    RemoveAllKey_(attrSet[i]);
                }
            }
        }

        /// <summary>
        /// ペインの持つ複数のアトリビュートのキーをすべて削除します
        /// </summary>
        void RemoveAllKey_(IAnmAttribute iattr)
        {
            AnmAttribute attr = iattr as AnmAttribute;

            if (attr.NumSubAttribute == 0 &&
                attr.CanGetOrSetValue())
            {
                AnmCurveManipulator anmCurveMnp = new AnmCurveManipulator();
                anmCurveMnp.BindTarget(attr.ICurrentAnimationCurve);
                foreach (IAnmKeyFrame key in attr.ICurrentAnimationCurve.IKeyFrameSet)
                {
                    anmCurveMnp.RemoveKeyFrame(key, false);
                }
                anmCurveMnp.Update(true);
            }

            for (int i = 0; i < attr.NumSubAttribute; i++)
            {
                RemoveAllKey_(attr.SubAttrArray.ElementAt(i));
            }
        }

        /// <summary>
        /// サブシーンを作成し、ペインを追加します
        /// </summary>
        ISubScene CreateSubSceneAndAddNewPane(string paneName)
        {
            ISubScene subScene = (LayoutEditorCore.Scene as Scene).AddSubScene();

            // ペインを追加する
            SubSceneManipulator subSceneMnp = new SubSceneManipulator();
            subSceneMnp.BindTarget(subScene);
            subSceneMnp.AddNullPane(paneName);

            return subScene;
        }

        /// <summary>
        /// サブシーンを閉じます
        /// </summary
        void CloseCubScene(ISubScene subScene)
        {
            SceneManipulator sceneMnp = new SceneManipulator();
            sceneMnp.BindTarget(LayoutEditorCore.Scene);
            sceneMnp.RemoveSubScene(subScene);
        }

        /// <summary>
        /// サブシーンを保存して閉じます
        /// </summary>
        void SaveAndClose(ISubScene subScene, string filePath)
        {
            // アニメーションが消えた場合に対応できるように、書き出し前に、flan をクリーンアップします。
            {
                string flanPath = Path.ChangeExtension(filePath, ".flan");
                if (File.Exists(flanPath))
                {
                    File.Delete(flanPath);
                }
            }

            // 保存
            ExportOption exportOption = new ExportOption()
            {
                UseBaseValue = true,
                Frame = GlobalTime.Inst.Time,
            };
            LayoutEditorCore.BeforeExportEventArgs prepareOption = new LayoutEditorCore.BeforeExportEventArgs(filePath, exportOption);

            LayoutEditorCore.PrepareExport(prepareOption);
            LayoutEditorCore.ExportToFileAll(subScene, filePath, exportOption);

            // 閉じる
            CloseCubScene(subScene);
        }

        /// <summary>
        /// 位置のアトリビュートを取得します
        /// </summary>
        IAnmAttribute GetTransAttribute(IPane pane)
        {
            IAnmAttribute[] attrSet = pane.GetAnimationTargetAttributeSet(ParamaterKind.Animation_PaneSRT);
            foreach (IAnmAttribute attr in attrSet)
            {
                if (attr.Name == "pos")
                {
                    return attr;
                }
            }

            return null;
        }
    }

    #endregion private method
}
