﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using LECore;
using LECore.Manipulator;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using LECore.Save_Load;
using LECore.Structures;
using System.IO;
using LECore.Structures.Core.Command;
using System.Reflection;
using System.Linq;
using LECore.Structures.Core;

namespace LECoreTest
{
    /// <summary>
    /// チェックされていない項目（TODO）
    ///
    /// * アニメーションの新規作成
    /// * アニメーションのコピー貼り付け
    ///
    /// </summary>
    [TestClass]
    public class AnimationTest
    {
        //----------------------------------------------------------
        void CheckCurvesSame_(IAnmCurve curve0, IAnmCurve curve1)
        {
            // 失敗箇所が分かりやすいように、IsSameWithoutKeis でばらしてチェックする。
            {
                Assert.IsTrue(curve0.IsSameWithoutKeis(curve1));
                Assert.AreEqual(curve0.NumKeyFrame, curve1.NumKeyFrame);
                for (int i = 0; i < curve0.NumKeyFrame; i++)
                {
                    var anmKey0 = curve0.IKeyFrameSet[i];
                    var anmKey1 = curve1.IKeyFrameSet[i];

                    Assert.IsTrue(anmKey0.IsSame(anmKey1));
                }
            }

            // 統合版 IsSame もチェックする
            Assert.IsTrue(curve0.IsSame(curve1));
        }

        //----------------------------------------------------------
        void CheckPaneAnimationsSame_(IAnmAttribute attr0, IAnmAttribute attr1)
        {
            Assert.IsTrue(attr0.IsSameCurveCount(attr1));
            if(attr0.Curves != null)
            {
                TestUtility.WriteLog(TestUtility.GetAttributeTreeString(attr0));
                for (int i = 0; i < attr0.Curves.Length; i++)
                {
                    CheckCurvesSame_(attr0.Curves[i], attr1.Curves[i]);
                }
            }

            Assert.AreEqual(attr0.NumSubAttribute, attr1.NumSubAttribute);
            for (int i = 0; i < attr0.NumSubAttribute; i++)
            {
                CheckPaneAnimationsSame_(attr0.FindSubAttributeByIdx(i), attr1.FindSubAttributeByIdx(i));
            }
        }

        //----------------------------------------------------------
        void CheckPaneAnimationsSame_(IPane pane0, IPane pane1)
        {
            foreach (ParamaterKind kind in ParamaterKindHelper.AnimationKindSet)
            {
                IAnmAttribute[] attrSet0 = PaneAnimationHelper.GetAnimationTargetAttributeSet(pane0, kind);
                IAnmAttribute[] attrSet1 = PaneAnimationHelper.GetAnimationTargetAttributeSet(pane1, kind);

                Assert.AreEqual(attrSet0.Length, attrSet1.Length);
                for(int i = 0;i < attrSet0.Length;i++)
                {
                    CheckPaneAnimationsSame_(attrSet0[i], attrSet1[i]);
                }
            }
        }

        //----------------------------------------------------------
        void CheckFileIO_(string flytFileName)
        {
            SceneManipulator mnp = new SceneManipulator();
            mnp.BindTarget(LayoutEditorCore.Scene);

            LECore.Util.TemporaryFileUtil.Initialize();

            // 準備
            {
                // 部品ルートの設定
                mnp.RefreshPartsSubScenes(Path.Combine(AppConstants.NwUnitTestDataRoot, "parts"));
            }

            // アニメーションの読み込み・書き出しで変化が起こらないことの確認。
            {
                PartsLayoutManipulator partsLayoutMnp = new PartsLayoutManipulator();

                string filePath = Path.Combine(AppConstants.NwUnitTestDataRoot, "Parts\\" + flytFileName);
                LayoutEditorCore.LoadLayoutFile(filePath, LayoutEditorCore.LoadOption.TryToOpenRlan);
                var loaded = LayoutEditorCore.Scene.CurrentISubScene;

                string outPath = Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), "Output");

                // 一時フォルダをまっさらに消す。
                if (Directory.Exists(outPath))
                {
                    Directory.Delete(outPath, true);
                }
                Directory.CreateDirectory(outPath);

                // 書き出す
                ExportOption exportOption = new ExportOption()
                {
                    UseBaseValue = true,
                    Frame = 0,
                    CreateThumbnail = false,
                };

                string tempOutFilePath = Path.Combine(outPath, flytFileName);
                {
                    bool isSuccess = LayoutEditorCore.ExportToFileAllWithoutEventNotifications(LayoutEditorCore.Scene.CurrentISubScene, tempOutFilePath, exportOption);
                    Assert.IsTrue(isSuccess);
                }

                // 書き出したのを読む
                var loadedDocument = LayoutEditorCore.LoadLayoutFile(tempOutFilePath, LayoutEditorCore.LoadOption.TryToOpenRlan);
                Assert.IsNotNull(loadedDocument);
                var loaded2 = LayoutEditorCore.Scene.CurrentISubScene;

                // 区間タグを比較する。
                {
                    AnimFrameSectionSetHelper.IsAnimFrameSectionSetSame(loaded.IAnimFrameSectionSet, loaded2.IAnimFrameSectionSet);

                    // SubSceneHelper.GetTags が一致することも確認する。
                    {
                        var tags0 = SubSceneHelper.GetTags(loaded);
                        var tags1 = SubSceneHelper.GetTags(loaded2);

                        Assert.AreEqual(tags0.Length, tags1.Length);
                        for (int i = 0; i < tags0.Count(); i++)
                        {
                            Assert.AreEqual(tags0.ElementAt(i), tags1.ElementAt(i));
                        }
                    }
                }

                // 比較して完全に一致する
                Assert.AreEqual(loaded.IPaneArray.Count(), loaded2.IPaneArray.Count());
                for (int i = 0; i < loaded.IPaneArray.Count(); i++)
                {
                    IPane pane0 = loaded.IPaneArray.ElementAt(i);
                    IPane pane1 = loaded2.IPaneArray.ElementAt(i);

                    CheckPaneAnimationsSame_(pane0, pane1);
                }

                // アンロード
                mnp.RemoveSubScene(loaded);
                mnp.RemoveSubScene(loaded2);
            }

            // 読み込まれている部品レイアウトをすべて破棄。
            mnp.RefreshPartsSubScenes(Path.Combine(AppConstants.NwUnitTestDataRoot, "parts"));
        }

        //----------------------------------------------------------
        [TestMethod]
        public void AnimationTest_AnimationCopyTest()
        {
            CheckFileIO_("animation.flyt");
            CheckFileIO_("animation_derived.flyt");
            CheckFileIO_("animation_separated_mode.flyt");
        }

        [TestMethod]
        public void AnimationTest_CheckHaveKeyRecursiveInTargetTagTest()
        {
            ISubScene subScene = TestUtility.CreateLayout();
            IPane pane = TestUtility.AddNullPane(subScene, "pane0");

            // アトリビュートを取得
            IAnmAttribute paneAttr = ((Pane)pane).PaneAttrRef;
            IAnmAttribute posXAttr = pane.IPaneAttribute.TransIAttr.FindSubAttributeByIdx(0);

            const string tag0 = "Tag0";
            const string tag1 = "Tag1";
            TestUtility.AddAnimFrameSection(subScene, tag0, 0, 10);
            TestUtility.AddAnimFrameSection(subScene, tag1, 10, 20);

            // モードの確認
            Assert.IsFalse(subScene.IsAnimEditSeparateMode());

            // キーの有無の確認
            Assert.IsFalse(paneAttr.CheckHaveKeyRecursiveInTargetTag(0, 10, tag0, subScene.IsAnimEditSeparateMode()));
            Assert.IsFalse(paneAttr.CheckHaveKeyRecursiveInTargetTag(10, 20, tag1, subScene.IsAnimEditSeparateMode()));

            // キーを追加
            TestUtility.AddAnimKey(posXAttr, 5, 0);

            // キーの有無の確認
            Assert.IsTrue(paneAttr.CheckHaveKeyRecursiveInTargetTag(0, 10, tag0, subScene.IsAnimEditSeparateMode()));
            Assert.IsFalse(paneAttr.CheckHaveKeyRecursiveInTargetTag(10, 20, tag1, subScene.IsAnimEditSeparateMode()));

            // 区間と無関係な範囲を指定したとき
            Assert.IsFalse(paneAttr.CheckHaveKeyRecursiveInTargetTag(10, 20, tag0, subScene.IsAnimEditSeparateMode()));
            Assert.IsTrue(paneAttr.CheckHaveKeyRecursiveInTargetTag(0, 10, tag1, subScene.IsAnimEditSeparateMode()));

            // 誤ったモードを指定したとき
            Assert.IsFalse(paneAttr.CheckHaveKeyRecursiveInTargetTag(0, 10, tag0, !subScene.IsAnimEditSeparateMode()));

            // 分割モードに変更
            TestUtility.ChangeAnimEditMode(subScene);

            // モードの確認
            Assert.IsTrue(subScene.IsAnimEditSeparateMode());

            // キーの有無の確認
            Assert.IsTrue(paneAttr.CheckHaveKeyRecursiveInTargetTag(0, 10, tag0, subScene.IsAnimEditSeparateMode()));
            Assert.IsFalse(paneAttr.CheckHaveKeyRecursiveInTargetTag(0, 10, tag1, subScene.IsAnimEditSeparateMode()));

            // 区間と無関係な範囲を指定したとき
            Assert.IsFalse(paneAttr.CheckHaveKeyRecursiveInTargetTag(10, 20, tag0, subScene.IsAnimEditSeparateMode()));
            Assert.IsFalse(paneAttr.CheckHaveKeyRecursiveInTargetTag(10, 20, tag1, subScene.IsAnimEditSeparateMode()));

            // 誤ったモードを指定したとき
            Assert.IsFalse(paneAttr.CheckHaveKeyRecursiveInTargetTag(0, 10, tag0, !subScene.IsAnimEditSeparateMode()));
        }
    }
}
