﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Threading.Tasks;

using Microsoft.VisualStudio.TestTools.UnitTesting;

using Nintendo.HtcTools.Htclow;

namespace Nintendo.HtcTools.Htclow
{
    internal class HandshakeUtil
    {
        private PacketFactory m_Factory;
        private HtclowLibraryTestService m_Service;

        public HandshakeUtil(PacketFactory factory, HtclowLibraryTestService service)
        {
            m_Factory = factory;
            m_Service = service;
        }

        /// <summary>
        /// 以下のハンドシェークを実行する。ホストが先に Connect を呼ぶパターン。
        /// 1. ホスト側 Connect() 実行
        /// 2. ターゲット側 Syn パケット送信
        /// 3. ターゲット側 SynAck パケット受信
        /// </summary>
        public void DoHandshakeByHost(HtclowChannel htclowChannel, Channel channel)
        {
            var task = Task.Run(() =>
            {
                htclowChannel.Connect();
            });

            m_Service.SendPacket(m_Factory.MakeSynPacket(channel));

            var expected = m_Factory.MakeSynAckPacket(channel);
            var actual = m_Service.ReceivePacket();

            Assert.AreEqual(expected, actual);

            task.Wait();
        }

        /// <summary>
        /// 以下のハンドシェークを実行する。ターゲットが先に Connect を呼ぶパターン。
        /// 1. ターゲット側 Syn パケット送信
        /// 2. ホスト側 Connect() 実行
        /// 3. ターゲット側 SynAck パケット受信
        /// </summary>
        public void DoHandshakeByTarget(HtclowChannel htclowChannel, Channel channel)
        {
            m_Service.SendPacket(m_Factory.MakeSynPacket(channel));

            htclowChannel.Connect();

            var expected = m_Factory.MakeSynAckPacket(channel);
            var actual = m_Service.ReceivePacket();

            Assert.AreEqual(expected, actual);
        }

        /// <summary>
        /// 以下のシャットダウンシーケンスを実行する。ホストが先にシーケンスを開始するパターン。
        /// 1. ホスト側 Shutdown() 実行 (並列)
        /// 2. ターゲット側 Fin パケット受信
        /// 3. ターゲット側 FinAck パケット送信
        /// 4. ターゲット側 Fin パケット送信
        /// 5. ターゲット側 FinAck パケット受信
        /// </summary>
        public void DoShutdownByHost(HtclowChannel htclowChannel, Channel channel)
        {
            var task = Task.Run(() =>
            {
                htclowChannel.Shutdown();
            });

            var packet1 = m_Service.ReceivePacket();
            Assert.AreEqual(PacketType.Fin, packet1.PacketType);

            var finAck = m_Factory.MakeFinAckPacket(channel, packet1.SequenceId);
            m_Service.SendPacket(finAck);

            var fin = m_Factory.MakeFinPacket(channel);
            m_Service.SendPacket(fin);

            var packet2 = m_Service.ReceivePacket();
            Assert.AreEqual(PacketType.FinAck, packet2.PacketType);

            task.Wait();
        }

        /// <summary>
        /// 以下のシャットダウンシーケンスを実行する。ターゲットが先にシーケンスを開始するパターン。
        /// 1. ターゲット側 Fin パケット送信
        /// 2. ターゲット側 FinAck パケット受信
        /// 3. ホスト側 Shutdown() 実行 (並列)
        /// 4. ターゲット側 Fin パケット受信
        /// 5. ターゲット側 FinAck パケット送信
        /// </summary>
        public void DoShutdownByTarget(HtclowChannel htclowChannel, Channel channel)
        {
            var fin = m_Factory.MakeFinPacket(channel);
            m_Service.SendPacket(fin);

            var packet1 = m_Service.ReceivePacket();
            Assert.AreEqual(PacketType.FinAck, packet1.PacketType);

            var task = Task.Run(() =>
            {
                htclowChannel.Shutdown();
            });

            var packet2 = m_Service.ReceivePacket();
            Assert.AreEqual(PacketType.Fin, packet2.PacketType);

            var finAck = m_Factory.MakeFinAckPacket(channel, packet2.SequenceId);
            m_Service.SendPacket(finAck);

            task.Wait();
        }

        /// <summary>
        /// 以下のシャットダウンシーケンスを実行する。ホストとターゲットが同時にシーケンスを開始するパターン。
        /// 1.  ホスト側 Shutdown() 実行 (並列)
        /// 2.  ターゲット側 Fin パケット送信
        /// 3a. ターゲット側 Fin パケット受信
        /// 3b. ターゲット側 FinAck パケット受信
        /// 4.  ターゲット側 FinAck パケット送信
        /// </summary>
        public void DoShutdownSimultaneously(HtclowChannel htclowChannel, Channel channel)
        {
            var task = Task.Run(() =>
            {
                htclowChannel.Shutdown();
            });

            var fin = m_Factory.MakeFinPacket(channel);
            m_Service.SendPacket(fin);

            var packet1 = m_Service.ReceivePacket();

            // ホストが Fin と FinAck のどちらを先に送信するかは制御できないので、どちらの場合でも正しく検証できるようにする
            if (packet1.PacketType == PacketType.Fin)
            {
                var packet2 = m_Service.ReceivePacket();
                Assert.AreEqual(PacketType.FinAck, packet2.PacketType);
            }
            else if (packet1.PacketType == PacketType.FinAck)
            {
                var packet2 = m_Service.ReceivePacket();
                Assert.AreEqual(PacketType.Fin, packet2.PacketType);
            }
            else
            {
                Assert.Fail();
            }

            var finAck = m_Factory.MakeFinAckPacket(channel, packet1.SequenceId);
            m_Service.SendPacket(finAck);

            task.Wait();
        }
    }
}
