﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using Microsoft.VisualStudio.TestTools.UnitTesting;

using Nintendo.HtcTools.HtcfsLibrary;

namespace Nintendo.HtcTools.HtcfsLibraryTest
{
    /// <summary>
    /// testFs_WriteOption に相当するテストを HtcfsLibrary 単体で行う
    /// </summary>
    [TestClass]
    public sealed class HtcfsWriteOptionTest : IDisposable
    {
        private readonly OpenMode[] OpenFileModes =
        {
            OpenMode.Read,
            OpenMode.Read | OpenMode.AllowAppend,
            OpenMode.Write,
            OpenMode.Write | OpenMode.AllowAppend,
        };

        private readonly Random random;
        private readonly HtcfsLibraryTestService testService;
        private readonly string filePath;

        public HtcfsWriteOptionTest()
        {
            random = new Random();
            testService = new HtcfsLibraryTestService();
            filePath = $"{testService.TestRootPath}/test.file";
        }

        public void Dispose()
        {
            testService.Dispose();
        }

        [TestMethod]
        public void WriteFile()
        {
            CompoundResult result;

            const int fileSize = 32;
            const int bufferSize = 32;

            // WriteOption のフラグ (0) 指定時の挙動が正しく動作していること
            {
                var writeBuffer = new byte[bufferSize];

                random.NextBytes(writeBuffer);

                result = testService.CreateFile(filePath, fileSize);
                Assert.IsTrue(result.IsSuccess());

                result = testService.OpenFile(out var handle, filePath, OpenMode.Read | OpenMode.Write | OpenMode.AllowAppend);
                Assert.IsTrue(result.IsSuccess());

                result = testService.WriteFile(handle, writeBuffer, 0, 0);
                Assert.IsTrue(result.IsSuccess());

                result = testService.FlushFile(handle);
                Assert.IsTrue(result.IsSuccess());

                result = testService.ReadFile(out var readBuffer, handle, 0, bufferSize);
                Assert.IsTrue(result.IsSuccess());
                Assert.AreEqual(bufferSize, readBuffer.Length);
                CollectionAssert.AreEqual(writeBuffer, readBuffer);

                result = testService.CloseFile(handle);
                Assert.IsTrue(result.IsSuccess());

                result = testService.DeleteFile(filePath);
                Assert.IsTrue(result.IsSuccess());
            }

            // WriteOption のフラグ (WriteOptionFlag_Flush) 指定時の挙動が正しく動作していること
            {
                var writeBuffer = new byte[bufferSize];

                random.NextBytes(writeBuffer);

                result = testService.CreateFile(filePath, fileSize);
                Assert.IsTrue(result.IsSuccess());

                result = testService.OpenFile(out var handle, filePath, OpenMode.Read | OpenMode.Write | OpenMode.AllowAppend);
                Assert.IsTrue(result.IsSuccess());

                result = testService.WriteFile(handle, writeBuffer, 0, WriteOption.Flush);
                Assert.IsTrue(result.IsSuccess());

                result = testService.ReadFile(out var readBuffer, handle, 0, bufferSize);
                Assert.IsTrue(result.IsSuccess());
                Assert.AreEqual(bufferSize, readBuffer.Length);
                CollectionAssert.AreEqual(writeBuffer, readBuffer);

                result = testService.CloseFile(handle);
                Assert.IsTrue(result.IsSuccess());

                result = testService.DeleteFile(filePath);
                Assert.IsTrue(result.IsSuccess());
            }

            // 0 バイト書き込み時の WriteOption のフラグ (WriteOptionFlag_Flush) 指定時の挙動が正しく動作していること
            {
                var writeBuffer = new byte[bufferSize];
                var zeroBuffer = new byte[0];

                random.NextBytes(writeBuffer);

                result = testService.CreateFile(filePath, fileSize);
                Assert.IsTrue(result.IsSuccess());

                result = testService.OpenFile(out var handle, filePath, OpenMode.Read | OpenMode.Write | OpenMode.AllowAppend);
                Assert.IsTrue(result.IsSuccess());

                result = testService.WriteFile(handle, writeBuffer, 0, 0);
                Assert.IsTrue(result.IsSuccess());

                result = testService.WriteFile(handle, zeroBuffer, 0, WriteOption.Flush);
                Assert.IsTrue(result.IsSuccess());

                result = testService.ReadFile(out var readBuffer, handle, 0, bufferSize);
                Assert.IsTrue(result.IsSuccess());
                Assert.AreEqual(bufferSize, readBuffer.Length);
                CollectionAssert.AreEqual(writeBuffer, readBuffer);

                result = testService.CloseFile(handle);
                Assert.IsTrue(result.IsSuccess());

                result = testService.DeleteFile(filePath);
                Assert.IsTrue(result.IsSuccess());
            }
        }
    }
}
