﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace Nintendo.Hid
{
    using System;
    using System.IO;

    internal class NetworkBinaryReader : IDisposable
    {
        private bool disposed = false;
        private BinaryReader binaryReader;

        internal NetworkBinaryReader(Stream input)
        {
            this.binaryReader = new BinaryReader(input);
        }

        ~NetworkBinaryReader()
        {
            this.Dispose(false);
        }

        internal Stream BaseStream
        {
            get { return this.binaryReader.BaseStream; }
        }

        public void Dispose()
        {
            this.Dispose(true);

            GC.SuppressFinalize(this);
        }

        internal byte ReadByte()
        {
            return this.ReadBytes(1)[0];
        }

        internal byte[] ReadBytes(int count)
        {
            byte[] bytes = this.binaryReader.ReadBytes(count);

            if (bytes.Length != count)
            {
                throw new ObjectDisposedException("NetworkStream");
            }

            return bytes;
        }

        internal long ReadInt64()
        {
            return BitConverter.ToInt64(Utility.ConvertToBigEndian(this.ReadBytes(8)), 0);
        }

        internal ushort ReadUInt16()
        {
            return BitConverter.ToUInt16(Utility.ConvertToBigEndian(this.ReadBytes(2)), 0);
        }

        private void Dispose(bool disposing)
        {
            if (this.disposed)
            {
                return;
            }

            if (disposing)
            {
                this.binaryReader.Dispose();
            }

            this.disposed = true;
        }
    }
}
