﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace Nintendo.Hid
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;

    public abstract class BaseModule
    {
        private List<BaseSubModule> subModules = new List<BaseSubModule>();

        internal BaseModule(byte id)
        {
            this.ModuleId = id;
        }

        internal abstract byte ModuleTypeValue { get; }

        internal byte ModuleId { get; private set; }

        internal void AddSubModule(BaseSubModule subModule)
        {
            this.subModules.Add(subModule);
        }

        internal void Reset()
        {
            foreach (var m in this.subModules)
            {
                m.Reset();
            }
        }

        internal void Update()
        {
            foreach (var m in this.subModules)
            {
                m.Update();
            }
        }

        internal byte[] GenerateAutoInputCommand()
        {
            using (var stream = new MemoryStream())
            using (var writer = new NetworkBinaryWriter(stream))
            {
                foreach (var m in this.subModules)
                {
                    writer.Write((byte[])m.GenerateAutoInputCommand());
                }

                return stream.ToArray();
            }
        }

        internal byte[] GenerateRequestCommand()
        {
            using (var stream = new MemoryStream())
            using (var writer = new NetworkBinaryWriter(stream))
            {
                foreach (var m in this.subModules)
                {
                    writer.Write((byte[])m.GenerateRequestCommand());
                }

                return stream.ToArray();
            }
        }

        internal void AcceptInputState(NetworkBinaryReader reader)
        {
            var subModuleType = reader.ReadByte();
            reader.ReadByte();
            var subModule = this.subModules.FirstOrDefault(m => m.SubModuleTypeValue == subModuleType);
            if (subModule == null)
            {
                throw new ObjectDisposedException("NetworkStream");
            }

            reader.ReadBytes(3);
            var count = reader.ReadByte();
            reader.ReadBytes(4);
            reader.ReadInt64();
            while (count-- > 0)
            {
                subModule.AcceptInputState(reader);
            }
        }
    }
}
