﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using EffectMaker.Foundation.Input;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Extensions;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.EffectMaker.UIControls.TestTypes;

namespace Test.EffectMaker.UIControls.UnitTests
{
    /// <summary>
    /// Class containing unit test related to the UIRadioButtonGroup control.
    /// </summary>
    [TestClass]
    public class UIRadioButtonGroupUnitTests
    {
        /// <summary>
        /// Test the Checked property through data binding.
        /// </summary>
        [TestMethod]
        public void UIRadioButtonGroupTestMethod()
        {
            var radioGroup = new UIRadioButtonGroup();
            var dataSource = new RadioButtonGroupTestViewModel();

            radioGroup.AddBinding("SelectedItem", "SelectedItem");
            radioGroup.AddBinding("AvailableItems", "AvailableItems");

            radioGroup.DataContext = dataSource;

            radioGroup.SelectedIndex = 0;
            Assert.AreEqual(dataSource.SelectedItem, RadioButtonGroupTestViewModel.Items.A, "Failed");
            radioGroup.SelectedIndex = 1;
            Assert.AreEqual(dataSource.SelectedItem, RadioButtonGroupTestViewModel.Items.B, "Failed");
            radioGroup.SelectedIndex = 2;
            Assert.AreEqual(dataSource.SelectedItem, RadioButtonGroupTestViewModel.Items.C, "Failed");

            dataSource.SelectedItem = RadioButtonGroupTestViewModel.Items.A;
            Assert.AreEqual(radioGroup.SelectedIndex, 0, "Failed");
            dataSource.SelectedItem = RadioButtonGroupTestViewModel.Items.B;
            Assert.AreEqual(radioGroup.SelectedIndex, 1, "Failed");
            dataSource.SelectedItem = RadioButtonGroupTestViewModel.Items.C;
            Assert.AreEqual(radioGroup.SelectedIndex, 2, "Failed");
        }

        /// <summary>
        /// Add test.
        /// </summary>
        [TestMethod]
        public void UIRadioButtonGroupUnitTestAdd()
        {
            var radioGroup = new UIRadioButtonGroup();
            var dataSource = new RadioButtonGroupTestViewModel();

            radioGroup.AddBinding("SelectedItem", "SelectedItem");
            radioGroup.AddBinding("AvailableItems", "AvailableItems");

            radioGroup.DataContext = dataSource;

            dataSource.AvailableItems.Add(
                new KeyValuePair<string, object>("Radio Button Group Item D", RadioButtonGroupTestViewModel.Items.D));

            // Add a new available item in the end.
            Assert.AreEqual(radioGroup.StackControls.Count, 4, "Failed");

            var button = radioGroup.StackControls[3] as UIRadioButton;
            Assert.AreEqual(button.Text, "Radio Button Group Item D", "Failed");

            // Insert a new available item at index 1.
            dataSource.AvailableItems.Insert(
                1,
                new KeyValuePair<string, object>("Radio Button Group Item E", ComboBoxTestViewModel.Items.E));

            Assert.AreEqual(radioGroup.StackControls.Count, 5, "Failed");

            button = radioGroup.StackControls[1] as UIRadioButton;
            Assert.AreEqual(button.Text, "Radio Button Group Item E", "Failed");
        }

        /// <summary>
        /// Replace test.
        /// </summary>
        [TestMethod]
        public void UIRadioButtonGroupUnitTestReplace()
        {
            var radioGroup = new UIRadioButtonGroup();
            var dataSource = new RadioButtonGroupTestViewModel();

            radioGroup.AddBinding("SelectedItem", "SelectedItem");
            radioGroup.AddBinding("AvailableItems", "AvailableItems");

            radioGroup.DataContext = dataSource;

            // Replace available item B with E.
            dataSource.AvailableItems[1] =
                new KeyValuePair<string, object>("Radio Button Group Item E", ComboBoxTestViewModel.Items.E);

            // Add a new available item in the end.
            Assert.AreEqual(radioGroup.StackControls.Count, 3, "Failed");

            var item = radioGroup.StackControls[1] as UIRadioButton;
            Assert.AreEqual(item.Text, "Radio Button Group Item E", "Failed");
        }

        /// <summary>
        /// Move test.
        /// </summary>
        [TestMethod]
        public void UIRadioButtonGroupUnitTestMove()
        {
            var radioGroup = new UIRadioButtonGroup();
            var dataSource = new RadioButtonGroupTestViewModel();

            radioGroup.AddBinding("SelectedItem", "SelectedItem");
            radioGroup.AddBinding("AvailableItems", "AvailableItems");

            radioGroup.DataContext = dataSource;

            // Clear the available items.
            dataSource.AvailableItems.Move(0, 1);

            // Add a new available item in the end.
            Assert.AreEqual(radioGroup.StackControls.Count, 3, "Failed");

            var item = radioGroup.StackControls[1] as UIRadioButton;
            Assert.AreEqual(item.Text, "Radio Button Group Item A", "Failed");

            item = radioGroup.StackControls[0] as UIRadioButton;
            Assert.AreEqual(item.Text, "Radio Button Group Item B", "Failed");
        }
    }

    /// <summary>
    /// ラジオボタンテスト用ビューモデル
    /// </summary>
    public class RadioButtonGroupTestViewModel : DummyViewModel
    {
        /// <summary>
        /// 選択中の値
        /// </summary>
        private object selectedItem;

        /// <summary>
        /// アイテム
        /// </summary>
        private ObservableCollection<KeyValuePair<string, object>> availableItems =
            new ObservableCollection<KeyValuePair<string, object>>();

        /// <summary>
        /// Constructor.
        /// </summary>
        public RadioButtonGroupTestViewModel()
        {
            this.availableItems.Add(new KeyValuePair<string, object>("Radio Button Group Item A", Items.A));
            this.availableItems.Add(new KeyValuePair<string, object>("Radio Button Group Item B", Items.B));
            this.availableItems.Add(new KeyValuePair<string, object>("Radio Button Group Item C", Items.C));
        }

        /// <summary>
        /// The items for the combo box.
        /// </summary>
        public enum Items
        {
            /// <summary>Item A</summary>
            A,

            /// <summary>Item B</summary>
            B,

            /// <summary>Item C</summary>
            C,

            /// <summary>Item D</summary>
            D,

            /// <summary>Item E</summary>
            E,
        }

        /// <summary>
        /// The selected item.
        /// </summary>
        public object SelectedItem
        {
            get { return this.selectedItem; }
            set { this.SetValue(ref this.selectedItem, value); }
        }

        /// <summary>
        /// アイテム
        /// </summary>
        public ObservableCollection<KeyValuePair<string, object>> AvailableItems
        {
            get { return this.availableItems; }
        }
    }
}
