﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Linq;
using System.Windows.Forms;
using EffectMaker.UIControls;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UIControls.ValueConverters;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.EffectMaker.UIControls.TestTypes;

namespace Test.EffectMaker.UIControls.UnitTests
{
    /// <summary>
    /// Test class that contains the data binding test methods.
    /// This class mainly tests the Binder class,
    /// as well as the IControl and IValueConverter implementations.
    /// </summary>
    [TestClass]
    public class DataBindingUnitTests
    {
        /// <summary>
        /// Test with DataContext, with property and matching property type.
        /// </summary>
        [TestMethod]
        public void TestBinder01()
        {
            var txt = new UITextBox();
            var dataSource = new StringViewModel { Value = "A" };

            // bind string to string
            txt.AddBinding("Text", "Value");

            txt.DataContext = dataSource;

            if (txt.Text != "A")
            {
                throw new Exception();
            }

            dataSource.Value = "B";
            if (txt.Text != "B")
            {
                throw new Exception();
            }

            txt.Text = "C";
            if (dataSource.Value != "C")
            {
                throw new Exception();
            }
        }

        /// <summary>
        /// Test with DataContext, with property and matching property type.
        /// </summary>
        [TestMethod]
        public void TestBinder02()
        {
            var button = new UIButton();
            var dataSource = new StringViewModel { Value = "A" };

            // bind string to object
            button.AddBinding("Tag", "Value");

            button.DataContext = dataSource;

            if (button.Tag as string != "A")
            {
                throw new Exception();
            }

            dataSource.Value = "B";
            if (button.Tag as string != "B")
            {
                throw new Exception();
            }

            button.Tag = "C";

            // Tag property does not raise PropertyChanged event
            if (dataSource.Value != "B")
            {
                throw new Exception();
            }
        }

        /// <summary>
        /// Test with DataContext, with property and missmatching property type.
        /// </summary>
        [TestMethod]
        public void TestBinder03()
        {
            var txt = new UITextBox();
            var dataSource = new IntegerViewModel { Value = 32 };

            // bind int to string
            txt.AddBinding("Text", "Value");

            txt.DataContext = dataSource;

            if (txt.Text != "32")
            {
                throw new Exception();
            }

            dataSource.Value = 33;
            if (txt.Text != "33")
            {
                throw new Exception();
            }

            txt.Text = "34";
            if (dataSource.Value != 34)
            {
                throw new Exception();
            }
        }

        /// <summary>
        /// Test with DataContext, with property and missmatching property type.
        /// </summary>
        [TestMethod]
        public void TestBinder04()
        {
            var txt = new UITextBox();

            var dt = DateTime.Now;
            var dataSource = new DateTimeViewModel { Value = dt };

            // bind DateTime to string
            txt.AddBinding("Text", "Value");

            txt.DataContext = dataSource;

            // not sure about the DateTime format used by TypeConverter
            /* depend too much on local machine culture and settings
            if (txt.Text != dt.ToString("yyyy/MM/dd HH:mm"))
            {
                throw new Exception();
            }

            dt = dt.AddYears(1);

            dataSource.Value = dt;
            if (txt.Text != dt.ToString("yyyy/MM/dd HH:mm"))
            {
                throw new Exception();
            }
            */

            txt.Text = "2015/03/25 18:04:00";
            if (dataSource.Value != new DateTime(2015, 3, 25, 18, 4, 0))
            {
                throw new Exception();
            }
        }

        //// ================================================================================

        /// <summary>
        /// Test with DataContext, without property and matching property type.
        /// </summary>
        [TestMethod]
        public void TestBinder05()
        {
            var button = new UIButton();
            var dataSource = new StringViewModel { Value = "A" };

            // bind StringViewModel to string
            button.AddBinding("Text", null);

            button.DataContext = dataSource;

            if (button.Text != dataSource.ToString())
            {
                throw new Exception();
            }

            dataSource.Value = "B";

            // cannot be affected when there is no property
            if (button.Text != dataSource.ToString())
            {
                throw new Exception();
            }

            button.Text = "C";

            // cannot be affected when there is no property
            if (dataSource.Value != "B")
            {
                throw new Exception();
            }
        }

        /// <summary>
        /// Test with DataContext, without property and matching property type.
        /// </summary>
        [TestMethod]
        public void TestBinder06()
        {
            var button = new UIButton();
            var dataSource = new StringViewModel { Value = "A" };

            // bind StringViewModel to object
            button.AddBinding("Tag", null);

            button.DataContext = dataSource;

            if (button.Tag != dataSource)
            {
                throw new Exception();
            }

            dataSource.Value = "B";

            // cannot be affected when there is no property
            if (button.Tag != dataSource)
            {
                throw new Exception();
            }

            button.Tag = "C";

            // cannot be affected when there is no property
            if (dataSource.Value != "B")
            {
                throw new Exception();
            }
        }

        //// ================================================================================

        /// <summary>
        /// Test with DataContext, with property and wrong property type.
        /// </summary>
        [TestMethod]
        public void TestBinder07()
        {
            var txt = new UITextBox();

            var dt = DateTime.Now;
            var dataSource = new ObjectViewModel { Value = (object)dt };

            // bind object to string
            txt.AddBinding("Text", "Value");

            txt.DataContext = dataSource;

            var converter = System.ComponentModel.TypeDescriptor.GetConverter(typeof(DateTime));
            var str = converter.ConvertTo(dt, typeof(string)) as string;
            if (txt.Text != str)
            {
                throw new Exception();
            }

            dataSource.Value = (object)"B";
            if (txt.Text != "B")
            {
                throw new Exception();
            }

            txt.Text = "C";
            if (dataSource.Value as string != "C")
            {
                throw new Exception();
            }
        }

        //// =======================================================================
        //// =======================================================================
        //// =======================================================================

        // All the bellow tests are the same as the above ones, but binding to dynamic properties.

        /// <summary>
        /// Test with DataContext, with property and matching property type.
        /// </summary>
        [TestMethod]
        public void TestBinder08()
        {
            var txt = new UITextBox();
            dynamic dataSource = new DummyViewModel();
            dataSource.StringValue = "A";

            // bind string to string
            txt.AddBinding("Text", "StringValue");

            txt.DataContext = dataSource;

            if (txt.Text != "A")
            {
                throw new Exception();
            }

            dataSource.StringValue = "B";
            if (txt.Text != "B")
            {
                throw new Exception();
            }

            txt.Text = "C";
            if (dataSource.StringValue != "C")
            {
                throw new Exception();
            }
        }

        /// <summary>
        /// Test with DataContext, with property and matching property type.
        /// </summary>
        [TestMethod]
        public void TestBinder09()
        {
            var button = new UIButton();
            dynamic dataSource = new DummyViewModel();
            dataSource.StringValue = "A";

            // bind string to object
            button.AddBinding("Tag", "StringValue");

            button.DataContext = dataSource;

            if (button.Tag as string != "A")
            {
                throw new Exception();
            }

            dataSource.StringValue = "B";
            if (button.Tag as string != "B")
            {
                throw new Exception();
            }

            button.Tag = "C";

            // Tag property does not raise PropertyChanged event
            if (dataSource.StringValue != "B")
            {
                throw new Exception();
            }
        }

        /// <summary>
        /// Test with DataContext, with property and missmatching property type.
        /// </summary>
        [TestMethod]
        public void TestBinder10()
        {
            var txt = new UITextBox();
            dynamic dataSource = new DummyViewModel();
            dataSource.IntegerValue = 32;
            var test = dataSource.StringValue;

            dataSource.StringValue = null;

            // bind int to string
            txt.AddBinding("Text", "IntegerValue");

            txt.DataContext = dataSource;

            if (txt.Text != "32")
            {
                throw new Exception();
            }

            dataSource.IntegerValue = 33;
            if (txt.Text != "33")
            {
                throw new Exception();
            }

            txt.Text = "34";
            if (dataSource.IntegerValue != 34)
            {
                throw new Exception();
            }
        }

        /// <summary>
        /// Test with DataContext, with property and missmatching property type.
        /// </summary>
        [TestMethod]
        public void TestBinder11()
        {
            var txt = new UITextBox();

            var dt = DateTime.Now;
            dynamic dataSource = new DummyViewModel();
            dataSource.DateTimeValue = dt;

            // bind DateTime to string
            txt.AddBinding("Text", "DateTimeValue");

            txt.DataContext = dataSource;

            // not sure about the DateTime format used by TypeConverter
            /* depends too much on local machine culture and settings
            if (txt.Text != dt.ToString("yyyy/MM/dd HH:mm"))
            {
                throw new Exception();
            }

            dt = dt.AddYears(1);

            dataSource.DateTimeValue = dt;
            if (txt.Text != dt.ToString("yyyy/MM/dd HH:mm"))
            {
                throw new Exception();
            }
            */

            txt.Text = "2015/03/25 18:04:00";
            if (dataSource.DateTimeValue != new DateTime(2015, 3, 25, 18, 4, 0))
            {
                throw new Exception();
            }
        }

        //// ================================================================================

        /// <summary>
        /// Test with DataContext, with property and wrong property type.
        /// </summary>
        [TestMethod]
        public void TestBinder12()
        {
            var txt = new UITextBox();

            var dt = DateTime.Now;
            dynamic dataSource = new DummyViewModel();
            dataSource.ObjectValue = (object)dt;

            // bind object to string
            txt.AddBinding("Text", "ObjectValue");

            txt.DataContext = dataSource;

            /* depends too much on local machine culture and settings
            if (txt.Text != dt.ToString("yyyy/MM/dd HH:mm"))
            {
                throw new Exception();
            }
            */

            dataSource.ObjectValue = (object)"B";
            if (txt.Text != "B")
            {
                throw new Exception();
            }

            txt.Text = "C";
            if (dataSource.ObjectValue != "C")
            {
                throw new Exception();
            }
        }

        //// ================================================================================

        /// <summary>
        /// Test with DataContext, with property and matching property type using ValueConverter.
        /// </summary>
        [TestMethod]
        public void TestBinder13()
        {
#if DEBUG
            var control = new FakeControl();
            dynamic dataSource = new DummyViewModel();
            dataSource.StringValue = "A B C D";

            // bind string to string
            var binder = new Binder(control, "TagEx", "StringValue")
            {
                ValueConverter = new SplitValueConverter(),
            };

            control.DataContext = dataSource;

            //// -----------------

            var array = control.TagEx as string[];

            if (array == null || array.Length != 4)
            {
                throw new Exception();
            }

            var testArray = new[] { "A", "B", "C", "D" };
            if (array.SequenceEqual(testArray) == false)
            {
                throw new Exception();
            }

            //// -----------------

            dataSource.StringValue = "E F G H";

            array = control.TagEx as string[];

            if (array == null || array.Length != 4)
            {
                throw new Exception();
            }

            testArray = new[] { "E", "F", "G", "H" };
            if (array.SequenceEqual(testArray) == false)
            {
                throw new Exception();
            }

            //// -----------------

            control.TagEx = new[] { "I", "J", "K", "L" };
            if (dataSource.StringValue != "I J K L")
            {
                throw new Exception();
            }
#endif // DEBUG
        }

        /// <summary>
        /// Test with DataContext, with property and matching property type using ValueConverter.
        /// </summary>
        [TestMethod]
        public void TestBinder14()
        {
            var control = new UICheckBox();
            var dataSource = new BooleanViewModel();
            dataSource.Value = true;

            // bind string to string
            var binder = new Binder(control, "CheckState", "Value")
            {
                ValueConverter = new CheckedCheckStateValueConverter(),
            };

            control.DataContext = dataSource;

            //// -----------------

            if (control.CheckState != CheckState.Checked)
            {
                throw new Exception();
            }

            dataSource.Value = false;

            if (control.CheckState != CheckState.Unchecked)
            {
                throw new Exception();
            }

            //// -----------------

            control.CheckState = CheckState.Indeterminate;
            if (dataSource.Value != true)
            {
                throw new Exception();
            }
        }

        /// <summary>
        /// Test with DataContext, with property and matching property type using ValueConverter.
        /// </summary>
        [TestMethod]
        public void TestBinder15()
        {
            var control = new UICheckBox();
            var dataSource = new BooleanViewModel();
            dataSource.Value = true;

            // bind string to string
            var binder = new Binder(control, "CheckState", "Value")
            {
                ValueConverter = new UncheckedCheckStateValueConverter(),
            };

            control.DataContext = dataSource;

            //// -----------------

            if (control.CheckState != CheckState.Checked)
            {
                throw new Exception();
            }

            dataSource.Value = false;

            if (control.CheckState != CheckState.Unchecked)
            {
                throw new Exception();
            }

            //// -----------------

            control.CheckState = CheckState.Indeterminate;
            if (dataSource.Value != false)
            {
                throw new Exception();
            }
        }

        /// <summary>
        /// Test with DataContext, with property and matching property type using one way mode.
        /// </summary>
        [TestMethod]
        public void TestBinder16()
        {
            var control = new UICheckBox();
            var dataSource = new BooleanViewModel();
            dataSource.Value = true;

            // bind string to string
            var binder = new Binder(control, "Checked", "Value")
            {
                Mode = BindingMode.OneWay,
            };

            control.DataContext = dataSource;

            //// -----------------

            if (control.Checked != true)
            {
                throw new Exception();
            }

            dataSource.Value = false;

            if (control.Checked != false)
            {
                throw new Exception();
            }

            //// -----------------

            control.Checked = true;
            if (dataSource.Value != false)
            {
                throw new Exception();
            }
        }

        /// <summary>
        /// Test with DataContext, with property and matching property type
        /// using one way to source mode.
        /// </summary>
        [TestMethod]
        public void TestBinder17()
        {
            var control = new UICheckBox();
            var dataSource = new BooleanViewModel();
            dataSource.Value = true;

            // bind string to string
            var binder = new Binder(control, "Checked", "Value")
            {
                Mode = BindingMode.OneWayToSource,
            };

            control.DataContext = dataSource;

            //// -----------------

            if (control.Checked != false)
            {
                throw new Exception();
            }

            dataSource.Value = false;

            if (control.Checked != false)
            {
                throw new Exception();
            }

            //// -----------------

            control.Checked = true;
            if (dataSource.Value != true)
            {
                throw new Exception();
            }
        }

        /// <summary>
        /// Test with DataContext, with property and matching property type
        /// using one time mode.
        /// </summary>
        [TestMethod]
        public void TestBinder18()
        {
            var control = new UICheckBox();
            var dataSource = new BooleanViewModel();
            dataSource.Value = true;

            // bind string to string
            var binder = new Binder(control, "Checked", "Value")
            {
                Mode = BindingMode.OneTime,
            };

            control.DataContext = dataSource;

            //// -----------------

            if (control.Checked != true)
            {
                throw new Exception();
            }

            dataSource.Value = false;

            if (control.Checked != true)
            {
                throw new Exception();
            }

            dataSource.Value = true;

            //// -----------------

            control.Checked = false;
            if (dataSource.Value != true)
            {
                throw new Exception();
            }
        }

        /// <summary>
        /// Test with DataContext, with property and matching property type
        /// using one time mode with different update timings.
        /// </summary>
        [TestMethod]
        public void TestBinder19()
        {
            var control = new UICheckBox();
            var dataSource = new BooleanViewModel();
            dataSource.Value = true;
            control.DataContext = dataSource;

            // bind string to string
            var binder = new Binder(control, "Checked", "Value")
            {
                Mode = BindingMode.OneTime,
            };

            binder.UpdateElement();

            //// -----------------

            if (control.Checked != true)
            {
                throw new Exception();
            }

            dataSource.Value = false;

            if (control.Checked != true)
            {
                throw new Exception();
            }

            dataSource.Value = true;

            //// -----------------

            control.Checked = false;
            if (dataSource.Value != true)
            {
                throw new Exception();
            }
        }

        /// <summary>
        /// Test setting DataContext through binding.
        /// </summary>
        [TestMethod]
        public void TestBinder20()
        {
            var subViewModel1 = new SubViewModel1();
            var subViewModel2 = new SubViewModel2();

            var rootPanel = new UIPanel { Name = "rootPanel" };

            var buttonA = new UIButton { Name = "buttonA" };
            var buttonB = new UIButton { Name = "buttonB" };
            var buttonC = new UIButton { Name = "buttonC" };

            var subViewModelPanel = new UIPanel { Name = "subViewModelPanel" };

            var buttonValue1 = new UIButton { Name = "buttonValue1" };
            var buttonValue2 = new UIButton { Name = "buttonValue2" };

            var leafPanel = new UIPanel { Name = "leafPanel" };

            var leafButton1 = new UIButton { Name = "leafButton1" };
            var leafButton2 = new UIButton { Name = "leafButton2" };

            subViewModelPanel.Controls.Add(leafPanel);

            leafPanel.Controls.Add(leafButton1);
            leafPanel.Controls.Add(leafButton2);

            //// ----------------------------------

            rootPanel.Controls.Add(buttonA);
            rootPanel.Controls.Add(buttonB);
            rootPanel.Controls.Add(buttonC);
            rootPanel.Controls.Add(subViewModelPanel);

            subViewModelPanel.Controls.Add(buttonValue1);
            subViewModelPanel.Controls.Add(buttonValue2);

            ((dynamic)subViewModel1.A).StringValue = "Text A";
            ((dynamic)subViewModel1.B).StringValue = "Text B";
            ((dynamic)subViewModel1.C).StringValue = "Text C";

            subViewModel1.SubViewModel.Value = 51.0f;
            subViewModel1.SubViewModel.StringViewModel.Value = "Text Leaf";

            buttonA.AddBinding("DataContext", "A");
            buttonA.AddBinding("Text", "StringValue");

            buttonB.AddBinding("DataContext", "B");
            buttonB.AddBinding("Text", "StringValue");

            buttonC.AddBinding("DataContext", "C");
            buttonC.AddBinding("Text", "StringValue");

            subViewModelPanel.AddBinding("DataContext", "SubViewModel");

            buttonValue1.AddBinding("Text", "Value");

            leafButton1.AddBinding("DataContext", "StringViewModel");
            leafButton1.AddBinding("Text", "Value");
            leafButton2.AddBinding("Text", "Value");

            rootPanel.DataContext = subViewModel1;

            if (buttonA.Text != "Text A")
            {
                throw new Exception();
            }

            if (buttonB.Text != "Text B")
            {
                throw new Exception();
            }

            if (buttonC.Text != "Text C")
            {
                throw new Exception();
            }

            ((dynamic)subViewModel1.A).StringValue = "Text AAA";

            if (buttonA.Text != "Text AAA")
            {
                throw new Exception();
            }

            if (buttonValue1.Text != "51")
            {
                throw new Exception();
            }

            if (buttonValue2.Text != string.Empty)
            {
                throw new Exception();
            }

            if (leafButton1.Text != "Text Leaf")
            {
                throw new Exception();
            }

            if (leafButton2.Text != "51")
            {
                throw new Exception();
            }

            var newDataContext = new SubViewModel1();

            ((dynamic)newDataContext.A).StringValue = "New Text A";
            ((dynamic)newDataContext.B).StringValue = "New Text B";
            ((dynamic)newDataContext.C).StringValue = "New Text C";

            newDataContext.SubViewModel.Value = 37.0f;
            newDataContext.SubViewModel.StringViewModel.Value = "New Text Leaf";

            //// SECOND TIME DATACONTEXT SET

            rootPanel.DataContext = newDataContext;

            //// --------------------------------------

            if (buttonA.Text != "New Text A")
            {
                throw new Exception();
            }

            if (buttonB.Text != "New Text B")
            {
                throw new Exception();
            }

            if (buttonC.Text != "New Text C")
            {
                throw new Exception();
            }

            if (buttonValue1.Text != "37")
            {
                throw new Exception();
            }

            if (buttonValue2.Text != string.Empty)
            {
                throw new Exception();
            }

            if (leafButton1.Text != "New Text Leaf")
            {
                throw new Exception();
            }

            if (leafButton2.Text != "37")
            {
                throw new Exception();
            }
        }
    }
}
