﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Windows.Forms;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UIControls.Xaml;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.EffectMaker.UIControls.TestTypes;

namespace Test.EffectMaker.UIControls.UnitTests
{
    /// <summary>
    /// Test class for behaviors
    /// </summary>
    [TestClass]
    public class BehaviorUnitTests
    {
        /// <summary>
        /// Test a simple behavior
        /// </summary>
        [TestMethod]
        public void BehaviorTest01()
        {
            var button = new UIButton();
            button.Text = "Original";

            var behavior = new ChangeTextBehavior("Different");
            button.AddBehavior(behavior);

            Assert.AreEqual(button.Text, "Different");

            behavior.Detach();

            Assert.AreEqual(button.Text, "Original");
        }

        /// <summary>
        /// Test a simple behavior
        /// </summary>
        [TestMethod]
        public void BehaviorTest02()
        {
            var button = new UIButton();
            button.Text = "Original";

            button.AddBehavior(new ChangeTextBehavior("Different"));

            Assert.AreEqual(button.Text, "Different");

            button.Behaviors.Dispose();

            Assert.AreEqual(button.Text, "Original");
        }

        /// <summary>
        /// Test a simple behavior
        /// </summary>
        [TestMethod]
        public void BehaviorTest03()
        {
            var button1 = new UIButton { Text = "Original1" };
            var button2 = new UIButton { Text = "Original2" };

            var behavior1 = new ChangeTextBehavior("Different1");
            var behavior2 = new ChangeTextBehavior("Different2");

            button1.AddBehavior(behavior1);
            button2.AddBehavior(behavior2);

            Assert.AreEqual(button1.Text, "Different1");

            behavior1.Detach();

            Assert.AreEqual(button1.Text, "Original1");
            Assert.AreEqual(button2.Text, "Different2");

            behavior2.Detach();

            Assert.AreEqual(button2.Text, "Original2");
        }

        /// <summary>
        /// Test a simple behavior
        /// </summary>
        [TestMethod]
        public void BehaviorTest04()
        {
            var button = new UIButton { Text = "Original" };

            //// attach the two following behaviors to the same button

            var behavior1 = new ChangeTextBehavior("Different1");
            var behavior2 = new ChangeTextBehavior("Different2");

            button.AddBehavior(behavior1);

            Assert.AreEqual(button.Text, "Different1");

            button.AddBehavior(behavior2);

            Assert.AreEqual(button.Text, "Different2");

            behavior2.Detach();

            Assert.AreEqual(button.Text, "Different1");

            behavior1.Detach();

            Assert.AreEqual(button.Text, "Original");
        }

        /// <summary>
        /// Test a simple behavior
        /// </summary>
        [TestMethod]
        public void BehaviorTest05()
        {
            var button = new UIButton { Text = "Original" };

            //// attach the two following behaviors to the same button

            var behavior1 = new ChangeTextBehavior("Different1");
            var behavior2 = new ChangeTextBehavior("Different2");

            button.AddBehavior(behavior1);

            Assert.AreEqual(button.Text, "Different1");

            button.AddBehavior(behavior2);

            Assert.AreEqual(button.Text, "Different2");

            // detach first attached behavior before second one
            behavior1.Detach();

            Assert.AreEqual(button.Text, "Original");

            // detach second behavior at last
            behavior2.Detach();

            Assert.AreEqual(button.Text, "Different1");
        }

        /// <summary>
        /// Test a simple behavior
        /// </summary>
        [TestMethod]
        public void BehaviorTest06()
        {
            var viewModel = new DummyViewModel();

            IControl rootControl = null;

            try
            {
                // XAMLリソースを読み込む
                rootControl = (IControl)this.LoadControl("test_behavior.xaml");
            }
            catch (Exception ex)
            {
                throw new Exception("Failed to load XAML.", ex);
            }

            rootControl.DataContext = viewModel;

            var flattenTree = rootControl.FlattenControlTree();

            var button = (UIButton)flattenTree["btn"];
            var behavior = (ButtonBehavior)button.Behaviors.ElementAt(0);

            Assert.AreEqual(viewModel.Proxy.IntegerValue, 0);
            Assert.AreEqual(behavior.Value, 0);

            ((dynamic)viewModel).IntegerValue = 3;

            Assert.AreEqual(viewModel.Proxy.IntegerValue, 3);
            Assert.AreEqual(behavior.Value, 3);

            button.PerformClick();

            Assert.AreEqual(viewModel.Proxy.IntegerValue, 4);
            Assert.AreEqual(behavior.Value, 4);
        }

        /// <summary>
        /// Loads a control from a XAML file using a name relative to the executable directory.
        /// </summary>
        /// <param name="relativeResourceName">Relative XAML resource name.</param>
        /// <returns>Returns a control, or throws an exception.</returns>
        private Control LoadControl(string relativeResourceName)
        {
            Control control;

            var assembly = Assembly.GetExecutingAssembly();

            using (Stream stream = ResourceUtility.LoadEmbedded(assembly, relativeResourceName))
            {
                control = XamlLoader.Load(stream);
            }

            return control;
        }
    }
}
