﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/typeDef2.h>
#if EFT_IS_WIN

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cmath>
#include <gl/glew.h>

#include "tsu_SavecurrentBufferAsBmp.h"

enum
{
    FILEHEADERSIZE = 14,                //ファイルヘッダのサイズ
    INFOHEADERSIZE = 40                 //情報ヘッダのサイズ

};
#define HEADERSIZE (FILEHEADERSIZE+INFOHEADERSIZE)


typedef struct
{
    unsigned char b;
    unsigned char g;
    unsigned char r;
}Rgb;

typedef struct
{
    unsigned int height;
    unsigned int width;
    Rgb *data;
}Image;



bool Write_Bmp( const char *filename, Image *img, bool linear )
{
    int i, j;
    FILE *fp;
    int real_width;
    unsigned char *bmp_line_data;
    unsigned char header_buf[HEADERSIZE];
    unsigned int file_size;
    unsigned int offset_to_data;
    unsigned long info_header_size;
    unsigned int planes;
    unsigned int color;
    unsigned long compress;
    unsigned long data_size;
    long xppm;
    long yppm;
    bool ret = false;

    fopen_s( &fp, filename, "wb" );
    if ( fp == NULL )
    {
        return ret;
    }

    real_width = img->width * 3 + img->width % 4;

    // ヘッダ作成
    file_size = img->height * real_width + HEADERSIZE;
    offset_to_data      = HEADERSIZE;
    info_header_size    = INFOHEADERSIZE;
    planes              = 1;
    color               = 24;
    compress            = 0;
    data_size           = img->height * real_width;
    xppm                = 1;
    yppm                = 1;

    header_buf[0]       = 'B';
    header_buf[1]       = 'M';
    memcpy(header_buf + 2, &file_size, sizeof(file_size));
    header_buf[6]       = 0;
    header_buf[7]       = 0;
    header_buf[8]       = 0;
    header_buf[9]       = 0;
    memcpy(header_buf + 10, &offset_to_data, sizeof(file_size));
    header_buf[11]      = 0;
    header_buf[12]      = 0;
    header_buf[13]      = 0;

    memcpy(header_buf + 14, &info_header_size, sizeof(info_header_size));
    header_buf[15]      = 0;
    header_buf[16]      = 0;
    header_buf[17]      = 0;
    memcpy(header_buf + 18, &img->width, sizeof(img->width));
    memcpy(header_buf + 22, &img->height, sizeof(img->height));
    memcpy(header_buf + 26, &planes, sizeof(planes));
    memcpy(header_buf + 28, &color, sizeof(color));
    memcpy(header_buf + 30, &compress, sizeof(compress));
    memcpy(header_buf + 34, &data_size, sizeof(data_size));
    memcpy(header_buf + 38, &xppm, sizeof(xppm));
    memcpy(header_buf + 42, &yppm, sizeof(yppm));
    header_buf[46]      = 0;
    header_buf[47]      = 0;
    header_buf[48]      = 0;
    header_buf[49]      = 0;
    header_buf[50]      = 0;
    header_buf[51]      = 0;
    header_buf[52]      = 0;
    header_buf[53]      = 0;

    // ヘッダの書き込み
    fwrite(header_buf, sizeof(unsigned char), HEADERSIZE, fp);

    bmp_line_data = reinterpret_cast<unsigned char*>( malloc(sizeof(unsigned char) * real_width) );
    if ( !bmp_line_data )
    {
        goto exit;
    }

    // RGB情報の書き込み
    for(i=0; i<(int)img->height; i++)
    {
        for(j=0; j<(int)img->width; j++)
        {
            if ( linear )
            {
                double dr = img->data[i * img->width + j].r / 255.0;
                double dg = img->data[i * img->width + j].g / 255.0;
                double db = img->data[i * img->width + j].b / 255.0;
                bmp_line_data[j * 3]      = static_cast<u32>( pow( dr, 1.0 / 2.2 ) * 255.f );
                bmp_line_data[j * 3 + 1]  = static_cast<u32>( pow( dg, 1.0 / 2.2 ) * 255.f );
                bmp_line_data[j * 3 + 2]  = static_cast<u32>( pow( db, 1.0 / 2.2 ) * 255.f );
            }
            else
            {
                bmp_line_data[j * 3]      = img->data[i * img->width + j].r;
                bmp_line_data[j * 3 + 1]  = img->data[i * img->width + j].g;
                bmp_line_data[j * 3 + 2]  = img->data[i * img->width + j].b;
            }
        }

        for( j=img->width * 3; j<real_width; j++ )
        {
            bmp_line_data[j] = 0;
        }

        fwrite(bmp_line_data, sizeof(unsigned char), real_width, fp);
    }

    free(bmp_line_data);
    ret = true;

exit:
    fclose(fp);
    return ret;
}


void SaveCurrentBufferAsBmpFile( const char* filename, unsigned int width, unsigned int height, bool linear )
{
    GLubyte* pixel_data = reinterpret_cast<GLubyte*>( malloc(((size_t)width) * ((size_t)height) * 3 * (sizeof(GLubyte))) );

    glPixelStorei( GL_PACK_ALIGNMENT, 1 );
    glReadPixels( 0,0,
                  (GLsizei)width,(GLsizei)height,
                  GL_RGB,
                  GL_UNSIGNED_BYTE,
                  pixel_data);

    Image image;
    image.height    = height;
    image.width     = width;
    image.data      = (Rgb *)pixel_data;
    Write_Bmp( filename, &image, linear );
    free(pixel_data);
}

#endif
