﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <eftdemo_GlareEffect.h>
#include <eftdemo_Texture.h>

namespace nw      {
namespace eftdemo {


// ガウスフィルタの重みを計算
static void CalcWeight( f32* weight, f32 mpp, f32 sigma );


//---------------------------------------------------------------------------
//! @brief        コンストラクタです。
//---------------------------------------------------------------------------
GlareEffect::GlareEffect()
{
}


//---------------------------------------------------------------------------
//! @brief        初期化処理です。
//!
//! @param[in] allocator フレームバッファに使用するアロケータ。
//! @param[in] width     画面の横幅。
//! @param[in] height    画面の縦幅。
//---------------------------------------------------------------------------
void GlareEffect::Initialize( nw::ut::MemoryAllocator* allocator, f32 width, f32 height )
{
    // フレームバッファのスケール
    f32 scale[GLARE_BUFFER_MAX] = {
        1.0f /  4.0f,
        1.0f /  8.0f,
        1.0f / 16.0f,
        1.0f / 32.0f,
        1.0f / 64.0f
    };

    // フレームバッファを初期化
    for ( int i = 0; i < GLARE_BUFFER_MAX; ++i )
    {
        mFrameBufferTextureGlareX[i].Initialize( allocator, width * scale[i],  height * scale[i], nw::eftdemo::FrameBuffer::FRAMEBUFFER_TYPE_FLOAT16 );

        mFrameBufferTextureGlareY[i].Initialize( allocator, width * scale[i],  height * scale[i], nw::eftdemo::FrameBuffer::FRAMEBUFFER_TYPE_FLOAT16 );
    }

    // ガウスフィルタの重みを計算
    for ( int i = 0; i < GLARE_BUFFER_MAX; ++i )
    {
        f32 mpp = ( 1.0f / scale[i] ) / 1.6f;
        CalcWeight( mWeightValue[i], mpp, 3.0f * mpp );
    }
}


//---------------------------------------------------------------------------
//! @brief        描画処理です。
//!
//! @param[in] drawer             スクリーン描画クラス
//! @param[in] currentFrameBuffer フレームバッファ
//! @param[in] glareNum           利用するグレアバッファの数
//---------------------------------------------------------------------------
void GlareEffect::Draw( nw::eftdemo::ScreenDrawer* drawer, nw::eftdemo::FrameBuffer* currentFrameBuffer, u32 glareNum )
{
    nw::gfnd::Graphics* graphics = nw::gfnd::Graphics::GetInstance();

    graphics->SetBlendEnable( false );
    graphics->SetDepthEnable( false, false );
    graphics->SetAlphaTestEnable( false );

    {
        // グレアバッファ0に切り替えて
        mFrameBufferTextureGlareY[0].Bind();
        mFrameBufferTextureGlareY[0].Clear( true, false );

        // currentFrameBuffer を縮小しつつ、輝度抽出しながらコピー
        drawer->DrawScreen( nw::eftdemo::ScreenDrawer::SHADER_TYPE_LUMINANCE_SELECTION, currentFrameBuffer );
    }

    for ( u32 i = 1; i < glareNum; ++i )
    {
        // グレアバッファiに切り替えて
        mFrameBufferTextureGlareY[i].Bind();
        mFrameBufferTextureGlareY[i].Clear( true, false );

        // mFrameBufferTextureGlareY[i-1] を縮小してコピー
        drawer->DrawScreen( nw::eftdemo::ScreenDrawer::SHADER_TYPE_COPY, &mFrameBufferTextureGlareY[i - 1] );
    }

    // 横方向にガウスをかけてコピー
    for ( u32 i = 0; i < glareNum; ++i )
    {
        drawer->SetWeight( mWeightValue[i] );

        mFrameBufferTextureGlareX[i].Bind();
        mFrameBufferTextureGlareX[i].Clear( true, false );

        mFrameBufferTextureGlareY[i].Invalidate( true, false );
        drawer->DrawScreen( nw::eftdemo::ScreenDrawer::SHADER_TYPE_GAUSS_X, &mFrameBufferTextureGlareY[i] );
    }

    // 縦方向にガウスをかけてコピー
    for ( u32 i = 0; i < glareNum; ++i )
    {
        drawer->SetWeight( mWeightValue[i] );

        mFrameBufferTextureGlareY[i].Bind();
        mFrameBufferTextureGlareY[i].Clear( true, false );

        mFrameBufferTextureGlareX[i].Invalidate( true, false );
        drawer->DrawScreen( nw::eftdemo::ScreenDrawer::SHADER_TYPE_GAUSS_Y, &mFrameBufferTextureGlareX[i] );
    }

    // グレアバッファを合成
    graphics->SetBlendEnable( true );
    graphics->SetBlendFactorSeparate( nw::gfnd::Graphics::BLEND_FACTOR_ONE,
                                      nw::gfnd::Graphics::BLEND_FACTOR_ONE,
                                      nw::gfnd::Graphics::BLEND_FACTOR_ONE,
                                      nw::gfnd::Graphics::BLEND_FACTOR_ONE );

    for ( u32 i = glareNum - 1; i > 0; --i )
    {
        mFrameBufferTextureGlareY[i - 1].Bind();
        mFrameBufferTextureGlareY[i - 1].Invalidate( true, false );

        mFrameBufferTextureGlareY[i].Invalidate( true, false );
        drawer->DrawScreen( nw::eftdemo::ScreenDrawer::SHADER_TYPE_COPY, &mFrameBufferTextureGlareY[i] );
    }

    // フレームバッファに戻す。
    currentFrameBuffer->Bind();

    mFrameBufferTextureGlareY[0].Invalidate( true, false );
    drawer->DrawScreen( nw::eftdemo::ScreenDrawer::SHADER_TYPE_COPY, &mFrameBufferTextureGlareY[0] );

    graphics->SetBlendFactorSeparate( nw::gfnd::Graphics::BLEND_FACTOR_SRC_ALPHA,
                                      nw::gfnd::Graphics::BLEND_FACTOR_INV_SRC_ALPHA,
                                      nw::gfnd::Graphics::BLEND_FACTOR_SRC_ALPHA,
                                      nw::gfnd::Graphics::BLEND_FACTOR_INV_SRC_ALPHA );
}


//---------------------------------------------------------------------------
//! @brief        ガウスフィルタの重みを計算します。
//!
//! @param[out] weight 計算した重み
//! @param[in]  mpp    グレアバッファのスケール
//! @param[in]  sigma  σ
//---------------------------------------------------------------------------
static void CalcWeight( f32* weight, f32 mpp, f32 sigma )
{
    f32 sum = 0.0f;

    for ( int i = 0; i < ScreenDrawer::GAUSS_LENGTH; ++i )
    {
        f32 r = ( 2.0f * i ) * mpp;
        if ( i == 0 )
        {
            r = 3.0f * mpp;
        }

        weight[i] = expf(- (r * r) / (2 * sigma * sigma));
        sum += weight[i] * mpp;
    }

    for ( int i = 0; i < ScreenDrawer::GAUSS_LENGTH; ++i )
    {
        weight[i] /= sum;
    }
}

} // namespace nw::eftdemo
} // namespace nw
