﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace ConverterTestApp
{
    internal class ConverterTest
    {
        private static readonly List<string> ConvXmlList = new List<string>();

        /// <summary>
        /// ファイル検索
        /// </summary>
        /// <param name="inDir">ディレクトリパス</param>
        /// <param name="fileName">検索するファイル名</param>
        public static void SearchFile(string inDir, string fileName)
        {
            try
            {
                foreach (var dir in Directory.GetDirectories(inDir))
                {
                    foreach (var file in Directory.GetFiles(dir, fileName))
                    {
                        ConvXmlList.Add(file);
                    }
                    SearchFile(dir, fileName);
                }
            }
            catch (System.Exception excpt)
            {
                Console.WriteLine(excpt.Message);
            }
        }

        /// <summary>
        /// テストモード
        /// </summary>
        [Flags]
        private enum TestModeFlags : int
        {
            None = 0,        // なし
            Convert = 1,        // コンバートテスト
            Compare = 1 << 1,   // 比較テスト
        }

        /// <summary>
        /// メイン
        /// </summary>
        private static int Main(string[] args)
        {
            var testMode        = TestModeFlags.None;           // テストモード
            var exePath         = string.Empty;                           // EffectMaker.exeのパス
            var outputDir       = string.Empty;                           // 出力ディレクトリ
            var originalDir     = string.Empty;                           // オリジナルディレクトリ
            var cmd             = string.Empty;                           // EffectMakerに渡すコマンド
            var convertDirArray = new List<string>();           // コンバートするディレクトリ配列
            var inputDirArray   = new string[] { };             // コンバートするディレクトリ
            var platformArray   = new string[] { };             // プラットフォーム配列
            var flagRecursive   = false;                        // 再帰的に検索するかのフラグ
            const string ConvertXmlFileName = "convlist.xml";   // コンバートリストファイル名

            var convertTest = new ConvertTest();    // コンバートテスト
            var binaryTest = new BinaryTest();      // バイナリ比較テスト

            for (var i = 0; i < args.Count(); i++)
            {
                // コンバートテスト
                if (args[i] == "--convert-test")
                {
                    testMode |= TestModeFlags.Convert;
                }

                // 比較テスト
                if (args[i] == "--compare-test")
                {
                    testMode |= TestModeFlags.Compare;
                }

                // EffectMaker.exeのパス
                if (args[i] == "--exe")
                {
                    exePath = args[i + 1];
                }

                // コンバートするディレクトリ
                if (args[i] == "--input")
                {
                    inputDirArray = args[i + 1].Split(',');
                }

                // 出力ディレクトリ
                if (args[i] == "--output")
                {
                    outputDir = args[i + 1];
                }

                if (args[i] == "--original-dir")
                {
                    originalDir = args[i + 1];
                }

                // 再帰的に検索
                if (args[i] == "--recursive")
                {
                    flagRecursive = true;
                }

                // プラットフォーム
                if (args[i] == "--platforms")
                {
                    platformArray = args[i + 1].Split(',');
                }

                // EffectMakerに渡すコマンド
                if (args[i] == "--command")
                {
                    cmd = args[i + 1];
                }
            }

            // convlist.xmlを検索
            foreach (var inputDir in inputDirArray)
            {
                if (flagRecursive)
                {
                    SearchFile(inputDir, ConvertXmlFileName);

                    convertDirArray.AddRange(ConvXmlList.Select(System.IO.Path.GetDirectoryName));
                }
                else
                {
                    var convListPath = inputDir + "/" + ConvertXmlFileName;
                    if (!Utility.CheckFile(convListPath))
                    {
                        continue;
                    }
                    convertDirArray.Add(inputDir);
                }
            }

            var ret = 0;
            // テスト開始
            foreach (var dir in convertDirArray)
            {
                var uAttribute = System.IO.File.GetAttributes(dir);

                // 本当にディレクトリか判断する
                if ((uAttribute & System.IO.FileAttributes.Directory) != System.IO.FileAttributes.Directory)
                {
                    Console.WriteLine("ディレクトリではありません：" + dir);
                    continue;
                }

                foreach (var platform in platformArray)
                {
                    var fileName = System.IO.Path.GetFileName(dir) + "_" + platform + ".ptcl";
                    // 出力ファイルの設定
                    var outfile = outputDir + "/" + fileName;

                    // コンバート
                    if ((testMode & TestModeFlags.Convert) != 0)
                    {
                        if (!convertTest.Convert(exePath, dir, ConvertXmlFileName, outfile, platform, cmd))
                        {
                            ret = 1;
                        }
                    }

                    // バイナリ比較
                    if ((testMode & TestModeFlags.Compare) == 0)
                    {
                        continue;
                    }

                    // オリジナルファイル
                    var originalFile = originalDir + "/" + fileName;

                    if (!binaryTest.CompareBinary(originalFile, outfile))
                    {
                        Console.WriteLine("一致しません");
                        Console.WriteLine("  ファイル1：" + originalFile);
                        Console.WriteLine("  ファイル2：" + outfile);
                        ret = 1;
                    }
                }
            }

            return ret;
        }
    }
}
