﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Web.Script.Serialization;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using TestUtility;


namespace DevMenuCommandTest
{
    /*
     * 再起動不要NUPが伴うテストを追加するためのクラス
     */
    [TestClass]
    public class RebootlessSystemUpdateTest : TestBase
    {
        private const ulong TestAppId = 0x0100ccccaaaa0000;
        private const string TestAppIdString = "0x0100ccccaaaa0000";
        private const ulong TestSystemUpdateId = 0x0100000000000816;
        private const string TestSystemUpdateIdString = "0x0100000000000816";
        private const ulong TestBlackListId = 0x0100000000000825;
        private const string TestBlackListIdString = "0x0100000000000825";

        private const int BlackListTargetVersion = 2;
        private const int SendSystemAndPatchVersion = 4;

        private static List<NspInfo> AppList;
        private static UInt32 OriginalSystemUpdateVersion;
        private static UInt32 LupSystemUpdateVersion;
        private static UInt32 RebootlessSystemUpdateVersion;
        private static NspInfo OriginalSystemUpdate;
        private static NspInfo LupSystemUpdate;
        private static NspInfo RebootlessSystemUpdate;
        private static NspInfo BlackList;
        private const UInt32 BlackListVersion = 0xFFFFFFFF;

        private static string OriginalBlackListPath;


        [ClassInitialize]
        public static void ClassInitialize(TestContext context)
        {
            SetupTestEnvironment(context);

            AppList = new List<NspInfo>();

            AppList.Add(AppMaker.MakeApplication(TestAppId, additionalOptions: "--small-code"));
            AppList.Add(AppMaker.MakePatch(TestAppId, 1, AppList[0].Path, additionalOptions: "--small-code"));
            AppList.Add(AppMaker.MakePatch(TestAppId, 2, AppList[0].Path, additionalOptions: "--small-code"));
            AppList.Add(AppMaker.MakePatch(TestAppId, 3, AppList[0].Path, additionalOptions: "--small-code"));

            OriginalBlackListPath = Path.Combine(new TestPath(context).GetSigloRoot(), "Programs/Eris/Outputs/NX-NXFP2-a64/SystemData/ApplicationBlackList/Release/ApplicationBlackList.nsp");

            var makeNsData = new MakeNsData(context);
            var blacklistRecordList = new List<MakeNsData.BlackListRecord>();
            blacklistRecordList.Add(new MakeNsData.BlackListRecord(TestAppId, BlackListTargetVersion << 16));
            BlackList = makeNsData.MakeBlackListNsp(AppMaker.GetOutputDirectory().Replace("\"", ""), "blacklist", TestBlackListIdString, BlackListVersion, blacklistRecordList);

            var systemProgramList = GetSystemProgramList(HostCommand);
            var systemUpdateInfo = systemProgramList.Single(x => x.id.ToLower() == TestSystemUpdateIdString.ToLower());
            OriginalSystemUpdateVersion = systemUpdateInfo.version;

            OriginalSystemUpdate = SysMaker.MakeTestSystemUpdateMeta(TestSystemUpdateId, OriginalSystemUpdateVersion, new MakeTestSystemContent.ContentMetaInfo[] { });
            LupSystemUpdateVersion = OriginalSystemUpdateVersion + 1;
            LupSystemUpdate = SysMaker.MakeTestSystemUpdateMeta(TestSystemUpdateId, LupSystemUpdateVersion, new MakeTestSystemContent.ContentMetaInfo[] { });
            RebootlessSystemUpdateVersion = LupSystemUpdateVersion + 1;
            RebootlessSystemUpdate = SysMaker.MakeTestSystemUpdateMeta(TestSystemUpdateId, RebootlessSystemUpdateVersion, new MakeTestSystemContent.ContentMetaInfo[]
            {
                new MakeTestSystemContent.ContentMetaInfo(BlackList, BlackListVersion),
            });
            AppList.Add(AppMaker.MakePatch(TestAppId, SendSystemAndPatchVersion, AppList[0].Path, requiredSystemVersion: LupSystemUpdateVersion, additionalOptions: "--small-code"));

            CleanupImpl(context);
        }

        public static void CleanupImpl(TestContext context)
        {
            var cleanupCommandList = new string[]
            {
                string.Format("systemprogram install {0} --force", OriginalSystemUpdate.Path),
                string.Format("systemprogram install {0} --force", OriginalBlackListPath),
            };

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;
                command.Execute(cleanupCommandList, ignoreError: true);
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;
                command.Execute(cleanupCommandList, ignoreError: true);
            });

            hostTask.Wait();
            clientTask.Wait();

            CleanupBase(context);
        }

        [TestCleanup]
        public void TestCleanup()
        {
            CleanupImpl(this.TestContext);
        }

        private void SwitchToTestSystem(DevMenuCommandSystem command, NspInfo system)
        {
            // 前の環境をリセット
            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("systemprogram uninstall {0}", TestSystemUpdateIdString),
                string.Format("systemprogram install {0}", system.Path),
            }));
            ForceReset(command);
        }

        private void UpdateSystemRebootless(DevMenuCommandSystem command)
        {
            // 前の環境をリセット
            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("systemprogram uninstall {0} ", TestSystemUpdateIdString),
                string.Format("systemprogram install {0}", RebootlessSystemUpdate.Path),
                string.Format("systemprogram uninstall {0}", TestBlackListIdString),
                string.Format("systemprogram install {0}", BlackList.Path),
                "systemupdate notify-systemdata-update"
            }));
        }

        private void InstallTestApplication(DevMenuCommandSystem command, int version, string appStorage = null, string patchStorage = null)
        {
            InstallTestApplicationImpl(command, version, AppList, appStorage, patchStorage);
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void SendBlackListPatch()
        {
            var barrier = new Barrier(2);
            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;

                Action successTest = () =>
                {
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare send-application {0} --timeout 300", TestAppIdString)
                    }));
                    {
                        var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                    }
                };

                Action failureTest = () =>
                {
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare send-application {0} --timeout 300", TestAppIdString)
                    }));
                    {
                        var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.ContentsShareFailed, output.state);
                        Assert.AreEqual((UInt32)LcsFailureReason.UnshareableContents, output.reason);
                    }
                };

                InstallTestApplication(command, BlackListTargetVersion);

                // 通常は送れることの確認
                barrier.SignalAndWait();

                successTest();

                // BlackList のバージョンと同じものは送れない

                UpdateSystemRebootless(command);

                barrier.SignalAndWait();

                failureTest();

                // BlackList のバージョンより小さいものは送れない

                InstallTestApplication(command, BlackListTargetVersion - 1);

                barrier.SignalAndWait();

                failureTest();

                // BlackList のバージョンより新しいものは送れる

                InstallTestApplication(command, BlackListTargetVersion + 1);

                barrier.SignalAndWait();

                successTest();

                // マスターが受信者になる場合も受け取れない
                InstallTestApplication(command, BlackListTargetVersion - 1);

                barrier.SignalAndWait();

                failureTest();
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                Action successTest = () =>
                {
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare receive-application {0} --timeout 300", TestAppIdString)
                    }));
                    {
                        var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                    }
                };

                Action failureTest = () =>
                {
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare receive-application {0} --timeout 300", TestAppIdString)
                    }));
                    {
                        var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.ContentsShareFailed, output.state);
                        Assert.AreEqual((UInt32)LcsFailureReason.UnshareableContents, output.reason);
                    }
                };


                InstallTestApplication(command, 0);

                barrier.SignalAndWait();

                // 成功
                successTest();

                // patch == BlackListVersion
                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("patch uninstall {0} {1}", TestAppIdString, HasClientSdCard ? "-s sdcard" : "")
                }));
                barrier.SignalAndWait();

                failureTest();

                // patch < BlackListVersion
                barrier.SignalAndWait();

                failureTest();

                // patch > BlackListVersion
                barrier.SignalAndWait();

                successTest();

                // Send

                InstallTestApplication(command, BlackListTargetVersion);
                barrier.SignalAndWait();

                failureTest();

            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void ReceiverHasBlacklist()
        {
            var barrier = new Barrier(2);
            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;

                Action successTest = () =>
                {
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare send-application {0} --timeout 300", TestAppIdString)
                    }));
                    {
                        var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                    }
                };

                Action failureTest = () =>
                {
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare send-application {0} --timeout 300", TestAppIdString)
                    }));
                    {
                        var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.ContentsShareFailed, output.state);
                        Assert.AreEqual((UInt32)LcsFailureReason.UnshareableContents, output.reason);
                    }
                };

                InstallTestApplication(command, BlackListTargetVersion);

                // BlackList のバージョンと同じものは送れない

                barrier.SignalAndWait();

                failureTest();

                // BlackList のバージョンより小さいものは送れない

                InstallTestApplication(command, BlackListTargetVersion - 1);

                barrier.SignalAndWait();

                failureTest();

                // BlackList のバージョンより新しいものは送れる

                InstallTestApplication(command, BlackListTargetVersion + 1);

                barrier.SignalAndWait();

                successTest();

                // マスターが受信者になる場合も受け取れない
                InstallTestApplication(command, BlackListTargetVersion - 1);

                barrier.SignalAndWait();

                failureTest();
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                Action successTest = () =>
                {
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare receive-application {0} --timeout 300", TestAppIdString)
                    }));
                    {
                        var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                    }
                };

                Action failureTest = () =>
                {
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare receive-application {0} --timeout 300", TestAppIdString)
                    }));
                    {
                        var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.ContentsShareFailed, output.state);
                        Assert.AreEqual((UInt32)LcsFailureReason.UnshareableContents, output.reason);
                    }
                };


                InstallTestApplication(command, 0);

                UpdateSystemRebootless(command);

                barrier.SignalAndWait();

                // patch == BlackListVersion
                failureTest();

                // patch < BlackListVersion
                barrier.SignalAndWait();

                failureTest();

                // patch > BlackListVersion
                barrier.SignalAndWait();

                successTest();

                // Send

                InstallTestApplication(command, BlackListTargetVersion);
                barrier.SignalAndWait();

                failureTest();

            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void SendRebootlessSystemUpdate()
        {
            var barrier = new Barrier(2);
            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;

                InstallTestApplication(command, SendSystemAndPatchVersion);

                UpdateSystemRebootless(command);

                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare send-application {0} --timeout 300", TestAppIdString)
                }));
                {
                    var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                }
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                InstallTestApplication(command, 0);

                var resumeFile = Path.Combine(SysMaker.OutputDirPath, "resume.txt");
                if (File.Exists(resumeFile))
                {
                    File.Delete(resumeFile);
                }

                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare receive-application {0} --timeout 300 --output-resume-file \"{1}\"", TestAppIdString, resumeFile)
                }));
                {
                    var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Suspened, output.state);
                    Assert.AreEqual((UInt32)LcsSuspendedReason.RebootRequired, output.reason);
                    Assert.IsTrue(!string.IsNullOrEmpty(output.context));
                    Assert.IsTrue(File.Exists(resumeFile));

                    var systemProgramList = GetSystemProgramList(command);
                    Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemUpdateIdString && x.version == RebootlessSystemUpdateVersion));
                    Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestBlackListIdString && x.version == BlackListVersion));
                }

                Reboot(command);

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare receive-application {0} --timeout 300 --resume \"{1}\"", TestAppIdString, resumeFile)
                }));
                {
                    var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                }
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void ExecuteUpdateRebootlessSystemUpdateOnApplyingLup()
        {
            var barrier = new Barrier(2);
            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;

                InstallTestApplication(command, SendSystemAndPatchVersion);

                SwitchToTestSystem(command, LupSystemUpdate);

                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare send-application {0} --timeout 300", TestAppIdString),
                }));
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                InstallTestApplication(command, 0);

                var resumeFile = Path.Combine(SysMaker.OutputDirPath, "resume.txt");
                if (File.Exists(resumeFile))
                {
                    File.Delete(resumeFile);
                }

                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare receive-application {0} --timeout 300 --output-resume-file \"{1}\"", TestAppIdString, resumeFile)
                }));
                {
                    var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Suspened, output.state);
                    Assert.AreEqual((UInt32)LcsSuspendedReason.RebootRequired, output.reason);
                    Assert.IsTrue(!string.IsNullOrEmpty(output.context));
                    Assert.IsTrue(File.Exists(resumeFile));

                    var systemProgramList = GetSystemProgramList(command);
                    Assert.IsTrue(systemProgramList.Any(x => x.id.ToLower() == TestSystemUpdateIdString && x.version == LupSystemUpdateVersion));
                }

                Reboot(command);

                UpdateSystemRebootless(command);

                Assert.IsTrue(command.FailureExecute(
                    string.Format("localcontentshare receive-application {0} --timeout 300 --resume \"{1}\"", TestAppIdString, resumeFile),
                    expect: "Unexpected error. result = 0x000380d3." // lcs::ResultUnexpectedSystemVersion
                ));
            });

            hostTask.Wait();
            clientTask.Wait();
        }
    }
}
