﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Web.Script.Serialization;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace DevMenuCommandTest
{
    /*
     * システム更新が伴わないパッチを送受信するテストを追加するためのクラス
     */
    [TestClass]
    public class PatchTest : TestBase
    {
        private const ulong TestAppId = 0x0100bbbbaaaa0000;
        private const string TestAppIdString = "0x0100bbbbaaaa0000";

        private static List<NspInfo> AppList;

        [ClassInitialize]
        public static void ClassInitialize(TestContext context)
        {
            SetupTestEnvironment(context);

            AppList = new List<NspInfo>();

            AppList.Add(AppMaker.MakeApplication(TestAppId, requiredSystemVersion: 0, additionalOptions: "--small-code"));
            AppList.Add(AppMaker.MakePatch(TestAppId, 1, AppList[0].Path, requiredSystemVersion: 0, additionalOptions: "--small-code"));
            AppList.Add(AppMaker.MakePatch(TestAppId, 2, AppList[0].Path, requiredSystemVersion: 0, additionalOptions: "--small-code"));

            CleanupBase(context);
        }

        [TestCleanup]
        public void TestCleanup()
        {
            var cleanupCommand = new string[]
            {
                "application uninstall --all",
                "application reset-required-version --all",
                "ticket delete-all",
            };

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;
                command.Execute(cleanupCommand, ignoreError: true);
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;
                command.Execute(cleanupCommand, ignoreError: true);
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [AssemblyCleanup]
        public static void AssemblyCleanup()
        {
            CleanupExternalStorage(true);
        }

        private void InstallTestApplication(DevMenuCommandSystem command, int version, string appStorage = null, string patchStorage = null)
        {
            InstallTestApplicationImpl(command, version, AppList, appStorage, patchStorage);
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void SendPatchToNoPatchUser()
        {
            var barrier = new Barrier(2);

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;

                InstallTestApplication(command, 1);

                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("localcontentshare send-application {0} --timeout 180", TestAppIdString)
                }));
                var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                InstallTestApplication(command, 0);

                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare receive-application {0} --timeout 180", TestAppIdString)
                }));
                var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);

                Assert.IsTrue(command.Execute(new string[]
                {
                    "application list-view"
                }));
                var view = ApplicationViewForJson.Deserialize(command.LastOutput);
                Assert.AreEqual(1, view.Count);
                Assert.AreEqual(1u << 16, view[0].version);
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void SendPatchToHasPatchUser()
        {
            var barrier = new Barrier(2);

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;
                InstallTestApplication(command, 2);

                barrier.SignalAndWait();
                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("localcontentshare send-application {0} --timeout 180", TestAppIdString)
                }));
                var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;
                InstallTestApplication(command, 1);

                barrier.SignalAndWait();
                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare receive-application {0} --timeout 180", TestAppIdString)
                }));
                var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);

                Assert.IsTrue(command.Execute(new string[]
                {
                    "application list-view"
                }));
                var view = ApplicationViewForJson.Deserialize(command.LastOutput);
                Assert.AreEqual(1, view.Count);
                Assert.AreEqual(2u << 16, view[0].version);
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void SendPatchOnSdCard()
        {
            if (!HasHostSdCard)
            {
                return;
            }

            using (ScopeExit _ = new ScopeExit(() => { CleanupExternalStorage(true); }))
            {

                var barrier = new Barrier(2);

                var hostTask = Task.Run(() =>
                {
                    var command = HostCommand;

                    InstallTestApplication(command, 1, patchStorage: "sdcard");

                    barrier.SignalAndWait();

                    Assert.IsTrue(command.Execute(new string[] {
                        string.Format("localcontentshare send-application {0} --timeout 180", TestAppIdString)
                    }));
                    var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);
                });

                var clientTask = Task.Run(() =>
                {
                    var command = ClientCommand;

                    InstallTestApplication(command, 0);

                    barrier.SignalAndWait();

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare receive-application {0} --timeout 180", TestAppIdString)
                    }));
                    var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "application list-view"
                    }));
                    var view = ApplicationViewForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual(1, view.Count);
                    Assert.AreEqual(1u << 16, view[0].version);
                });

                hostTask.Wait();
                clientTask.Wait();
            }
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void SendOnCardPatch()
        {
            if (!HasHostGameCard)
            {
                return;
            }

            using (ScopeExit _ = new ScopeExit(() => { CleanupExternalStorage(true); }))
            {
                var barrier = new Barrier(2);

                var hostTask = Task.Run(() =>
                {
                    var command = HostCommand;
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("gamecard write {0} --on-card-patch {1}", AppList[0].Path, AppList[1].Path),
                    }));

                    barrier.SignalAndWait();

                    Assert.IsTrue(command.Execute(new string[] {
                        string.Format("localcontentshare send-application {0} --timeout 180", TestAppIdString)
                    }));
                    var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);
                });

                var clientTask = Task.Run(() =>
                {
                    var command = ClientCommand;
                    InstallTestApplication(command, 0);

                    barrier.SignalAndWait();

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare receive-application {0} --timeout 180", TestAppIdString)
                    }));
                    var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "application list-view"
                    }));
                    var view = ApplicationViewForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual(1, view.Count);
                    Assert.AreEqual(1u << 16, view[0].version);
                });

                hostTask.Wait();
                clientTask.Wait();
            }
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void SendPatchFromNoExFatToHasExFat()
        {
            using (var recover = new ExFatRecovery(HostCommand, TestContext))
            {
                var barrier = new Barrier(2);
                var hostTask = Task.Run(() =>
                {
                    var command = HostCommand;
                    UninstallExFat(command);

                    InstallTestApplication(command, 1);

                    barrier.SignalAndWait();
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare send-application {0} --timeout 300", TestAppIdString)
                    }));
                    {
                        var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                    }
                });

                var clientTask = Task.Run(() =>
                {
                    var command = ClientCommand;
                    InstallTestApplication(command, 0);

                    barrier.SignalAndWait();
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare receive-application {0} --timeout 300", TestAppIdString)
                    }));
                    {
                        var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                        Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                    }

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "application list-view"
                    }));
                    var view = ApplicationViewForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual(1, view.Count);
                    Assert.AreEqual(1u << 16, view[0].version);

                });

                hostTask.Wait();
                clientTask.Wait();
            }
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void SendCorruptPatch()
        {
            var barrier = new Barrier(2);

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;

                InstallTestApplication(command, 1);
                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("application corrupt {0} --patch", TestAppIdString)
                }));

                barrier.SignalAndWait();

                // 相手が切断するタイミングによって結果が変わるので、結果を無視する
                command.Execute(new string[] {
                    string.Format("localcontentshare send-application {0} --timeout 60", TestAppIdString)
                });
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                InstallTestApplication(command, 0);

                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare receive-application {0} --timeout 60", TestAppIdString)
                }));
                var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                Assert.AreEqual((UInt32)LcsState.ContentsShareFailed, outputs.state);
                Assert.AreEqual((UInt32)LcsFailureReason.ContentsCorrupted, outputs.reason);

                Assert.IsTrue(command.Execute(new string[]
                {
                    "application list-view"
                }));
                var view = ApplicationViewForJson.Deserialize(command.LastOutput);
                Assert.AreEqual(1, view.Count);
                Assert.AreEqual(0u, view[0].version);
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestApplicationRequiredVersion()
        {
            var barrier = new Barrier(2);

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;

                InstallTestApplication(command, 1);

                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("localcontentshare send-application {0} --timeout 60", TestAppIdString)
                }));
                var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);

                Assert.IsTrue(command.Execute(new string[] {
                    "patch install " + AppList[2].Path
                }));

                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("localcontentshare send-application {0} --timeout 60", TestAppIdString)
                }));
                outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                InstallTestApplication(command, 2);
                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("application launch {0}", TestAppIdString)
                }));
                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("patch uninstall {0}", TestAppIdString)
                }));

                // 記録を作っておく
                Assert.IsTrue(command.Execute(new string[]
                {
                    "patch install " + AppList[1].Path,
                    string.Format("application delete-entity {0} --patch", TestAppIdString),
                }));

                Assert.IsTrue(command.Execute(new string[]
                {
                    "application list-view"
                }));
                var view = ApplicationViewForJson.Deserialize(command.LastOutput);
                Assert.AreEqual(1, view.Count);
                Assert.AreEqual(1u << 16, view[0].version);
                Assert.IsTrue(view[0].hasPatchRecord);
                Assert.IsTrue(!view[0].hasPatchEntity);

                barrier.SignalAndWait();

                // 必須バージョンより小さいバージョンのパッチは受け取れない
                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare receive-application {0} --timeout 60", TestAppIdString)
                }));
                var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                Assert.AreEqual((UInt32)LcsState.ContentsShareFailed, outputs.state);
                Assert.AreEqual((UInt32)LcsFailureReason.NotExistDownloadContents, outputs.reason);

                Assert.IsTrue(command.Execute(new string[]
                {
                    "application list-view"
                }));
                view = ApplicationViewForJson.Deserialize(command.LastOutput);
                Assert.AreEqual(1, view.Count);
                Assert.AreEqual(1u << 16, view[0].version);

                barrier.SignalAndWait();

                // 必須バージョン と同じバージョンのパッチは受け取れる
                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare receive-application {0} --timeout 60", TestAppIdString)
                }));
                outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);

                Assert.IsTrue(command.Execute(new string[]
                {
                    "application list-view"
                }));
                view = ApplicationViewForJson.Deserialize(command.LastOutput);
                Assert.AreEqual(1, view.Count);
                Assert.AreEqual(2u << 16, view[0].version);
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestApplicationRecordVersion()
        {
            if (!HasHostSdCard || !HasClientSdCard)
            {
                return;
            }

            using (ScopeExit _ = new ScopeExit(() => { CleanupExternalStorage(true); }))
            {
                var barrier = new Barrier(2);

                var hostTask = Task.Run(() =>
                {
                    var command = HostCommand;

                    InstallTestApplication(command, 1, patchStorage: "sdcard");

                    barrier.SignalAndWait();

                    Assert.IsTrue(command.Execute(new string[] {
                        string.Format("localcontentshare send-application {0} --timeout 180", TestAppIdString)
                    }));
                    var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);

                    Assert.IsTrue(command.Execute(new string[] {
                        "patch install " + AppList[2].Path
                    }));

                    barrier.SignalAndWait();

                    Assert.IsTrue(command.Execute(new string[] {
                        string.Format("localcontentshare send-application {0} --timeout 180", TestAppIdString)
                    }));
                    outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);
                });

                var resumeFile = Path.Combine(SysMaker.OutputDirPath, "resume.txt");
                if (File.Exists(resumeFile))
                {
                    File.Delete(resumeFile);
                }
                var clientTask = Task.Run(() =>
                {
                    var command = ClientCommand;

                    InstallTestApplication(command, 2, patchStorage: "sdcard");

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "sdcard remove"
                    }));

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "application list-view"
                    }));
                    var view = ApplicationViewForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual(1, view.Count);
                    Assert.AreEqual(2u << 16, view[0].version);
                    Assert.IsTrue(view[0].hasPatchRecord);
                    Assert.IsTrue(!view[0].hasPatchEntity);

                    // 記録より小さいバージョンのパッチは受け取れない
                    barrier.SignalAndWait();

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare receive-application {0} --output-resume-file \"{1}\" --timeout 180", TestAppIdString, resumeFile)
                    }));
                    var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.ContentsShareFailed, outputs.state);
                    Assert.AreEqual((UInt32)LcsFailureReason.NotExistDownloadContents, outputs.reason);

                    barrier.SignalAndWait();

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare receive-application {0} --output-resume-file \"{1}\" --timeout 180", TestAppIdString, resumeFile)
                    }));
                    outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);
                    Assert.IsTrue(!File.Exists(resumeFile));


                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "application list-view"
                    }));
                    view = ApplicationViewForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual(1, view.Count);
                    Assert.AreEqual(2u << 16, view[0].version);
                    Assert.IsTrue(view[0].hasPatchRecord);
                    Assert.IsTrue(view[0].hasPatchEntity);
                });

                hostTask.Wait();
                clientTask.Wait();
            }
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void CleanupPlaceHolderToSendPatch()
        {
            var barrier = new Barrier(2);

            ulong id = 0x0100ffffaaaa0000;
            // 転送時間を稼ぐため、大き目のパッチを作る
            var app = AppMaker.MakeApplication(id);
            var patch = AppMaker.MakePatch(id, 1, app.Path, size: 0x20000000); // 512 MB

            using (ScopeExit _ = new ScopeExit(() => { CleanupExternalStorage(true); }))
            {
                var hostTask = Task.Run(() =>
                {
                    var command = HostCommand;

                    Assert.IsTrue(command.Execute(new string[] {
                        "application install " + app.Path,
                        "patch install " + patch.Path
                    }));

                    barrier.SignalAndWait();

                    Assert.IsTrue(command.Execute(new string[] {
                        string.Format("localcontentshare send-application {0} --timeout 180", IdString(id))
                    }));
                    var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);

                    if (!HasClientSdCard)
                    {
                        return;
                    }

                    barrier.SignalAndWait();

                    Assert.IsTrue(command.Execute(new string[] {
                        string.Format("localcontentshare send-application {0} --timeout 180", IdString(id))
                    }));
                    outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);
                });

                var clientTask = Task.Run(() =>
                {
                    var command = ClientCommand;

                    //
                    // NAND の PlaceHolder がクリーンナップされる
                    //
                    Assert.IsTrue(command.Execute(new string[] {
                        "application install " + app.Path,
                    }));

                    if (HasClientSdCard)
                    {
                        Assert.IsTrue(command.Execute(new string[]
                        {
                            "sdcard remove",
                        }));
                    }
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "application list-place-holder"
                    }));
                    var placeHolderList = new JavaScriptSerializer().Deserialize<List<string>>(command.LastOutput);
                    Assert.AreEqual(0, placeHolderList.Count);

                    DevMenuCommandBase.RunOnTargetOptions options = new DevMenuCommandBase.RunOnTargetOptions();
                    options.NoWait = true;
                    command.Execute(new string[]
                    {
                        string.Format("localcontentshare receive-application {0} --timeout 180", IdString(id))
                    }, options: options);

                    // 先に受信操作を行ってから、送信を行わせる
                    barrier.SignalAndWait();

                    bool hasPlaceHolder = false;
                    while (!hasPlaceHolder)
                    {
                        options.NoWait = false;
                        options.SuppressAutoKill = true;
                        Assert.IsTrue(command.Execute(new string[]
                        {
                            "application list-place-holder"
                        }, options: options));
                        placeHolderList = new JavaScriptSerializer().Deserialize<List<string>>(command.LastOutput);
                        hasPlaceHolder = placeHolderList.Count > 0;
                    }

                    // コマンドの途中で強制電源断
                    ForceReset(command);

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "application list-place-holder"
                    }));
                    placeHolderList = new JavaScriptSerializer().Deserialize<List<string>>(command.LastOutput);
                    Assert.AreEqual(0, placeHolderList.Count);

                    if (!HasClientSdCard)
                    {
                        return;
                    }
                    //
                    // SD カードの PlaceHolder がクリーンナップされる
                    //
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "application list-place-holder -s sdcard"
                    }));
                    placeHolderList = new JavaScriptSerializer().Deserialize<List<string>>(command.LastOutput);
                    Assert.AreEqual(0, placeHolderList.Count);

                    options = new DevMenuCommandBase.RunOnTargetOptions();
                    options.NoWait = true;
                    command.Execute(new string[]
                    {
                        string.Format("localcontentshare receive-application {0} --timeout 180", IdString(id))
                    }, options: options);

                    // 先に受信操作を行ってから、送信を行わせる
                    barrier.SignalAndWait();

                    hasPlaceHolder = false;
                    while (!hasPlaceHolder)
                    {
                        options.NoWait = false;
                        options.SuppressAutoKill = true;
                        Assert.IsTrue(command.Execute(new string[]
                        {
                            "application list-place-holder -s sdcard"
                        }, options: options));
                        placeHolderList = new JavaScriptSerializer().Deserialize<List<string>>(command.LastOutput);
                        hasPlaceHolder = placeHolderList.Count > 0;
                    }

                    // コマンドの途中で強制電源断
                    ForceReset(command);

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "application list-place-holder -s sdcard"
                    }));
                    placeHolderList = new JavaScriptSerializer().Deserialize<List<string>>(command.LastOutput);
                    Assert.AreEqual(0, placeHolderList.Count);
                });

                hostTask.Wait();
                clientTask.Wait();
            }
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void SendPatchWithFilledSdCard()
        {
            var barrier = new Barrier(2);

            if (!HasClientSdCard)
            {
                return;
            }

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;

                InstallTestApplication(command, 1);

                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("localcontentshare send-application {0} --timeout 180", TestAppIdString)
                }));
                var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                InstallTestApplication(command, 0);

                using (var _ = new ScopeExit(() => { command.Execute(new string[] { "debug empty-sdcard-free-space" }); }))
                {
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "debug fill-sdcard-free-space"
                    }));

                    barrier.SignalAndWait();

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        string.Format("localcontentshare receive-application {0} --timeout 180", TestAppIdString)
                    }));
                    var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);

                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "application list-record " + TestAppIdString
                    }));
                    var recordList = ApplicationRecordListForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual(2, recordList.Count);
                    Assert.IsTrue(recordList.Any(x => x.type == "Patch" && x.version == (1u << 16) && x.storage == "BuiltInUser"));
                }
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void SendPatchToDifferentStorage()
        {
            var barrier = new Barrier(2);

            if (!HasClientSdCard)
            {
                return;
            }

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;

                InstallTestApplication(command, 2);

                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("localcontentshare send-application {0} --timeout 180", TestAppIdString)
                }));
                var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                InstallTestApplication(command, 1, patchStorage: "builtin");

                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare receive-application {0} --timeout 180", TestAppIdString)
                }));
                var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);

                Assert.IsTrue(command.Execute(new string[]
                {
                    "application list-view"
                }));
                var view = ApplicationViewForJson.Deserialize(command.LastOutput);
                Assert.AreEqual(1, view.Count);
                Assert.AreEqual(2u << 16, view[0].version);

                Assert.IsTrue(command.Execute(new string[]
                {
                    "application list-record " + TestAppIdString
                }));
                var recordList = ApplicationRecordListForJson.Deserialize(command.LastOutput);
                Assert.AreEqual(2, recordList.Count);
                Assert.IsTrue(recordList.Any(x => x.type == "Patch" && x.version == (2u << 16) && x.storage == "SdCard"));
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestErrorContextForPatch()
        {
            var barrier = new Barrier(2);

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;

                InstallTestApplication(command, 2);

                barrier.SignalAndWait();

                command.Execute(new string[] {
                    string.Format("localcontentshare send-application {0} --timeout 60", TestAppIdString)
                });
                // 結果は見ない


                UInt32 resultValue = 0x202;
                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("localcontentshare set-async-task-result 0x{0:x16}", resultValue),
                }));
                Reboot(command);

                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("localcontentshare send-application {0} --timeout 60", TestAppIdString)
                }));
                var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                Assert.AreEqual((UInt32)LcsState.ContentsShareFailed, outputs.state);
                Assert.AreEqual((int)ErrorContextType.LocalContentShare, outputs.errorContext.type);
                var errorContext = outputs.errorContext.localContentShare;
                Assert.AreEqual(TestAppIdString.ToLower(), errorContext.applicationId.ToLower());
                Assert.AreEqual(resultValue, errorContext.resultInnerValue);
                Assert.AreNotEqual(0u, errorContext.ip);
                Assert.IsTrue(errorContext.isSender);
                Assert.IsTrue(errorContext.isApplicationRequest);
                Assert.AreEqual(1, errorContext.numKey);
                Assert.AreEqual(2u << 16, errorContext.keyList[0].version);
                Assert.AreEqual(129, errorContext.keyList[0].type);
                Assert.AreEqual(0, errorContext.keyList[0].installType);
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                InstallTestApplication(command, 0);

                UInt32 resultValue = 0x202;
                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("localcontentshare set-async-task-result 0x{0:x16}", resultValue),
                }));
                Reboot(command);

                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare receive-application {0} --timeout 60", TestAppIdString)
                }));
                var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                Assert.AreEqual((UInt32)LcsState.ContentsShareFailed, outputs.state);
                Assert.AreEqual((int)ErrorContextType.LocalContentShare, outputs.errorContext.type);
                var errorContext = outputs.errorContext.localContentShare;
                Assert.AreEqual(TestAppIdString.ToLower(), errorContext.applicationId.ToLower());
                Assert.AreEqual(resultValue, errorContext.resultInnerValue);
                Assert.AreNotEqual(0u, errorContext.ip);
                Assert.IsTrue(!errorContext.isSender);
                Assert.IsTrue(errorContext.isApplicationRequest);
                Assert.AreEqual(1, errorContext.numKey);
                Assert.AreEqual(2u << 16, errorContext.keyList[0].version);
                Assert.AreEqual(129, errorContext.keyList[0].type);
                Assert.AreEqual(0, errorContext.keyList[0].installType);

                resultValue = 0;
                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("localcontentshare set-async-task-result 0x{0:x16}", resultValue),
                }));
                Reboot(command);

                barrier.SignalAndWait();

                command.Execute(new string[]
                {
                    string.Format("localcontentshare receive-application {0} --timeout 60", TestAppIdString)
                });
                // 結果は見ない

            });

            hostTask.Wait();
            clientTask.Wait();
        }
    }
}
