﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Web.Script.Serialization;
using Microsoft.VisualStudio.TestTools.UnitTesting;


namespace DevMenuCommandTest
{
    [TestClass]
    public class AccountTest : TestBase
    {
        private const ulong TestAppId = 0x0100e9c00192c000;
        private const string TestAppIdString = "0x0100e9c00192c000";
        private const ulong TestPatchId = 0x0100e9c00192c800;
        private const string TestPatchIdString = "0x0100e9c00192c800";

        private static bool IsHostNetworkEnabled;
        private static bool IsClientNetworkEnabled;
        private static bool IsNetworkEnabled;

        [ClassInitialize]
        public static void ClassInitialize(TestContext context)
        {
            SetupTestEnvironment(context);

            var hostTask = Task.Run(() =>
            {
                IsHostNetworkEnabled = VerifyNetworkEnvironment(HostCommand);
            });

            var clientTask = Task.Run(() =>
            {
                IsClientNetworkEnabled = VerifyNetworkEnvironment(ClientCommand);
            });

            hostTask.Wait();
            clientTask.Wait();

            IsNetworkEnabled = IsHostNetworkEnabled && IsClientNetworkEnabled;

            if (!IsNetworkEnabled)
            {
                return;
            }

            CleanupBase(context);

            var cleanupCommand = new string[]
            {
                "account clear_all",
                "shop unlink-device-all"
            };

            hostTask = Task.Run(() =>
            {
                var command = HostCommand;
                command.Execute(cleanupCommand, ignoreError: true);
            });

            clientTask = Task.Run(() =>
            {
                var command = ClientCommand;
                command.Execute(cleanupCommand, ignoreError: true);
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestCleanup]
        public void TestCleanup()
        {
            if (!IsNetworkEnabled)
            {
                return;
            }

            var cleanupCommand = new string[]
            {
                "application uninstall --all",
                "application reset-required-version --all",
                "ticket delete-all",
                "account clear_all",
                "shop unlink-device-all"
            };

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;
                command.Execute(cleanupCommand, ignoreError: true);
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;
                command.Execute(cleanupCommand, ignoreError: true);
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        private ApplicationViewForJson GetApplicationView(DevMenuCommandSystem command, string id = TestAppIdString)
        {
            return GetApplicationViewImpl(command, id);
        }

        private void InstallTestApplication(DevMenuCommandSystem command, UInt32 version)
        {
            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("application download {0}", TestAppIdString)
            }));

            if (version > 0)
            {
                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("application create-download-task {0} --type Patch --id {1} --version {2}", TestAppIdString, TestPatchIdString, version),
                    string.Format("application wait-download {0}", TestAppIdString)
                }));
            }
        }

        private void LinkDevice(DevMenuCommandSystem command)
        {
            Assert.IsTrue(command.Execute(new string[]
            {
                "account add --register_nsa",
                "account link --index 0 --id shimada_hiromasa@nintendo.co.jp --password NintendoShimada",
                "shop start"
            }));
        }

        private void UnlinkDevice(DevMenuCommandSystem command)
        {
            Assert.IsTrue(command.Execute(new string[]
            {
                "account clear_all",
                "shop unlink-device-all"
            }));
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void SendPatchOfNoTicketApplication()
        {
            if (!IsNetworkEnabled)
            {
                return;
            }

            const UInt32 PatchVersion = 1 << 16;

            var barrier = new Barrier(2);

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;
                LinkDevice(command);
                InstallTestApplication(command, PatchVersion);
                UnlinkDevice(command);

                // アカウントリンクを切ったことを通知
                barrier.SignalAndWait();

                Assert.IsFalse(command.Execute(new string[]
                {
                    string.Format("application check-launch-rights {0}", TestAppIdString)
                }));

                // クライアントの準備待ち
                barrier.SignalAndWait();
                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare send-application {0} --timeout 180", TestAppIdString)
                }));
                var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                // ホストのリンク切れ待ち
                barrier.SignalAndWait();

                LinkDevice(command);
                InstallTestApplication(command, 0);

                // 準備完了通知
                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare receive-application {0} --timeout 180", TestAppIdString)
                }));
                var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);

                var view = GetApplicationView(command, TestAppIdString);
                Assert.AreEqual(PatchVersion, view.version);
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void ReceivePatchOfNoTicketApplication()
        {
            if (!IsNetworkEnabled)
            {
                return;
            }

            const UInt32 PatchVersion = 1 << 16;

            var barrier = new Barrier(2);

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;

                // クライアントのリンク切れ待ち
                barrier.SignalAndWait();

                LinkDevice(command);
                InstallTestApplication(command, PatchVersion);

                // 準備完了通知
                barrier.SignalAndWait();
                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare send-application {0} --timeout 180", TestAppIdString)
                }));
                var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;

                LinkDevice(command);
                InstallTestApplication(command, 0);
                UnlinkDevice(command);

                // アカウントリンクを切ったことを通知
                barrier.SignalAndWait();

                Assert.IsFalse(command.Execute(new string[]
                {
                    string.Format("application check-launch-rights {0}", TestAppIdString)
                }));

                // ホストの準備待ち
                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare receive-application {0} --timeout 180", TestAppIdString)
                }));
                var outputs = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                Assert.AreEqual((UInt32)LcsState.Completed, outputs.state);

                var view = GetApplicationView(command, TestAppIdString);
                Assert.AreEqual(PatchVersion, view.version);
            });

            hostTask.Wait();
            clientTask.Wait();
        }
    }
}
