﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.IO;
using System.Text;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace CodeFormatterTest
{
    [TestClass]
    public class CodeFormatterExecutionTest
    {
        public TestContext TestContext { get; set; }

        [TestMethod]
        public void ExecutionTest()
        {
            string testDir = Path.Combine(TestContext.TestRunDirectory, nameof(ExecutionTest));
            Directory.CreateDirectory(testDir);
            string testFile = Path.Combine(testDir, "test.txt");

            // 整形なし
            string testText = "test.\r\n";
            File.WriteAllText(testFile, testText, new UTF8Encoding(true));
            Assert.AreEqual(testText, ExecuteCodeFormatter(testFile));

            // 整形あり
            testText = "test.    \r\n";
            string expected = "test.\r\n";
            File.WriteAllText(testFile, testText, new UTF8Encoding(true));
            Assert.AreEqual(expected, ExecuteCodeFormatter(testFile));
        }

        [TestMethod]
        public void OutputFileTest()
        {
            string testDir = Path.Combine(TestContext.TestRunDirectory, nameof(OutputFileTest));
            Directory.CreateDirectory(testDir);
            string testFile = Path.Combine(testDir, "test.txt");

            // 整形なし
            string outputFile = Path.Combine(testDir, "output.txt");
            string testText = "test.\r\n";
            File.WriteAllText(testFile, testText, new UTF8Encoding(true));
            ExecuteCodeFormatter(testFile, outputFile);
            Assert.AreEqual(testText, File.ReadAllText(outputFile));
            File.Delete(outputFile);

            // 整形あり
            testText = "test.    \r\n";
            string expected = "test.\r\n";
            File.WriteAllText(testFile, testText, new UTF8Encoding(true));
            ExecuteCodeFormatter(testFile, outputFile);
            Assert.AreEqual(expected, File.ReadAllText(outputFile));
            File.Delete(outputFile);
        }

        private static string ExecuteCodeFormatter(string path, string output = "")
        {
            string args = string.Join(" ",
                new string[]
                {
                    $"\"{path}\"",
                    (!string.IsNullOrEmpty(output) ? $"--output \"{output}\"" : "")
                });

            var psi = new ProcessStartInfo()
            {
                FileName = Path.Combine(Environment.CurrentDirectory, "CodeFormatter.exe"),
                Arguments = args,
                CreateNoWindow = true,
                UseShellExecute = false,
                RedirectStandardInput = false,
                RedirectStandardOutput = true,
                RedirectStandardError = false,
            };

            using (var p = Process.Start(psi))
            {
                var outputTask = p.StandardOutput.ReadToEndAsync();
                p.WaitForExit();
                outputTask.Wait();

                return outputTask.Result;
            }
        }
    }
}
