﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include "TestUtil.h"
#include "HidInputerWrapper.h"

using namespace hidInputer;

namespace
{
    // 乱数生成器
    std::random_device      g_RandomDevice;
    std::mt19937            g_Mersenne(g_RandomDevice());
}

// シングルタッチテスト
TEST( HidInputerSuite, SingleTouchTest )
{
    // ターゲットシリアル取得
    SerialNumberString serialNumber;
    ASSERT_TRUE(GetTargetSerial(&serialNumber));

    // Hid ライブラリのラッパークラス
    HidInputerWrapper       hidInputerWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadHidLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadFunctiontions());
    hidInputerWrapper.m_InitializeFunction(serialNumber);

    //-------------------------------------
    // 全タッチ ID でシングルタッチ操作
    //-------------------------------------
    for (int i = 0; i < g_MaxTouchNum; i++)
    {
        int x = g_Mersenne() % g_ScreenWidth;
        int y = g_Mersenne() % g_ScreenHeight;

        // 押す
        EXPECT_EQ(hidInputerWrapper.m_TouchPressDownFunction(i, x, y), TouchResult::TouchResult_Success);
        EXPECT_EQ(hidInputerWrapper.m_TouchPressDownWithFingerIdFunction(i, x, y, static_cast<int>(g_Mersenne())), TouchResult::TouchResult_Success);

        // 移動
        int movePosX = g_Mersenne() % g_ScreenWidth;
        int movePosY = g_Mersenne() % g_ScreenHeight;
        EXPECT_EQ(hidInputerWrapper.m_TouchMoveFunction(i, movePosX, movePosY), TouchResult::TouchResult_Success);


        // 離す
        EXPECT_EQ(hidInputerWrapper.m_TouchPressUpFunction(i), TouchResult::TouchResult_Success);

        // タップ
        EXPECT_EQ(hidInputerWrapper.m_TapFunction(i, x, y, g_DefaultWaitPress), TouchResult::TouchResult_Success);
    }

    //-------------------------------------
    // 不正なタッチIDで操作
    //-------------------------------------
    for (int i = 0; i < g_MaxTouchNum; i++)
    {
        int id = CreateInvalidTouchId();
        int x = g_Mersenne() % g_ScreenWidth;
        int y = g_Mersenne() % g_ScreenHeight;

        int movePosX = g_Mersenne() % g_ScreenWidth;
        int movePosY = g_Mersenne() % g_ScreenHeight;


        // 押す
        EXPECT_EQ(hidInputerWrapper.m_TouchPressDownFunction(id, x, y), TouchResult::TouchResult_InvalidTouchId);
        EXPECT_EQ(hidInputerWrapper.m_TouchPressDownWithFingerIdFunction(id, x, y, static_cast<int>(g_Mersenne())), TouchResult::TouchResult_InvalidTouchId);

        // 移動
        EXPECT_EQ(hidInputerWrapper.m_TouchMoveFunction(id, movePosX, movePosY), TouchResult::TouchResult_InvalidTouchId);

        // 離す
        EXPECT_EQ(hidInputerWrapper.m_TouchPressUpFunction(id), TouchResult::TouchResult_InvalidTouchId);

        // タップ
        EXPECT_EQ(hidInputerWrapper.m_TapFunction(id, x, y, g_DefaultWaitPress), TouchResult::TouchResult_InvalidTouchId);
    }

    //-------------------------------------
    // 画面外を操作
    // 画面外を操作しても内部的にクランプするため特に正常に動作する
    //-------------------------------------
    for (int i = 0; i < g_MaxTouchNum; i++)
    {
        int x = (g_Mersenne() + g_ScreenWidth) * CreateRandomSign();
        int y = (g_Mersenne() + g_ScreenHeight) * CreateRandomSign();

        int movePosX = (g_Mersenne() + g_ScreenWidth) * CreateRandomSign();
        int movePosY = (g_Mersenne() + g_ScreenHeight) * CreateRandomSign();


        // 押す
        EXPECT_EQ(hidInputerWrapper.m_TouchPressDownFunction(i, x, y), TouchResult::TouchResult_Success);
        EXPECT_EQ(hidInputerWrapper.m_TouchPressDownWithFingerIdFunction(i, x, y, static_cast<int>(g_Mersenne())), TouchResult::TouchResult_Success);

        // 移動
        EXPECT_EQ(hidInputerWrapper.m_TouchMoveFunction(i, movePosX, movePosY), TouchResult::TouchResult_Success);

        // 離す
        EXPECT_EQ(hidInputerWrapper.m_TouchPressUpFunction(i), TouchResult::TouchResult_Success);

        // タップ
        EXPECT_EQ(hidInputerWrapper.m_TapFunction(i, x, y, g_DefaultWaitPress), TouchResult::TouchResult_Success);
    }

    // 終了処理
    hidInputerWrapper.m_FinalizeFunction();
    // ライブラリ解放テスト
    ASSERT_TRUE(hidInputerWrapper.ReleaseHidLibrary());
}

// シングルタッチドラッグテスト
TEST( HidInputerSuite, SingleTouchDragTest )
{
    // ターゲットシリアル取得
    SerialNumberString serialNumber;
    ASSERT_TRUE(GetTargetSerial(&serialNumber));

     // Hid ライブラリのラッパークラス
    HidInputerWrapper       hidInputerWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadHidLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadFunctiontions());
    hidInputerWrapper.m_InitializeFunction(serialNumber);

    //-------------------------------------
    // 全タッチ ID でドラッグ操作
    //-------------------------------------
    for (int i = 0; i < g_MaxTouchNum; i++)
    {
        int startPosX = g_Mersenne() % g_ScreenWidth;
        int startPosY = g_Mersenne() % g_ScreenHeight;

        int endPosX = g_Mersenne() % g_ScreenWidth;
        int endPosY = g_Mersenne() % g_ScreenHeight;

        float angle = CreateDegree(false);
        float power = CreatePower(false);

        bool releaseAfter = (g_Mersenne() % 2) == 0 ? false : true;


        // 座標指定のドラッグ
        EXPECT_EQ(hidInputerWrapper.m_DragPosFunction(i, startPosX, startPosY, endPosX, endPosY, g_DefaultWaitDuration, g_DefaultWaitSpan, releaseAfter), TouchResult::TouchResult_Success);

        // 方向指定のドラッグ
        EXPECT_EQ(hidInputerWrapper.m_DragAngleFunction(i, startPosX, startPosY, angle, power, g_DefaultWaitDuration, g_DefaultWaitSpan, releaseAfter), TouchResult::TouchResult_Success);
    }

    //-------------------------------------
    // 不正なタッチIDでドラッグ操作
    //-------------------------------------
    for (int i = 0; i < g_MaxTouchNum; i++)
    {
        int id = CreateInvalidTouchId();

        int startPosX = g_Mersenne() % g_ScreenWidth;
        int startPosY = g_Mersenne() % g_ScreenHeight;

        int endPosX = g_Mersenne() % g_ScreenWidth;
        int endPosY = g_Mersenne() % g_ScreenHeight;

        float angle = CreateDegree(false);
        float power = CreatePower(false);

        bool releaseAfter = (g_Mersenne() % 2) == 0 ? false : true;

        // 座標指定のドラッグ
        EXPECT_EQ(hidInputerWrapper.m_DragPosFunction(id, startPosX, startPosY, endPosX, endPosY, g_DefaultWaitDuration, g_DefaultWaitSpan, releaseAfter), TouchResult::TouchResult_InvalidTouchId);

        // 方向指定のドラッグ
        EXPECT_EQ(hidInputerWrapper.m_DragAngleFunction(id, startPosX, startPosY, angle, power, g_DefaultWaitDuration, g_DefaultWaitSpan, releaseAfter), TouchResult::TouchResult_InvalidTouchId);
    }

    //-------------------------------------
    // 画面外を操作
    // 画面外を操作しても内部的にクランプするため特に正常に動作する
    //-------------------------------------
    for (int i = 0; i < g_MaxTouchNum; i++)
    {
        int startPosX = (g_Mersenne() + g_ScreenWidth) * CreateRandomSign();
        int startPosY = (g_Mersenne() + g_ScreenHeight) * CreateRandomSign();

        int endPosX = (g_Mersenne() + g_ScreenWidth) * CreateRandomSign();
        int endPosY = (g_Mersenne() + g_ScreenHeight) * CreateRandomSign();

        float angle = CreateDegree(true);
        float power = CreatePower(true);

        bool releaseAfter = (g_Mersenne() % 2) == 0 ? false : true;


        // 座標指定のドラッグ
        EXPECT_EQ(hidInputerWrapper.m_DragPosFunction(i, startPosX, startPosY, endPosX, endPosY, g_DefaultWaitDuration, g_DefaultWaitSpan, releaseAfter), TouchResult::TouchResult_Success);

        // 方向指定のドラッグ
        EXPECT_EQ(hidInputerWrapper.m_DragAngleFunction(i, startPosX, startPosY, angle, power, g_DefaultWaitDuration, g_DefaultWaitSpan, releaseAfter), TouchResult::TouchResult_Success);
    }


    // 終了処理
    hidInputerWrapper.m_FinalizeFunction();
    // ライブラリ解放テスト
    ASSERT_TRUE(hidInputerWrapper.ReleaseHidLibrary());
}
