﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>

#include "CaptureProcessorWrapper.h"

using namespace captureProcessor;

TEST(CaptureProcessorSuite, CreateWindowTest)
{
    CaptureProcessorWrapper       captureProcessorWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadCaptureLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadFunctiontions());

    ImageIdType imageId;
    const char testWindow[] = "test window";

    //-------------------------------------
    // 未初期化での実行
    //-------------------------------------
    EXPECT_EQ(CaptureResult::CaptureResult_NotInitialized,
        captureProcessorWrapper.m_GetImageFromCaptureDeviceFunction(&imageId));

    EXPECT_EQ(CaptureResult::CaptureResult_NotInitialized,
        captureProcessorWrapper.m_OpenImageWindowFunction(testWindow, imageId));

    EXPECT_EQ(CaptureResult::CaptureResult_NotInitialized,
        captureProcessorWrapper.m_CloseImageWindowFunction(testWindow));

    //-------------------------------------
    // 初期化済みでの実行
    //-------------------------------------
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();
    SerialNumberString serialNumber;
    ASSERT_GE(argc, 2);
    strncpy_s(serialNumber.value, argv[1], sizeof(serialNumber.value));

    ASSERT_EQ(captureProcessorWrapper.m_InitializeForDevkitCapture(
        serialNumber, false), CaptureResult::CaptureResult_Success);

    // キャプチャ画像取得 -> ウィンドウ表示を繰り返す
    for (int i = 0; i < 10; i++)
    {
        EXPECT_EQ(CaptureResult::CaptureResult_Success,
            captureProcessorWrapper.m_GetImageFromCaptureDeviceFunction(&imageId));

        EXPECT_EQ(CaptureResult::CaptureResult_Success,
            captureProcessorWrapper.m_OpenImageWindowFunction(testWindow, imageId));

        EXPECT_EQ(CaptureResult::CaptureResult_Success,
            captureProcessorWrapper.m_ReleaseImageFunction(imageId));
    }

    // ウィンドウを閉じる
    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_CloseImageWindowFunction(testWindow));

    // TIPS: 未表示のウィンドウに対して、特に処理は行われず CaptureResult_Success を返すことを確認する。
    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_CloseImageWindowFunction("dummy window"));

    // 不正な画像 ID に対して実行
    ImageIdType invalidImageId = { 0 };
    EXPECT_EQ(CaptureResult::CaptureResult_InvalidImageId,
        captureProcessorWrapper.m_OpenImageWindowFunction(testWindow, invalidImageId));

    // ライブラリ解放テスト
    ASSERT_TRUE(captureProcessorWrapper.ReleaseCaptureLibrary());
}

TEST(CaptureProcessorSuite, ExtractImageTest)
{
    CaptureProcessorWrapper       captureProcessorWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadCaptureLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadFunctiontions());

    //-------------------------------------
    // 未初期化での実行
    //-------------------------------------
    ImageIdType imageId;
    ImageIdType extractedImageId;
    ImageRect extractRect = { 100, 100, 200, 200 };
    EXPECT_EQ(CaptureResult::CaptureResult_NotInitialized,
        captureProcessorWrapper.m_GetImageFromCaptureDeviceFunction(&imageId));

    EXPECT_EQ(CaptureResult::CaptureResult_NotInitialized,
        captureProcessorWrapper.m_ExtractImageFunction(&extractedImageId, imageId, extractRect));

    //-------------------------------------
    // 初期化済みでの実行
    //-------------------------------------
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();
    SerialNumberString serialNumber;
    ASSERT_GE(argc, 2);
    strncpy_s(serialNumber.value, argv[1], sizeof(serialNumber.value));

    ASSERT_EQ(captureProcessorWrapper.m_InitializeForDevkitCapture(
        serialNumber, false), CaptureResult::CaptureResult_Success);

    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_GetImageFromCaptureDeviceFunction(&imageId));

    // 正常なパラメーターでの実行
    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_ExtractImageFunction(&extractedImageId, imageId, extractRect));

    // 基画像のサイズが変わっていないことを確認
    int width = 0;
    int height = 0;
    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_GetImageSizeFunction(&width, &height, imageId));
    EXPECT_EQ(1280, width);
    EXPECT_EQ(720, height);

    // 不正な範囲に対して実行
    ImageRect invalidImageRect = { 0, 0, 1281, 720 };
    EXPECT_EQ(CaptureResult::CaptureResult_InvalidParameter, captureProcessorWrapper.m_ExtractImageFunction(&extractedImageId, imageId, invalidImageRect));

    // 不正な画像 ID に対して実行
    ImageIdType invalidImageId = { 0 };
    EXPECT_EQ(CaptureResult::CaptureResult_InvalidImageId, captureProcessorWrapper.m_ExtractImageFunction(&extractedImageId, invalidImageId, extractRect));


    // ExtractImage 後の画像サイズを確認
    EXPECT_EQ(CaptureResult::CaptureResult_Success, captureProcessorWrapper.m_GetImageSizeFunction(&width, &height, extractedImageId));
    EXPECT_EQ(200, width);
    EXPECT_EQ(200, height);

    // ライブラリ解放テスト
    ASSERT_TRUE(captureProcessorWrapper.ReleaseCaptureLibrary());
}
