﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include "CaptureProcessorWrapper.h"
#include "testCaptureProcessor_TestDefinition.h"

using namespace captureProcessor;


TEST(CaptureProcessorSuite, GetImageInfoTestInUninitialized)
{
     // Capture ライブラリのラッパークラス
    CaptureProcessorWrapper       captureProcessorWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadCaptureLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadFunctiontions());

    int width;
    int height;
    ImageIdType imageId = { 0 };

    const size_t bufferSize = 256;
    uint8_t buffer[bufferSize];


    //-------------------------------------
    // 未初期化での実行
    //-------------------------------------
    // 画像情報取得
    EXPECT_EQ(captureProcessorWrapper.m_GetImageSizeFunction(&width, &height, imageId),
        CaptureResult::CaptureResult_NotInitialized);

    // 画像データ取得
    EXPECT_EQ(captureProcessorWrapper.m_GetImageDataFunction(buffer, bufferSize, imageId),
        CaptureResult::CaptureResult_NotInitialized);


    // ライブラリ解放テスト
    ASSERT_TRUE(captureProcessorWrapper.ReleaseCaptureLibrary());
}

TEST_P(CaptureModeTest, GetImageInfoTest)
{
     // Capture ライブラリのラッパークラス
    CaptureProcessorWrapper       captureProcessorWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadCaptureLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadFunctiontions());

    int width;
    int height;
    ImageIdType imageId = { 0 };

    const size_t bufferSize = 256;
    uint8_t buffer[bufferSize];


    //-------------------------------------
    // 初期化済みでの実行
    //-------------------------------------
    int expectWidth;
    int expectHeight;
    auto mode = GetParam();
    if (mode == CaptureMode::CaptureMode_720p)
    {
        captureProcessorWrapper.m_InitializeForCaptureDeviceFunction(
            0, false, CaptureResolution::CaptureResolution_720p);

        expectWidth = 1280;
        expectHeight = 720;
    }
    else if (mode == CaptureMode::CaptureMode_1080p)
    {
        captureProcessorWrapper.m_InitializeForCaptureDeviceFunction(
            0, false, CaptureResolution::CaptureResolution_1080p);

        expectWidth = 1920;
        expectHeight = 1080;
    }
    else if (mode == CaptureMode::CaptureMode_DevKit)
    {
        int     argc = nnt::GetHostArgc();
        char**  argv = nnt::GetHostArgv();
        SerialNumberString serialNumber;
        ASSERT_GE(argc, 2);
        strncpy_s(serialNumber.value, argv[1], sizeof(serialNumber.value));

        ASSERT_EQ(captureProcessorWrapper.m_InitializeForDevkitCapture(
            serialNumber, false), CaptureResult::CaptureResult_Success);

        expectWidth = 1280;
        expectHeight = 720;
    }
    else
    {
        ASSERT_TRUE(false);
    }

    // 画像読み込み
    EXPECT_EQ(captureProcessorWrapper.m_GetImageFromCaptureDeviceFunction(&imageId),
        CaptureResult::CaptureResult_Success);

    // 画像情報取得
    EXPECT_EQ(captureProcessorWrapper.m_GetImageSizeFunction(&width, &height, imageId),
        CaptureResult::CaptureResult_Success);
    EXPECT_EQ(width, expectWidth);
    EXPECT_EQ(height, expectHeight);


    // 画像データ取得
    EXPECT_EQ(captureProcessorWrapper.m_GetImageDataFunction(buffer, bufferSize, imageId),
        CaptureResult::CaptureResult_Success);


    //-------------------------------------
    // 不正なパラメーターでの実行
    //-------------------------------------
    // 不正な画像 ID に対して実行
    ImageIdType invalidId = { 0 };
    EXPECT_EQ(captureProcessorWrapper.m_GetImageSizeFunction(&width, &height, invalidId),
        CaptureResult::CaptureResult_InvalidImageId);
    EXPECT_EQ(captureProcessorWrapper.m_GetImageDataFunction(buffer, bufferSize, invalidId),
        CaptureResult::CaptureResult_InvalidImageId);

    // ライブラリ解放テスト
    ASSERT_TRUE(captureProcessorWrapper.ReleaseCaptureLibrary());
}
