﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;
using System.Threading;

namespace ProdTestTool
{
    internal class Common
    {
        public static string ExecuteCommand(string exe, string args, bool ignoreError = false)
        {
            Console.WriteLine(string.Format("{0} {1}", exe, args));

            var process = new Process();
            bool eventHandled = false;

            try
            {
                process.StartInfo.FileName = exe;
                process.StartInfo.Arguments = args;
                process.StartInfo.CreateNoWindow = true;
                process.StartInfo.UseShellExecute = false;
                process.StartInfo.RedirectStandardOutput = true;
                process.StartInfo.RedirectStandardError = true;

                Action<object, System.EventArgs> exitedHandler = (sender, e) =>
                    {
                        eventHandled = true;
                    };

                process.EnableRaisingEvents = true;
                process.Exited += new EventHandler(exitedHandler);

                process.Start();
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                throw ex;
            }

            var stdout = process.StandardOutput.ReadToEnd();
            var errorMsg = process.StandardError.ReadToEnd();

            while (!eventHandled)
            {
                Thread.Sleep(200);
            }

            if (!string.IsNullOrEmpty(errorMsg) && !ignoreError)
            {
                Console.WriteLine(errorMsg);
                throw new InvalidOperationException();
            }

            return stdout;
        }

        public static string MakeInfoDir(string romId)
        {
            return Constant.OutDir + string.Format("/info/{0}", romId);
        }

        public static void Cleanup(string appId)
        {
            var devPath = Constant.OutDir + string.Format("/dev/{0}", appId);
            var prodPath = Constant.OutDir + string.Format("/prod/{0}", appId);
            Console.WriteLine("Deleting {0}...", appId);
            Action<string> deleteDir = (path) =>
            {
                if (!Directory.Exists(path))
                {
                    return;
                }
                Directory.Delete(path, true);
                const int WaitForDeleteDirectoryTimeOutMilliSec = 1000 * 3;
                const int WaitForDeleteDirectoryWaitUnitMilliSec = 100;
                for (int waitMilliSec = 0; waitMilliSec < WaitForDeleteDirectoryTimeOutMilliSec; waitMilliSec += WaitForDeleteDirectoryWaitUnitMilliSec)
                {
                    if (!Directory.Exists(path))
                    {
                        break;
                    }
                    Thread.Sleep(WaitForDeleteDirectoryWaitUnitMilliSec);
                }
            };
            deleteDir(devPath);
            deleteDir(prodPath);
            Console.WriteLine("Done");
        }
    }

    internal class Constant
    {
        public const string OutDir = "";

        public const string Aspera = "";
        public const string HostName = "";
        public const string UserName = "";
        public const string PassWord = "";
        public const string Port = "";
        public const string RomRootDir = "";

        public const string Curl = "";
        public const string DbHostUrl = "";
        public const string Proxy = "";

        public const string Authoring = "";
        public const string KeyConfig = "";
        public const string ReleaseBinariesDir = "";
    }
}
