﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.IO.Compression;
using System.Linq;
using System.Reflection;
using System.Runtime.InteropServices;
using System.Text.RegularExpressions;
using System.Threading;
using System.Security.Cryptography;
using System.Xml.Serialization;
using System.Xml;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.Authoring.AuthoringLibrary;
using Nintendo.Authoring.FileSystemMetaLibrary;
using TestUtility;

namespace AuthoringToolsTest
{
    public class TestEnvironment
    {
        public string ToolPath { get; set; }
        public string OutputDir { get; set; }
        public string SourceDir { get; set; }
        public string TestCodeDir { get; set; }
        public string TestCodeFile { get; set; }
        public string NpdmFile { get; set; }
        public string DefaultMetaFile { get; set; }
        public string DefaultDescFile { get; set; }
        public string DefaultKeyConfigFile { get; set; }
        public string DefaultIcon { get; set; }

        private TestPath TestPath;
        private string TestName;

        public TestEnvironment(TestPath testPath, string testName)
        {
            TestPath = testPath;
            TestName = testName;


            ToolPath = Path.Combine(TestPath.GetSigloRoot(), "Tools/CommandLineTools/AuthoringTool/AuthoringTool.exe");
            SourceDir = Path.Combine(TestPath.GetSigloRoot(), "Tests/Tools/Sources/Tests/AuthoringToolsTest/TestResources");
            OutputDir = Path.Combine(Directory.GetCurrentDirectory().Replace("\\" + Assembly.GetExecutingAssembly().GetName().Name, string.Empty), testName);
            TestCodeDir = Path.Combine(OutputDir, "testCode");
            TestCodeFile = Path.Combine(TestCodeDir, "code.dat");
            DefaultMetaFile = Path.Combine(testPath.GetSigloRoot(), "Programs/Iris/Resources/SpecFiles/Application.aarch64.lp64.nmeta");
            DefaultDescFile = Path.Combine(testPath.GetSigloRoot(), "Programs/Iris/Resources/SpecFiles/Application.desc");
            NpdmFile = Path.Combine(TestCodeDir, "main.npdm");
            DefaultKeyConfigFile = Path.Combine(SourceDir, "AuthoringToolsTest.keyconfig.xml");
            DefaultIcon = SourceDir + "\\Icon\\describe_all.bmp";

            Utils.DeleteIncludingJunctionDirectoryIfExisted(OutputDir);
            Directory.CreateDirectory(OutputDir);
            Directory.CreateDirectory(TestCodeDir);

            using (FileStream stream = File.Create(TestCodeFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }
        }

        // TODO: 整理
        static public string GetOutputPath(string testName)
        {
            return Path.Combine(Directory.GetCurrentDirectory().Replace("\\" + Assembly.GetExecutingAssembly().GetName().Name, string.Empty), testName);
        }

        public void MakeNpdm()
        {
            MakeNpdm(DefaultMetaFile, DefaultDescFile);
        }

        public void MakeNpdm(string metaFile, string descFile)
        {
            Process process = new Process();

            process.StartInfo.FileName = Path.Combine(TestPath.GetSigloRoot(), "Tools/CommandLineTools/MakeMeta/MakeMeta.exe");
            process.StartInfo.Arguments = String.Format("-o {0} --meta {1} --desc {2} --no_check_programid", NpdmFile, metaFile, descFile);
            process.StartInfo.CreateNoWindow = true;
            process.StartInfo.UseShellExecute = false;
            process.StartInfo.RedirectStandardError = true;
            process.Start();

            string errorMsg = process.StandardError.ReadToEnd();
            process.WaitForExit();
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
        }
    }

    public class TestMetaFile
    {
        public string FileName { get; set; }
        public string OutputDir { get; set; }

        public TestMetaFile()
        {
            m_ApplicationList = new List<string>();
            FileName = Path.GetRandomFileName() + ".nmeta";
            OutputDir = Path.GetTempPath();
        }

        ~TestMetaFile()
        {
            // ファイル名はランダムで、ファイルが再利用されることはないはずなので、いつ消されてもかまわないはずなので、デストラクタで消す
            string path = GetFilePath();
            if (File.Exists(path))
            {
                File.Delete(path);
            }
        }

        public string GetFilePath()
        {
            return Path.Combine(OutputDir, FileName);
        }

        public void Write()
        {
            using (var fs = File.OpenWrite(GetFilePath()))
            {
                var sw = new StreamWriter(fs);
                WriteTmpMetaFileHeader(sw);

                WriteApplicationTags(sw);

                WriteTmpMetaFileFooter(sw);

                sw.Flush();
            }
        }

        public void AddApplicationTags(string line)
        {
            m_ApplicationList.Add(line);
        }

        private void WriteTmpMetaFileHeader(StreamWriter sw)
        {
            sw.WriteLine("<?xml version=\"1.0\"?>");
            sw.WriteLine("<NintendoSdkMeta>");
            sw.WriteLine("<Core>");
            sw.WriteLine("<ApplicationId>0x0100000000002802</ApplicationId>");
            sw.WriteLine("</Core>");
        }

        private void WriteTmpMetaFileFooter(StreamWriter sw)
        {
            sw.WriteLine("</NintendoSdkMeta>");
        }

        private void WriteApplicationTags(StreamWriter sw)
        {
            if (m_ApplicationList.Count() <= 0)
            {
                return;
            }

            sw.WriteLine("<Application>");
            foreach(var line in m_ApplicationList)
            {
                sw.WriteLine(line);
            }
            sw.WriteLine("</Application>");
        }

        private List<string> m_ApplicationList;
    }

    public abstract class ExcecutionTestBase
    {
        public TestContext TestContext { get; set; }

        protected void ForceGCImpl()
        {
            System.Runtime.GCSettings.LargeObjectHeapCompactionMode = System.Runtime.GCLargeObjectHeapCompactionMode.CompactOnce;
            GC.Collect();
        }

        protected void MakeNpdm(string outFile, string metaFile, string descFile)
        {
            TestUtility.TestPath testPath = new TestUtility.TestPath(this.TestContext);
            MakeMetaUtil.MakeNpdm(testPath, outFile, metaFile, descFile);
        }

        protected void SafeDeleteDirectory(string path)
        {
            Utils.DeleteIncludingJunctionDirectoryIfExisted(path);
        }

        protected string ExecuteProgram(string args)
        {
            return ExecuteProgram(args, false);
        }

        protected string ExecuteProgram(string args, bool isStandardOut)
        {
            Console.WriteLine(args);

            Process process = new Process();
            TestUtility.TestPath testPath = new TestUtility.TestPath(this.TestContext);

            process.StartInfo.FileName = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\AuthoringTool\\AuthoringTool.exe";
            process.StartInfo.Arguments = args;
            process.StartInfo.CreateNoWindow = true;
            process.StartInfo.UseShellExecute = false;
            process.StartInfo.RedirectStandardOutput = isStandardOut;
            process.StartInfo.RedirectStandardError = !isStandardOut;
            process.Start();

            string errorMsg = (isStandardOut ? process.StandardOutput : process.StandardError).ReadToEnd();
            process.WaitForExit();

            return errorMsg;
        }

        protected void ExecuteProgram(out string standardError, out string standardOut, string args)
        {
            Console.WriteLine(args);

            var standardOutputString = string.Empty;
            var standardErrorString = string.Empty;

            Process process = new Process();
            TestUtility.TestPath testPath = new TestUtility.TestPath(this.TestContext);

            process.StartInfo.FileName = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\AuthoringTool\\AuthoringTool.exe";
            process.StartInfo.Arguments = args;
            process.StartInfo.CreateNoWindow = true;
            process.StartInfo.UseShellExecute = false;
            process.StartInfo.RedirectStandardOutput = true;
            process.StartInfo.RedirectStandardError = true;

            // StandardError.ReadToEnd()とStandardOutput.ReadToEnd()で標準出力と標準エラー出力を取得すると
            // デッドロックが発生することがあるので、非同期で標準出力と標準エラー出力を取得する
            process.OutputDataReceived += new DataReceivedEventHandler((sender, e) =>
            {
                if (!string.IsNullOrEmpty(e.Data))
                {
                    standardOutputString += e.Data + Environment.NewLine;
                }
            });
            process.ErrorDataReceived += new DataReceivedEventHandler((sender, e) =>
            {
                if (!string.IsNullOrEmpty(e.Data))
                {
                    standardErrorString += e.Data + Environment.NewLine;
                }
            });

            process.Start();
            process.BeginErrorReadLine();
            process.BeginOutputReadLine();
            process.WaitForExit();

            standardError = standardErrorString;
            standardOut = standardOutputString;
            return;
        }

        protected void CreateNsp(string outputPath, TestEnvironment env, string metaFile, string addCmd)
        {
            var cmd = string.Format("creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} {3} --icon AmericanEnglish {4} Japanese {4}", outputPath, metaFile, env.DefaultDescFile, env.TestCodeDir, env.DefaultIcon) + addCmd;
            MakeNpdm(env.NpdmFile, metaFile, env.DefaultDescFile);
            var error = ExecuteProgram(cmd);
            Assert.IsTrue(error == string.Empty, error);
//            Nintendo.Authoring.AuthoringTool.Program.Main(cmd.Split());
            VerifyNsp(outputPath, null);
        }

        protected void CreatePatch(string outputPath, TestEnvironment env, string originalPath, string currentPath, string addCmd)
        {
            var cmd = string.Format("makepatch -o {0} --desc {1} --original {2} --current {3}", outputPath, env.DefaultDescFile, originalPath, currentPath) + addCmd;
            var error = ExecuteProgram(cmd);
            Assert.IsTrue(error == string.Empty, error);
//            Nintendo.Authoring.AuthoringTool.Program.Main(cmd.Split());
            VerifyNsp(outputPath, null);
        }

        protected void CheckRequiredSystemVersion(string outputPath, uint expectedVersion)
        {
            List<ContentMetaModel> models;
            using (var nspReader = new NintendoSubmissionPackageReader(outputPath))
            {
                models = ArchiveReconstructionUtils.ReadContentMetaInNsp(nspReader);
            }

            foreach (var model in models)
            {
                switch (model.Type)
                {
                    case "Application":
                        {
                            var appModel = model as ApplicationContentMetaModel;
                            Assert.IsTrue(appModel.RequiredSystemVersion == expectedVersion);
                        }
                        break;
                    case "Patch":
                        {
                            var patchModel = model as PatchContentMetaModel;
                            Assert.IsTrue(patchModel.RequiredSystemVersion == expectedVersion);
                        }
                        break;
                    default:
                        continue;
                }
            }
        }

        protected void CheckXmlEncoding(string xmlPath)
        {
            Console.WriteLine("Check {0}", xmlPath);
            string text;
            using (var xml = new FileStream(xmlPath, FileMode.Open, FileAccess.Read))
            {
                var bom = new byte[3];
                xml.Read(bom, 0, bom.Length);

                // BOM 付き UTF-8 であることをチェック
                Assert.IsTrue(bom[0] == 0xef && bom[1] == 0xbb && bom[2] == 0xbf);

                var st = new StreamReader(xml, new System.Text.UTF8Encoding(true, true));
                text = st.ReadToEnd();
            }

            // 改行コードが CR+LF であることをチェック
            var noCrlf = text.Replace("\r\n", string.Empty);
            Assert.IsTrue(noCrlf.IndexOf("\n") == -1);
        }

        protected static ContentMetaModel GetContentMetaModel(NintendoSubmissionPackageReader nspReader)
        {
            var entry = nspReader.ListFileInfo().Find(s => s.Item1.EndsWith(".cnmt.xml"));
            return ArchiveReconstructionUtils.ReadModel(nspReader, entry.Item1, entry.Item2);
        }

        protected static NintendoContentArchiveReader GetContentArchiveReader(NintendoSubmissionPackageReader nspReader, ContentMetaModel metaModel, string contentType, KeyConfiguration config)
        {
            var content = metaModel.ContentList.Find(m => m.Type == contentType);
            var ncaReader = nspReader.OpenNintendoContentArchiveReader(content.Id + ".nca", new NcaKeyGenerator(config));
            TicketUtility.SetExternalKey(ref ncaReader, nspReader);
            return ncaReader;
        }

        protected string CreateLegalInfoXml(string outputDir)
        {
            string testXmlFilePath = Path.Combine(outputDir, LegalInfoXmlFileName);
            MakeFileImpl(
                testXmlFilePath,
                "<?xml version=\"1.0\" encoding=\"utf-8\"?>\r\n" +
                "<SoftwareLegalInformation>\r\n" +
                "  <ProductRegions>\r\n" +
                "    <Usa>true</Usa>\r\n" +
                "    <Japan>true</Japan>\r\n" +
                "    <Europe>true</Europe>\r\n" +
                "  </ProductRegions>\r\n" +
                "  <DataHash>aa2f134b8dd791c6fe46d1af14e9cd1a</DataHash>\r\n" +
                "  <CopyrightNotations>\r\n" +
                "    <TechnologyName>NintendoSDK_libcurl</TechnologyName>\r\n" +
                "    <TechnologyName>NintendoSDK_movie</TechnologyName>\r\n" +
                "  </CopyrightNotations>\r\n" +
                "  <DeclarationRequiredNotations>\r\n" +
                "    <TechnologyName>NintendoSDK_libcurl</TechnologyName>\r\n" +
                "    <TechnologyName>NintendoSDK_movie</TechnologyName>\r\n" +
                "    <TechnologyName>NintendoWare_Bezel_Engine</TechnologyName>\r\n" +
                "  </DeclarationRequiredNotations>\r\n" +
                "  <ApplicationId>0x0100000000001000</ApplicationId>\r\n" +
                "  <OutputDateTime>2016-11-21T05:20:07.108Z</OutputDateTime>\r\n" +
                "  <FormatVersion>1.0.0</FormatVersion>\r\n" +
                "</SoftwareLegalInformation>",
                System.Text.Encoding.UTF8
            );
            return testXmlFilePath;
        }

        protected void MakeMetaFileWithProgramIndex(string baseMetaFile, string newMetaFile, byte index, int version, ulong? changedApplicationId = null)
        {
            var document = new XmlDocument();
            document.Load(baseMetaFile);

            // Core/ApplicationId を Core/ProgramId に変更
            var applicationId = string.Empty;
            var coreNode = document.SelectSingleNode("//Core");
            {
                var applicationIdNode = coreNode.SelectSingleNode("//ApplicationId");
                if (changedApplicationId.HasValue)
                {
                    applicationId = "0x" + changedApplicationId.Value.ToString("x16");
                }
                else
                {
                    applicationId = applicationIdNode.InnerText;
                }
                coreNode.RemoveChild(applicationIdNode);
                var programIdNode = document.CreateNode(XmlNodeType.Element, "ProgramId", string.Empty);
                programIdNode.InnerText = "0x" + (Convert.ToUInt64(applicationId, 16) + (ulong)index).ToString("x16");
                coreNode.AppendChild(programIdNode);
            }

            // Application/ApplicationId, Application/ProgramIndex を追加
            var applicationNode = document.SelectSingleNode("//Application");
            {
                var applicationIdNode = document.CreateNode(XmlNodeType.Element, "ApplicationId", string.Empty);
                if (changedApplicationId.HasValue)
                {
                    applicationIdNode.InnerText = "0x" + changedApplicationId.Value.ToString("x16");
                }
                else
                {
                    applicationIdNode.InnerText = applicationId;
                }
                applicationNode.AppendChild(applicationIdNode);
                var programIndexNode = document.CreateNode(XmlNodeType.Element, "ProgramIndex", string.Empty);
                programIndexNode.InnerText = index.ToString("d");
                applicationNode.AppendChild(programIndexNode);
                var versionNode = applicationNode.SelectSingleNode("Version");
                if (versionNode == null)
                {
                    versionNode = document.CreateNode(XmlNodeType.Element, "Version", string.Empty);
                    versionNode.InnerText = version.ToString();
                    applicationNode.AppendChild(versionNode);
                }
                else
                {
                    versionNode.InnerText = version.ToString();
                }
            }

            document.Save(newMetaFile);
        }

        protected void MakeSpecifiedVersionMetaFile(string baseMetaFile, string newMetaFile, int version, ulong? applicationId = null)
        {
            var document = new XmlDocument();
            document.Load(baseMetaFile);

            if (applicationId.HasValue)
            {
                var idNode = document.SelectSingleNode("//Core").SelectSingleNode("ApplicationId");
                idNode.InnerText = "0x" + applicationId.Value.ToString("x16");
            }

            var applicationNode = document.SelectSingleNode("//Application");
            var versionNode = applicationNode.SelectSingleNode("Version");
            if (versionNode == null)
            {
                versionNode = document.CreateNode(XmlNodeType.Element, "Version", string.Empty);
                versionNode.InnerText = version.ToString();

                applicationNode.AppendChild(versionNode);
            }
            else
            {
                int oldVersion;
                if (int.TryParse(versionNode.InnerText, out oldVersion))
                {
                    Debug.Assert(oldVersion <= version);
                }
                versionNode.InnerText = version.ToString();
            }

            document.Save(newMetaFile);
        }

        protected void MakeAocMetaFile(string baseMetaFile, string newMetaFile, int version, int index, string dataPath, ulong applicationId)
        {
            var document = new XmlDocument();
            string rootName;
            {
                var baseDocument = new XmlDocument();
                baseDocument.Load(baseMetaFile);
                rootName = baseDocument.DocumentElement.Name;
            }

            document.LoadXml(
$@"<?xml version=""1.0""?>
<{rootName}>
  <AddOnContent>
    <Index>{index}</Index>
    <ReleaseVersion>{version}</ReleaseVersion>
    <ApplicationId>{"0x" + applicationId.ToString("x16")}</ApplicationId>
    <DataPath>{dataPath}</DataPath>
    <Tag>Item{index}</Tag>
  </AddOnContent>
</{rootName}>");
            document.Save(newMetaFile);
        }

        protected void VerifyNsp(string nspPath, string keyConfigFile)
        {
            var command = string.Format("verify {0}", nspPath);
            if (!string.IsNullOrEmpty(keyConfigFile))
            {
                command += string.Format(" --keyconfig {0}", keyConfigFile);
            }
            string errorMsg = ExecuteProgram(command);
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
        }

        protected void VerifyPatch(string currentPatchPath, string previousPatchPath, string keyConfigFile)
        {
            var command = string.Format("verify {0} --previous {1}", currentPatchPath, previousPatchPath);
            if (!string.IsNullOrEmpty(keyConfigFile))
            {
                command += string.Format(" --keyconfig {0}", keyConfigFile);
            }
            string errorMsg = ExecuteProgram(command);
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
        }

        protected void ExtractAndShowNsp(string nspPath)
        {
            var extractedDirectoryPath = nspPath + ".extract";
            SafeDeleteDirectory(extractedDirectoryPath);
            Directory.CreateDirectory(extractedDirectoryPath);
            var errorMsg = ExecuteProgram(string.Format("extractnsp -o {0} {1}", extractedDirectoryPath, nspPath));
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);

            foreach (var file in Directory.EnumerateFiles(extractedDirectoryPath))
            {
//                Console.WriteLine("{0}", file);
                if (Path.GetExtension(file) == ".xml")
                {
//                    Console.WriteLine("Show {0}...", file);

                    // xml のエンコーディングをチェックする
                    CheckXmlEncoding(file);

//                    using (var streamReader = new StreamReader(file))
//                    {
//                        Console.WriteLine(streamReader.ReadToEnd());
//                    }
                }
            }
        }

        protected void CheckRequiredSystemVersion(string outputDir, string outputPath, uint expectedVersion)
        {
            var nspPath = outputDir + "/" + Path.GetFileNameWithoutExtension(outputPath) + "_prod.nsp";

            CheckRequiredSystemVersion(nspPath, expectedVersion);
        }

        protected void CheckResultXml(string outputDir, string outputPath)
        {
            var nspPath = outputDir + "/" + Path.GetFileNameWithoutExtension(outputPath) + "_prod.nsp";
            var xmlPath = outputDir + "/" + Path.GetFileNameWithoutExtension(outputPath) + "_prod.nsp.result.xml";

            CheckXmlEncoding(xmlPath);

            ResultModel model;
            using (var fs = new FileStream(xmlPath, FileMode.Open, FileAccess.Read))
            {
                var serializer = new XmlSerializer(typeof(ResultModel));
                model = (ResultModel)serializer.Deserialize(fs);
            }

            Assert.IsTrue(model.Code == "Pass");
            Assert.IsTrue(model.ErrorMessage == string.Empty);

            using (var fs = new FileStream(nspPath, FileMode.Open, FileAccess.Read))
            {
                Assert.IsTrue(model.Size == fs.Length);
                var hashCalculator = new SHA256CryptoServiceProvider();
                var hash = "0x" + BitConverter.ToString(hashCalculator.ComputeHash(fs), 0, 32).Replace("-", string.Empty).ToLower();
                Assert.IsTrue(model.Hash == hash);
            }

            // TORIAEZU
            Assert.IsTrue(model.Date != string.Empty);
            Assert.IsTrue(model.Command != string.Empty);
            Assert.IsTrue(model.ToolVersion != string.Empty);
            Assert.IsTrue(model.ContentMetaList.Count == 1);
            foreach (var cnmtModel in model.ContentMetaList)
            {
                if (cnmtModel.Type == NintendoContentMetaConstant.ContentMetaTypePatch)
                {
                    Assert.IsTrue(!string.IsNullOrEmpty((cnmtModel as PatchContentMetaModel)?.ApplicationId));
                }
                else if (cnmtModel.Type == NintendoContentMetaConstant.ContentMetaTypeApplication)
                {
                    Assert.IsTrue(!string.IsNullOrEmpty((cnmtModel as ApplicationContentMetaModel)?.PatchId));
                }
                else
                {
                    Assert.IsTrue(false);
                }
            }
        }

        protected static void MakeFileImpl(string filePath, IEnumerable<string> lines, System.Text.Encoding encoding = null)
        {
            var dirPath = Path.GetDirectoryName(filePath);
            if (!Directory.Exists(dirPath))
            {
                Directory.CreateDirectory(dirPath);
            }

            var swEncoding = encoding != null ? encoding : System.Text.Encoding.Default;

            using (StreamWriter sw = new StreamWriter(filePath, false, swEncoding))
            {
                foreach (var line in lines)
                {
                    sw.WriteLine(line);
                }
            }
        }

        protected static void MakeFileImpl(string filePath, string value, System.Text.Encoding encoding = null)
        {
            var dirPath = Path.GetDirectoryName(filePath);
            if (!Directory.Exists(dirPath))
            {
                Directory.CreateDirectory(dirPath);
            }

            var swEncoding = encoding != null ? encoding : System.Text.Encoding.Default;

            using (StreamWriter sw = new StreamWriter(filePath, false, swEncoding))
            {
                sw.Write(value);
            }
        }

        protected const string TestZipFileName = "legal-information.txt";
        protected const string LegalInfoXmlFileName = "legalinfo.xml";
        protected string MakeLegalInfoZipfile(string outputDir, bool hasXmlFile = true)
        {
            string tmpDir = Path.GetTempPath();
            string dirName = Path.GetRandomFileName();
            string dirPath = Path.Combine(tmpDir, dirName);
            Directory.CreateDirectory(dirPath);

            string testFilePath = Path.Combine(dirPath, TestZipFileName);
            using (var file = File.Create(testFilePath))
            {
                StreamWriter sw = new StreamWriter(file);
                sw.WriteLine("Legal Information");
                sw.Flush();
            }

            if (hasXmlFile)
            {
                CreateLegalInfoXml(dirPath);
            }

            string zipPath = Path.Combine(outputDir, Path.GetRandomFileName() + ".zip");
            ZipFile.CreateFromDirectory(dirPath, zipPath);

            Directory.Delete(dirPath, true);

            return zipPath;
        }

        protected static void MakeAccessibleUrlsFile(string accessibleUrlsFile)
        {
            string accessibleUrlsDir = Path.GetDirectoryName(accessibleUrlsFile);

            if (!Directory.Exists(accessibleUrlsDir))
            {
                Directory.CreateDirectory(accessibleUrlsDir);
            }

            MakeFileImpl(accessibleUrlsFile, new string[]
            {
                "https://www.nintendo.co.jp",
                "https://www.nintendo.co.jp/index.html&hoge=soge",
            });
        }

        protected static void MakeHtmlDocumentXmlFile(string filePath)
        {
            string dirName = Path.GetDirectoryName(filePath);

            if (!Directory.Exists(dirName))
            {
                Directory.CreateDirectory(dirName);
            }

            // 最後に改行を含めないようにするため、改行コードを明示して作る
            MakeFileImpl(filePath,
                "<?xml version=\"1.0\" encoding=\"utf-8\"?>\r\n" +
                "<HtmlDocument>\r\n" +
                "  <AccessibleUrls>\r\n" +
                "    <Url>https://www.nintendo.co.jp</Url>\r\n" +
                "    <Url>https://www.nintendo.co.jp/index.html&amp;hoge=soge</Url>\r\n" +
                "  </AccessibleUrls>\r\n" +
                "</HtmlDocument>"
            , System.Text.Encoding.UTF8);
        }

        protected static void MakeEmptyHtmlDocumentXmlFile(string filePath)
        {
            string dirName = Path.GetDirectoryName(filePath);

            if (!Directory.Exists(dirName))
            {
                Directory.CreateDirectory(dirName);
            }

            // 最後に改行を含めないようにするため、改行コードを明示して作る
            MakeFileImpl(filePath,
                "<?xml version=\"1.0\" encoding=\"utf-8\"?>\r\n" +
                "<HtmlDocument>\r\n" +
                "  <AccessibleUrls />\r\n" +
                "</HtmlDocument>"
            , System.Text.Encoding.UTF8);
        }

        protected void WriteTmpMetaFileHeader(StreamWriter sw, bool isOldRoot)
        {
            sw.WriteLine("<?xml version=\"1.0\"?>");
            if (isOldRoot)
            {
                sw.WriteLine("<Meta>");
            }
            else
            {
                sw.WriteLine("<NintendoSdkMeta>");
            }
            sw.WriteLine("<Core>");
            sw.WriteLine("<ApplicationId>0x0100000000002802</ApplicationId>");
            sw.WriteLine("</Core>");
            sw.WriteLine("<Application>");
        }

        // Extract のテスト
        protected void TestExtractWithNca(string ncaPath, string testSrcDir, string testExtractDir, int numLoop)
        {
            TestExtractWithNca(null, ncaPath, testSrcDir, testExtractDir, numLoop);
        }

        // Extract のテスト
        protected void TestExtractWithNca(string originalNcaPath, string ncaPath, string testSrcDir, string testExtractDir, int numLoop)
        {
            SafeDeleteDirectory(testExtractDir);

            string errorMsg;
            if (originalNcaPath == null)
            {
                errorMsg = ExecuteProgram(string.Format("extract {0} -o {1}", ncaPath, testExtractDir));
            }
            else
            {
                errorMsg = ExecuteProgram(string.Format("extract {0} -o {1} --original {2}", ncaPath, testExtractDir, originalNcaPath));
            }
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);

            var expected = new List<Tuple<string, long>>();
            for (int i = 0; i < numLoop; i++)
            {
                foreach (var file in Directory.EnumerateFiles(testSrcDir, "*", SearchOption.AllDirectories))
                {
                    var target = string.Format("fs{0}/{1}", i, file.Substring(testSrcDir.Length).Replace("\\", "/"));
                    expected.Add(new Tuple<string, long>(target, new System.IO.FileInfo(file).Length));
                }
            }

            var actual = new List<Tuple<string, long>>();
            foreach (var file in Directory.EnumerateFiles(testExtractDir, "*", SearchOption.AllDirectories))
            {
                var target = file.Substring(testExtractDir.Length + 1).Replace("\\", "/");
                actual.Add(new Tuple<string, long>(target, new System.IO.FileInfo(file).Length));
            }
            Assert.IsTrue(Enumerable.SequenceEqual<Tuple<string, long>>(expected, actual));
        }

        // 制限：チェック対象はマニュアル指定、.nca の名前違いはチェックしない、ファイルサイズもチェックしない
        protected void TestExtractWithNsp(string nspPath, string testExtractDir, string testExtractEntryDir, string[] entryRuleStringArray)
        {
            TestExtractWithNsp(null, nspPath, testExtractDir, testExtractEntryDir, entryRuleStringArray);
        }

        // 制限：チェック対象はマニュアル指定、.nca の名前違いはチェックしない、ファイルサイズもチェックしない
        protected void TestExtractWithNsp(string originalNspPath, string nspPath, string testExtractDir, string testExtractEntryDir, string[] entryRuleStringArray)
        {
            TestExtractWithNsp(originalNspPath, nspPath, testExtractDir, testExtractEntryDir, entryRuleStringArray, null);
        }

        // 制限：チェック対象はマニュアル指定、.nca の名前違いはチェックしない、ファイルサイズもチェックしない
        protected void TestExtractWithNsp(string originalNspPath, string nspPath, string testExtractDir, string testExtractEntryDir, string[] entryRuleStringArray, string keyConfigFilePath)
        {
            // 全 Extract のテスト
            var entryRuleList = new List<Regex>(Array.ConvertAll<string, Regex>(entryRuleStringArray, x => new Regex(x)));

            var keyConfigFileOption = string.IsNullOrEmpty(keyConfigFilePath) ? string.Empty : "--keyconfig " + keyConfigFilePath;

            var errorMsg = string.Empty;
            if (string.IsNullOrEmpty(originalNspPath))
            {
                errorMsg = ExecuteProgram(string.Format("extract {0} -o {1} {2}", nspPath, testExtractDir, keyConfigFileOption));
            }
            else
            {
                errorMsg = ExecuteProgram(string.Format("extract {0} -o {1} --original {2} {3}", nspPath, testExtractDir, originalNspPath, keyConfigFileOption));
            }
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);

            // 全て取り出して比較
            var entryList = new List<string>();
            foreach (var entry in Directory.EnumerateFiles(testExtractDir, "*", SearchOption.AllDirectories))
            {
                var entryPath = entry.Substring(testExtractDir.Length + 1).Replace("\\", "/");
                foreach (var rule in entryRuleList)
                {
                    if (rule.IsMatch(entryPath))
                    {
                        entryList.Add(entryPath);
                        break;
                    }
                }
                if (entryList.Count() == 0)
                {
                    Utils.WriteLine(entry + " is not matched.");
                }
                Assert.IsTrue(entryList.Count > 0 && entryList.Last() == entryPath);
            }
            foreach (var entry in entryRuleList.Where(rule => !entryList.Any(e => rule.IsMatch(e))))
            {
                Utils.WriteLineToError(string.Format("rule not found: '{0}'", entry));
            }
            Assert.IsTrue(entryList.Count == entryRuleList.Count);

            // nca の単独取り出しをテストできるように追加しておく
            foreach (var entry in Directory.GetDirectories(testExtractDir))
            {
                var entryPath = entry.Substring(testExtractDir.Length + 1).Replace("\\", "/");
                if (Regex.IsMatch(entryPath, @"^.*\.nca$"))
                {
                    entryList.Add(entryPath);
                }
            }

            // 単独 Extract のテスト
            foreach (var test in ExtractEntryTestList)
            {
                if (test.Item2 == false)
                {
                    Assert.IsTrue(TestExtractEntry(originalNspPath, nspPath, testExtractEntryDir, test.Item1, keyConfigFilePath) == test.Item2);
                }
            }
            foreach (var entry in entryList)
            {
                bool extractResult = TestExtractEntry(originalNspPath, nspPath, testExtractEntryDir, entry, keyConfigFilePath);
                if (!extractResult)
                {
                    Utils.WriteLine("extract: " + entry + " is missing.");
                }
                Assert.IsTrue(extractResult);
            }
        }

        // first: path, second: isExisted
        protected static readonly Tuple<string, bool>[] ExtractEntryTestList = new Tuple<string, bool>[] {
            new Tuple<string, bool>("fs0/data.dat", true),
            new Tuple<string, bool>("fs1/main.npdm", true),
            new Tuple<string, bool>("hoge", false),
            new Tuple<string, bool>("fs0/data.da", false),
            new Tuple<string, bool>("fs0/data.datt", false),
            new Tuple<string, bool>("fs2/data.dat", false),
            new Tuple<string, bool>("fs0data.dat", false),
            new Tuple<string, bool>("fs0", false),
        };

        protected bool TestExtractEntry(string nspncaPath, string outputDir, string targetPath)
        {
            return TestExtractEntry(null, nspncaPath, outputDir, targetPath);
        }

        protected bool TestExtractEntry(string originalPath, string nspncaPath, string outputDir, string targetPath)
        {
            return TestExtractEntry(originalPath, nspncaPath, outputDir, targetPath, null);
        }

        protected bool TestExtractEntry(string originalPath, string nspncaPath, string outputDir, string targetPath, string keyConfigFilePath)
        {
            var keyConfigFileOption = string.IsNullOrEmpty(keyConfigFilePath) ? string.Empty : "--keyconfig " + keyConfigFilePath;

            string errorMsg = string.Empty;
            if (string.IsNullOrEmpty(originalPath))
            {
                errorMsg = ExecuteProgram(string.Format("extract {0} -o {1} --target {2} {3}", nspncaPath, outputDir, targetPath, keyConfigFileOption));
            }
            else
            {
                errorMsg = ExecuteProgram(string.Format("extract {0} -o {1} --target {2} --original {3} {4}", nspncaPath, outputDir, targetPath, originalPath, keyConfigFileOption));
            }
            var entryPath = Path.Combine(outputDir, Path.GetFileName(targetPath));
            bool ret = File.Exists(entryPath);
            if (ret)
            {
                File.Delete(entryPath);
            }
            return ret;
        }

        protected string GetDefaultApplicationMetaFile(string metaDir)
        {
            return Path.Combine(metaDir, "describe_all.nmeta");
        }

        public static ModelType ReadXmlFromNsp<ModelType>(NintendoSubmissionPackageReader nspReader, string fileName, long fileSize)
        {
            var serializer = new XmlSerializer(typeof(ModelType));
            ModelType model;
            using (var ms = new MemoryStream((int)fileSize))
            {
                var xmlData = nspReader.ReadFile(fileName, 0, fileSize);
                ms.Write(xmlData, 0, xmlData.Length);
                ms.Seek(0, SeekOrigin.Begin);
                model = (ModelType)serializer.Deserialize(ms);
            }
            return model;
        }

        protected void CheckNcaKeyGeneration(string nspPath, List<byte> expectedGenerationList, KeyConfiguration config = null)
        {
            // expectedGenerationList には、nsp に含まれる nca 毎の期待値と順番を揃えて入力すること

            ForceGCImpl();

            var generationList = new List<byte>();
            using (var nspReader = new NintendoSubmissionPackageReader(nspPath))
            {
                if (config == null)
                {
                    // cnmt.xml を確認
                    foreach (var file in nspReader.ListFileInfo().Where(x => x.Item1.EndsWith(".cnmt.xml")))
                    {
                        var model = ReadXmlFromNsp<ContentMetaModel>(nspReader, file.Item1, file.Item2);
                        foreach (var content in model.ContentList)
                        {
                            generationList.Add((byte)content.KeyGeneration);
                        }
                    }
                }
                else
                {
                    foreach (var info in nspReader.ListFileInfo().FindAll(x => Path.GetExtension(x.Item1) == ".nca"))
                    {
                        using (var ncaReader = nspReader.OpenNintendoContentArchiveReader(info.Item1, new NcaKeyGenerator(config)))
                        {
                            generationList.Add(ncaReader.GetKeyGeneration());
                        }
                    }
                }
            }
            Trace.Assert(generationList.Count == expectedGenerationList.Count);

            for (int i = 0; i < expectedGenerationList.Count; i++)
            {
                Assert.AreEqual(generationList[i], expectedGenerationList[i]);
            }
        }

        protected byte[] Checksum(string filePath, int notCheckedTailSize = 0)
        {
            using (var md5 = MD5.Create())
            {
                using (var stream = File.OpenRead(filePath))
                {
                    byte[] data = new byte[stream.Length - notCheckedTailSize];
                    stream.Read(data, 0, data.Length);
                    return md5.ComputeHash(data);
                }
            }
        }

        protected List<string> GetEntryFilePathList(string directoryPath)
        {
            var entryList = new List<string>();
            foreach (var entry in Directory.EnumerateFiles(directoryPath, "*", SearchOption.AllDirectories))
            {
                var entryPath = entry.Substring(directoryPath.Length + 1).Replace("\\", "/");
                entryList.Add(entryPath);
            }
            return entryList;
        }

        protected bool IsTargetPathIncluded(List<Regex> ruleList, string targetPath)
        {
            foreach (var rule in ruleList)
            {
                if (rule.IsMatch(targetPath))
                {
                    return true;
                }
            }
            return false;
        }

        protected string FindTargetEntryPath(List<string> entryList, Regex pattern)
        {
            foreach (var entry in entryList)
            {
                if (pattern.IsMatch(entry))
                {
                    return entry;
                }
            }
            return null;
        }

        protected void TestReplaceWithNspOrNca(string archivePath, string testReplaceDir, string inExtractedDir, string testExtractDir, Tuple<Regex, bool>[] targetRuleArray)
        {
            TestReplaceWithNspOrNca(archivePath, testReplaceDir, inExtractedDir, testExtractDir, targetRuleArray, null);
        }

        protected int replaceTestCount = 0;
        protected void TestReplaceWithNspOrNca(string archivePath, string testReplaceDir, string inExtractedDir, string testExtractDir, Tuple<Regex, bool>[] targetRuleArray, List<int[]> combination)
        {
            // 名前リストを取得する
            var numEntries = 0;
            var replaceInfoList = new List<Tuple<string, Regex, bool>>();
            foreach (var targetRule in targetRuleArray)
            {
                var entryList = GetEntryFilePathList(inExtractedDir);
                numEntries = entryList.Count;
                var targetPath = FindTargetEntryPath(entryList, targetRule.Item1);
                replaceInfoList.Add(new Tuple<string, Regex, bool>(targetPath, targetRule.Item1, targetRule.Item2));
            }

            if (combination == null)
            {
                combination = new List<int[]>() { new int[2] { 0, 1 } };
            }

            // 複数差し替えの組み合わせを作成する
            var replaceInfoTupleArrayList = new List<Tuple<string, Regex, bool>[]>();
            foreach (var comb in combination)
            {
                var element = new Tuple<string, Regex, bool>[comb.Count()];
                for(int index = 0; index < comb.Count(); index++)
                {
                    element[index] = replaceInfoList[comb[index]];
                }
                replaceInfoTupleArrayList.Add(element);
            }

            TestReplaceWithNspOrNca(archivePath, testReplaceDir, inExtractedDir, testExtractDir, numEntries, replaceInfoTupleArrayList);
        }

        protected void TestReplaceWithNspOrNca(string archivePath, string testReplaceDir, string inExtractedDir, string testExtractDir, int numEntries, List<Tuple<string, Regex, bool>[]> replaceInfoList)
        {
            Directory.CreateDirectory(testReplaceDir);
            string fileExtension = Path.GetExtension(archivePath);
            bool isNsp = fileExtension == ".nsp";

            TestUtility.TestPath testPath = new TestUtility.TestPath(this.TestContext);
            var descPath = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";

            // replace → extract でファイルのチェック
            foreach (var repInfoSet in replaceInfoList)
            {
                // 差替え用ファイルの作成（サイズ増減で3パターン）
                var sizeOffsetList = new int[] { 0, -1000, 1000 };
                foreach (int sizeOffset in sizeOffsetList)
                {
                    // 差し替えで必要なもの
                    var targetPathList = Array.ConvertAll<Tuple<string, Regex, bool>, string>(repInfoSet, x => x.Item1).ToList();
                    var entryPathList = new List<string>();
                    bool isProgramContent = false;

                    // 差し替え後のチェックで必要なもの
                    var replacedEntryInfoList = new List<Tuple<Regex, string, string>>();

                    foreach (var repInfo in repInfoSet)
                    {
                        SafeDeleteDirectory(testExtractDir);

                        var targetPath = repInfo.Item1;
                        var entryRule = repInfo.Item2;
                        isProgramContent |= repInfo.Item3;

                        var entryFileSrcPath = Path.Combine(inExtractedDir, targetPath);
                        var entryFileModifiedSrcPath = Path.Combine(testReplaceDir, Path.GetFileName(entryFileSrcPath) + ".modified");
                        entryPathList.Add(entryFileModifiedSrcPath);
                        replacedEntryInfoList.Add(new Tuple<Regex, string, string>(entryRule, entryFileSrcPath, entryFileModifiedSrcPath));

                        // replace 用ファイルの作成
                        {
                            byte[] data = File.ReadAllBytes(entryFileSrcPath);
                            if (entryFileSrcPath.EndsWith(".cnmt"))
                            {
                                // cnmt の Version（9 バイト目）をインクリメント
                                data[9] += 1;
                                File.WriteAllBytes(entryFileModifiedSrcPath, data);
                            }
                            else if (entryFileSrcPath.EndsWith(".nacp"))
                            {
                                // nacp の末尾（リザーブ領域）をインクリメント
                                data[data.Length - 1] += 1;
                                File.WriteAllBytes(entryFileModifiedSrcPath, data);
                            }
                            else
                            {
                                data[0] += 1;
                                if (sizeOffset != 0)
                                {
                                    int length = Math.Max(1, data.Length + sizeOffset);
                                    byte[] data2 = new byte[length];
                                    Buffer.BlockCopy(data, 0, data2, 0, Math.Min(data.Length, data2.Length));
                                    File.WriteAllBytes(entryFileModifiedSrcPath, data2);
                                }
                                else
                                {
                                    File.WriteAllBytes(entryFileModifiedSrcPath, data);
                                }
                            }
                        }
                    }

                    // replace
                    string errorMsg = "";
                    string replaceRuleListFilePath = Path.Combine(testReplaceDir, "../replaceRuleFilePath.csv");
                    string replaceArgsStr = string.Format("{0} {1}", String.Join("|", targetPathList), String.Join("|", entryPathList));
                    string replaceArgCommand = (replaceTestCount++ % 2 == 0) ? replaceRuleListFilePath : replaceArgsStr;
                    using (var fw = new System.IO.StreamWriter(replaceRuleListFilePath))
                    {
                        for(int i = 0; i < targetPathList.Count; i++)
                        {
                            fw.Write(string.Format("{0}\t{1}\n", targetPathList[i], entryPathList[i]));
                        }
                    }

                    if (isProgramContent)
                    {
                        // desc 無しでの失敗チェック
                        errorMsg = ExecuteProgram(string.Format("replace {0} {1} -o {2}", archivePath, replaceArgCommand, testReplaceDir));
                        Assert.IsTrue(errorMsg.IndexOf("Replacing 'Program' content needs desc file.") >= 0, errorMsg);

                        errorMsg = ExecuteProgram(string.Format("replace {0} {1} --desc {2} -o {3}", archivePath, replaceArgCommand, descPath, testReplaceDir));
                        Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                    }
                    else
                    {
                        errorMsg = ExecuteProgram(string.Format("replace {0} {1} -o {2} --ignore-unpublishable-error", archivePath, replaceArgCommand, testReplaceDir));
                        Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                        // TODO: cnmt の ID 変更検知
                    }

                    // extract
                    var replacedNspPath = Path.Combine(testReplaceDir, Path.GetFileNameWithoutExtension(archivePath) + "_replaced" + fileExtension);
                    errorMsg = ExecuteProgram(string.Format("extract {0} -o {1}", replacedNspPath, testExtractDir));
                    Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                    // list check
                    var replacedEntryList = GetEntryFilePathList(testExtractDir);
                    Assert.IsTrue(numEntries == replacedEntryList.Count);  // 数のみチェック

                    // 修正後のファイルの一致確認
                    if (isNsp)
                    {
                        foreach (var replacedEntryInfo in replacedEntryInfoList)
                        {
                            var entryRule = replacedEntryInfo.Item1;
                            var entryFileSrcPath = replacedEntryInfo.Item2;
                            var entryFileModifiedSrcPath = replacedEntryInfo.Item3;

                            // TORIAEZU: 全エントリのハッシュチェックは nca 側のみ
                            var entryFileOutPath = FindTargetEntryPath(replacedEntryList, entryRule);
                            entryFileOutPath = Path.Combine(testExtractDir, entryFileOutPath);

                            if (entryFileSrcPath.EndsWith(".cnmt"))
                            {
                                // cnmt の Digest は一致しない
                                var orgFileHash = Checksum(entryFileModifiedSrcPath, 32);
                                var replacedFileHash = Checksum(entryFileOutPath, 32);
                                // cnmt 以外も変えた場合は一致しない
                                if (replacedEntryInfoList.Count > 1)
                                {
                                    Assert.IsTrue(! orgFileHash.SequenceEqual(replacedFileHash));
                                }
                                else
                                {
                                    Assert.IsTrue(orgFileHash.SequenceEqual(replacedFileHash));
                                }
                            }
                            else
                            {
                                var orgFileHash = Checksum(entryFileModifiedSrcPath);
                                var replacedFileHash = Checksum(entryFileOutPath);
                                Assert.IsTrue(orgFileHash.SequenceEqual(replacedFileHash));
                            }

                            var srcEntryFileList = Directory.EnumerateFiles(inExtractedDir, "*", SearchOption.AllDirectories);
                            var replacedEntryFileList = Directory.EnumerateFiles(testExtractDir, "*", SearchOption.AllDirectories);
                            Assert.IsTrue(srcEntryFileList.Count() == replacedEntryFileList.Count());
                        }
                    }
                    else
                    {
                        var entryFileSrcPathList = replacedEntryInfoList.ConvertAll<string>(x => x.Item2).ToList();
                        var entryFileSrcPathDict = new Dictionary<string, string>();
                        foreach(var replacedEntryInfo in replacedEntryInfoList)
                        {
                            entryFileSrcPathDict[replacedEntryInfo.Item2] = replacedEntryInfo.Item3;
                        }

                        bool isReplaced = false;
                        var srcEntryFileList = Directory.EnumerateFiles(inExtractedDir, "*", SearchOption.AllDirectories);
                        foreach (var entryFilePath in srcEntryFileList)
                        {
                            var entryPath = entryFilePath.Substring(inExtractedDir.Length + 1).Replace("\\", "/");
                            var replacedEntryFilePath = Path.Combine(testExtractDir, entryPath);
                            var originalSrcPath = entryFilePath;
                            var replacedEntryPath = entryFileSrcPathList.Find(path => Path.GetFullPath(entryFilePath) == Path.GetFullPath(path));
                            if (replacedEntryPath != null)
                            {
                                originalSrcPath = entryFileSrcPathDict[replacedEntryPath];
                                isReplaced = true;
                            }
                            var orgFileHash = Checksum(originalSrcPath);
                            var replacedFileHash = Checksum(replacedEntryFilePath);
                            Assert.IsTrue(orgFileHash.SequenceEqual(replacedFileHash));
                        }
                        var replacedEntryFileList = Directory.EnumerateFiles(testExtractDir, "*", SearchOption.AllDirectories);
                        Assert.IsTrue(srcEntryFileList.Count() == replacedEntryFileList.Count());
                        Assert.IsTrue(isReplaced);
                    }
                }
            }

            // Replace が発生しなかった際バイナリ一致すること
            {
                var replaceRule = replaceInfoList.First().First();
                var entry = replaceRule.Item1;
                var errorMsg = ExecuteProgram(string.Format("replace {0} {1} {2} --desc {3} -o {4} --ignore-unpublishable-error", archivePath, entry, Path.Combine(inExtractedDir, entry), descPath, testReplaceDir));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                var replacedNspPath = Path.Combine(testReplaceDir, Path.GetFileNameWithoutExtension(archivePath) + "_replaced" + fileExtension);
                CheckFileDiff(archivePath, replacedNspPath, 0);
            }
        }

        protected void TestReplaceNcaInNsp(string archivePath, string testTempDir, string targetNcaName, string targetEntryName)
        {
            TestReplaceNcaInNsp(archivePath, testTempDir, targetNcaName, targetEntryName, null, false);
        }

        // packagePath の nsp から targetNcaName の nca を取り出し、その nca に含まれる targetEntryName を修正し、修正済み nca を nsp に replace する
        // addNcaTarget が指定されている場合、修正済み nca を addNcaTarget.Item2 で指定された nsp に追加する
        protected void TestReplaceNcaInNsp(string packagePath, string testTempDir, string targetNcaName, string targetEntryName, Tuple<string, string> addNcaTarget, bool isFailTest)
        {
            // create temp dir
            SafeDeleteDirectory(testTempDir);
            Directory.CreateDirectory(testTempDir);
            string errorMsg = "";

            TestUtility.TestPath testPath = new TestUtility.TestPath(this.TestContext);
            var descPath = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";

            // extract nca to temp dir
            errorMsg = ExecuteProgram(string.Format("extract {0} --target {1} -o {2}", packagePath, targetNcaName, testTempDir));
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
            string ncaPath = Path.Combine(testTempDir, targetNcaName);

            // extract nsp to temp dir
            string tempExtractDir = Path.Combine(testTempDir, "ext");
            errorMsg = ExecuteProgram(string.Format("extract {0} -o {1}", packagePath, tempExtractDir));
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);

            // select target
            var entryList = GetEntryFilePathList(tempExtractDir);
            var targetPath = entryList.Find(entry => entry.EndsWith(targetEntryName));
            var extractedFilePath = Path.Combine(tempExtractDir, targetPath);
            var editedFilePath = Path.Combine(testTempDir, Path.GetFileName(extractedFilePath) + ".edited");

            // edit extracted file
            File.Copy(extractedFilePath, editedFilePath);
            using (var fs = new FileStream(editedFilePath, FileMode.Append, FileAccess.Write))
            using (var sw = new StreamWriter(fs))
            {
                sw.Write("test");
            }

            if(addNcaTarget == null)
            {
                // replace nca to temp dir
                string targetPathInNca = targetPath.Substring(targetNcaName.Count() + 1);
                var com = string.Format("replace {0} {1} {2} -o {3} --desc {4}", ncaPath, targetPathInNca, editedFilePath, testTempDir, descPath);
                errorMsg = ExecuteProgram(string.Format("replace {0} {1} {2} -o {3} --desc {4}", ncaPath, targetPathInNca, editedFilePath, testTempDir, descPath));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                string replacedNcaPath = Path.Combine(testTempDir, ncaPath.Replace(".nca", "") + "_replaced.nca");

                // replace nsp with replaced nca
                errorMsg = ExecuteProgram(string.Format("replace {0} {1} {2} -o {3}", packagePath, targetNcaName, replacedNcaPath, testTempDir));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                string replacedNspPath = Path.Combine(testTempDir, Path.GetFileNameWithoutExtension(packagePath) + "_replaced.nsp");

                // extract files from replaced nsp
                string testReplacedTempDir = Path.Combine(testTempDir, "rep_ext");
                errorMsg = ExecuteProgram(string.Format("extract {0} -o {1}", replacedNspPath, testReplacedTempDir));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                // find replaced file
                entryList = GetEntryFilePathList(testReplacedTempDir);
                var extractedReplacedNcaPath = Path.Combine(testReplacedTempDir, entryList.Find(entry => entry.EndsWith(targetEntryName)));

                // check hash and compare
                CheckFileDiff(editedFilePath, extractedReplacedNcaPath, 0);
            }
            else
            {
                // replace fail test
                if(!isFailTest){
                    var replacedNspFailPath = addNcaTarget.Item1;
                    var testRepFailExtractDir = Path.Combine(testTempDir, "rep_fail_ext");
                    errorMsg = ExecuteProgram(string.Format("extract {0} -o {1}", replacedNspFailPath, testRepFailExtractDir));
                    Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                    entryList = GetEntryFilePathList(testRepFailExtractDir);
                    var targetName = "data.dat";
                    var ncaName = entryList.Find(entry => entry.EndsWith("/" + targetName)).Replace("/fs0/" + targetName, "");

                    errorMsg = ExecuteProgram(string.Format("replace {0} {1} {2} -o {3}", replacedNspFailPath, ncaName, ncaPath, testTempDir));
                    Assert.IsTrue(errorMsg != string.Empty, errorMsg);
                }

                // add nca to nsp
                errorMsg = ExecuteProgram(string.Format("replace {0} {1} {2} -o {3}", addNcaTarget.Item1, addNcaTarget.Item2, ncaPath, testTempDir));
                if(isFailTest)
                {
                    Assert.IsTrue(errorMsg != string.Empty, errorMsg);
                    return;
                }
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                string replacedNspPath = Path.Combine(testTempDir, Path.GetFileNameWithoutExtension(addNcaTarget.Item1) + "_replaced.nsp");

                // extract nsp
                string testReplacedTempDir = Path.Combine(testTempDir, "rep_ext");
                errorMsg = ExecuteProgram(string.Format("extract {0} -o {1}", replacedNspPath, testReplacedTempDir));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                // prepare check
                entryList = GetEntryFilePathList(testReplacedTempDir);
                var extractedReplacedFilePath = entryList.FindAll(entry => entry.EndsWith("/" + targetEntryName));
                Assert.IsTrue(extractedReplacedFilePath.Count() > 0);
            }
        }

        protected static void CheckFileDiff(string file1, string file2, long length) // length 0 で全一致確認
        {
            using (var fs1 = new FileStream(file1, FileMode.Open, FileAccess.Read))
            {
                using (var fs2 = new FileStream(file2, FileMode.Open, FileAccess.Read))
                {
                    long checkLength;
                    if (length == 0)
                    {
                        Assert.IsTrue(fs1.Length == fs2.Length);
                        checkLength = fs1.Length;
                    }
                    else
                    {
                        checkLength = length;
                    }

                    var fs1Data = new byte[checkLength];
                    var fs2Data = new byte[checkLength];

                    fs1.Read(fs1Data, 0, fs1Data.Length);
                    fs2.Read(fs2Data, 0, fs2Data.Length);

                    Assert.IsTrue(fs1Data.SequenceEqual(fs2Data));
                }
            }
        }

        public struct TestFileInfo
        {
            public bool IsRomFs;
            public string NcaName;
            public int FsIndex;
            public string FileName;
            public long Offset;
            public long Size;

            public bool IsSamePartition(TestFileInfo other)
            {
                return NcaName == other.NcaName && FsIndex == other.FsIndex;
            }
        }

        public static List<TestFileInfo> CheckRomFsFileAlignment(string nspPath, KeyConfiguration keyConfig)
        {
            var fileList = new List<TestFileInfo>();

            using (var stream = File.OpenRead(nspPath))
            using (var nsp = new NintendoSubmissionPackageReader(stream))
            {
                Action<string, NintendoContentArchiveReader> CheckArchive = (name, nca) =>
                {
                    for (int i = 0; i < 4; ++i)
                    {
                        if (nca.HasFsInfo(i))
                        {
                            using (var fs = nca.OpenFileSystemArchiveReader(i))
                            {
                                var isRomFs = fs is RomFsFileSystemArchiveReader;

                                foreach (var fileInfo in fs.ListFileInfo())
                                {
                                    var pair = fs.GetFileFragmentList(fileInfo.Item1).First();
                                    fileList.Add(new TestFileInfo
                                    {
                                        IsRomFs = isRomFs,
                                        NcaName = name,
                                        FsIndex = i,
                                        FileName = fileInfo.Item1,
                                        Offset = pair.Item1,
                                        Size = pair.Item2
                                    });
                                }
                            }
                        }
                    }
                };
                var keyGenerator = new NcaKeyGenerator(keyConfig);
                foreach (var ncaList in nsp.ListFileInfo().Where(info => info.Item1.EndsWith(".nca")))
                {
                    using (var nca = nsp.OpenNintendoContentArchiveReader(ncaList.Item1, keyGenerator))
                    {
                        CheckArchive(Path.GetFileName(ncaList.Item1), nca);
                    }
                }
            }

            return fileList;
        }
    }
}
