﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.IO.Compression;
using System.Runtime.InteropServices;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.Authoring.AuthoringLibrary;
using Nintendo.Authoring.AuthoringTool;
using TestUtility;

namespace AuthoringToolsTest
{
    [TestClass]
    public class OptionTest
    {
        private void DeleteDirectoryIfExisted(string path)
        {
            if (Directory.Exists(path))
            {
                Directory.Delete(path, true);
            }
            // Directory.Delete は非同期なので、削除されるまで待つ
            const int WaitForDeleteDirectoryTimeOutMilliSec = 1000 * 3;
            const int WaitForDeleteDirectoryWaitUnitMilliSec = 100;
            for (int waitMilliSec = 0; waitMilliSec < WaitForDeleteDirectoryTimeOutMilliSec; waitMilliSec += WaitForDeleteDirectoryWaitUnitMilliSec)
            {
                if (Directory.Exists(path) == false)
                {
                    break;
                }
                System.Threading.Thread.Sleep(WaitForDeleteDirectoryWaitUnitMilliSec);
            }
        }

        private const string TestNroDir = "TestTmpNroDir";
        private string MakeTmpNroDir()
        {
            string tmpDir = Path.GetTempPath();
            string dirName = TestNroDir;
            string dirPath = Path.Combine(tmpDir, dirName);
            Directory.CreateDirectory(dirPath);
            return dirPath;
        }

        private const string TestZipFileName = "legal-information.txt";
        private string MakeTmpZipFile()
        {
            string tmpDir = Path.GetTempPath();
            string dirName = Path.GetRandomFileName();
            string dirPath = Path.Combine(tmpDir, dirName);
            Directory.CreateDirectory(dirPath);

            string testFilePath = Path.Combine(dirPath, TestZipFileName);
            using (var file = File.Create(testFilePath))
            {
                StreamWriter sw = new StreamWriter(file);
                sw.WriteLine("Legal Information");
                sw.Flush();
            }

            string zipPath = Path.Combine(tmpDir, Path.GetRandomFileName() + ".zip");
            ZipFile.CreateFromDirectory(dirPath, zipPath);

            Directory.Delete(dirPath, true);

            return zipPath;
        }

        [TestMethod]
        public void TestNoneOption()
        {
            // 不明なオプションの指定
            {
                Option option = new Option();
                string[] args = new string[] { };
                option.Parse(args);
                Assert.IsTrue(option.IsShowUsage);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "--unknown" };
                Utils.CheckReturnException(new InvalidOptionException("subcommand --unknown is not implemented."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }

            // -h, --help
            {
                Option option = new Option();
                string[] args = new string[] { "-h" };
                option.Parse(args);
                Assert.IsTrue(option.IsShowUsage);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "--help" };
                option.Parse(args);
                Assert.IsTrue(option.IsShowUsage);
            }
        }

        [TestMethod]
        public void TestHelpOption()
        {
            // 不明なオプションの指定
            {
                Option option = new Option();
                string[] args = new string[] { "help" };
                option.Parse(args);
                Assert.IsTrue(option.Help != null);
                Assert.IsTrue(option.IsShowUsage);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "help", "--unknown" };
                Utils.CheckReturnException(new InvalidOptionException("subcommand --unknown is not implemented."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }

            // -h, --help
            {
                Option option = new Option();
                string[] args = new string[] { "help", "-h" };
                option.Parse(args);
                Assert.IsTrue(option.Help != null);
                Assert.IsTrue(option.IsShowUsage);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "help", "--help" };
                option.Parse(args);
                Assert.IsTrue(option.Help != null);
                Assert.IsTrue(option.IsShowUsage);
            }

            // 各サブコマンドのテスト
            {
                Option option = new Option();
                string[] args = new string[] { "help", "unknown" };
                Utils.CheckReturnException(new InvalidOptionException("subcommand unknown is not implemented."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "help", "help", "createfs", "createnca" };
                option.Parse(args);
                Assert.IsTrue(option.Help != null);
                Assert.IsTrue(option.IsShowUsage);
                Assert.IsTrue(option.Help.SubCommandList.FindIndex(e => e == Option.SubCommandType.Help) != -1);
                Assert.IsTrue(option.Help.SubCommandList.FindIndex(e => e == Option.SubCommandType.CreateFs) != -1);
                Assert.IsTrue(option.Help.SubCommandList.FindIndex(e => e == Option.SubCommandType.CreateNca) != -1);
            }
        }

        [TestMethod]
        public void TestCreateFsOption()
        {
            // 不明なオプションの指定
            {
                Option option = new Option();
                string[] args = new string[] { "createfs" };
                Utils.CheckReturnException(new InvalidOptionException("too few arguments for createfs subcommand."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createfs", "--unknown" };
                option.Parse(args);
                // Todo: ヘルプ表示
            }

            // -h, --help
            {
                Option option = new Option();
                string[] args = new string[] { "createfs", "-h" };
                option.Parse(args);
                Assert.IsTrue(option.IsShowUsage);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createfs", "--help" };
                option.Parse(args);
                Assert.IsTrue(option.IsShowUsage);
            }

            // --format
            {
                Option option = new Option();
                string[] args = new string[] { "createfs", "--format", "unknown", "dir1" };
                Utils.CheckReturnException(new InvalidOptionException("invalid option --format."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createfs", "--format", "partitionfs", "dir1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateFs != null);
                Assert.IsTrue(option.CreateFs.Format == ArchiveFormatType.PartitionFs);
                Assert.IsTrue(option.CreateFs.InputDir == "dir1");
                Assert.IsTrue(!option.CreateFs.IsSaveAdf);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createfs", "--format", "partitionfs" };
                Utils.CheckReturnException(new InvalidOptionException("too few arguments for createfs subcommand."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createfs", "--format", "partitionfs", "dir1", "dir2" };
                Utils.CheckReturnException(new InvalidOptionException("too many arguments for createfs subcommand."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }

            // input adf
            {
                Option option = new Option();
                string[] args = new string[] { "createfs", "file1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateFs != null);
                Assert.IsTrue(option.CreateFs.Format == ArchiveFormatType.Invalid);
                Assert.IsTrue(option.CreateFs.InputAdfFile == "file1");
                Assert.IsTrue(option.CreateFs.IsSaveAdf);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createfs", "file1", "file2" };
                Utils.CheckReturnException(new InvalidOptionException("too many arguments for createfs subcommand."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }

            // --save-adf
            {
                Option option = new Option();
                string[] args = new string[] { "createfs", "--save-adf", "file1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateFs != null);
                Assert.IsTrue(option.CreateFs.IsSaveAdf);
            }

            // -o
            {
                Option option = new Option();
                string[] args = new string[] { "createfs", "-o", "c:/Windows/Temp/output", "file1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateFs != null);
                Assert.IsTrue(option.CreateFs.OutputFile == "c:/Windows/Temp/output");
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createfs", "-o", "output", "file1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateFs != null);
                Assert.IsTrue(option.CreateFs.OutputFile == "./output");
            }
        }

        [TestMethod]
        public void TestCreateNcaOption()
        {
            // 不明なオプションの指定
            {
                Option option = new Option();
                string[] args = new string[] { "createnca" };
                Utils.CheckReturnException(new InvalidOptionException("too few arguments for createnca subcommand."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnca", "--unknown" };
                option.Parse(args);
                // Todo: ヘルプ表示
            }

            // -h, --help
            {
                Option option = new Option();
                string[] args = new string[] { "createnca", "-h" };
                option.Parse(args);
                Assert.IsTrue(option.IsShowUsage);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnca", "--help" };
                option.Parse(args);
                Assert.IsTrue(option.IsShowUsage);
            }

            // --program
            {
                Option option = new Option();
                string[] args = new string[] { "createnca", "--program" };
                Utils.CheckReturnException(new InvalidOptionException("no option argument for \"--program\"."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnca", "--program", "dir1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNca != null);
                Assert.IsTrue(option.CreateNca.ContentType == "Program");
                Assert.IsTrue(option.CreateNca.InputDirs.Count == 1);
                Assert.IsTrue(option.CreateNca.InputDirs[0].first == "dir1");
                Assert.IsTrue(!option.CreateNca.IsSaveAdf);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnca", "--program", "dir1", "dir2" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNca != null);
                Assert.IsTrue(option.CreateNca.ContentType == "Program");
                Assert.IsTrue(option.CreateNca.InputDirs.Count == 2);
                Assert.IsTrue(option.CreateNca.InputDirs[0].first == "dir1");
                Assert.IsTrue(option.CreateNca.InputDirs[1].first == "dir2");
                Assert.IsTrue(!option.CreateNca.IsSaveAdf);
            }

            // --control
            {
                Option option = new Option();
                string[] args = new string[] { "createnca", "--control" };
                Utils.CheckReturnException(new InvalidOptionException("no option argument for \"--control\"."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnca", "--control", "dir1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNca != null);
                Assert.IsTrue(option.CreateNca.ContentType == "Control");
                Assert.IsTrue(option.CreateNca.InputDirs.Count == 1);
                Assert.IsTrue(option.CreateNca.InputDirs[0].first == "dir1");
                Assert.IsTrue(!option.CreateNca.IsSaveAdf);
            }

            // --html-document
            {
                Option option = new Option();
                string[] args = new string[] { "createnca", "--html-document" };
                Utils.CheckReturnException(new InvalidOptionException("no option argument for \"--html-document\"."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnca", "--html-document", "dir1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNca != null);
                Assert.IsTrue(option.CreateNca.ContentType == "HtmlDocument");
                Assert.IsTrue(option.CreateNca.InputDirs.Count == 1);
                Assert.IsTrue(option.CreateNca.InputDirs[0].first == "dir1");
                Assert.IsTrue(!option.CreateNca.IsSaveAdf);
            }

            // --accessible-urls
            {
                Option option = new Option();
                string[] args = new string[] { "createnca", "--accessible-urls" };
                Utils.CheckReturnException(new InvalidOptionException("no option argument for \"--accessible-urls\"."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnca", "--accessible-urls", "dir1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNca != null);
                Assert.IsTrue(option.CreateNca.ContentType == "HtmlDocument");
                Assert.IsTrue(option.CreateNca.InputDirs.Count == 1);
                Assert.IsTrue(option.CreateNca.InputDirs[0].first == "dir1");
                Assert.IsTrue(!option.CreateNca.IsSaveAdf);
            }

            // --html-document & --accessible-urls
            {
                Option option = new Option();
                string[] args = new string[] { "createnca", "--html-document", "dir1", "--accessible-urls", "dir2" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNca != null);
                Assert.IsTrue(option.CreateNca.ContentType == "HtmlDocument");
                Assert.IsTrue(option.CreateNca.InputDirs.Count == 2);
                Assert.IsTrue(option.CreateNca.InputDirs[0].first == "dir1");
                Assert.IsTrue(option.CreateNca.InputDirs[1].first == "dir2");
                Assert.IsTrue(!option.CreateNca.IsSaveAdf);
            }

            // --legal-information
            {
                Option option = new Option();
                string[] args = new string[] { "createnca", "--legal-information" };
                Utils.CheckReturnException(new InvalidOptionException("no option argument for \"--legal-information\"."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnca", "--legal-information", "dir1" };
                Utils.CheckReturnException(new InvalidOptionException(".zip file should be specified for legal information."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                var zipPath = MakeTmpZipFile();
                Option option = new Option();
                string[] args = new string[] { "createnca", "--legal-information", zipPath };
                option.Parse(args);
                Assert.IsTrue(option.CreateNca != null);
                Assert.IsTrue(option.CreateNca.ContentType == "LegalInformation");
                Assert.IsTrue(option.CreateNca.InputDirs.Count == 1);
                Assert.IsTrue(Directory.Exists(option.CreateNca.InputDirs[0].first));
                Assert.IsTrue(File.Exists(Path.Combine(option.CreateNca.InputDirs[0].first, TestZipFileName)));
                Assert.IsTrue(!option.CreateNca.IsSaveAdf);

                File.Delete(zipPath);
            }

            // --html-document & --legal-information
            {
                var zipPath = MakeTmpZipFile();
                Option option = new Option();
                string[] args = new string[] { "createnca", "--html-document", "dir1", "--legal-information", zipPath };
                Utils.CheckReturnException(new InvalidOptionException("cannot set two content types to nca."), () =>
                {
                    option.Parse(args);
                    return true;
                });

                File.Delete(zipPath);
            }

            // --accessible-urls & --legal-information
            {
                var zipPath = MakeTmpZipFile();
                Option option = new Option();
                string[] args = new string[] { "createnca", "--accessible-urls", "dir1", "--legal-information", zipPath };
                Utils.CheckReturnException(new InvalidOptionException("cannot set two content types to nca."), () =>
                {
                    option.Parse(args);
                    return true;
                });

                File.Delete(zipPath);
            }

            // --nro
            {
                Option option = new Option();
                string[] args = new string[] { "createnca", "--nro", "nroDir", "--program", "dir1"};
                option.Parse(args);
                Assert.IsTrue(option.CreateNca != null);
                Assert.IsTrue(option.CreateNca.ContentType == "Program");
                Assert.IsTrue(option.CreateNca.InputDirs.Count == 1);
                Assert.IsTrue(option.CreateNca.InputDirs[0].first == "dir1");
                Assert.IsTrue(!option.CreateNca.IsSaveAdf);
            }
#if false
            // --addoncontent
            {
                Option option = new Option();
                string[] args = new string[] { "createnca", "--addoncontent" };
                Utils.CheckReturnException(new InvalidOptionException("no option argument for \"--addoncontent\"."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnca", "--addoncontent", "dir1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNca != null);
                Assert.IsTrue(option.CreateNca.ContentType == NintendoContentArchiveContentType.AddOnContent);
                Assert.IsTrue(option.CreateNca.InputDirs.Count == 1);
                Assert.IsTrue(option.CreateNca.InputDirs[0].first == "dir1");
                Assert.IsTrue(!option.CreateNca.IsSaveAdf);
            }
#endif

            // input adf
            {
                Option option = new Option();
                string[] args = new string[] { "createnca", "file1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNca != null);
                Assert.IsTrue(option.CreateNca.ContentType == null);
                Assert.IsTrue(option.CreateNca.InputAdfFile == "file1");
                Assert.IsTrue(option.CreateNca.IsSaveAdf);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnca", "file1", "file2" };
                Utils.CheckReturnException(new InvalidOptionException("too many arguments for createnca subcommand."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }

            // --save-adf
            {
                Option option = new Option();
                string[] args = new string[] { "createnca", "--save-adf", "file1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNca != null);
                Assert.IsTrue(option.CreateNca.IsSaveAdf);
            }

            // -o
            {
                Option option = new Option();
                string[] args = new string[] { "createnca", "-o", "c:/Windows/Temp/output", "file1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNca != null);
                Assert.IsTrue(option.CreateNca.OutputFile == "c:/Windows/Temp/output");
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnca", "-o", "output", "file1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNca != null);
                Assert.IsTrue(option.CreateNca.OutputFile == "./output");
            }
        }

        [TestMethod]
        public void TestCreateNspOption()
        {
            string testDir = Path.GetTempPath().Replace("\\", "/") + "/TestAuthoringTool/";
            string metaFile = testDir + "test.nmeta";
            Directory.CreateDirectory(testDir);
            using (FileStream stream = File.Create(metaFile))
            using (var sw = new StreamWriter(stream))
            {
                sw.WriteLine("<NintendoSdkMeta><KeyGeneration>0</KeyGeneration><Application></Application></NintendoSdkMeta>");
            }

            string metaFileForFdf = testDir + "testFdf.nmeta";
            using (FileStream stream = File.Create(metaFileForFdf))
            using (var sw = new StreamWriter(stream))
            {
                sw.WriteLine("<NintendoSdkMeta><KeyGeneration>0</KeyGeneration><Application><FilterDescriptionFilePath>dummy</FilterDescriptionFilePath></Application></NintendoSdkMeta>");
            }

            // 不明なオプションの指定
            {
                Option option = new Option();
                string[] args = new string[] { "creatensp" };
                Utils.CheckReturnException(new InvalidOptionException("too few arguments for creatensp subcommand."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "creatensp", "--unknown" };
                option.Parse(args);
                // Todo: ヘルプ表示
            }

            // -h, --help
            {
                Option option = new Option();
                string[] args = new string[] { "creatensp", "-h" };
                option.Parse(args);
                Assert.IsTrue(option.IsShowUsage);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "creatensp", "--help" };
                option.Parse(args);
                Assert.IsTrue(option.IsShowUsage);
            }

            // adf 指定しない場合、--meta, --type と --program/--control/--manual のいずれかは指定が必須
            {
                Option option = new Option();
                string[] args;
                args = new string[] { "creatensp", "--type", "Application", "--program", "dir1" };
                Utils.CheckReturnException(new InvalidOptionException("too few options for creatensp subcommand."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
                args = new string[] { "creatensp", "--meta", metaFile, "--program", "dir1" };
                Utils.CheckReturnException(new InvalidOptionException("too few options for creatensp subcommand."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
                args = new string[] { "creatensp", "--meta", metaFile, "--type", "Application", "--program", "dir1", "dir2", "dir3", "adfPath" };
                Utils.CheckReturnException(new InvalidOptionException("too many arguments for creatensp subcommand."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }

            // --meta
            {
                Option option = new Option();
                string[] args = new string[] { "creatensp", "--save-adf", "--meta", metaFile, "--type", "Application", "--program", "dir1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNsp != null);
                Assert.IsTrue(option.CreateNsp.NspContentInfos.Count == 1);
                Assert.IsTrue(option.CreateNsp.NspContentInfos[0].MetaFilePath == metaFile);
                Assert.IsTrue(option.CreateNsp.NspContentInfos[0].KeyGeneration.Value == 0);

                args = new string[] { "creatensp", "--save-adf", "--meta", testDir, "--type", "Application", "--program", "dir1" };
                Utils.CheckReturnException(new InvalidOptionException("file path should be specified for --meta."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }

            // --type
            {
                Option option = new Option();
                string[] args = new string[] { "creatensp", "--save-adf", "--meta", metaFile, "--type", "Application", "--program", "dir1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNsp != null);
                Assert.IsTrue(option.CreateNsp.NspContentInfos.Count == 1);
                Assert.IsTrue(option.CreateNsp.NspContentInfos[0].MetaType == "Application");

                args = new string[] { "creatensp", "--save-adf", "--meta", metaFile, "--type", "SystemProgram", "--program", "dir1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNsp != null);
                Assert.IsTrue(option.CreateNsp.NspContentInfos.Count == 1);
                Assert.IsTrue(option.CreateNsp.NspContentInfos[0].MetaType == "SystemProgram");

                args = new string[] { "creatensp", "--save-adf", "--meta", metaFile, "--type", "hoge", "--program", "dir1" };
                Utils.CheckReturnException(new InvalidOptionException("invalid option --type hoge."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }

            // --program
            {
                Option option = new Option();
                string[] args = new string[] { "creatensp", "--save-adf", "--meta", metaFile, "--type", "Application", "--program", "dir1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNsp != null);
                Assert.IsTrue(option.CreateNsp.NspContentInfos.Count == 1);
                Assert.IsTrue(option.CreateNsp.NspContentInfos[0].GetResource("Program").PathList.Count == 1);
                Assert.IsTrue(option.CreateNsp.NspContentInfos[0].GetResource("Program").PathList[0].first == "dir1");

                args = new string[] { "creatensp", "--save-adf", "--meta", metaFile, "--type", "Application", "--program", "dir1", "dir2" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNsp != null);
                Assert.IsTrue(option.CreateNsp.NspContentInfos.Count == 1);
                Assert.IsTrue(option.CreateNsp.NspContentInfos[0].GetResource("Program").PathList.Count == 2);
                Assert.IsTrue(option.CreateNsp.NspContentInfos[0].GetResource("Program").PathList[0].first == "dir1");
                Assert.IsTrue(option.CreateNsp.NspContentInfos[0].GetResource("Program").PathList[1].first == "dir2");
            }

            // --control
            {
                Option option = new Option();
                string[] args = new string[] { "creatensp", "--save-adf", "--meta", metaFile, "--type", "Application", "--control", "dir1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNsp != null);
                Assert.IsTrue(option.CreateNsp.NspContentInfos.Count == 1);
                Assert.IsTrue(option.CreateNsp.NspContentInfos[0].GetResource("Control").PathList.Count == 1);
                Assert.IsTrue(option.CreateNsp.NspContentInfos[0].GetResource("Control").PathList[0].first == "dir1");
            }

            // --html-document
            {
                Option option = new Option();
                string[] args = new string[] { "creatensp", "--save-adf", "--meta", metaFile, "--type", "Application", "--html-document", "dir1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNsp != null);
                Assert.IsTrue(option.CreateNsp.NspContentInfos.Count == 1);
                Assert.IsTrue(option.CreateNsp.NspContentInfos[0].GetResource("HtmlDocument").PathList.Count == 1);
                Assert.IsTrue(option.CreateNsp.NspContentInfos[0].GetResource("HtmlDocument").PathList[0].first == "dir1");
            }

            // --accessible-urls
            {
                Option option = new Option();
                string[] args = new string[] { "creatensp", "--save-adf", "--meta", metaFile, "--type", "Application", "--accessible-urls", "dir1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNsp != null);
                Assert.IsTrue(option.CreateNsp.NspContentInfos.Count == 1);
                Assert.IsTrue(option.CreateNsp.NspContentInfos[0].GetResource("HtmlDocument").PathList.Count == 1);
                Assert.IsTrue(option.CreateNsp.NspContentInfos[0].GetResource("HtmlDocument").PathList[0].first == "dir1");
            }

            // --accessible-urls & --html-document
            {
                Option option = new Option();
                string[] args = new string[] { "creatensp", "--save-adf", "--meta", metaFile, "--type", "Application", "--accessible-urls", "dir1", "--html-document", "dir2" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNsp != null);
                Assert.IsTrue(option.CreateNsp.NspContentInfos.Count == 1);
                Assert.IsTrue(option.CreateNsp.NspContentInfos[0].GetResource("HtmlDocument").PathList.Count == 2);
                Assert.IsTrue(option.CreateNsp.NspContentInfos[0].GetResource("HtmlDocument").PathList[0].first == "dir1");
                Assert.IsTrue(option.CreateNsp.NspContentInfos[0].GetResource("HtmlDocument").PathList[1].first == "dir2");
            }

            // --legal-information
            {
                Option option = new Option();
                string[] args = new string[] { "creatensp", "--save-adf", "--meta", metaFile, "--type", "Application", "--legal-information", "dir1" };
                Utils.CheckReturnException(new InvalidOptionException(".zip file should be specified for legal information."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }

            // --nro
            {
                Option option = new Option();
                string[] args = new string[] { "creatensp", "--save-adf", "--meta", metaFile, "--type", "Application", "--nro" , "disableNroDire" };
                Utils.CheckReturnException(new InvalidOptionException("directory path should be specified for --nro."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                var nroPath = MakeTmpNroDir();
                Option option = new Option();
                string[] args = new string[] { "creatensp", "--save-adf", "--meta", metaFile, "--type", "Application", "--nro", nroPath };
                option.Parse(args);
                Assert.IsTrue(option.CreateNsp != null);
                Directory.Delete(nroPath);
            }

            {
                var zipPath = MakeTmpZipFile();
                Option option = new Option();
                string[] args = new string[] { "creatensp", "--save-adf", "--meta", metaFile, "--type", "Application", "--legal-information", zipPath };
                option.Parse(args);
                Assert.IsTrue(option.CreateNsp != null);
                Assert.IsTrue(option.CreateNsp.NspContentInfos.Count == 1);
                Assert.IsTrue(option.CreateNsp.NspContentInfos[0].GetResource("LegalInformation").PathList.Count == 1);
                Assert.IsTrue(Directory.Exists(option.CreateNsp.NspContentInfos[0].GetResource("LegalInformation").PathList[0].first));
                Assert.IsTrue(File.Exists(Path.Combine(option.CreateNsp.NspContentInfos[0].GetResource("LegalInformation").PathList[0].first, TestZipFileName)));

                File.Delete(zipPath);
            }

            // multiple contents
            {
            }

            // input adf
            {
                Option option = new Option();
                string[] args = new string[] { "creatensp", "file1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNsp != null);
                Assert.IsTrue(option.CreateNsp.InputAdfFile == "file1");
                Assert.IsTrue(option.CreateNsp.IsSaveAdf);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "creatensp", "file1", "file2" };
                Utils.CheckReturnException(new InvalidOptionException("too many arguments for creatensp subcommand."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }

            // --save-adf
            {
                Option option = new Option();
                string[] args = new string[] { "creatensp", "--save-adf", "--meta", metaFile, "--type", "Application", "--program", "dir1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNsp != null);
                Assert.IsTrue(option.CreateNsp.IsSaveAdf);
            }

            // -o
            {
                Option option = new Option();
                string[] args = new string[] { "creatensp", "-o", "c:/Windows/Temp/output", "file1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNsp != null);
                Assert.IsTrue(option.CreateNsp.OutputFile == "c:/Windows/Temp/output");
            }
            {
                Option option = new Option();
                string[] args = new string[] { "creatensp", "-o", "output", "file1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNsp != null);
                Assert.IsTrue(option.CreateNsp.OutputFile == "./output");
                Assert.IsTrue(option.CreateNsp.NoCheckDirWarning == false);
            }

            // fdf が meta に書いてあるパターン
            {
                Option option = new Option();
                string[] args = new string[] { "creatensp", "--save-adf", "--meta", metaFileForFdf, "--type", "Application", "--program", "dir1" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNsp != null);
                Assert.IsTrue(option.CreateNsp.NspContentInfos[0].FilterDescriptionFilePath == "dummy");
            }

            // --filter が指定されていて且つ fdf が meta に書いてあるパターン
            {
                Option option = new Option();
                string[] args = new string[] { "creatensp", "--save-adf", "--meta", metaFileForFdf, "--type", "Application", "--program", "dir1", "--filter", "dummy" };
                Utils.CheckReturnException(new ArgumentException("It is impossible to specify fdf file from both nmeta and command line option."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }

            // --ignore-unpublishable-error
            {
                Option option = new Option();
                string[] args = new string[] { "creatensp", "--ignore-unpublishable-error", "arg1", "arg2" };
                Utils.CheckReturnException(new InvalidOptionException("too many arguments for --ignore-unpublishable-error option."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }

            // --no-check-dir-warning
            {
                Option option = new Option();
                string[] args = new string[] { "creatensp", "-o", "output", "file1", "--no-check-dir-warning" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNsp != null);
                Assert.IsTrue(option.CreateNsp.OutputFile == "./output");
                Assert.IsTrue(option.CreateNsp.NoCheckDirWarning == true);
            }

            DeleteDirectoryIfExisted(testDir);
        }

        [TestMethod]
        public void TestCreateNspMetaOption()
        {
            string testDir = Path.GetTempPath().Replace("\\", "/") + "/TestAuthoringTool/";
            string metaFile = testDir + "test.nmeta";
            string iconFile = testDir + "test.nmeta";
            Directory.CreateDirectory(testDir);
            using (FileStream stream = File.Create(metaFile))
            {
            }

            // -o
            {
                Option option = new Option();
                string[] args = new string[] { "createnspmeta", "--type", "Application", "--meta", metaFile };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspMeta != null);
                Assert.IsTrue(option.CreateNspMeta.OutputDirectory == "./output.nspd");
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnspmeta", "-o", testDir, "--type", "Application", "--meta", metaFile };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspMeta != null);
                Assert.IsTrue(option.CreateNspMeta.OutputDirectory == testDir);
            }

            DeleteDirectoryIfExisted(testDir);
        }

        [TestMethod]
        public void TestCreateNspdOption()
        {
            string testDir = Path.GetTempPath().Replace("\\", "/") + "/TestAuthoringTool/";
            string testNspd = testDir + "test.nspd";
            string metaFile = testDir + "test.nmeta";
            string iconFile = testDir + "test.nmeta";
            Directory.CreateDirectory(testDir);
            using (FileStream stream = File.Create(metaFile))
            using (var sw = new StreamWriter(stream))
            {
                sw.WriteLine("<NintendoSdkMeta><KeyGeneration>0</KeyGeneration><Application></Application></NintendoSdkMeta>");
            }

            string metaFileForFdf = testDir + "testFdf.nmeta";
            using (FileStream stream = File.Create(metaFileForFdf))
            using (var sw = new StreamWriter(stream))
            {
                sw.WriteLine("<NintendoSdkMeta><KeyGeneration>0</KeyGeneration><Application><FilterDescriptionFilePath>dummy</FilterDescriptionFilePath></Application></NintendoSdkMeta>");
            }

            // -o
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "--type", "Application", "--meta", metaFile, "--code", testDir };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspd != null);
                Assert.IsTrue(option.CreateNspd.OutputDirectory == "./output.nspd");
                Assert.IsTrue(option.CreateNspd.MetaType == "Application");
                Assert.IsTrue(option.CreateNspd.MetaFilePath == metaFile);
                Assert.IsTrue(option.CreateNspd.CodeDirectory == testDir);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "-o", testNspd, "--type", "Application", "--meta", metaFile, "--program", testDir };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspd != null);
                Assert.IsTrue(option.CreateNspd.OutputDirectory == testNspd);
                Assert.IsTrue(option.CreateNspd.MetaType == "Application");
                Assert.IsTrue(option.CreateNspd.MetaFilePath == metaFile);
                Assert.IsTrue(option.CreateNspd.CodeDirectory == testDir);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "-o", testDir, "--type", "Application", "--meta", metaFile, "--code", testDir };
                Utils.CheckReturnException(new InvalidOptionException("-o option should be used with file name which ends with \".nspd\""), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }

            // --type
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "-o", testNspd, "--type", "SystemProgram", "--meta", metaFile, "--code", testDir };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspd != null);
                Assert.IsTrue(option.CreateNspd.OutputDirectory == testNspd);
                Assert.IsTrue(option.CreateNspd.MetaType == "SystemProgram");
                Assert.IsTrue(option.CreateNspd.MetaFilePath == metaFile);
                Assert.IsTrue(option.CreateNspd.CodeDirectory == testDir);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "-o", testNspd, "--meta", metaFile, "--code", testDir };
                Utils.CheckReturnException(new InvalidOptionException("createnspd command needs --meta and --type options."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            string[] metaTypes = { "AddOnContent", "Patch", "SystemUpdate", "SystemData", "BootImagePackage", "BootImagePackageSafe" };
            foreach (var typeName in metaTypes)
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "-o", testNspd, "--type", typeName, "--meta", metaFile, "--code", testDir };
                Utils.CheckReturnException(new InvalidOptionException("createnspd command should be used with --type Application or --type SystemProgram"), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }

            // --meta
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "-o", testNspd, "--type", "Application", "--code", testDir };
                Utils.CheckReturnException(new InvalidOptionException("createnspd command needs --meta and --type options."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }

            // --program
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "-o", testNspd, "--type", "Application", "--meta", metaFile, "--program", testDir };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspd != null);
                Assert.IsTrue(option.CreateNspd.OutputDirectory == testNspd);
                Assert.IsTrue(option.CreateNspd.MetaType == "Application");
                Assert.IsTrue(option.CreateNspd.MetaFilePath == metaFile);
                Assert.IsTrue(option.CreateNspd.CodeDirectory == testDir);
                Assert.IsTrue(option.CreateNspd.DataDirectory == null);
                Assert.IsTrue(option.CreateNspd.LogoDirectory == null);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "-o", testNspd, "--type", "Application", "--meta", metaFile, "--program", testDir, testDir };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspd != null);
                Assert.IsTrue(option.CreateNspd.OutputDirectory == testNspd);
                Assert.IsTrue(option.CreateNspd.MetaType == "Application");
                Assert.IsTrue(option.CreateNspd.MetaFilePath == metaFile);
                Assert.IsTrue(option.CreateNspd.CodeDirectory == testDir);
                Assert.IsTrue(option.CreateNspd.DataDirectory == testDir);
                Assert.IsTrue(option.CreateNspd.LogoDirectory == null);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "-o", testNspd, "--type", "Application", "--meta", metaFile, "--program", testDir, testDir, testDir };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspd != null);
                Assert.IsTrue(option.CreateNspd.OutputDirectory == testNspd);
                Assert.IsTrue(option.CreateNspd.MetaType == "Application");
                Assert.IsTrue(option.CreateNspd.MetaFilePath == metaFile);
                Assert.IsTrue(option.CreateNspd.CodeDirectory == testDir);
                Assert.IsTrue(option.CreateNspd.DataDirectory == testDir);
                Assert.IsTrue(option.CreateNspd.LogoDirectory == testDir);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "-o", testNspd, "--type", "Application", "--meta", metaFile, "--program", testDir, testDir, testDir, "--code", testDir };
                Utils.CheckReturnException(new InvalidOptionException("--program option cannot be used with --code and --data options."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "-o", testNspd, "--type", "Application", "--meta", metaFile, "--program", testDir, testDir, testDir, "--data", testDir };
                Utils.CheckReturnException(new InvalidOptionException("--program option cannot be used with --code and --data options."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "-o", testNspd, "--type", "Application", "--meta", metaFile, "--program", testDir, testDir, testDir, "--logo", testDir };
                Utils.CheckReturnException(new InvalidOptionException("--program option cannot be used with --code, --data and --logo options."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "-o", testNspd, "--type", "Application", "--meta", metaFile };
                Utils.CheckReturnException(new InvalidOptionException("createnspd command needs code directory. Please specify --program or --code option."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }

            // --code
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "-o", testNspd, "--type", "Application", "--meta", metaFile, "--code", testDir };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspd != null);
                Assert.IsTrue(option.CreateNspd.OutputDirectory == testNspd);
                Assert.IsTrue(option.CreateNspd.MetaType == "Application");
                Assert.IsTrue(option.CreateNspd.MetaFilePath == metaFile);
                Assert.IsTrue(option.CreateNspd.CodeDirectory == testDir);
                Assert.IsTrue(option.CreateNspd.DataDirectory == null);
                Assert.IsTrue(option.CreateNspd.LogoDirectory == null);
            }
            // --code & --forcibly-copy-*
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "-o", testNspd, "--type", "Application", "--meta", metaFile, "--code", testDir, "--forcibly-copy-code --forcibly-copy-data --forcibly-copy-control --forcibly-copy-control --forcibly-copy-html-document --forcibly-copy-accessible-urls --forcibly-copy-legal-information" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspd != null);
                Assert.IsTrue(option.CreateNspd.OutputDirectory == testNspd);
                Assert.IsTrue(option.CreateNspd.MetaType == "Application");
                Assert.IsTrue(option.CreateNspd.MetaFilePath == metaFile);
                Assert.IsTrue(option.CreateNspd.CodeDirectory == testDir);
                Assert.IsTrue(option.CreateNspd.DataDirectory == null);
                Assert.IsTrue(option.CreateNspd.LogoDirectory == null);
            }
            // --data
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "-o", testNspd, "--type", "Application", "--meta", metaFile, "--code", testDir, "--data", testDir };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspd != null);
                Assert.IsTrue(option.CreateNspd.OutputDirectory == testNspd);
                Assert.IsTrue(option.CreateNspd.MetaType == "Application");
                Assert.IsTrue(option.CreateNspd.MetaFilePath == metaFile);
                Assert.IsTrue(option.CreateNspd.CodeDirectory == testDir);
                Assert.IsTrue(option.CreateNspd.DataDirectory == testDir);
                Assert.IsTrue(option.CreateNspd.LogoDirectory == null);
                Assert.IsTrue(option.CreateNspd.InputFdfPath == null);
            }
            // fdf が meta に書いてあるパターン
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "-o", testNspd, "--type", "Application", "--meta", metaFileForFdf, "--code", testDir, "--data", testDir };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspd != null);
                Assert.IsTrue(option.CreateNspd.OutputDirectory == testNspd);
                Assert.IsTrue(option.CreateNspd.MetaType == "Application");
                Assert.IsTrue(option.CreateNspd.MetaFilePath == metaFileForFdf);
                Assert.IsTrue(option.CreateNspd.CodeDirectory == testDir);
                Assert.IsTrue(option.CreateNspd.DataDirectory == testDir);
                Assert.IsTrue(option.CreateNspd.LogoDirectory == null);
                Assert.IsTrue(option.CreateNspd.InputFdfPath == "dummy");
            }
            // --data & --filter
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "-o", testNspd, "--type", "Application", "--meta", metaFile, "--code", testDir, "--data", testDir, "--filter", "dummy" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspd != null);
                Assert.IsTrue(option.CreateNspd.OutputDirectory == testNspd);
                Assert.IsTrue(option.CreateNspd.MetaType == "Application");
                Assert.IsTrue(option.CreateNspd.MetaFilePath == metaFile);
                Assert.IsTrue(option.CreateNspd.CodeDirectory == testDir);
                Assert.IsTrue(option.CreateNspd.DataDirectory == testDir);
                Assert.IsTrue(option.CreateNspd.LogoDirectory == null);
                Assert.IsTrue(option.CreateNspd.InputFdfPath == "dummy");
            }
            // --data & --filter & --forcibly-copy-with-filtering
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "-o", testNspd, "--type", "Application", "--meta", metaFile, "--code", testDir, "--data", testDir, "--filter", "dummy", "--forcibly-copy-with-filtering" };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspd != null);
                Assert.IsTrue(option.CreateNspd.OutputDirectory == testNspd);
                Assert.IsTrue(option.CreateNspd.MetaType == "Application");
                Assert.IsTrue(option.CreateNspd.MetaFilePath == metaFile);
                Assert.IsTrue(option.CreateNspd.CodeDirectory == testDir);
                Assert.IsTrue(option.CreateNspd.DataDirectory == testDir);
                Assert.IsTrue(option.CreateNspd.LogoDirectory == null);
                Assert.IsTrue(option.CreateNspd.InputFdfPath == "dummy");
            }
            // --filter が指定されていて且つ fdf が meta に書いてあるパターン
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "-o", testNspd, "--type", "Application", "--meta", metaFileForFdf, "--code", testDir, "--data", testDir, "--filter", "dummy" };
                Utils.CheckReturnException(new ArgumentException("It is impossible to specify fdf file from both nmeta and command line option."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            // --logo
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "-o", testNspd, "--type", "Application", "--meta", metaFile, "--code", testDir, "--logo", testDir };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspd != null);
                Assert.IsTrue(option.CreateNspd.OutputDirectory == testNspd);
                Assert.IsTrue(option.CreateNspd.MetaType == "Application");
                Assert.IsTrue(option.CreateNspd.MetaFilePath == metaFile);
                Assert.IsTrue(option.CreateNspd.CodeDirectory == testDir);
                Assert.IsTrue(option.CreateNspd.DataDirectory == null);
                Assert.IsTrue(option.CreateNspd.LogoDirectory == testDir);
            }
            // --code & --data & --logo
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "-o", testNspd, "--type", "Application", "--meta", metaFile, "--code", testDir, "--data", testDir, "--logo", testDir };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspd != null);
                Assert.IsTrue(option.CreateNspd.OutputDirectory == testNspd);
                Assert.IsTrue(option.CreateNspd.MetaType == "Application");
                Assert.IsTrue(option.CreateNspd.MetaFilePath == metaFile);
                Assert.IsTrue(option.CreateNspd.CodeDirectory == testDir);
                Assert.IsTrue(option.CreateNspd.DataDirectory == testDir);
                Assert.IsTrue(option.CreateNspd.LogoDirectory == testDir);
            }

            // --control
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "--type", "Application", "--meta", metaFile, "--control", testDir, "--program", testDir };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspd != null);
                Assert.IsTrue(option.CreateNspd.OutputDirectory == "./output.nspd");
                Assert.IsTrue(option.CreateNspd.MetaType == "Application");
                Assert.IsTrue(option.CreateNspd.MetaFilePath == metaFile);
                Assert.IsTrue(option.CreateNspd.ControlDirectory == testDir);
            }

            // --html-document
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "--type", "Application", "--meta", metaFile, "--html-document", testDir, "--program", testDir };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspd != null);
                Assert.IsTrue(option.CreateNspd.OutputDirectory == "./output.nspd");
                Assert.IsTrue(option.CreateNspd.MetaType == "Application");
                Assert.IsTrue(option.CreateNspd.MetaFilePath == metaFile);
                Assert.IsTrue(option.CreateNspd.HtmlDocumentDirectory == testDir);
            }

            // --accessible-urls
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "--type", "Application", "--meta", metaFile, "--accessible-urls", testDir, "--program", testDir };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspd != null);
                Assert.IsTrue(option.CreateNspd.OutputDirectory == "./output.nspd");
                Assert.IsTrue(option.CreateNspd.MetaType == "Application");
                Assert.IsTrue(option.CreateNspd.MetaFilePath == metaFile);
                Assert.IsTrue(option.CreateNspd.AccessibleUrlsDirectory == testDir);
            }

            // --legal-information-dir
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "--type", "Application", "--meta", metaFile, "--legal-information-dir", testDir, "--program", testDir };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspd != null);
                Assert.IsTrue(option.CreateNspd.OutputDirectory == "./output.nspd");
                Assert.IsTrue(option.CreateNspd.MetaType == "Application");
                Assert.IsTrue(option.CreateNspd.MetaFilePath == metaFile);
                Assert.IsTrue(option.CreateNspd.LegalInformationDirectory == testDir);
            }

            // --nro
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "--type", "Application", "--meta", metaFile, "--nro", testDir, "--program", testDir };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspd != null);
                Assert.IsTrue(option.CreateNspd.OutputDirectory == "./output.nspd");
                Assert.IsTrue(option.CreateNspd.MetaType == "Application");
                Assert.IsTrue(option.CreateNspd.MetaFilePath == metaFile);
            }

            // --icon
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "--type", "Application", "--meta", metaFile, "--icon", "Japanese", iconFile, "--program", testDir };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspd != null);
                Assert.IsTrue(option.CreateNspd.OutputDirectory == "./output.nspd");
                Assert.IsTrue(option.CreateNspd.MetaType == "Application");
                Assert.IsTrue(option.CreateNspd.MetaFilePath == metaFile);
                Assert.IsTrue(option.CreateNspd.IconFileList.Count == 1);
                Assert.IsTrue(option.CreateNspd.IconFileList[0].Item1 == "Japanese");
                Assert.IsTrue(option.CreateNspd.IconFileList[0].Item2 == iconFile);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "--type", "Application", "--meta", metaFile, "--icon", "Japanese", iconFile, "AmericanEnlish", iconFile, "--program", testDir };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspd != null);
                Assert.IsTrue(option.CreateNspd.OutputDirectory == "./output.nspd");
                Assert.IsTrue(option.CreateNspd.MetaType == "Application");
                Assert.IsTrue(option.CreateNspd.MetaFilePath == metaFile);
                Assert.IsTrue(option.CreateNspd.IconFileList.Count == 2);
                Assert.IsTrue(option.CreateNspd.IconFileList[0].Item1 == "Japanese");
                Assert.IsTrue(option.CreateNspd.IconFileList[0].Item2 == iconFile);
                Assert.IsTrue(option.CreateNspd.IconFileList[1].Item1 == "AmericanEnlish");
                Assert.IsTrue(option.CreateNspd.IconFileList[1].Item2 == iconFile);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "--type", "SystemProgram", "--meta", metaFile, "--icon", "Japanese", "--program", testDir };
                Utils.CheckReturnException(new InvalidOptionException("--icon <language> <iconPath>..."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "--type", "SystemProgram", "--meta", metaFile, "--icon", "Japanese", iconFile, "--program", testDir };
                Utils.CheckReturnException(new InvalidOptionException("--icon option should be used with --type Application."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }

            // --nx-icon
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "--type", "Application", "--meta", metaFile, "--nx-icon", "Japanese", iconFile, "--program", testDir };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspd != null);
                Assert.IsTrue(option.CreateNspd.OutputDirectory == "./output.nspd");
                Assert.IsTrue(option.CreateNspd.MetaType == "Application");
                Assert.IsTrue(option.CreateNspd.MetaFilePath == metaFile);
                Assert.IsTrue(option.CreateNspd.NxIconFileList.Count == 1);
                Assert.IsTrue(option.CreateNspd.NxIconFileList[0].Item1 == "Japanese");
                Assert.IsTrue(option.CreateNspd.NxIconFileList[0].Item2 == iconFile);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "--type", "Application", "--meta", metaFile, "--nx-icon", "Japanese", iconFile, "AmericanEnlish", iconFile, "--program", testDir };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspd != null);
                Assert.IsTrue(option.CreateNspd.OutputDirectory == "./output.nspd");
                Assert.IsTrue(option.CreateNspd.MetaType == "Application");
                Assert.IsTrue(option.CreateNspd.MetaFilePath == metaFile);
                Assert.IsTrue(option.CreateNspd.NxIconFileList.Count == 2);
                Assert.IsTrue(option.CreateNspd.NxIconFileList[0].Item1 == "Japanese");
                Assert.IsTrue(option.CreateNspd.NxIconFileList[0].Item2 == iconFile);
                Assert.IsTrue(option.CreateNspd.NxIconFileList[1].Item1 == "AmericanEnlish");
                Assert.IsTrue(option.CreateNspd.NxIconFileList[1].Item2 == iconFile);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "--type", "SystemProgram", "--meta", metaFile, "--nx-icon", "Japanese", "--program", testDir };
                Utils.CheckReturnException(new InvalidOptionException("--nx-icon <language> <iconPath>..."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "--type", "SystemProgram", "--meta", metaFile, "--nx-icon", "Japanese", iconFile, "--program", testDir };
                Utils.CheckReturnException(new InvalidOptionException("--nx-icon option should be used with --type Application."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }

            // --nx-icon-max-size
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "--type", "Application", "--meta", metaFile, "--nx-icon", "Japanese", iconFile, "--nx-icon-max-size", "1000", "--program", testDir };
                option.Parse(args);
                Assert.IsTrue(option.CreateNspd != null);
                Assert.IsTrue(option.CreateNspd.OutputDirectory == "./output.nspd");
                Assert.IsTrue(option.CreateNspd.MetaType == "Application");
                Assert.IsTrue(option.CreateNspd.MetaFilePath == metaFile);
                Assert.IsTrue(option.CreateNspd.NxIconFileList.Count == 1);
                Assert.IsTrue(option.CreateNspd.NxIconFileList[0].Item1 == "Japanese");
                Assert.IsTrue(option.CreateNspd.NxIconFileList[0].Item2 == iconFile);
                Assert.IsTrue(option.CreateNspd.NxIconMaxSize == 1000);
            }

            // --ignore-unpublishable-error
            {
                Option option = new Option();
                string[] args = new string[] { "createnspd", "--ignore-unpublishable-error", "arg1", "arg2" };
                Utils.CheckReturnException(new InvalidOptionException("too many arguments for --ignore-unpublishable-error option."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }

            DeleteDirectoryIfExisted(testDir);
        }

        [TestMethod]
        public void TestMergeNspOption()
        {
            string testDir = Path.GetTempPath().Replace("\\", "/") + "/TestAuthoringTool/";
            string nsp1 = testDir + "1.nsp";
            string nsp2 = testDir + "2.nsp";
            string nsp3 = testDir + "3.nsp";
            string nsp4 = testDir + "4.nsp";
            Directory.CreateDirectory(testDir);
            using (FileStream stream1 = File.Create(nsp1))
            using (FileStream stream2 = File.Create(nsp2))
            using (FileStream stream3 = File.Create(nsp3))
            using (FileStream stream4 = File.Create(nsp4))
            {
            }

            {
                Option option = new Option();
                string[] args = new string[] { "mergensp", nsp1 };
                Utils.CheckReturnException(new InvalidOptionException("too few arguments for mergensp subcommand."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "mergensp", nsp1, nsp2};
                option.Parse(args);
                Assert.IsTrue(option.MergeNsp != null);
                Assert.IsTrue(option.MergeNsp.InputFiles.Count == 2);
                Assert.IsTrue(option.MergeNsp.InputFiles[0] == nsp1);
                Assert.IsTrue(option.MergeNsp.InputFiles[1] == nsp2);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "mergensp", nsp1, nsp2, nsp3, nsp4 };
                option.Parse(args);
                Assert.IsTrue(option.MergeNsp != null);
                Assert.IsTrue(option.MergeNsp.InputFiles.Count == 4);
                Assert.IsTrue(option.MergeNsp.InputFiles[0] == nsp1);
                Assert.IsTrue(option.MergeNsp.InputFiles[1] == nsp2);
                Assert.IsTrue(option.MergeNsp.InputFiles[2] == nsp3);
                Assert.IsTrue(option.MergeNsp.InputFiles[3] == nsp4);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "mergensp", nsp1, nsp2, testDir, nsp4 };
                Utils.CheckReturnException(new InvalidOptionException("file path should be specified for arg[2]."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }

            DeleteDirectoryIfExisted(testDir);
        }

        [TestMethod]
        public void TestProdEncryptionOption()
        {
            using (var fs = File.Create("test.nsp")) {}
            using (var fs = File.Create("upp.nsp")) {}
            using (var fs = File.Create("patch.nsp")) {}
            using (var fs = File.Create("aoc.nsp")) {}
            using (var fs = File.Create("mac.nsp")) {}
            {
                Option option = new Option();
                string[] args = new string[] { "prodencryption" };
                Utils.CheckReturnException(new InvalidOptionException("input nsp file must be specified for prodencryption subcommand."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "prodencryption", "test.nsp", "patch.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("too many arguments for prodencryption subcommand."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "prodencryption", "test.nsp" };
                option.Parse(args);
                Assert.IsTrue(option.ProdEncryption != null);
                Assert.IsTrue(option.ProdEncryption.OutputDirectory == ".");
                Assert.IsTrue(option.ProdEncryption.InputNspFile == "./test.nsp");
            }
            {
                Option option = new Option();
                string[] args = new string[] { "prodencryption", "-o", "output", "test.nsp" };
                option.Parse(args);
                Assert.IsTrue(option.ProdEncryption != null);
                Assert.IsTrue(option.ProdEncryption.CreateNspu);
                Assert.IsTrue(option.ProdEncryption.InputNspFile == "./test.nsp");
                Assert.IsTrue(option.ProdEncryption.OutputDirectory == "./output");
            }
            {
                Option option = new Option();
                string[] args = new string[] { "prodencryption", "-o", "output", "--no-nspu", "test.nsp" };
                option.Parse(args);
                Assert.IsTrue(option.ProdEncryption != null);
                Assert.IsFalse(option.ProdEncryption.CreateNspu);
                Assert.IsTrue(option.ProdEncryption.InputNspFile == "./test.nsp");
                Assert.IsTrue(option.ProdEncryption.OutputDirectory == "./output");
            }
            {
                Option option = new Option();
                string[] args = new string[] { "prodencryption", "test.nsp", "test.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("too many arguments for prodencryption subcommand."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "prodencryption", "--gamecard", "--upp", "upp.nsp", "test.nsp" };
                option.Parse(args);
                Assert.IsTrue(option.ProdEncryption != null);
                Assert.IsTrue(option.ProdEncryption.OutputDirectory == ".");
                Assert.IsTrue(option.ProdEncryption.InputNspFile == "./test.nsp");
                Assert.IsTrue(option.ProdEncryption.CreateXci);
                Assert.IsTrue(option.ProdEncryption.CreateXcie);
                Assert.IsTrue(option.ProdEncryption.LaunchFlags == 0);
                Assert.IsTrue(option.ProdEncryption.InputUppNspFile == "./upp.nsp");
            }
            {
                Option option = new Option();
                string[] args = new string[] { "prodencryption", "--gamecard", "--no-xcie", "--auto-boot", "--history-erase", "--repair-time-reviser-tool", "--patch", "patch.nsp", "--aoc", "aoc.nsp", "test.nsp" };
                option.Parse(args);
                Assert.IsTrue(option.ProdEncryption != null);
                Assert.IsTrue(option.ProdEncryption.OutputDirectory == ".");
                Assert.IsTrue(option.ProdEncryption.InputNspFile == "./test.nsp");
                Assert.IsTrue(option.ProdEncryption.CreateXci);
                Assert.IsFalse(option.ProdEncryption.CreateXcie);
                Assert.IsTrue(option.ProdEncryption.LaunchFlags == 7);
                Assert.IsTrue(option.ProdEncryption.InputPatchNspFile == "./patch.nsp");
                Assert.IsTrue(option.ProdEncryption.InputAocNspFile == "./aoc.nsp");
            }
            {
                Option option = new Option();
                string[] args = new string[] { "prodencryption", "-o", "output", "--no-xcie", "test.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("--no-padding, --no-xcie, --for-repair, --auto-boot, --history-erase, --repair-time-reviser-tool, --patch and --aoc option requires to be used with --gamecard option."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "prodencryption", "-o", "output", "--no-padding", "test.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("--no-padding, --no-xcie, --for-repair, --auto-boot, --history-erase, --repair-time-reviser-tool, --patch and --aoc option requires to be used with --gamecard option."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "prodencryption", "-o", "output", "--gamecard", "--no-nspu", "test.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("--no-nspu option requires to be used without --gamecard option."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "prodencryption", "-o", "output", "--gamecard", "--upp", "upp.nsp", "--auto-boot", "--keep-generation", "test.nsp" };
                option.Parse(args);
                Assert.IsTrue(option.ProdEncryption.LaunchFlags == 1);
                Assert.IsTrue(option.ProdEncryption.InputUppNspFile == "./upp.nsp");
                Assert.IsTrue(option.ProdEncryption.KeepGeneration);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "prodencryption", "-o", "output", "--gamecard", "--auto-boot", "--keep-generation", "test.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("--keep-generation option requires to be used with --upp option."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "prodencryption", "-o", "output", "--multiapplicationgamecard", "mac.nsp", "--auto-boot", "test.nsp", "patch.nsp", "aoc.nsp" };
                option.Parse(args);
                Assert.IsTrue(option.ProdEncryption != null);
                Assert.IsTrue(option.ProdEncryption.OutputDirectory == "./output");
                Assert.AreEqual(option.ProdEncryption.InputNspFiles.Count, 3);
                Assert.IsTrue(option.ProdEncryption.InputNspFile == "./test.nsp");
                Assert.IsTrue(option.ProdEncryption.InputNspFiles[1] == "./patch.nsp");
                Assert.IsTrue(option.ProdEncryption.InputNspFiles[2] == "./aoc.nsp");
                Assert.IsTrue(option.ProdEncryption.InputMultiApplicationCardIndicatorNspFile == "./mac.nsp");
                Assert.IsTrue(option.ProdEncryption.CreateXci);
                Assert.IsTrue(option.ProdEncryption.CreateXcie);
                Assert.IsTrue(option.ProdEncryption.LaunchFlags == 1);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "prodencryption", "-o", "output", "--gamecard", "--multiapplicationgamecard", "mac.nsp", "test.nsp", "patch.nsp", "aoc.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("--multiapplicationgamecard option requires to be used without --gamecard option."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "prodencryption", "-o", "output", "--multiapplicationgamecard", "mac.nsp", "--patch", "patch.nsp", "test.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("--patch and --aoc option requires to be used with --gamecard option."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }
            File.Delete("test.nsp");
            File.Delete("upp.nsp");
            File.Delete("patch.nsp");
            File.Delete("aoc.nsp");
            File.Delete("mac.nsp");
        }

        [TestMethod]
        public void TestProdEncryptionPatchOption()
        {
            using (var fs = File.Create("test.nsp")) {}
            using (var fs = File.Create("test2.nsp")) {}
            using (var fs = File.Create("upp.nsp")) {}
            using (var fs = File.Create("upp2.nsp")) {}
            using (var fs = File.Create("patch.nsp")) {}
            {
                Option option = new Option();
                string[] args = new string[] { "prodencryption-patch", "--upp", "upp.nsp", "--upp-for-desired-safe-keygen", "upp2.nsp", "--original", "test.nsp", "--original-prod", "test2.nsp", "--use-safe-keygen-forcibly", "patch.nsp" };
                option.Parse(args);
                Assert.IsTrue(option.ProdEncryptionPatch != null);
                Assert.IsTrue(option.ProdEncryptionPatch.OutputDirectory == ".");
                Assert.IsTrue(option.ProdEncryptionPatch.InputNspFile == "./patch.nsp");
                Assert.IsTrue(option.ProdEncryptionPatch.InputUppNspFile == "./upp.nsp");
                Assert.IsTrue(option.ProdEncryptionPatch.InputOriginalNspFile == "./test.nsp");
                Assert.IsTrue(option.ProdEncryptionPatch.InputOriginalProdNspFile == "./test2.nsp");
                Assert.IsTrue(option.ProdEncryptionPatch.InputUppForDesiredSafeKeyGenNspFile == "./upp2.nsp");
                Assert.IsTrue(option.ProdEncryptionPatch.SizeThresholdForKeyGenerationUpdate == (long)128 * 1024 * 1024 * 1024);
            }
            File.Delete("test.nsp");
            File.Delete("test2.nsp");
            File.Delete("upp.nsp");
            File.Delete("upp2.nsp");
            File.Delete("patch.nsp");
        }

        [TestMethod]
        public void TestMakePatchOption()
        {
            using (var fs = File.Create("file1.txt")) { }
            using (var fs = File.Create("file1.nca")) { }
            using (var fs = File.Create("file2.nca")) { }
            using (var fs = File.Create("file1.nsp")) { }
            using (var fs = File.Create("file2.nsp")) { }
            using (var fs = File.Create("file3.nsp")) { }
            Directory.CreateDirectory("dir1");

            // 不明なオプションの指定
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch" };
                Utils.CheckReturnException(new InvalidOptionException("original archive file should be specified."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "--unknown" };
                Utils.CheckReturnException(new InvalidOptionException("original archive file should be specified."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }

            // -h, --help
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "-h" };
                option.Parse(args);
                Assert.IsTrue(option.IsShowUsage);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "--help" };
                option.Parse(args);
                Assert.IsTrue(option.IsShowUsage);
            }

            // input nca
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "file1" };
                Utils.CheckReturnException(new InvalidOptionException("original archive file should be specified."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "--current", "file1.nca" };
                Utils.CheckReturnException(new InvalidOptionException("original archive file should be specified."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "--original", "file1.nca" };
                Utils.CheckReturnException(new InvalidOptionException("current archive file should be specified."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "--original", "file1.nca", "--current", "file2.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("current archive file must be .nca file."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "--original", "file1.nsp", "--current", "file2.nca" };
                Utils.CheckReturnException(new InvalidOptionException("current archive file must be .nsp file."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "--original", "file1.txt", "--current", "file2.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("original archive file must be .nca or .nsp file."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "--original", "file1.nsp", "--current", "file2.nsp" };
                option.Parse(args);
                Assert.IsTrue(option.MakePatch != null);
                Assert.IsTrue(option.MakePatch.OutputFile == "./output.nsp");
                Assert.IsTrue(option.MakePatch.OriginalFilePath == "./file1.nsp");
                Assert.IsTrue(option.MakePatch.CurrentFilePath == "./file2.nsp");
            }
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "--original", "file1.nsp", "--current", "file2.nsp", "--previous", "file3.nsp" };
                option.Parse(args);
                Assert.IsTrue(option.MakePatch != null);
                Assert.IsTrue(option.MakePatch.OutputFile == "./output.nsp");
                Assert.IsTrue(option.MakePatch.OriginalFilePath == "./file1.nsp");
                Assert.IsTrue(option.MakePatch.CurrentFilePath == "./file2.nsp");
                Assert.IsTrue(option.MakePatch.PreviousFilePath == "./file3.nsp");
            }

            // -o
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "-o", "c:/Windows/Temp/output.nca", "--original", "file1.nca", "--current", "file2.nca" };
                option.Parse(args);
                Assert.IsTrue(option.MakePatch != null);
                Assert.IsTrue(option.MakePatch.OutputFile == "c:/Windows/Temp/output.nca");
                Assert.IsTrue(option.MakePatch.OriginalFilePath == "./file1.nca");
                Assert.IsTrue(option.MakePatch.CurrentFilePath == "./file2.nca");
            }
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "-o", "output.nca", "--original", "file1.nsp", "--current", "file2.nsp" };
                option.Parse(args);
                Assert.IsTrue(option.MakePatch != null);
                Assert.IsTrue(option.MakePatch.OutputFile == "./output.nsp");
                Assert.IsTrue(option.MakePatch.OriginalFilePath == "./file1.nsp");
                Assert.IsTrue(option.MakePatch.CurrentFilePath == "./file2.nsp");
            }
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "-o", "foo.nsp", "--original", "file1.nsp", "--current", "file2.nsp", "--previous", "file3.nsp" };
                option.Parse(args);
                Assert.IsTrue(option.MakePatch != null);
                Assert.IsTrue(option.MakePatch.OutputFile == "./foo.nsp");
                Assert.IsTrue(option.MakePatch.OriginalFilePath == "./file1.nsp");
                Assert.IsTrue(option.MakePatch.CurrentFilePath == "./file2.nsp");
                Assert.IsTrue(option.MakePatch.PreviousFilePath == "./file3.nsp");
            }
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "-o", "file1.nsp", "--original", "file1.nsp", "--current", "file2.nsp", "--previous", "file3.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("Original and ouptut are same file"), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "-o", "file2.nsp", "--original", "file1.nsp", "--current", "file2.nsp", "--previous", "file3.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("Current and output are same file"), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "-o", "file3.nsp", "--original", "file1.nsp", "--current", "file2.nsp", "--previous", "file3.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("Previous and output are same file"), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }

#if false
            // -desc
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "-o", "c:/Windows/Temp/output", "--desc", "desc", "file1", "file2" };
                option.Parse(args);
                Assert.IsTrue(option.MakePatch != null);
                Assert.IsTrue(option.MakePatch.OutputFile == "c:/Windows/Temp/output");
                Assert.IsTrue(option.MakePatch.OriginalFilePath == "file1");
                Assert.IsTrue(option.MakePatch.CurrentFilePath == "file2");
                Assert.IsTrue(option.MakePatch.DescFilePath == "desc");
            }
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "-o", "output", "--desc", "desc", "file1", "file2" };
                option.Parse(args);
                Assert.IsTrue(option.MakePatch != null);
                Assert.IsTrue(option.MakePatch.OutputFile == "./output");
                Assert.IsTrue(option.MakePatch.OriginalFilePath == "file1");
                Assert.IsTrue(option.MakePatch.CurrentFilePath == "file2");
                Assert.IsTrue(option.MakePatch.DescFilePath == "desc");
            }
#endif
            // --minimum-matching-size
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "-o", "file4.nsp", "--original", "file1.nsp", "--current", "file2.nsp", "--previous", "file3.nsp", "--minimum-matching-size", "2" };
                option.Parse(args);
                Assert.IsTrue(option.MakePatch.MinimumMatchingSize == 2 * 1024);
                // NOTE: オプションのテストではないもののここでチェック
                Utils.CheckReturnException(new ArgumentException("match size should be large than or equal to 4KiB."), () =>
                {
                    IndirectStorageSource.SetMinimumMatchingSize(option.MakePatch.MinimumMatchingSize);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "-o", "file4.nsp", "--original", "file1.nsp", "--current", "file2.nsp", "--previous", "file3.nsp", "--minimum-matching-size", "30" };
                option.Parse(args);
                Assert.IsTrue(option.MakePatch.MinimumMatchingSize == 30 * 1024);
                // NOTE: オプションのテストではないもののここでチェック
                Utils.CheckReturnException(new ArgumentException("match size should be power of 2."), () =>
                {
                    IndirectStorageSource.SetMinimumMatchingSize(option.MakePatch.MinimumMatchingSize);
                    return true;
                });
            }

            // --defragment
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "-o", "file4.nsp", "--original", "file1.nsp", "--current", "file2.nsp", "--previous", "file3.nsp", "--defragment" };
                option.Parse(args);
                Assert.IsTrue(option.MakePatch.NeedsDefragment);
            }

            // --defragment-size
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "-o", "file4.nsp", "--original", "file1.nsp", "--current", "file2.nsp", "--previous", "file3.nsp", "--defragment-size", "16" };
                Utils.CheckReturnException(new InvalidOptionException("'--defragment' option should be specified."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "-o", "file4.nsp", "--original", "file1.nsp", "--current", "file2.nsp", "--previous", "file3.nsp", "--defragment", "--defragment-size", "2" };
                option.Parse(args);
                Assert.IsTrue(option.MakePatch.DefragmentSize == 2 * 1024);
                // NOTE: オプションのテストではないもののここでチェック
                Utils.CheckReturnException(new ArgumentException("defragment size should be large than or equal to 4KiB."), () =>
                {
                    RelocatedIndirectStorageSource.SetDefragmentSize(option.MakePatch.DefragmentSize);
                    return true;
                });
            }

            // --cache-directory
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "--original", "file1.nsp", "--current", "file2.nsp", "--cache-directory", "dir1" };
                option.Parse(args);
                Assert.IsTrue(option.MakePatch != null);
                Assert.IsTrue(option.MakePatch.OutputFile == "./output.nsp");
                Assert.IsTrue(option.MakePatch.OriginalFilePath == "./file1.nsp");
                Assert.IsTrue(option.MakePatch.CurrentFilePath == "./file2.nsp");
                Assert.IsTrue(option.MakePatch.CacheDirectory == "./dir1");
            }
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "--original", "file1.nsp", "--current", "file2.nsp", "--cache-directory", "no-exist-directory" };
                option.Parse(args);
                Assert.IsTrue(option.MakePatch != null);
                Assert.IsTrue(option.MakePatch.OutputFile == "./output.nsp");
                Assert.IsTrue(option.MakePatch.OriginalFilePath == "./file1.nsp");
                Assert.IsTrue(option.MakePatch.CurrentFilePath == "./file2.nsp");
                Assert.IsTrue(option.MakePatch.CacheDirectory == "./no-exist-directory");
            }

            // --do-application-compaction
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "--original", "file1.nsp", "--current", "file2.nsp", "--do-application-compaction" };
                option.Parse(args);
                Assert.IsTrue(option.MakePatch.NeedsSparse);
            }

            // --compaction-block-size
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "--original", "file1.nsp", "--current", "file2.nsp", "--compaction-block-size", "3" };
                option.Parse(args);
                Assert.IsTrue(option.MakePatch.SparseBlockSize == 3 * 1024);
                // NOTE: オプションのテストではないもののここでチェック
                Utils.CheckReturnException(new ArgumentException("block size should be power of 2."), () =>
                {
                    SparseStorageSource.SetOption(option.MakePatch.SparseBlockSize, option.MakePatch.SparseEraseSize);
                    return true;
                });
            }

            // --compaction-erase-size
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "--original", "file1.nsp", "--current", "file2.nsp", "--compaction-erase-size", "2" };
                option.Parse(args);
                Assert.IsTrue(option.MakePatch.SparseEraseSize == 2 * 1024);
                // NOTE: オプションのテストではないもののここでチェック
                Utils.CheckReturnException(new ArgumentException("minimum erase size larger than or equal to block size."), () =>
                {
                    SparseStorageSource.SetOption(option.MakePatch.SparseBlockSize, option.MakePatch.SparseEraseSize);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "makepatch", "--original", "file1.nsp", "--current", "file2.nsp", "--compaction-erase-size", "120" };
                option.Parse(args);
                Assert.IsTrue(option.MakePatch.SparseEraseSize == 120 * 1024);
                // NOTE: オプションのテストではないもののここでチェック
                Utils.CheckReturnException(new ArgumentException("erase size should be an integral multiple of block size."), () =>
                {
                    SparseStorageSource.SetOption(option.MakePatch.SparseBlockSize, option.MakePatch.SparseEraseSize);
                    return true;
                });
            }

            Directory.Delete("dir1");
            File.Delete("file1.txt");
            File.Delete("file1.nca");
            File.Delete("file2.nca");
            File.Delete("file1.nsp");
            File.Delete("file2.nsp");
            File.Delete("file3.nsp");
        }

        [TestMethod]
        public void TestOptimizePatchOption()
        {
            using (var fs = File.Create("file1.txt")) { }
            using (var fs = File.Create("file1.nca")) { }
            using (var fs = File.Create("file2.nca")) { }
            using (var fs = File.Create("file1.nsp")) { }
            using (var fs = File.Create("file2.nsp")) { }
            Directory.CreateDirectory("dir1");

            // 不明なオプションの指定
            {
                Option option = new Option();
                string[] args = new string[] { "optimizepatch" };
                Utils.CheckReturnException(new InvalidOptionException("previous archive file should be specified."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "optimizepatch", "--unknown" };
                Utils.CheckReturnException(new InvalidOptionException("previous archive file should be specified."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }

            // -h, --help
            {
                Option option = new Option();
                string[] args = new string[] { "optimizepatch", "-h" };
                option.Parse(args);
                Assert.IsTrue(option.IsShowUsage);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "optimizepatch", "--help" };
                option.Parse(args);
                Assert.IsTrue(option.IsShowUsage);
            }

            // input nsp
            {
                Option option = new Option();
                string[] args = new string[] { "optimizepatch", "file1" };
                Utils.CheckReturnException(new InvalidOptionException("previous archive file should be specified."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "optimizepatch", "--current", "file1.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("previous archive file should be specified."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "optimizepatch", "--previous", "file1.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("current archive file should be specified."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "optimizepatch", "--previous", "file1.nca", "--current", "file2.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("previous archive file must be .nsp file."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "optimizepatch", "--previous", "file1.nsp", "--current", "file2.nca" };
                Utils.CheckReturnException(new InvalidOptionException("current archive file must be .nsp file."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "optimizepatch", "--previous", "file1.txt", "--current", "file2.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("previous archive file must be .nsp file."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "optimizepatch", "--previous", "file1.nsp", "--current", "file2.nsp" };
                option.Parse(args);
                Assert.IsTrue(option.OptimizePatch != null);
                Assert.IsTrue(option.OptimizePatch.OutputPath == "./output.nsp");
                Assert.IsTrue(option.OptimizePatch.PreviousFilePath == "./file1.nsp");
                Assert.IsTrue(option.OptimizePatch.CurrentFilePath == "./file2.nsp");
            }

            // -o
            {
                Option option = new Option();
                string[] args = new string[] { "optimizepatch", "-o", "c:/Windows/Temp/output.nsp", "--previous", "file1.nsp", "--current", "file2.nsp" };
                option.Parse(args);
                Assert.IsTrue(option.OptimizePatch != null);
                Assert.IsTrue(option.OptimizePatch.OutputPath == "c:/Windows/Temp/output.nsp");
                Assert.IsTrue(option.OptimizePatch.PreviousFilePath == "./file1.nsp");
                Assert.IsTrue(option.OptimizePatch.CurrentFilePath == "./file2.nsp");
            }
#if false
            // -desc
            {
                Option option = new Option();
                string[] args = new string[] { "optimizepatch", "-o", "c:/Windows/Temp/output", "--desc", "desc", "file1", "file2" };
                option.Parse(args);
                Assert.IsTrue(option.OptimizePatch != null);
                Assert.IsTrue(option.OptimizePatch.OutputFile == "c:/Windows/Temp/output");
                Assert.IsTrue(option.OptimizePatch.OriginalFilePath == "file1");
                Assert.IsTrue(option.OptimizePatch.CurrentFilePath == "file2");
                Assert.IsTrue(option.OptimizePatch.DescFilePath == "desc");
            }
            {
                Option option = new Option();
                string[] args = new string[] { "optimizepatch", "-o", "output", "--desc", "desc", "file1", "file2" };
                option.Parse(args);
                Assert.IsTrue(option.OptimizePatch != null);
                Assert.IsTrue(option.OptimizePatch.OutputFile == "./output");
                Assert.IsTrue(option.OptimizePatch.OriginalFilePath == "file1");
                Assert.IsTrue(option.OptimizePatch.CurrentFilePath == "file2");
                Assert.IsTrue(option.OptimizePatch.DescFilePath == "desc");
            }
#endif
            // --defragment
            {
                Option option = new Option();
                string[] args = new string[] { "optimizepatch", "--previous", "file1.nsp", "--current", "file2.nsp", "--defragment" };
                option.Parse(args);
                Assert.IsTrue(option.OptimizePatch.NeedsDefragment);
            }

            // --defragment-size
            {
                Option option = new Option();
                string[] args = new string[] { "optimizepatch", "--previous", "file1.nsp", "--current", "file2.nsp", "--defragment-size", "16" };
                Utils.CheckReturnException(new InvalidOptionException("'--defragment' option should be specified."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "optimizepatch", "--previous", "file1.nsp", "--current", "file2.nsp", "--defragment", "--defragment-size", "2" };
                option.Parse(args);
                Assert.IsTrue(option.OptimizePatch.DefragmentSize == 2 * 1024);
                // NOTE: オプションのテストではないもののここでチェック
                Utils.CheckReturnException(new ArgumentException("defragment size should be large than or equal to 4KiB."), () =>
                {
                    RelocatedIndirectStorageSource.SetDefragmentSize(option.OptimizePatch.DefragmentSize);
                    return true;
                });
            }

            // --cache-directory
            {
                Option option = new Option();
                string[] args = new string[] { "optimizepatch", "--previous", "file1.nsp", "--current", "file2.nsp", "--cache-directory", "dir1" };
                option.Parse(args);
                Assert.IsTrue(option.OptimizePatch != null);
                Assert.IsTrue(option.OptimizePatch.OutputPath == "./output.nsp");
                Assert.IsTrue(option.OptimizePatch.PreviousFilePath == "./file1.nsp");
                Assert.IsTrue(option.OptimizePatch.CurrentFilePath == "./file2.nsp");
                Assert.IsTrue(option.OptimizePatch.CacheDirectory == "./dir1");
            }
            {
                Option option = new Option();
                string[] args = new string[] { "optimizepatch", "--previous", "file1.nsp", "--current", "file2.nsp", "--cache-directory", "no-exist-directory" };
                option.Parse(args);
                Assert.IsTrue(option.OptimizePatch != null);
                Assert.IsTrue(option.OptimizePatch.OutputPath == "./output.nsp");
                Assert.IsTrue(option.OptimizePatch.PreviousFilePath == "./file1.nsp");
                Assert.IsTrue(option.OptimizePatch.CurrentFilePath == "./file2.nsp");
                Assert.IsTrue(option.OptimizePatch.CacheDirectory == "./no-exist-directory");
            }

            Directory.Delete("dir1");
            File.Delete("file1.txt");
            File.Delete("file1.nca");
            File.Delete("file2.nca");
            File.Delete("file1.nsp");
            File.Delete("file2.nsp");
        }

        [TestMethod]
        public void TestBinaryDiffOption()
        {
            // 不明なオプションの指定
            {
                Option option = new Option();
                string[] args = new string[] { "bdiff" };
                Utils.CheckReturnException(new InvalidOptionException("source nsp file should be specified."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "bdiff", "--unknown" };
                Utils.CheckReturnException(new InvalidOptionException("too many arguments for makedelta subcommand."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }

            // -h, --help
            {
                Option option = new Option();
                string[] args = new string[] { "bdiff", "-h" };
                option.Parse(args);
                Assert.IsTrue(option.IsShowUsage);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "bdiff", "--help" };
                option.Parse(args);
                Assert.IsTrue(option.IsShowUsage);
            }

            // any arguments
            {
                Option option = new Option();
                string[] args = new string[] { "bdiff", "file1" };
                Utils.CheckReturnException(new InvalidOptionException("too many arguments for makedelta subcommand."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }

            var file1 = @"c:\Windows\Temp\file1";
            var file2 = @"c:\Windows\Temp\file2";

            if (File.Exists(file1))
            {
                File.Delete(file1);
            }
            using (File.Create(file1))
            {
                // nothing to do
            }

            if (File.Exists(file2))
            {
                File.Delete(file2);
            }
            using (File.Create(file2))
            {
                // nothing to do
            }

            // -o
            {
                Option option = new Option();
                string[] args = new string[] { "bdiff", "-o", "c:/Windows/Temp/output", "--source", file1, "--destination", file2 };
                option.Parse(args);
                Assert.IsTrue(option.MakeDelta != null);
                Assert.IsTrue(option.MakeDelta.OutputFile == "c:/Windows/Temp/output");
                Assert.IsTrue(option.MakeDelta.SourceFilePath == file1.Replace(@"\", "/"));
                Assert.IsTrue(option.MakeDelta.DestinationFilePath == file2.Replace(@"\", "/"));
            }
            {
                Option option = new Option();
                string[] args = new string[] { "bdiff", "-o", "output", "--source", file1, "--destination", file2 };
                option.Parse(args);
                Assert.IsTrue(option.MakeDelta != null);
                Assert.IsTrue(option.MakeDelta.OutputFile == "./output");
                Assert.IsTrue(option.MakeDelta.SourceFilePath == file1.Replace(@"\", "/"));
                Assert.IsTrue(option.MakeDelta.DestinationFilePath == file2.Replace(@"\", "/"));
            }
        }

        [TestMethod]
        public void TestMakeDeltaSaveDeltaOption()
        {
            // 不明なオプションの指定やヘルプは TestMakeDeltaOption に任せる
            // ユーザー向けでない隠しオプションのテスト

            var file1 = @"c:\Windows\Temp\file1";
            var file2 = @"c:\Windows\Temp\file2";
            var file3 = @"c:\Windows\Temp\file3";

            if (File.Exists(file1))
            {
                File.Delete(file1);
            }
            using (File.Create(file1))
            {
                // nothing to do
            }

            if (File.Exists(file2))
            {
                File.Delete(file2);
            }
            using (File.Create(file2))
            {
                // nothing to do
            }

            if (File.Exists(file3))
            {
                File.Delete(file3);
            }
            using (File.Create(file3))
            {
                // nothing to do
            }

            Option option = new Option();
            string[] args = new string[] { "makedelta", "-o", "output", "--source", file1, "--destination", file2, "--meta", file3, "--save-delta" };
            option.Parse(args);
            Assert.IsTrue(option.MakeDelta != null);
            Assert.IsTrue(option.MakeDelta.OutputFile == "./output");
            Assert.IsTrue(option.MakeDelta.SourceFilePath == file1.Replace(@"\", "/"));
            Assert.IsTrue(option.MakeDelta.DestinationFilePath == file2.Replace(@"\", "/"));
        }
        [TestMethod]
        public void TestGetUnpublishableErrorOption()
        {
            // no argument
            {
                Option option = new Option();
                string[] args = new string[] { "get-unpublishable-error" };
                Utils.CheckReturnException(new InvalidOptionException("input nsp file must be specified for get-unpublishable-error subcommand."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            // invalid file
            {
                using (var fs = File.Create("file1")) { }

                Option option = new Option();
                string[] args = new string[] { "get-unpublishable-error" , "file1" };
                Utils.CheckReturnException(new InvalidOptionException("input archive file must be .nsp file."), () =>
                {
                    option.Parse(args);
                    return true;
                });
                File.Delete("file1");
            }
        }

        [TestMethod]
        public void TestListOption()
        {
            {
                Option option = new Option();
                string[] args = new string[] { "list" };
                Utils.CheckReturnException(new InvalidOptionException("input archive file must be specified for list subcommand."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "list", "file1", "file2" };
                Utils.CheckReturnException(new InvalidOptionException("input archive file must be specified for list subcommand."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "list", "--patched-only", "file1" };
                Utils.CheckReturnException(new InvalidOptionException("--patched-only or --patched-only-in-detail option must be used with --original option."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "list", "--patched-only-in-detail", "file1" };
                Utils.CheckReturnException(new InvalidOptionException("--patched-only or --patched-only-in-detail option must be used with --original option."), () =>
                        {
                            option.Parse(args);
                            return true;
                        });
            }
        }

        [TestMethod]
        public void TestSparsifyPatchOption()
        {
            using (var fs = File.Create("file1.txt")) { }
            using (var fs = File.Create("file1.nsp")) { }
            using (var fs = File.Create("file2.nsp")) { }
            using (var fs = File.Create("file3.nsp")) { }

            // 不明なオプションの指定
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp" };
                Utils.CheckReturnException(new InvalidOptionException("original archive file should be specified."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "--unknown" };
                Utils.CheckReturnException(new InvalidOptionException("original archive file should be specified."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }

            // -h, --help
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "-h" };
                option.Parse(args);
                Assert.IsTrue(option.IsShowUsage);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "--help" };
                option.Parse(args);
                Assert.IsTrue(option.IsShowUsage);
            }

            // input nsp
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "file1" };
                Utils.CheckReturnException(new InvalidOptionException("original archive file should be specified."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "--patch", "file1.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("original archive file should be specified."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "--original", "file1.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("patch archive file should be specified."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "--original", "file1.txt", "--patch", "file2.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("original archive file must be .nsp file."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "--original", "file1.nsp", "--patch", "file1.txt" };
                Utils.CheckReturnException(new InvalidOptionException("patch archive file must be .nsp file."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "--original", "file1.nsp", "--patch", "file2.nsp", "--previous", "file1.txt" };
                Utils.CheckReturnException(new InvalidOptionException("previous archive file must be .nsp file."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "--original", "file1.nsp", "--patch", "file1.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("input files are same file."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "--original", "file1.nsp", "--patch", "file2.nsp", "--previous", "file1.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("input files are same file."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "--original", "file1.nsp", "--patch", "file2.nsp", "--previous", "file2.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("input files are same file."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "--original", "file1.nsp", "--patch", "file2.nsp" };
                option.Parse(args);
                Assert.IsTrue(option.SparsifyNsp != null);
                Assert.IsTrue(option.SparsifyNsp.OutputOriginalFile == "./output.nsp");
                Assert.IsTrue(option.SparsifyNsp.OutputPatchFile == "./output.patch.nsp");
                Assert.IsTrue(option.SparsifyNsp.OriginalFilePath == "./file1.nsp");
                Assert.IsTrue(option.SparsifyNsp.PatchFilePath == "./file2.nsp");
                Assert.IsTrue(option.SparsifyNsp.PreviousFilePath == null);
            }
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "--original", "file1.nsp", "--patch", "file2.nsp", "--previous", "file3.nsp" };
                option.Parse(args);
                Assert.IsTrue(option.SparsifyNsp != null);
                Assert.IsTrue(option.SparsifyNsp.OutputOriginalFile == "./output.nsp");
                Assert.IsTrue(option.SparsifyNsp.OutputPatchFile == "./output.patch.nsp");
                Assert.IsTrue(option.SparsifyNsp.OriginalFilePath == "./file1.nsp");
                Assert.IsTrue(option.SparsifyNsp.PatchFilePath == "./file2.nsp");
                Assert.IsTrue(option.SparsifyNsp.PreviousFilePath == "./file3.nsp");
            }

            // -o
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "-o", "c:/Windows/Temp/output.nsp", "--original", "file1.nsp", "--patch", "file2.nsp" };
                option.Parse(args);
                Assert.IsTrue(option.SparsifyNsp != null);
                Assert.IsTrue(option.SparsifyNsp.OutputOriginalFile == "c:/Windows/Temp/output.nsp");
                Assert.IsTrue(option.SparsifyNsp.OutputPatchFile == "c:/Windows/Temp/output.patch.nsp");
                Assert.IsTrue(option.SparsifyNsp.OriginalFilePath == "./file1.nsp");
                Assert.IsTrue(option.SparsifyNsp.PatchFilePath == "./file2.nsp");
            }
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "-o", "output.nsp", "--original", "file1.nsp", "--patch", "file2.nsp" };
                option.Parse(args);
                Assert.IsTrue(option.SparsifyNsp != null);
                Assert.IsTrue(option.SparsifyNsp.OutputOriginalFile == "./output.nsp");
                Assert.IsTrue(option.SparsifyNsp.OutputPatchFile == "./output.patch.nsp");
                Assert.IsTrue(option.SparsifyNsp.OriginalFilePath == "./file1.nsp");
                Assert.IsTrue(option.SparsifyNsp.PatchFilePath == "./file2.nsp");
            }
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "-o", "foo.nsp", "--original", "file1.nsp", "--patch", "file2.nsp", "--previous", "file3.nsp" };
                option.Parse(args);
                Assert.IsTrue(option.SparsifyNsp != null);
                Assert.IsTrue(option.SparsifyNsp.OutputOriginalFile == "./foo.nsp");
                Assert.IsTrue(option.SparsifyNsp.OutputPatchFile == "./foo.patch.nsp");
                Assert.IsTrue(option.SparsifyNsp.OriginalFilePath == "./file1.nsp");
                Assert.IsTrue(option.SparsifyNsp.PatchFilePath == "./file2.nsp");
                Assert.IsTrue(option.SparsifyNsp.PreviousFilePath == "./file3.nsp");
            }
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "-o", "output.nsp", "-op", "output2.nsp", "--original", "file1.nsp", "--patch", "file2.nsp" };
                option.Parse(args);
                Assert.IsTrue(option.SparsifyNsp != null);
                Assert.IsTrue(option.SparsifyNsp.OutputOriginalFile == "./output.nsp");
                Assert.IsTrue(option.SparsifyNsp.OutputPatchFile == "./output2.nsp");
                Assert.IsTrue(option.SparsifyNsp.OriginalFilePath == "./file1.nsp");
                Assert.IsTrue(option.SparsifyNsp.PatchFilePath == "./file2.nsp");
            }
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "-o", "file1.nsp", "--original", "file1.nsp", "--patch", "file2.nsp", "--previous", "file3.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("Original and output are same file."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "-o", "output.nsp", "-op", "file1.nsp", "--original", "file1.nsp", "--patch", "file2.nsp", "--previous", "file3.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("Original and output are same file."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "-o", "file2.nsp", "--original", "file1.nsp", "--patch", "file2.nsp", "--previous", "file3.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("Patch and output are same file."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "-o", "output.nsp", "-op", "file2.nsp", "--original", "file1.nsp", "--patch", "file2.nsp", "--previous", "file3.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("Patch and output are same file."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "-o", "file3.nsp", "--original", "file1.nsp", "--patch", "file2.nsp", "--previous", "file3.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("Previous and output are same file."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "-o", "output.nsp", "-op", "file3.nsp", "--original", "file1.nsp", "--patch", "file2.nsp", "--previous", "file3.nsp" };
                Utils.CheckReturnException(new InvalidOptionException("Previous and output are same file."), () =>
                {
                    option.Parse(args);
                    return true;
                });
            }

            // --block-size
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "--original", "file1.nsp", "--patch", "file2.nsp", "--block-size", "3" };
                option.Parse(args);
                Assert.IsTrue(option.SparsifyNsp.BlockSize == 3 * 1024);
                // NOTE: オプションのテストではないもののここでチェック
                Utils.CheckReturnException(new ArgumentException("block size should be power of 2."), () =>
                {
                    SparseStorageSource.SetOption(option.SparsifyNsp.BlockSize, option.SparsifyNsp.MinimumEraseSize);
                    return true;
                });
            }

            // --minimum-erase-size
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "--original", "file1.nsp", "--patch", "file2.nsp", "--minimum-erase-size", "2" };
                option.Parse(args);
                Assert.IsTrue(option.SparsifyNsp.MinimumEraseSize == 2 * 1024);
                // NOTE: オプションのテストではないもののここでチェック
                Utils.CheckReturnException(new ArgumentException("minimum erase size larger than or equal to block size."), () =>
                {
                    SparseStorageSource.SetOption(option.SparsifyNsp.BlockSize, option.SparsifyNsp.MinimumEraseSize);
                    return true;
                });
            }
            {
                Option option = new Option();
                string[] args = new string[] { "sparsifynsp", "--original", "file1.nsp", "--patch", "file2.nsp", "--minimum-erase-size", "120" };
                option.Parse(args);
                Assert.IsTrue(option.SparsifyNsp.MinimumEraseSize == 120 * 1024);
                // NOTE: オプションのテストではないもののここでチェック
                Utils.CheckReturnException(new ArgumentException("erase size should be an integral multiple of block size."), () =>
                {
                    SparseStorageSource.SetOption(option.SparsifyNsp.BlockSize, option.SparsifyNsp.MinimumEraseSize);
                    return true;
                });
            }

            File.Delete("file1.txt");
            File.Delete("file1.nsp");
            File.Delete("file2.nsp");
            File.Delete("file3.nsp");
        }
    }
}
