﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;
using System.IO.Compression;
using System.Linq;
using System.Collections.Generic;
using System.Reflection;
using System.Runtime.InteropServices;
using System.Security.Cryptography;
using System.Xml.Serialization;
using System.Xml;
using System.Text.RegularExpressions;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.Authoring.AuthoringLibrary;
using Nintendo.Authoring.AuthoringTool;
using Nintendo.Authoring.ETicketLibrary;
using Nintendo.Authoring.FileSystemMetaLibrary;
using TestUtility;

namespace AuthoringToolsTest
{
    [TestClass]
    public class ExcecutionPatchTest : ExcecutionTestBase
    {
        [TestInitialize]
        public void ForceGC()
        {
            ForceGCImpl();
            Environment.SetEnvironmentVariable("AOC_TEST_DATA_PATH", "data", EnvironmentVariableTarget.Process);
        }

        private class OptimizePatchTestHelper
        {
            private ExcecutionPatchTest Test { get; set; }

            public OptimizePatchTestHelper(ExcecutionPatchTest test)
            {
                Test = test;
            }
            // ランダムなバイト列を作成する関数
            public byte[] GenerateRandomArray(long size, int seed)
            {
                var random = new Random(seed);
                var sequence = new byte[size];
                random.NextBytes(sequence);
                return sequence;
            }
            // バイト列をnspに保存する関数
            private int MakeNspImpl(
                string dstNspPath,
                string dstDataPath,
                string dstNpdmPath,
                byte[] data,
                string srcMetaPath,
                string srcDescPath,
                string srcIconPath,
                string srcLogoPath,
                string srcCodeDirPath,
                string srcDataDirPath,
                string srcLogoDirPath,
                string srcHtmlDocumentDirPath,
                string srcLegalDirPath,
                string keyConfigFile,
                int keyIndex,
                int keyGeneration,
                bool hasTicket)
            {
                if (Directory.Exists(srcCodeDirPath) == false)
                {
                    Directory.CreateDirectory(srcCodeDirPath);
                }
                if (srcDataDirPath != null && Directory.Exists(srcDataDirPath) == false)
                {
                    Directory.CreateDirectory(srcDataDirPath);
                }
                if (srcLogoDirPath != null && Directory.Exists(srcLogoDirPath) == false)
                {
                    Directory.CreateDirectory(srcLogoDirPath);
                }
                if (srcHtmlDocumentDirPath != null && Directory.Exists(srcHtmlDocumentDirPath) == false)
                {
                    Directory.CreateDirectory(srcHtmlDocumentDirPath);
                }
                if (srcLegalDirPath != null && !Directory.Exists(srcLegalDirPath))
                {
                    Directory.CreateDirectory(srcLegalDirPath);
                }

                if (dstDataPath != null)
                {
                    using (FileStream stream = File.Create(dstDataPath))
                    {
                        stream.Write(data, 0, data.Length);
                    }
                }
                Test.MakeNpdm(dstNpdmPath, srcMetaPath, srcDescPath);

                var programArguments = string.Format("--program {0} {1} {2}",
                                                 srcCodeDirPath,
                                                 srcDataDirPath != null ? srcDataDirPath : string.Empty,
                                                 (srcDataDirPath != null && srcLogoDirPath != null) ? srcLogoDirPath : string.Empty);

                // ロゴデータ作成
                if (srcLogoDirPath != null)
                {
                    File.Copy(srcLogoPath, srcLogoDirPath + "\\" + Path.GetFileName(srcLogoPath));
                }

                // htmlDocument 作成
                if (srcHtmlDocumentDirPath != null)
                {
                    var htmlDocumentPath = srcHtmlDocumentDirPath + "\\" + "htmlDocument";
                    using (FileStream stream = File.Create(htmlDocumentPath))
                    {
                        var htmlData = GenerateRandomArray(16 * 1024, 0);
                        stream.Write(htmlData, 0, htmlData.Length);
                    }
                }

                // LegalInformation 作成
                if (srcLegalDirPath != null)
                {
                    Test.CreateLegalInfoXml(srcLegalDirPath);
                }

                Func<string, string, string> setOptionIfNotNull = (option, argument) =>
                {
                    return argument != null ? $"{option} {argument}" : string.Empty;
                };

                string errorMsg = null;
                errorMsg = Test.ExecuteProgram(string.Join(
                    " ",
                    new string[]
                    {
                        "creatensp",
                        $"-o {dstNspPath}",
                        "--type Application",
                        $"--meta {srcMetaPath}",
                        $"--desc {srcDescPath}",
                        programArguments,
                        $"--icon AmericanEnglish {srcIconPath} Japanese {srcIconPath}",
                        setOptionIfNotNull("--html-document", srcHtmlDocumentDirPath),
                        setOptionIfNotNull("--legal-information-dir", srcLegalDirPath),
                        $"--keyconfig {keyConfigFile}",
                        $"--keyindex {keyIndex}",
                        $"--keygeneration {keyGeneration}",
                        hasTicket ? "--ticket" : string.Empty,
                        "--save-adf",
                    }));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                Assert.IsTrue(File.Exists(dstNspPath));
                Test.VerifyNsp(dstNspPath, keyConfigFile);

                errorMsg = Test.ExecuteProgram($"list {dstNspPath} --keyconfig {keyConfigFile}");
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                Test.ExtractAndShowNsp(dstNspPath);

                return 0;
            }
            public void MakeNsp(string outputNspPath, string outputDirArg, string iconDirArg, string metaFilePath, string descFilePath, string logoFilePath, byte[] data, bool htmlDocument, string keyConfigFile, int keyIndex, int keyGeration, bool hasTicket)
            {
                // 与えられたバイト列を data.dat として Program に持つ nsp を作成する
                var codeDir = outputDirArg + "\\program";
                var dataDir = data != null ? outputDirArg + "\\data" : null;
                var dataPath = dataDir != null ? dataDir + "\\data.dat" : null;
                var npdmPath = codeDir + "\\main.npdm";
                var iconPath = iconDirArg + "\\" + Path.GetFileNameWithoutExtension(metaFilePath) + ".bmp";
                var logoDir = logoFilePath != null ? outputDirArg + "\\logo" : null;
                var legalDir = htmlDocument ? outputDirArg + "\\legal" : null;
                var htmlDocumentDir = htmlDocument ? outputDirArg + "\\html" : null;

                MakeNspImpl(outputNspPath, dataPath, npdmPath, data, metaFilePath, descFilePath, iconPath, logoFilePath, codeDir, dataDir, logoDir, htmlDocumentDir, legalDir, keyConfigFile, keyIndex, keyGeration, hasTicket);
            }
            public void MakePatch(string outputNspPath, string originalNspPath, string currentNspPath, string metaFilePath, string descFilePath, int minimumMatchingSize, int stronglyOptimizeSize, string keyConfigFile, bool isOutputBuildLog)
            {
                string argument = $"makepatch -o {outputNspPath} --meta {metaFilePath} --desc {descFilePath} --original {originalNspPath} --current {currentNspPath} --keyconfig {keyConfigFile}";
                if (0 < minimumMatchingSize)
                {
                    argument += string.Format(" --minimum-matching-size {0}", minimumMatchingSize / 1024);
                }
                if (0 < stronglyOptimizeSize)
                {
                    argument += string.Format(" --optimize-size-strongly {0}", stronglyOptimizeSize);
                }
                if (isOutputBuildLog)
                {
                    argument += " --save-patch-build-log";
                }
                // パッチを作成する
                var errorMsg = Test.ExecuteProgram(argument);
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                Assert.IsTrue(File.Exists(outputNspPath));
                Test.VerifyNsp(outputNspPath, keyConfigFile);
            }
            public void OptimizePatch(string outputNspPath, string previousPatchNspPath, string currentPatchNspPath, string descFilePath, int defragmentSize, int stronglyOptimizeSize, string keyConfigFile, bool isOutputBuildLog)
            {
                string argument = $"optimizepatch -o {outputNspPath} --desc {descFilePath} --previous {previousPatchNspPath} --current {currentPatchNspPath} --keyconfig {keyConfigFile}";
                if (0 < defragmentSize)
                {
                    argument += " --defragment --defragment-size " + (defragmentSize / 1024).ToString();
                }
                if (0 < stronglyOptimizeSize)
                {
                    argument += string.Format(" --optimize-size-strongly {0}", stronglyOptimizeSize);
                }
                if (isOutputBuildLog)
                {
                    argument += " --save-patch-build-log";
                }
                // IS-C パッチを作成する
                var errorMsg = Test.ExecuteProgram(argument);
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                Assert.IsTrue(File.Exists(outputNspPath));
                Test.VerifyNsp(outputNspPath, keyConfigFile);
            }
            public void SparsifyPatch(string outputNspPath, string outputPatchNspPath, string originalNspPath, string patchNspPath, string keyConfigFile)
            {
                // スパース化 nsp を作成する
                var errorMsg = Test.ExecuteProgram(
                    string.Join(" ", new string[]
                    {
                        "sparsifynsp",
                        $"-o {outputNspPath}",
                        $"-op {outputPatchNspPath}",
                        $"--original {originalNspPath}",
                        $"--patch {patchNspPath}",
                        "--block-size 8",
                        "--minimum-erase-size 8",
                        "--save-patch-build-log",
                        $"--keyconfig {keyConfigFile}",

                        // 以下は使用されないことを確認するために渡す
                        "--keyindex 2",
                        "--keygeration 0",
                    }));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                Assert.IsTrue(File.Exists(outputNspPath));
                Assert.IsTrue(File.Exists(outputPatchNspPath));
                Test.VerifyNsp(outputNspPath, keyConfigFile);
                Test.VerifyNsp(outputPatchNspPath, keyConfigFile);
            }
            public int ProdEncryptPatch(string keyConfigFile, string outputDir, string originalNspPath, string previousPatchNspPath, string patchNspPath)
            {
                // previous を製品化
                var errorMsg = Test.ExecuteProgram(string.Format("prodencryption-patch --no-check --keyconfig {0} --requiredSystemVersion {4} -o {1} --original {2} {3}",
                    keyConfigFile, outputDir, originalNspPath, previousPatchNspPath, NintendoContentMeta.GetRequiredSystemVersion() + 1));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                // optimized patch を製品化
                var previousProdPatchNspPath = previousPatchNspPath.Replace(".nsp", "_prod.nsp");
                errorMsg = Test.ExecuteProgram(string.Format("prodencryption-patch --no-check --keyconfig {0} --requiredSystemVersion {5} -o {1} --original {2} --previous-prod {3} {4}",
                    keyConfigFile, outputDir, originalNspPath, previousProdPatchNspPath, patchNspPath, NintendoContentMeta.GetRequiredSystemVersion() + 1));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                // on-card-patch 製品化
                var prodPatchNspPath = patchNspPath.Replace(".nsp", "_prod.nsp");
                errorMsg = Test.ExecuteProgram(string.Format("prodencryption --no-check --keyconfig {0} -o {1} --gamecard --no-padding --no-xcie --patch {2} {3}",
                    keyConfigFile, outputDir, prodPatchNspPath, originalNspPath));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                // TODO: 歴史情報等のチェック

                return 0;
            }
            public Tuple<long,long> CompareByBlocks(string lhsNspPath, string rhsNspPath, int blockSize)
            {
                // ブロック単位でファイルを比較し、一致ブロック数とトータルブロック数を返す
                long totalBlocksCount = 0;
                long sameBlocksCount = 0;

                using (FileStream lhsStream = File.OpenRead(lhsNspPath))
                using (FileStream rhsStream = File.OpenRead(rhsNspPath))
                {
                    totalBlocksCount = (Math.Max(lhsStream.Length, rhsStream.Length) + (blockSize - 1)) / blockSize;

                    long compareSize = Math.Min(lhsStream.Length, rhsStream.Length);
                    for (long offset = 0; offset < compareSize; )
                    {
                        // ブロックの読み込み
                        var readSize = (int)Math.Min(blockSize, compareSize - offset);
                        var lhsBuf = new byte[readSize];
                        var rhsBuf = new byte[readSize];
                        var lhsReadSize = lhsStream.Read(lhsBuf, 0, readSize);
                        var rhsReadSize = rhsStream.Read(rhsBuf, 0, readSize);
                        Assert.IsTrue(lhsReadSize == rhsReadSize);
                        Assert.IsTrue(lhsReadSize == readSize);

                        // ブロックの比較
                        if (lhsBuf.SequenceEqual(rhsBuf))
                        {
                            ++sameBlocksCount;
                        }

                        offset += lhsReadSize;
                    }
                }
                return new Tuple<long, long>(sameBlocksCount, totalBlocksCount);
            }
        }

        private class OptimizePatchTestArgument
        {
            // input
            public string TestName { get; set; }
            public byte[] DataV0 { get; set; }
            public byte[] DataV1 { get; set; }
            public byte[] DataV2 { get; set; }
            public byte[] DataV3 { get; set; }
            public int MinimumMatchingSize { get; set; }
            public int DefragmentSize { get; set; }
            public int StronglyOptimizeSize { get; set; }
            public bool NeedsHtmlDocument { get; set; }
            public bool IsEnabledOptimizationSizeCheck { get; set; }
            public bool IsOutputBuildLog { get; set; }
            public int KeyIndex { get; set; } = 0;
            public Func<int, int> GetKeyGeneration { get; set; } = (v) => NintendoContentFileSystemMetaConstant.CurrentKeyGeneration;
            public bool HasTicket { get; set; } = false;
            // output
            public string KeyConfigFilePath { get; set; }
            public string DescFilePath { get; set; }
            public string OriginalNspPath { get; set; }
            public string V1PatchNspPath { get; set; }
            public string V2PatchNspPath { get; set; }
            public string V3PatchNspPath { get; set; }
        }

        private void ExecutionOptimizePatchCore(OptimizePatchTestArgument arg)
        {
            var helper = new OptimizePatchTestHelper(this);

            var isUsedData = true;
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var iconDir = testSourceDir + "\\Icon";
            var outputDirRoot = TestEnvironment.GetOutputPath(arg.TestName);
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var keyConfigFile = testSourceDir + "\\AuthoringToolsTest.keyconfig.xml";
            var metaFile = testSourceDir + "\\ApplicationMeta\\describe_all.nmeta";
            var metaFileV1 = outputDirRoot + "\\describe_all_v1.nmeta";
            var metaFileV2 = outputDirRoot + "\\describe_all_v2.nmeta";
            var metaFileV3 = outputDirRoot + "\\describe_all_v3.nmeta";

            arg.KeyConfigFilePath = keyConfigFile;
            arg.DescFilePath = descFile;

            // 別のテストのlogoデータを流用
            var logoFile = testPath.GetSigloRoot() + "\\Tests\\Patch\\NactBuild\\logo.png";

            SafeDeleteDirectory(outputDirRoot);

            var outputNspPathV0 = outputDirRoot + "\\v0.nsp";
            {
                var outputDir = outputDirRoot + "\\v0";
                var logoFileV0 = logoFile;
                // v0 ROM(元ROM)を作成
                helper.MakeNsp(outputNspPathV0, outputDir, iconDir, metaFile, descFile, logoFileV0, arg.DataV0, arg.NeedsHtmlDocument, keyConfigFile, arg.KeyIndex, arg.GetKeyGeneration(0), arg.HasTicket);
            }
            arg.OriginalNspPath = outputNspPathV0;

            var outputNspPathV1 = outputDirRoot + "\\v1.nsp";
            var outputNspPathV1Patch = outputDirRoot + "\\v1_patch.nsp";
            {
                var outputDir = outputDirRoot + "\\v1";
                var logoFileV1 = logoFile;
                MakeSpecifiedVersionMetaFile(metaFile, metaFileV1, 1);
                // v1 ROM、v0-v1パッチを作成
                helper.MakeNsp(outputNspPathV1, outputDir, iconDir, metaFile, descFile, logoFileV1, arg.DataV1, arg.NeedsHtmlDocument, keyConfigFile, arg.KeyIndex, arg.GetKeyGeneration(1), arg.HasTicket);
                helper.MakePatch(outputNspPathV1Patch, outputNspPathV0, outputNspPathV1, metaFileV1, descFile, arg.MinimumMatchingSize, arg.StronglyOptimizeSize, keyConfigFile, arg.IsOutputBuildLog);
            }
            arg.V1PatchNspPath = outputNspPathV1Patch;

            var outputNspPathV2 = outputDirRoot + "\\v2.nsp";
            var outputNspPathV2Patch = outputDirRoot + "\\v2_patch.nsp";
            {
                var outputDir = outputDirRoot + "\\v2";
                var logoFileV2 = logoFile;
                MakeSpecifiedVersionMetaFile(metaFile, metaFileV2, 2);
                // v2 ROM、v0-v2パッチを作成
                helper.MakeNsp(outputNspPathV2, outputDir, iconDir, metaFile, descFile, logoFileV2, arg.DataV2, arg.NeedsHtmlDocument, keyConfigFile, arg.KeyIndex, arg.GetKeyGeneration(2), arg.HasTicket);
                helper.MakePatch(outputNspPathV2Patch, outputNspPathV0, outputNspPathV2, metaFileV2, descFile, arg.MinimumMatchingSize, arg.StronglyOptimizeSize, keyConfigFile, arg.IsOutputBuildLog);
            }

            // v0-v2 パッチを最適化（v0-v1パッチとの差分を減らす）
            var outputNspPathV2PatchIsc = outputDirRoot + "\\v2_patch_isc.nsp";
            helper.OptimizePatch(outputNspPathV2PatchIsc, outputNspPathV1Patch, outputNspPathV2Patch, descFile, defragmentSize: 0, stronglyOptimizeSize: arg.StronglyOptimizeSize, keyConfigFile: keyConfigFile, isOutputBuildLog: arg.IsOutputBuildLog);
            CheckPatchData(keyConfigFile, outputNspPathV2PatchIsc, arg.NeedsHtmlDocument, arg.NeedsHtmlDocument, isUsedData);
            CheckPatchIntegrity(outputNspPathV2PatchIsc, outputNspPathV0, keyConfigFile);
            VerifyPatch(outputNspPathV2PatchIsc, outputNspPathV1Patch, keyConfigFile);

            arg.V2PatchNspPath = outputNspPathV2PatchIsc;

            // 最適化後のパッチを製品化
            helper.ProdEncryptPatch(keyConfigFile, outputDirRoot, outputNspPathV0, outputNspPathV1Patch, outputNspPathV2PatchIsc);
            CheckResultXml(outputDirRoot, outputNspPathV2PatchIsc);
            CheckRequiredSystemVersion(outputDirRoot, outputNspPathV2PatchIsc, NintendoContentMeta.GetRequiredSystemVersion() + 1);

            // v0-v1パッチとv0-v2 パッチの差分の量を取る
            var compareBlockSize = 1 * 1024;
            if (arg.IsEnabledOptimizationSizeCheck)
            {
                var compareResultNotOptimized = helper.CompareByBlocks(outputNspPathV1Patch, outputNspPathV2Patch, compareBlockSize);
                var differentBlocksNotOptimized = compareResultNotOptimized.Item2 - compareResultNotOptimized.Item1;

                // v0-v1パッチとv0-v2 IS-Cパッチの差分の量を取る
                var compareResultOptimized = helper.CompareByBlocks(outputNspPathV1Patch, outputNspPathV2PatchIsc, compareBlockSize);
                var differentBlocksOptimized = compareResultOptimized.Item2 - compareResultOptimized.Item1;

                // 差分が減っていればOK
                Assert.IsTrue(differentBlocksOptimized < differentBlocksNotOptimized);
            }

            // さらに最適化パッチを作るテスト
            if (arg.DataV3 != null)
            {
                var outputNspPathV3 = outputDirRoot + "\\v3.nsp";
                var outputNspPathV3Patch = outputDirRoot + "\\v3_patch.nsp";
                {
                    var outputDir = outputDirRoot + "\\v3";
                    var logoFileV3 = logoFile;
                    MakeSpecifiedVersionMetaFile(metaFile, metaFileV3, 3);
                    // v3 ROM、v0-v3パッチを作成
                    helper.MakeNsp(outputNspPathV3, outputDir, iconDir, metaFile, descFile, logoFileV3, arg.DataV3, arg.NeedsHtmlDocument, keyConfigFile, arg.KeyIndex, arg.GetKeyGeneration(3), arg.HasTicket);
                    helper.MakePatch(outputNspPathV3Patch, outputNspPathV0, outputNspPathV3, metaFileV3, descFile, arg.MinimumMatchingSize, arg.StronglyOptimizeSize, keyConfigFile, arg.IsOutputBuildLog);
                }
                arg.V3PatchNspPath = outputNspPathV3Patch;

                // v0-v3 パッチを最適化（v0-v2 IS-Cパッチとの差分を減らす）
                var outputNspPathV3PatchIsc = outputDirRoot + "\\v3_patch_isc.nsp";
                helper.OptimizePatch(outputNspPathV3PatchIsc, outputNspPathV2PatchIsc, outputNspPathV3Patch, descFile, defragmentSize: 0, stronglyOptimizeSize: arg.StronglyOptimizeSize, keyConfigFile: keyConfigFile, isOutputBuildLog: arg.IsOutputBuildLog);
                CheckPatchData(keyConfigFile, outputNspPathV3PatchIsc, arg.NeedsHtmlDocument, arg.NeedsHtmlDocument, isUsedData);
                CheckPatchIntegrity(outputNspPathV3PatchIsc, outputNspPathV0, keyConfigFile);
                VerifyPatch(outputNspPathV3PatchIsc, outputNspPathV2PatchIsc, keyConfigFile);

                // v0-v3 パッチをデフラグして最適化
                if (0 < arg.DefragmentSize)
                {
                    var outputNspPathV3DefragmentedPatch = outputDirRoot + "\\v3_patch_defrag.nsp";
                    helper.OptimizePatch(outputNspPathV3DefragmentedPatch, outputNspPathV2PatchIsc, outputNspPathV3Patch, descFile, arg.DefragmentSize, arg.StronglyOptimizeSize, keyConfigFile, arg.IsOutputBuildLog);
                    CheckPatchData(keyConfigFile, outputNspPathV3DefragmentedPatch, arg.NeedsHtmlDocument, arg.NeedsHtmlDocument, isUsedData);
                    CheckPatchIntegrity(outputNspPathV3DefragmentedPatch, outputNspPathV0, keyConfigFile);
                    VerifyPatch(outputNspPathV3DefragmentedPatch, outputNspPathV2PatchIsc, keyConfigFile);

                    Assert.IsTrue((new FileInfo(outputNspPathV3PatchIsc)).Length != (new FileInfo(outputNspPathV3DefragmentedPatch)).Length);
                }

                if (arg.IsEnabledOptimizationSizeCheck)
                {
                    // v0-v2 IS-Cパッチとv0-v3 パッチの差分の量を取る
                    var compareResultNotOptimized = helper.CompareByBlocks(outputNspPathV2PatchIsc, outputNspPathV3Patch, compareBlockSize);
                    var differentBlocksNotOptimized = compareResultNotOptimized.Item2 - compareResultNotOptimized.Item1;

                    // v0-v2 IS-Cパッチとv0-v3 IS-Cパッチの差分の量を取る
                    var compareResultOptimized = helper.CompareByBlocks(outputNspPathV2PatchIsc, outputNspPathV3PatchIsc, compareBlockSize);
                    var differentBlocksOptimized = compareResultOptimized.Item2 - compareResultOptimized.Item1;

                    // 差分が減っていればOK
                    Assert.IsTrue(differentBlocksOptimized < differentBlocksNotOptimized);
                }
            }
        }

        private void ExecutionOptimizePatch(OptimizePatchTestArgument arg, bool isSecondOptimize)
        {
            const int c_V1Seed = 10;
            const int c_V2Seed = 20;
            var helper = new OptimizePatchTestHelper(this);

            // V0 のデータ
            var dataV0 = new byte[] { 0, 1, 2 };

            // V1 のデータ (ランダムなデータ列)
            var dataV1 = helper.GenerateRandomArray(2 * 1024 * 1024, c_V1Seed);

            // V2 のデータ
            byte[] dataV2 = null;
            {
                //  V1 のデータの中央に新規のデータを挿入

                var dataV2List = new List<byte>(dataV1);
                dataV2List.InsertRange(dataV2List.Count / 2, helper.GenerateRandomArray(16 * 1024, c_V2Seed));

                dataV2 = dataV2List.ToArray();
            }

            // V3 のデータ
            byte[] dataV3 = null;
            if (isSecondOptimize)
            {
                const int c_V3Seed = 30;

                //  データの3/4の場所に新規のデータを挿入
                var dataV3List = new List<byte>(dataV2);
                dataV3List.InsertRange(dataV3List.Count * 3 / 4, helper.GenerateRandomArray(1023, c_V3Seed));
                dataV3 = dataV3List.ToArray();
            }

            arg.DataV0 = dataV0;
            arg.DataV1 = dataV1;
            arg.DataV2 = dataV2;
            arg.DataV3 = dataV3;
            arg.IsEnabledOptimizationSizeCheck = true;

            ExecutionOptimizePatchCore(arg);
        }

        private static void CheckPatchData(string keyConfigFile, string patchPath, bool hasHtmlDocument, bool hasLegalInformation, bool hasDataInProgram)
        {
            Func<NintendoContentArchiveReader, int, bool> HasNcaPatchInfo = (reader, fsIndex) =>
            {
                // NcaPatchInfo の indirectSize が 0 かどうかで判定
                return BitConverter.ToInt64(reader.GetFsHeaderInfo(fsIndex).GetRawData(), NintendoContentFileSystemMeta.PatchInfoOffset + 8) != 0;
            };

            using (var fs = new FileStream(patchPath, FileMode.Open, FileAccess.Read))
            using (var nspReader = new NintendoSubmissionPackageReader(fs))
            {
                var config = new AuthoringConfiguration();
                config.KeyConfigFilePath = keyConfigFile;
                var model = GetContentMetaModel(nspReader);

                {
                    using (var reader = GetContentArchiveReader(nspReader, model, "Program", config.GetKeyConfiguration()))
                    {
                        // プログラムはインダイレクト形式ではない
                        Assert.IsFalse(HasNcaPatchInfo(reader, (int)NintendoContentArchivePartitionType.Code));

                        // データはインダイレクト形式
                        if (hasDataInProgram)
                        {
                            Assert.IsTrue(HasNcaPatchInfo(reader, (int)NintendoContentArchivePartitionType.Data));
                        }
                        else
                        {
                            Assert.IsFalse(reader.HasFsInfo((int)NintendoContentArchivePartitionType.Data));
                        }

                        // ロゴは存在しない
                        Assert.IsFalse(reader.HasFsInfo((int)NintendoContentArchivePartitionType.Logo));
                    }
                }

                if (hasHtmlDocument)
                {
                    using (var reader = GetContentArchiveReader(nspReader, model, "HtmlDocument", config.GetKeyConfiguration()))
                    {
                        Assert.IsTrue(HasNcaPatchInfo(reader, 0));
                    }
                }

                if (hasLegalInformation)
                {
                    using (var reader = GetContentArchiveReader(nspReader, model, "LegalInformation", config.GetKeyConfiguration()))
                    {
                        Assert.IsFalse(HasNcaPatchInfo(reader, 0));
                    }
                }
            }
        }

        private void CheckPatchIntegrity(string patchPath, string originalPath, string keyConfigFile)
        {
            var errorMsg = ExecuteProgram($"list {patchPath} --original {originalPath} --keyconfig {keyConfigFile}");
            Assert.AreEqual(errorMsg, string.Empty);

            var noTicketPatchPath = patchPath.Replace(".nsp", "_noTik.nsp");
            var noTicketPatchXmlPath = noTicketPatchPath + ".xml";
            using (var inStream = new FileStream(patchPath, FileMode.Open, FileAccess.Read))
            using (var outStream = new FileStream(noTicketPatchPath, FileMode.Create, FileAccess.ReadWrite))
            using (var outXmlStream = new FileStream(noTicketPatchXmlPath, FileMode.Create, FileAccess.ReadWrite))
            {
                ContentArchiveLibraryInterface.RemoveTicket(outStream, outXmlStream, inStream, null);
            }

            errorMsg = ExecuteProgram($"list {noTicketPatchPath} --original {originalPath} --keyconfig {keyConfigFile}");
            Assert.IsTrue(Regex.IsMatch(errorMsg, @"(Failed to OpenFile 0005000c10000801000000000000000)\d(\.tik)"));

            errorMsg = ExecuteProgram($"list {noTicketPatchPath} --patched-only --original {originalPath} --keyconfig {keyConfigFile}", true);
            Assert.IsFalse(errorMsg.IndexOf("Failed to open nca") >= 0, errorMsg);

            File.Delete(noTicketPatchPath);
            File.Delete(noTicketPatchXmlPath);
        }

        [TestMethod]
        public void TestExecutionCreateDeltaNsp()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = Path.Combine(testPath.GetSigloRoot(), "Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources");
            var metaDir = Path.Combine(testSourceDir, "DeltaMeta");
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var keyConfigFile = testSourceDir + "\\AuthoringToolsTest.keyconfig.xml";
            SafeDeleteDirectory(outputDir);

            // TODO: 本当は仮のパッチではなく、アプリを 3 バージョン作り、IS パッチを 2 バージョン作ってテストとしたい
            var pair = new List<Pair<string, string>>
                {
                    new Pair<string, string>(Path.Combine(outputDir, "patch1.nsp"), Path.Combine(metaDir, "patch-v1.nmeta")),
                    new Pair<string, string>(Path.Combine(outputDir, "patch2.nsp"), Path.Combine(metaDir, "patch-v2.nmeta")),
                };
            pair.ForEach(p => CreatePatch(p.first, p.second, keyConfigFile));

            var metaFile = Path.Combine(metaDir, "delta.nmeta");
            var testExtractDir = Path.Combine(outputDir, "extract");

            var outputPath = Path.Combine(outputDir, Path.GetFileName(metaFile) + ".nsp");

            var error = ExecuteProgram(string.Format(
                "makedelta -o {0} --meta {1} --source {2} --destination {3} --keyconfig {4}",
                outputPath,
                metaFile,
                pair[0].first,
                pair[1].first,
                keyConfigFile));
            Assert.IsTrue(error == string.Empty, error);
            Assert.IsTrue(File.Exists(outputPath));

            string errorMsg = ExecuteProgram($"list {outputPath} --keyconfig {keyConfigFile}");
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);

            ExtractAndShowNsp(outputPath);
        }

        [TestMethod]
        public void TestExecutionMakeDeltaBinaryWithDifferentFile()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var keyConfigFile = testSourceDir + "\\AuthoringToolsTest.keyconfig.xml";

            SafeDeleteDirectory(outputDir);
            Directory.CreateDirectory(outputDir);

            var randomSeed = Utils.GenerateRandomSeed();
            var random = new Random(randomSeed);
            Utils.WriteLine("Test random seed is " + randomSeed);

            var nspBaseDir = outputDir + @"\nsp";
            var binaryPatchPath = outputDir + @"\" + "output.program.bdiff";
            Action<Stream> generateData = f =>
            {
                var data = new byte[512];
                random.NextBytes(data);
                f.Write(data, 0, data.Length);
            };

            for (int i = 0; i < 3; i++)
            {
                CreateNspApplicationForDelta(
                    nspBaseDir + @"\file" + i + ".nsp",
                    CreateNspApplicationForDeltaOption.All(i, nspBaseDir, keyConfigFile, generateData));
            }

            for (int i = 0; i < 2; i++)
            {
                Create1stPatchForDelta(
                    nspBaseDir + @"\patch" + i + ".nsp",
                    keyConfigFile,
                    nspBaseDir + @"\file0.nsp",
                    nspBaseDir + @"\file" + (i + 1) + ".nsp");

                using (var stream = File.OpenRead(nspBaseDir + @"\patch" + i + ".nsp"))
                using (var nsp = new Nintendo.Authoring.FileSystemMetaLibrary.NintendoSubmissionPackageReader(stream))
                {
                    var extractNcaPath = nspBaseDir + @"\program" + i + ".nca";
                    if (File.Exists(extractNcaPath))
                    {
                        File.Delete(extractNcaPath);
                    }

                    var program = nsp.ListFileInfo().First();
                    using (var nca = File.OpenWrite(extractNcaPath))
                    {
                        var data = nsp.ReadFile(program.Item1, 0, nsp.GetFileSize(program.Item1));
                        nca.Write(data, 0, data.Length);
                    }
                }
            }

            {
                var patch0 = nspBaseDir + @"\patch0.nsp";
                var patch1 = nspBaseDir + @"\patch1.nsp";
                var programNca0 = nspBaseDir + @"\program0.nca";
                var programNca1 = nspBaseDir + @"\program1.nca";

                string errorMsg = ExecuteProgram(
                    string.Format(
                        "bdiff -o {0} --source {1} --destination {2}",
                        binaryPatchPath,
                        patch0,
                        patch1));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                Assert.IsTrue(File.Exists(binaryPatchPath));

                using (var file = File.OpenRead(binaryPatchPath))
                {
                    const int HeaderSize = 64;
                    const int SizeOffset = 8;
                    var data = new byte[HeaderSize];
                    file.Read(data, 0, data.Length);

                    Assert.AreEqual(new FileInfo(programNca0).Length, BitConverter.ToInt64(data, SizeOffset + 0));
                    Assert.AreEqual(new FileInfo(programNca1).Length, BitConverter.ToInt64(data, SizeOffset + 8));
                    Assert.AreEqual(HeaderSize, BitConverter.ToInt64(data, SizeOffset + 16));
                    Assert.AreEqual(new FileInfo(binaryPatchPath).Length - HeaderSize, BitConverter.ToInt64(data, SizeOffset + 24));
                }
            }
        }

        [TestMethod]
        public void TestExecutionMakeDeltaBinaryWithSameFile()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var keyConfigFile = testSourceDir + "\\AuthoringToolsTest.keyconfig.xml";

            SafeDeleteDirectory(outputDir);
            Directory.CreateDirectory(outputDir);

            var randomSeed = Utils.GenerateRandomSeed();
            var random = new Random(randomSeed);
            Utils.WriteLine("Test random seed is " + randomSeed);

            var nspBaseDir = outputDir + @"\nsp";
            var binaryPatchPath = outputDir + @"\" + "output.program.bdiff";
            Action<Stream> generateData = f =>
            {
                var data = new byte[512];
                random.NextBytes(data);
                f.Write(data, 0, data.Length);
            };

            for (int i = 0; i < 2; i++)
            {
                CreateNspApplicationForDelta(
                    nspBaseDir + @"\file" + i + ".nsp",
                    CreateNspApplicationForDeltaOption.All(
                        i,
                        nspBaseDir,
                        keyConfigFile,
                        generateData));
            }

            Create1stPatchForDelta(
                nspBaseDir + @"\patch0.nsp",
                keyConfigFile,
                nspBaseDir + @"\file0.nsp",
                nspBaseDir + @"\file1.nsp");
            File.Copy(
                nspBaseDir + @"\patch0.nsp",
                nspBaseDir + @"\patch1.nsp",
                true);
            using (var stream = File.OpenRead(nspBaseDir + @"\patch0.nsp"))
            using (var nsp = new Nintendo.Authoring.FileSystemMetaLibrary.NintendoSubmissionPackageReader(stream))
            {
                var extractNcaPath = nspBaseDir + @"\program0.nca";
                if (File.Exists(extractNcaPath))
                {
                    File.Delete(extractNcaPath);
                }

                var program = nsp.ListFileInfo().First();
                using (var nca = File.OpenWrite(extractNcaPath))
                {
                    var data = nsp.ReadFile(program.Item1, 0, nsp.GetFileSize(program.Item1));
                    nca.Write(data, 0, data.Length);
                }
            }

            {
                var patch0 = nspBaseDir + @"\patch0.nsp";
                var patch1 = nspBaseDir + @"\patch1.nsp";
                var programNca0 = nspBaseDir + @"\program0.nca";
                var programNca1 = nspBaseDir + @"\program0.nca";

                string errorMsg = ExecuteProgram(
                    string.Format(
                        "bdiff -o {0} --source {1} --destination {2}",
                        binaryPatchPath,
                        patch0,
                        patch1));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                Assert.IsTrue(File.Exists(binaryPatchPath));

                using (var file = File.OpenRead(binaryPatchPath))
                {
                    const int HeaderSize = 64;
                    const int SizeOffset = 8;
                    var data = new byte[HeaderSize];
                    file.Read(data, 0, data.Length);

                    Assert.AreEqual(new FileInfo(programNca0).Length, BitConverter.ToInt64(data, SizeOffset + 0));
                    Assert.AreEqual(new FileInfo(programNca1).Length, BitConverter.ToInt64(data, SizeOffset + 8));
                    Assert.AreEqual(HeaderSize, BitConverter.ToInt64(data, SizeOffset + 16));
                    Assert.AreEqual(new FileInfo(binaryPatchPath).Length - HeaderSize, BitConverter.ToInt64(data, SizeOffset + 24));
                }
            }
        }

        class CheckFragmentsParameters
        {
            public uint Generation { get; set; } = 0;
            public int HtmlDocumentFragmentCount { get; set; } = 0;
            public int ProgramFragmentCount { get; set; } = 0;
        }

        private static void CheckFragments(string keyFilePath, string patch0, CheckFragmentsParameters expect)
        {
            using (var fs = new FileStream(patch0, FileMode.Open, FileAccess.Read))
            using (var nsp = new NintendoSubmissionPackageReader(fs))
            {
                var entry = nsp.ListFileInfo().Find(s => s.Item1.EndsWith(".cnmt.xml"));
                var model = (DeltaContentMetaModel)ArchiveReconstructionUtils.ReadModel(nsp, entry.Item1, entry.Item2);

                Func<NintendoContentArchiveReader, int, Tuple<uint, uint>> GetAesCtrEncryptionParameters = (reader, fsIndex) =>
                {
                    var offset = NintendoContentFileSystemMeta.PatchInfoOffset + NintendoContentFileSystemMeta.PatchInfoSize;
                    var rawData = reader.GetFsHeaderInfo(fsIndex).GetRawData();
                    // generation, secure value の順
                    return new Tuple<uint, uint>(
                        BitConverter.ToUInt32(rawData, offset + 0),
                        BitConverter.ToUInt32(rawData, offset + 4));
                };

                var listContentInfoIndexAndFragmentIndex = new List<Tuple<int, uint>>();
                foreach (var fragmentSet in model.FragmentSetList)
                {
                    switch (fragmentSet.FragmentTargetContentType)
                    {
                        case "Meta":
                            Assert.AreEqual(1, fragmentSet.FragmentList.Count);
                            break;
                        case "Control":
                            Assert.AreEqual(1, fragmentSet.FragmentList.Count);
                            break;
                        case "LegalInformation":
                            Assert.AreEqual(1, fragmentSet.FragmentList.Count);
                            break;
                        case "Program":
                            Assert.AreEqual(expect.ProgramFragmentCount, fragmentSet.FragmentList.Count);
                            break;
                        case "HtmlDocument":
                            Assert.AreEqual(expect.HtmlDocumentFragmentCount, fragmentSet.FragmentList.Count);
                            break;
                        default:
                            Assert.Fail(fragmentSet.FragmentTargetContentType + " is not expected.");
                            break;
                    }

                    foreach (var fragment in fragmentSet.FragmentList)
                    {
                        var config = new AuthoringConfiguration();
                        config.KeyConfigFilePath = keyFilePath;

                        var content = model.ContentList[fragment.ContentInfoIndex];
                        using (var ncaReader = nsp.OpenNintendoContentArchiveReader(content.Id + ".nca", new NcaKeyGenerator(config.GetKeyConfiguration())))
                        {
                            var parameters = GetAesCtrEncryptionParameters(ncaReader, 0);
                            Assert.AreEqual(expect.Generation, parameters.Item1);
                            listContentInfoIndexAndFragmentIndex.Add(new Tuple<int, uint>(fragment.ContentInfoIndex, parameters.Item2));
                        }
                    }
                }

                listContentInfoIndexAndFragmentIndex.Sort((v1, v2) => v1.Item1 < v2.Item1 ? -1 : v1.Item1 > v2.Item1 ? 1 : 0);
                uint counter = 0;
                foreach (var value in listContentInfoIndexAndFragmentIndex)
                {
                    Assert.AreEqual(counter, value.Item2);
                    ++counter;
                }
            }
        }

        [TestMethod]
        public void TestExecutionMakeDeltaNspWithDifferentFile()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var keyConfigFile = testSourceDir + "\\AuthoringToolsTest.keyconfig.xml";
            var metaFile = testSourceDir + @"\DeltaMeta\default.nmeta";

            SafeDeleteDirectory(outputDir);
            Directory.CreateDirectory(outputDir);

            var randomSeed = Utils.GenerateRandomSeed();
            var random = new Random(randomSeed);
            Utils.WriteLine("Test random seed is " + randomSeed);

            var nspBaseDir = outputDir + @"\nsp";
            var deltaBetweenPatchesNsp = outputDir + @"\" + "output.nsp";
            Action<Stream> generateData = f =>
            {
                var data = new byte[32 * 1024 * 1024];
                for (int i = 0; i < 32 * 5 / 32; i++)
                {
                    random.NextBytes(data);
                    f.Write(data, 0, data.Length);
                }
            };

            for (int i = 0; i < 3; i++)
            {
                CreateNspApplicationForDelta(
                    nspBaseDir + @"\file" + i + ".nsp",
                    CreateNspApplicationForDeltaOption.All(
                        i,
                        nspBaseDir,
                        keyConfigFile,
                        generateData));
            }

            for (int i = 0; i < 2; i++)
            {
                Create1stPatchForDelta(
                    nspBaseDir + @"\patch" + i + ".nsp",
                    keyConfigFile,
                    nspBaseDir + @"\file0.nsp",
                    nspBaseDir + @"\file" + (i + 1) + ".nsp");
            }

            {
                var patch0 = nspBaseDir + @"\patch0.nsp";
                var patch1 = nspBaseDir + @"\patch1.nsp";

                string errorMsg = ExecuteProgram(
                    string.Format(
                        "makedelta -o {0} --meta {1} --source {2} --destination {3} --keyconfig {4} --save-build-log",
                        deltaBetweenPatchesNsp,
                        metaFile,
                        patch0,
                        patch1,
                        keyConfigFile));
                Assert.AreEqual(string.Empty, errorMsg);
                Assert.IsTrue(File.Exists(deltaBetweenPatchesNsp));

                using (var stream = File.OpenRead(deltaBetweenPatchesNsp))
                using (var nsp = new Nintendo.Authoring.FileSystemMetaLibrary.NintendoSubmissionPackageReader(stream))
                {
                    Assert.AreEqual(10, nsp.ListFileInfo().Count);
                }

                var expect = new CheckFragmentsParameters();
                expect.Generation = 2;
                expect.HtmlDocumentFragmentCount = 2;
                expect.ProgramFragmentCount = 2;
                CheckFragments(keyConfigFile, deltaBetweenPatchesNsp, expect);

                Assert.AreEqual(
                    4,
                    Directory.EnumerateFiles(outputDir).Where(s => Regex.Match(s, @"output\.nsp\.c\d+\.DeltaFragment\.delta\.csv").Success).Count());
            }

            SafeDeleteDirectory(outputDir);
        }

        public void TestExecutionMakeDeltaNspWithAddAndRemoveNca(bool willAdd)
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var outputDir = TestEnvironment.GetOutputPath("\\TestExecutionMakeDeltaNspWith" + (willAdd ? "NewNca" : "LostNca"));
            var keyConfigFile = testSourceDir + "\\AuthoringToolsTest.keyconfig.xml";
            var metaFile = testSourceDir + @"\DeltaMeta\default.nmeta";

            SafeDeleteDirectory(outputDir);
            Directory.CreateDirectory(outputDir);

            var nspBaseDir = outputDir + @"\nsp";
            var deltaBetweenPatchesNsp = outputDir + @"\" + "output.nsp";
            Action<Stream> generateData = f =>
            {
                f.SetLength(1);
            };

            var patch0 = nspBaseDir + @"\file0.nsp";
            var patch1 = nspBaseDir + @"\file1.nsp";

            var option = CreateNspApplicationForDeltaOption.Patch(
                0,
                nspBaseDir,
                keyConfigFile,
                generateData);
            CreateNspApplicationForDelta(patch0, option);

            {
                var metaDir = testSourceDir + "\\ApplicationMeta";
                var applicationMetaFile = GetDefaultApplicationMetaFile(metaDir);
                var legalNca = nspBaseDir + @"\legal.nca";
                var htmlDocumentNca = nspBaseDir + @"\htmlDocument.nca";

                {
                    var legalDir = nspBaseDir + @"\legal";

                    if (!Directory.Exists(legalDir))
                    {
                        Directory.CreateDirectory(legalDir);
                    }
                    CreateLegalInfoXml(legalDir);

                    string errorMsg = ExecuteProgram(
                        string.Format(
                            "createnca -o {0} --legal-information-dir {1} --meta {2} --meta-type Application --save-adf",
                            legalNca,
                            legalDir,
                            applicationMetaFile));
                    Assert.AreEqual(string.Empty, errorMsg);
                    Assert.IsTrue(File.Exists(legalNca));
                }
                {
                    var htmlDocumentDir = option.DataDirectoryPath;

                    string errorMsg = ExecuteProgram(
                        string.Format(
                            "createnca -o {0} --html-document {1} --meta {2} --meta-type Application --save-adf",
                            htmlDocumentNca,
                            htmlDocumentDir,
                            applicationMetaFile));
                    Assert.AreEqual(string.Empty, errorMsg);
                    Assert.IsTrue(File.Exists(htmlDocumentNca));
                }

                {
                    File.Copy(patch0, patch1);
                    var patch1_tmp = patch1.Replace(".nsp", "_replaced.nsp");

                    string errorMsg = ExecuteProgram(
                        string.Format(
                            "replace -o {0} {1} null.nca@LegalInformation {2}",
                            Path.GetDirectoryName(patch1_tmp),
                            patch1,
                            legalNca));
                    Assert.AreEqual(string.Empty, errorMsg);
                    Assert.IsTrue(File.Exists(patch1_tmp));
                    File.Delete(patch1);
                    File.Move(patch1_tmp, patch1);

                    errorMsg = ExecuteProgram(
                        string.Format(
                            "replace -o {0} {1} null.nca@HtmlDocument {2}",
                            Path.GetDirectoryName(patch1_tmp),
                            patch1,
                            htmlDocumentNca));
                    Assert.AreEqual(string.Empty, errorMsg);
                    Assert.IsTrue(File.Exists(patch1_tmp));
                    File.Delete(patch1);
                    File.Move(patch1_tmp, patch1);
                }
            }

            {
                string errorMsg = ExecuteProgram(
                    string.Format(
                        "makedelta -o {0} --meta {1} --source {2} --destination {3}",
                        deltaBetweenPatchesNsp,
                        metaFile,
                        willAdd ? patch0 : patch1,
                        willAdd ? patch1 : patch0));
                Assert.AreEqual(string.Empty, errorMsg);
                Assert.IsTrue(File.Exists(deltaBetweenPatchesNsp));

                using (var nsp = new NintendoSubmissionPackageReader(deltaBetweenPatchesNsp))
                {
                    var entry = nsp.ListFileInfo().Find(s => s.Item1.EndsWith(".cnmt.xml"));
                    var model = (DeltaContentMetaModel)ArchiveReconstructionUtils.ReadModel(nsp, entry.Item1, entry.Item2);

                    if (willAdd)
                    {
                        Assert.IsNotNull(model.FragmentSetList.Find(m => m.FragmentTargetContentType == "HtmlDocument"));
                        Assert.IsNotNull(model.FragmentSetList.Find(m => m.FragmentTargetContentType == "LegalInformation"));
                    }
                    else
                    {
                        // Note: 生成できなくてエラーになってもよい
                        Assert.IsNull(model.FragmentSetList.Find(m => m.FragmentTargetContentType == "HtmlDocument"));
                        Assert.IsNull(model.FragmentSetList.Find(m => m.FragmentTargetContentType == "LegalInformation"));
                    }
                }
            }

            SafeDeleteDirectory(outputDir);
        }

        [TestMethod]
        public void TestExecutionMakeDeltaNspWithNewNca()
        {
            TestExecutionMakeDeltaNspWithAddAndRemoveNca(true);
        }

        [TestMethod]
        public void TestExecutionMakeDeltaNspWithLostNca()
        {
            TestExecutionMakeDeltaNspWithAddAndRemoveNca(false);
        }

        [TestMethod]
        public void TestExecutionMakeDeltaNspWithDeltaFragmentCheck()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var keyConfigFile = testSourceDir + "\\AuthoringToolsTest.keyconfig.xml";
            var metaFile = testSourceDir + @"\DeltaMeta\default.nmeta";

            SafeDeleteDirectory(outputDir);
            Directory.CreateDirectory(outputDir);

            var nspBaseDir = outputDir + @"\nsp";
            Action<Stream> generateData = f =>
            {
                f.SetLength(1);
            };

            for (int i = 0; i < 2; ++i)
            {
                var baseFile = nspBaseDir + @"\base" + i + ".nsp";

                CreateNspApplicationForDelta(
                    baseFile,
                    CreateNspApplicationForDeltaOption.Patch(
                    i,
                    nspBaseDir,
                    keyConfigFile,
                    generateData));
            }

            for (int i = 0; i < 3; ++i)
            {
                var metaDir = testSourceDir + "\\ApplicationMeta";
                var applicationMetaFile = GetDefaultApplicationMetaFile(metaDir);
                var htmlDocumentNca = nspBaseDir + @"\htmlDocument" + i + ".nca";
                var patch = nspBaseDir + @"\file" + i + ".nsp";
                string baseFile;

                {
                    var htmlDocumentDir = nspBaseDir + @"\html";
                    var adfPath = htmlDocumentNca + ".adf";

                    if (i == 2)
                    {
                        baseFile = nspBaseDir + @"\base1.nsp";
                        var adfPathPrevious = nspBaseDir + @"\htmlDocument" + (i - 1) + ".nca.adf";
                        File.Copy(adfPathPrevious, adfPath);
                    }
                    else
                    {
                        baseFile = nspBaseDir + @"\base0.nsp";

                        SafeDeleteDirectory(htmlDocumentDir);
                        Directory.CreateDirectory(htmlDocumentDir);
                        using (var file = new FileStream(htmlDocumentDir + "\\data.bin", FileMode.Create, FileAccess.Write))
                        {
                            file.SetLength(256 * 1024 * 1024);
                            file.Seek(0, SeekOrigin.Begin);

                            var tmp = new byte[32 * 1024 * 1024];
                            for (int j = 0; j < tmp.Length; j++)
                            {
                                tmp[j] = (byte)i;
                            }
                            for (int j = 0; j < 256 / 32; j++)
                            {
                                file.Write(tmp, 0, tmp.Length);
                            }
                        }
                        string errorMsg = ExecuteProgram(
                            string.Format(
                                "createnca -o {0} --html-document {1} --meta {2} --meta-type Application --save-adf --only-adf",
                                htmlDocumentNca,
                                htmlDocumentDir,
                                applicationMetaFile));
                        Assert.AreEqual(string.Empty, errorMsg);

                        File.WriteAllText(
                            adfPath,
                            Regex.Replace(
                                File.ReadAllText(adfPath),
                                "encryptionType : \\d+",
                                "encryptionType : " + (int)NintendoContentArchiveEncryptionType.None
                                ));
                    }

                    {
                        var errorMsg = ExecuteProgram(
                            string.Format(
                                "createnca --meta-type Application -o {0} {1}",
                                htmlDocumentNca,
                                adfPath));
                        Assert.AreEqual(string.Empty, errorMsg);
                        Assert.IsTrue(File.Exists(htmlDocumentNca));
                    }
                }

                {
                    File.Copy(baseFile, patch);
                    var patch_tmp = patch.Replace(".nsp", "_replaced.nsp");

                    var errorMsg = ExecuteProgram(
                        string.Format(
                            "replace -o {0} {1} null.nca@HtmlDocument {2}",
                            Path.GetDirectoryName(patch_tmp),
                            patch,
                            htmlDocumentNca));
                    Assert.AreEqual(string.Empty, errorMsg);
                    Assert.IsTrue(File.Exists(patch_tmp));
                    File.Delete(patch);
                    File.Move(patch_tmp, patch);
                }
            }

            for (int i = 0; i < 2; ++i)
            {
                var deltaPath = nspBaseDir + @"\delta" + i + ".nsp";
                var patch0 = nspBaseDir + @"\file0.nsp";
                var patchN = nspBaseDir + @"\file" + (i + 1) + ".nsp";

                string errorMsg = ExecuteProgram(
                    string.Format(
                        "makedelta -o {0} --meta {1} --source {2} --destination {3} --keyconfig {4}",
                        deltaPath,
                        metaFile,
                        patch0,
                        patchN,
                        keyConfigFile));
                Assert.AreEqual(string.Empty, errorMsg);
                Assert.IsTrue(File.Exists(deltaPath));
            }

            {
                var delta0Path = nspBaseDir + @"\delta0.nsp";
                var delta1Path = nspBaseDir + @"\delta1.nsp";

                Func<string, int, byte[]> readFragmentData = (deltaPath, fragmentIndex) =>
                {
                    using (var fs = new FileStream(deltaPath, FileMode.Open, FileAccess.Read))
                    using (var nsp = new NintendoSubmissionPackageReader(fs))
                    {
                        var entry = nsp.ListFileInfo().Find(s => s.Item1.EndsWith(".cnmt.xml"));
                        var model = (DeltaContentMetaModel)ArchiveReconstructionUtils.ReadModel(nsp, entry.Item1, entry.Item2);
                        var fragmentSet = model.FragmentSetList.Find(m => m.FragmentTargetContentType == "HtmlDocument");

                        long compareOffsetStart = 0x000A0D00;
                        long compareOffsetEnd = 0x000A1D40;

                        var fragment = fragmentSet.FragmentList.Find(f => f.FragmentIndex == fragmentIndex);
                        var content = model.ContentList[fragment.ContentInfoIndex];
                        var info = nsp.ListFileInfo().Find(m => m.Item1 == content.Id + ".nca");

                        Assert.IsTrue(info.Item2 > compareOffsetEnd);
                        return nsp.ReadFile(info.Item1, compareOffsetStart, compareOffsetEnd - compareOffsetStart);
                    }
                };

                byte[] data0 = readFragmentData(delta0Path, 0);
                byte[] data1 = readFragmentData(delta0Path, 1);
                byte[] data2 = readFragmentData(delta1Path, 0);

                Assert.IsFalse(Enumerable.SequenceEqual(data0, data1));
                Assert.IsFalse(Enumerable.SequenceEqual(data0, data2));
                Assert.IsFalse(Enumerable.SequenceEqual(data1, data2));
            }

            SafeDeleteDirectory(outputDir);
        }

        private class TestExecutionMakeDeltaNspWithSizeOptimizationResult
        {
            internal TestExecutionMakeDeltaNspWithSizeOptimizationResult(string outputDir, string keyConfigFilePath)
            {
                OutputDirectory = outputDir;
                OriginalPath = outputDir + "\\original.nsp";
                CurrentPath = outputDir + "\\current.nsp";
                CurrentPatchPath = outputDir + "\\currentPatch.nsp";
                CurrentAesCtrExBuildLogPath = CurrentPatchPath + ".c0.Program.aesctrex_table.txt";
                CurrentIndirectStorageBuildLogPath = CurrentPatchPath + ".c0.Program.optimized_table.txt";
                PreviousPatchPath = outputDir + "\\previousPatch.nsp";
                CurrentDeltaPath = CurrentPatchPath.Replace(".nsp", ".delta.nsp");
                CurrentDeltaBuildLogPath = CurrentDeltaPath + ".c0.DeltaFragment.delta.csv";
                KeyConfigFilePath = keyConfigFilePath;
            }

            public string OutputDirectory { get; private set; }
            public string CurrentAesCtrExBuildLogPath { get; private set; }
            public string CurrentDeltaBuildLogPath { get; private set; }
            public string CurrentDeltaPath { get; private set; }
            public string CurrentIndirectStorageBuildLogPath { get; private set; }
            public string CurrentPatchPath { get; private set; }
            public string CurrentPath { get; private set; }
            public string OriginalPath { get; private set; }
            public string PreviousPatchPath { get; private set; }
            public string KeyConfigFilePath { get; private set; }
        }

        private Action<string, int, Random> MakeTestExecutionMakeDeltaNspWithSizeOptimizationCheckGenerateData(Action<Stream, int, Random> generateData)
        {
            return (dataDirectoryPath, generation, random) =>
            {
                var testFile = dataDirectoryPath + @"\test.bin";
                using (var f = File.OpenWrite(testFile))
                {
                    generateData(f, generation, random);
                }
            };
        }

        private void TestExecutionMakeDeltaNspWithSizeOptimizationCheckImpl(
            string name,
            int maxGeneration,
            Action<string, int, Random> generateData,
            bool needLogo,
            Action<TestExecutionMakeDeltaNspWithSizeOptimizationResult, int> checkResult,
            IndirectStorageSource.CacheType cacheType,
            bool needBuildLog)
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var outputDir = TestEnvironment.GetOutputPath("\\" + name);
            var keyConfigFile = testSourceDir + "\\AuthoringToolsTest.keyconfig.xml";
            var metaFile = testSourceDir + @"\DeltaMeta\default.nmeta";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var nspBaseDir = outputDir + @"\nsp";
            var result = new TestExecutionMakeDeltaNspWithSizeOptimizationResult(outputDir, keyConfigFile);

            SafeDeleteDirectory(outputDir);
            Directory.CreateDirectory(outputDir);

            var randomSeed = Utils.GenerateRandomSeed();
            var random = new Random(randomSeed);
            Utils.WriteLine("Test random seed is " + randomSeed);

            {
                var option = CreateNspApplicationForDeltaOption.NoManualForSizeCheck(
                    0,
                    keyConfigFile,
                    nspBaseDir,
                    true,
                    f => generateData(f, 0, random));
                CreateNspApplicationForDelta(result.OriginalPath, option);
            }

            for (int generationApplication = 1; generationApplication <= maxGeneration; generationApplication++)
            {
                {
                    var option = CreateNspApplicationForDeltaOption.NoManualForSizeCheck(
                        generationApplication,
                        keyConfigFile,
                        nspBaseDir,
                        false,
                        f => generateData(f, generationApplication, random));
                    CreateNspApplicationForDelta(result.CurrentPath, option);
                }

                var stopWatch = new StopWatch();
                stopWatch.Start();

                string errorMsg = ExecuteProgram(
                    string.Format(
                        "makepatch -o {0} --original {1} --current {2} --desc {3} {4} --data-cache-type {5} {6}",
                        result.CurrentPatchPath,
                        result.OriginalPath,
                        result.CurrentPath,
                        descFile,
                        generationApplication >= 2 ? "--with-delta  --delta-meta-path " + metaFile + " --previous " + result.PreviousPatchPath : string.Empty,
                        cacheType.ToString(),
                        needBuildLog ? "--save-patch-build-log" : string.Empty));
                Assert.AreEqual(string.Empty, errorMsg);
                Assert.IsTrue(File.Exists(result.CurrentPatchPath));
                VerifyNsp(result.CurrentPatchPath, keyConfigFile);

                var elapsed = stopWatch.ElapsedMilliSec();
                Utils.WriteLine(string.Format("{0},{1},{2}", cacheType.ToString(), generationApplication, (ulong)elapsed));

                checkResult(result, generationApplication);

                if (generationApplication >= 2)
                {
                    File.Delete(result.CurrentDeltaPath);
                }

                File.Delete(result.PreviousPatchPath);
                File.Move(result.CurrentPatchPath, result.PreviousPatchPath);
            }

            SafeDeleteDirectory(outputDir);
        }

        [TestMethod]
        public void TestExecutionMakeDeltaNspWithSizeOptimizationCheckSmall()
        {
            const int ModifySize = 128 * 1024;
            const int MaxGeneration = 7; // 0 ～ 7

            long previousDeltaSize = 0;

            Action<Stream, int, Random> generateData = (f, i, random) =>
            {
                if (i == 0)
                {
                    var data = new byte[1 * 1024 * 1024];
                    random.NextBytes(data);
                    f.Write(data, 0, data.Length);
                }
                else if (i == 1)
                {
                    var data = new byte[ModifySize * (MaxGeneration + 1) - 2];
                    random.NextBytes(data);
                    f.Write(data, 0, data.Length);
                }
                else
                {
                    var data = new byte[ModifySize];
                    random.NextBytes(data);
                    f.Position = ModifySize * (i - 2);
                    f.Write(data, 0, data.Length);
                }
            };

            Action<TestExecutionMakeDeltaNspWithSizeOptimizationResult, int> checkResult = (result, generationApplication) =>
            {
                long minOffset = 0;

                {
                    var config = new AuthoringConfiguration();
                    config.KeyConfigFilePath = result.KeyConfigFilePath;

                    using (var nsp = new NintendoSubmissionPackageReader(result.CurrentPatchPath))
                    {
                        Assert.AreEqual((uint)(generationApplication - 1), nsp.GetAesCtrGeneration(new NcaKeyGenerator(config.GetKeyConfiguration())));

                        var meta = ArchiveReconstructionUtils.ReadContentMetaInNsp(nsp).First();
                        var content = meta.ContentList.Find(v => v.Type == "Program");
                        using (var nca = nsp.OpenNintendoContentArchiveReader(content.Id + ".nca", new NcaKeyGenerator(config.GetKeyConfiguration())))
                        {
                            minOffset = nca.GetExistentFsIndices().Min(v => nca.GetFsStartOffset(v));
                        }
                    }
                }

                Assert.AreNotEqual(0, minOffset);

                // パッチ間差分は パッチ(0, 1) と パッチ(0, 2) から生成が始まる
                if (generationApplication >= 2)
                {
                    var currentDeltaSize = new FileInfo(result.CurrentDeltaPath).Length;

                    // パッチ間差分の比較は 差分 ((0, 1), (0, 2)) <=> 差分 ((0, 2), (0, 3)) から可能
                    if (generationApplication >= 3)
                    {
                        Utils.WriteLine((currentDeltaSize - previousDeltaSize) + " bytes added");
                        Assert.IsTrue(currentDeltaSize - previousDeltaSize <= 1024);
                    }

                    {
                        var aesCtrExBuildLog = AesCtrExBuildLogParser.Parse(result.CurrentAesCtrExBuildLogPath);
                        var indirectStorageBuildLog = IndirectStorageBuildLogParser.Parse(result.CurrentIndirectStorageBuildLogPath);
                        var deltaLog = DeltaBuildLogParser.Parse(result.CurrentDeltaBuildLogPath);

                        // 現世代のデータデルタが生成される (ヘッダーは先頭データにマージ)
                        var currentData = aesCtrExBuildLog.EntryEntries.Where(entry => entry.Generation == generationApplication - 1);
                        Assert.AreEqual(currentData.Count(), deltaLog.DeltaEntries.Count());

                        for (int i = 1; i < currentData.Count(); ++i)
                        {
                            // オフセットは一致する
                            var data = currentData.ElementAt(i);
                            var delta = deltaLog.DeltaEntries[i];
                            Assert.AreEqual(minOffset + data.Offset, delta.Offset);

                            // パッチ側として IS にエントリが存在する
                            var indirectStorageEntry = indirectStorageBuildLog.EntryEntries.Where(entry => entry.PhisicalOffset <= data.Offset && entry.StorageIndex == 1);
                            Assert.AreNotEqual(0, indirectStorageEntry.Count());
                        }
                    }

                    var expect = new CheckFragmentsParameters();
                    expect.Generation = (uint)generationApplication;
                    expect.ProgramFragmentCount = 1;
                    CheckFragments(result.KeyConfigFilePath, result.CurrentDeltaPath, expect);

                    previousDeltaSize = currentDeltaSize;
                }
            };

            TestExecutionMakeDeltaNspWithSizeOptimizationCheckImpl(
                MethodBase.GetCurrentMethod().Name,
                MaxGeneration,
                MakeTestExecutionMakeDeltaNspWithSizeOptimizationCheckGenerateData(generateData),
                true,
                checkResult,
                IndirectStorageSource.CacheType.Memory,
                true);
        }

        [TestMethod]
        public void TestExecutionMakeDeltaNspWithSizeOptimizationCheckLarge()
        {
            const int MaxGeneration = 3; // 0 ～ 3

            var data = new byte[32 * 1024 * 1024];
            Action<Stream, int, Random> generateData = (f, i, random) =>
            {
                switch (i)
                {
                    case 0:
                        random.NextBytes(data);
                        f.Write(data, 0, 8 * 1024 * 1024);
                        break;
                    case 1:
                        f.Write(data, 0, 16 * 1024 * 1024);
                        break;
                    case 2:
                        f.Write(data, 0, 32 * 1024 * 1024);
                        break;
                    case 3:
                        f.Write(data, 8 * 1024 * 1024, 16 * 1024 * 1024);
                        break;
                }
            };

            Action<TestExecutionMakeDeltaNspWithSizeOptimizationResult, int> checkResult = (result, generationApplication) =>
            {
                switch (generationApplication)
                {
                    case 2:
                        Assert.IsTrue(new FileInfo(result.CurrentDeltaPath).Length < 17 * 1024 * 1024);
                        break;
                    case 3:
                        Assert.IsTrue(new FileInfo(result.CurrentDeltaPath).Length < 9 * 1024 * 1024);
                        break;
                }
            };

            TestExecutionMakeDeltaNspWithSizeOptimizationCheckImpl(
                MethodBase.GetCurrentMethod().Name,
                MaxGeneration,
                MakeTestExecutionMakeDeltaNspWithSizeOptimizationCheckGenerateData(generateData),
                false,
                checkResult,
                IndirectStorageSource.CacheType.Memory,
                true);
        }

        [TestMethod]
#if DEBUG
        [Ignore]
#endif
        public void TestExecutionMakeDeltaNspWithSamePattern()
        {
            // 現状他のパラメータはテスト不要
            const int MaxGeneration = 3;
            var cacheType = IndirectStorageSource.CacheType.Memory;

            Action<string, int, Random> generateData = (path, generation, random) =>
                {
                    Action<string, string, string> createFile = (fileName, size, mode) =>
                    {
                        TestExecutionMakeDeltaNspTestData.CreateFile(path + "/" + fileName, size, mode, random);
                    };

                    switch (generation)
                    {
                        case 0:
                            foreach (var entry in TestExecutionMakeDeltaNspTestData.ResourceFileList)
                            {
                                createFile(entry.Item1, entry.Item2, entry.Item3);
                            }
                            break;
                        case 1:
                            foreach (var entry in TestExecutionMakeDeltaNspTestData.ResourceAddFileList)
                            {
                                createFile(entry.Item1, entry.Item2, entry.Item3);
                            }
                            break;
                        case 2:
                            foreach (var entry in TestExecutionMakeDeltaNspTestData.ResourceAddDirTestFileList)
                            {
                                createFile(entry.Item1, entry.Item2, entry.Item3);
                            }
                            break;
                        case 3:
                            foreach (var entry in TestExecutionMakeDeltaNspTestData.ResourceDirTestFileList)
                            {
                                Utils.DeleteDirectoryIfExisted(path + "/" + entry);
                            }
                            break;
                    }
                };

            Action<TestExecutionMakeDeltaNspWithSizeOptimizationResult, int> checkResult = (result, generationApplication) =>
                {
                    Utils.WriteLine("" + generationApplication);
                };

            TestExecutionMakeDeltaNspWithSizeOptimizationCheckImpl(
                System.Reflection.MethodBase.GetCurrentMethod().Name,
                MaxGeneration,
                generateData,
                false,
                checkResult,
                cacheType,
                false);
        }

        [TestMethod]
        public void TestExecutionMakeNca1stPatch()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            string outputFile = Path.Combine(outputDir, "test.nca");
            string outputFile2 = Path.Combine(outputDir, "test2.nca");
            string patchFile = Path.Combine(outputDir, "patch.nca");
            string metaFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.aarch64.lp64.nmeta";
            string descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            string testDir = Path.Combine(outputDir, "test/");
            string testFile = testDir + "data.dat";
            string testFile2 = testDir + "data2.dat";
            string testExtractDir = Path.Combine(outputDir, "extract");
            string testExtractDir2 = Path.Combine(outputDir, "extract2");

            Utils.DeleteDirectoryIfExisted(outputDir);
            Directory.CreateDirectory(testDir);
            using (FileStream stream = File.Create(testFile))
            {
                int fileSize = 128 * 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }
            using (FileStream stream = File.Create(testFile2))
            {
                var randomSeed = Utils.GenerateRandomSeed();
                Utils.WriteLine(testFile2 + " generation random seed is " + randomSeed);
                var random = new Random(randomSeed);
                int fileSize = 3 * 32 * 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);

                fileSize = 32 * 1024;
                for (int i = 0; i < fileSize; i++)
                {
                    if (random.Next(8 * 1024) == 0)
                    {
                        data[i] = (byte)random.Next(256);
                    }
                    else
                    {
                        data[i] = (byte)i;
                    }
                }
                stream.Write(data, 0, fileSize);
            }

            string testNpdm = testDir + "\\main.npdm";
            MakeNpdm(testNpdm, metaFile, descFile);

            string errorMsg = null;

            File.Delete(outputFile);
            errorMsg = ExecuteProgram(string.Format("createnca -o {0} --meta {1} --desc {2} --program {3} {3}", outputFile, metaFile, descFile, testDir));
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
            Assert.IsTrue(File.Exists(outputFile));
            TestExtractWithNca(outputFile, testDir, testExtractDir, 2);

            File.Delete(outputFile2);
            errorMsg = ExecuteProgram(string.Format("createnca -o {0} --meta {1} --desc {2} --program {3} {3}", outputFile2, metaFile, descFile, testDir));
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
            Assert.IsTrue(File.Exists(outputFile2));
            TestExtractWithNca(outputFile2, testDir, testExtractDir2, 2);

            File.Delete(patchFile);
            errorMsg = ExecuteProgram(string.Format("makepatch -o {0} --desc {1} --original {2} --current {3} --save-patch-build-log", patchFile, descFile, outputFile, outputFile2));
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
            Assert.IsTrue(File.Exists(patchFile));
            TestExtractWithNca(outputFile, patchFile, testDir, testExtractDir, 2);

            var patchSize = (new FileInfo(patchFile)).Length;
            var outputSize = (new FileInfo(outputFile2)).Length;
            Assert.IsTrue(patchSize < outputSize);

            File.Delete(outputFile);
            File.Delete(outputFile2);
            File.Delete(patchFile);
        }

        private class TestExecutionMakeNsp1stPatchConfiguration
        {
            public bool HasLegalInformationOriginal { get; set; } = true;
            public bool HasHtmlDocumentOriginal { get; set; } = true;
            public bool HasAccessibleUrlsOriginal { get; set; } = true;

            public bool HasLegalInformationCurrent { get; set; } = true;
            public bool HasHtmlDocumentCurrent { get; set; } = true;
            public bool HasAccessibleUrlsCurrent { get; set; } = true;

            public string TestName { get; set; } = "TestExecutionMakeNsp1stPatch";
            public int FileBlockSize { get; set; } = 32 * 1024;
            public bool CheckSizeResult { get; set; } = true;
            public string MakePatchErrorMessage { get; set; } = string.Empty;

            public string OriginalOutputPath { get; set; } = string.Empty;
            public string CurrentOutputPath { get; set; } = string.Empty;
            public string PatchOutputPath { get; set; } = string.Empty;
            public bool NeedsDeleteOutputFiles { get; set; } = true;

            public static TestExecutionMakeNsp1stPatchConfiguration All
            {
                get
                {
                    return new TestExecutionMakeNsp1stPatchConfiguration();
                }
            }
        }

        private void TestExecutionMakeNsp1stPatchImpl(TestExecutionMakeNsp1stPatchConfiguration config)
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var iconDir = testSourceDir + "\\Icon";
            var invalidIconDir = testSourceDir + "\\Icon\\Invalid";
            var mainVisualDir = testSourceDir + "\\MainVisual";
            var outputDir = TestEnvironment.GetOutputPath(config.TestName);
            var dataDir = outputDir + "\\program";
            var dataFile = dataDir + "\\data.dat";
            var legalDir = outputDir + "\\legal";
            var accessibleUrlsPath = outputDir + "\\accessible-urls\\accessible-urls.txt";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var keyConfigFile = testSourceDir + "\\AuthoringToolsTest.keyconfig.xml";
            var testExtractDir = Path.Combine(outputDir, "extract");
            var testExtractDir2 = Path.Combine(outputDir, "extract2");
            var testExtractEntryDir = Path.Combine(outputDir, "extractEntry");
            var testReplaceDir = Path.Combine(outputDir, "replace");

            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(testExtractDir);
            Directory.CreateDirectory(testExtractDir2);
            Directory.CreateDirectory(dataDir);
            Directory.CreateDirectory(legalDir);
            {
                string metaFile = testSourceDir + "\\ApplicationMeta\\describe_all.nmeta";
                string metaFile2 = testSourceDir + "\\ApplicationMeta\\describe_all_but_required_system_version.nmeta";
                string patchMetaFile = outputDir + "\\describe_all_v1.nmeta";
                MakeSpecifiedVersionMetaFile(metaFile2, patchMetaFile, 1);

                using (FileStream stream = File.Create(dataFile))
                {
                    int fileSize = 4 * config.FileBlockSize;
                    byte[] data = new byte[fileSize];
                    for (int i = 0; i < fileSize; i++)
                    {
                        data[i] = (byte)i;
                    }
                    stream.Write(data, 0, fileSize);
                }

                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile, descFile);

                CreateLegalInfoXml(legalDir);
                MakeAccessibleUrlsFile(accessibleUrlsPath);

                var outputPath = outputDir + "\\" + Path.GetFileName(metaFile) + ".nsp";
                var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(metaFile) + ".bmp";

                string errorMsg = null;
                errorMsg = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} {3} --icon AmericanEnglish {4} Japanese {4} {5} {6} {7} --save-adf",
                    outputPath,
                    metaFile,
                    descFile,
                    dataDir,
                    iconPath,
                    config.HasHtmlDocumentOriginal ? "--html-document " + dataDir : string.Empty,
                    config.HasLegalInformationOriginal ? "--legal-information-dir " + legalDir : string.Empty,
                    config.HasAccessibleUrlsOriginal ? "--accessible-urls " + Path.GetDirectoryName(accessibleUrlsPath) : string.Empty));
                Assert.AreEqual(string.Empty, errorMsg);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, keyConfigFile);

                errorMsg = ExecuteProgram(string.Format("list {0}", outputPath));
                Assert.AreEqual(string.Empty, errorMsg);

                using (var nsp = new NintendoSubmissionPackageReader(outputPath))
                {
                    var entry = nsp.ListFileInfo().Find(s => s.Item1.EndsWith(".cnmt.xml"));
                    var model = ArchiveReconstructionUtils.ReadModel(nsp, entry.Item1, entry.Item2);
                    var content = model.ContentList.Find(m => m.Type == "Program");
                    var keyConfig = new AuthoringConfiguration();
                    keyConfig.KeyConfigFilePath = keyConfigFile;

                    using (var nca = nsp.OpenNintendoContentArchiveReader(content.Id + ".nca", new NcaKeyGenerator(keyConfig.GetKeyConfiguration())))
                    {
                        Assert.IsTrue(nca.HasFsInfo((int)NintendoContentArchivePartitionType.Code));
                        Assert.IsTrue(nca.HasFsInfo((int)NintendoContentArchivePartitionType.Data));
                        Assert.IsTrue(nca.HasFsInfo((int)NintendoContentArchivePartitionType.Logo));
                    }
                }

                ExtractAndShowNsp(outputPath);

                var randomSeed = Utils.GenerateRandomSeed();
                Utils.WriteLine("Test resource generation random seed is " + randomSeed);

                using (FileStream stream = File.Create(dataFile))
                {
                    var random = new Random(randomSeed);
                    int fileSize = 3 * config.FileBlockSize;
                    byte[] data = new byte[fileSize];
                    for (int i = 0; i < fileSize; i++)
                    {
                        data[i] = (byte)i;
                    }
                    stream.Write(data, 0, fileSize);

                    fileSize = config.FileBlockSize;
                    for (int i = 0; i < fileSize; i++)
                    {
                        if (random.Next(8 * 1024) == 0)
                        {
                            data[i] = (byte)random.Next(256);
                        }
                        else
                        {
                            data[i] = (byte)i;
                        }
                    }
                    stream.Write(data, 0, fileSize);
                }

                var outputPath2 = outputDir + "\\" + Path.GetFileName(metaFile) + "2.nsp";
                errorMsg = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} {3} --icon AmericanEnglish {4} Japanese {4} {5} {6} {7} --save-adf",
                    outputPath2,
                    patchMetaFile,
                    descFile,
                    dataDir,
                    iconPath,
                    config.HasHtmlDocumentCurrent ? "--html-document " + dataDir : string.Empty,
                    config.HasLegalInformationCurrent ? "--legal-information-dir " + legalDir : string.Empty,
                    config.HasAccessibleUrlsCurrent ? "--accessible-urls " + Path.GetDirectoryName(accessibleUrlsPath) : string.Empty));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                Assert.IsTrue(File.Exists(outputPath2));
                VerifyNsp(outputPath2, keyConfigFile);

                errorMsg = ExecuteProgram($"list {outputPath2} --keyconfig {keyConfigFile}");
                Assert.AreEqual(string.Empty, errorMsg);

                ExtractAndShowNsp(outputPath2);

                var patchPath = outputDir + "\\" + Path.GetFileName(metaFile) + "_patch.nsp";

                // メタファイル指定なし
                File.Delete(patchPath);
                foreach (string path in Directory.GetFiles(outputDir, "*.brf"))
                {
                    File.Delete(path);
                }
                errorMsg = ExecuteProgram(string.Format("makepatch -o {0} --desc {1} --original {2} --current {3} --cache-directory {4}", patchPath, descFile, outputPath, outputPath2, outputDir));
                if (config.MakePatchErrorMessage == string.Empty)
                {
                    Assert.AreEqual(string.Empty, errorMsg);
                    Assert.IsTrue(File.Exists(patchPath));
                    Assert.IsTrue(1 <= Directory.GetFiles(outputDir, "*.brf").Length);

                    VerifyNsp(patchPath, keyConfigFile);
                    CheckRequiredSystemVersion(patchPath, NintendoContentMeta.GetRequiredSystemVersion());

                    CheckPatchIntegrity(patchPath, outputPath, keyConfigFile);
                    ExtractAndShowNsp(patchPath);

                    if (config.CheckSizeResult)
                    {
                        var patchSize = (new FileInfo(patchPath)).Length;
                        var outputSize = (new FileInfo(outputPath2)).Length;
                        Utils.WriteLine("full nsp file size is " + outputSize + ", patch nsp size is " + patchSize);
                        Assert.IsTrue(patchSize < outputSize);
                    }
                    CheckPatchData(
                        keyConfigFile,
                        patchPath,
                        config.HasHtmlDocumentCurrent,
                        config.HasLegalInformationCurrent,
                        true);
                }
                else
                {
                    Assert.IsTrue(errorMsg.Contains(config.MakePatchErrorMessage), errorMsg);
                }

                // メタファイル指定あり
                File.Delete(patchPath);
                foreach (string path in Directory.GetFiles(outputDir, "*.brf"))
                {
                    File.Delete(path);
                }
                errorMsg = ExecuteProgram(string.Format("makepatch -o {0} --meta {1} --desc {2} --original {3} --current {4} --cache-directory {5}", patchPath, patchMetaFile, descFile, outputPath, outputPath2, outputDir));
                if (config.MakePatchErrorMessage == string.Empty)
                {
                    Assert.AreEqual(string.Empty, errorMsg);
                    Assert.IsTrue(File.Exists(patchPath));
                    Assert.IsTrue(1 <= Directory.GetFiles(outputDir, "*.brf").Length);

                    VerifyNsp(patchPath, keyConfigFile);
                    CheckRequiredSystemVersion(patchPath, NintendoContentMeta.GetRequiredSystemVersion());

                    CheckPatchIntegrity(patchPath, outputPath, keyConfigFile);
                    ExtractAndShowNsp(patchPath);

                    if (config.CheckSizeResult)
                    {
                        var patchSize = (new FileInfo(patchPath)).Length;
                        var outputSize = (new FileInfo(outputPath2)).Length;
                        Utils.WriteLine("full nsp file size is " + outputSize + ", patch nsp size is " + patchSize);
                        Assert.IsTrue(patchSize < outputSize);
                    }
                    CheckPatchData(keyConfigFile, patchPath, false, false, true);

                    // 製品化
                    {
                        errorMsg = ExecuteProgram(string.Format("prodencryption-patch --no-check --keyconfig {0} --requiredSystemVersion {4} -o {1} --original {2} {3}", keyConfigFile, outputDir, outputPath, patchPath, NintendoContentMeta.GetRequiredSystemVersion() + 1));
                        Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                        CheckResultXml(outputDir, patchPath);
                        CheckRequiredSystemVersion(outputDir, patchPath, NintendoContentMeta.GetRequiredSystemVersion() + 1);

                        errorMsg = ExecuteProgram(string.Format("prodencryption-patch --no-check --keyconfig {0} -o {1} --original {2} {3}", keyConfigFile, outputDir, outputPath, patchPath));
                        Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                        CheckResultXml(outputDir, patchPath);
                        CheckRequiredSystemVersion(outputDir, patchPath, NintendoContentMeta.GetRequiredSystemVersion());

                        errorMsg = ExecuteProgram(string.Format("prodencryption-patch --no-check --keyconfig {0} --requiredSystemVersion {4} -o {1} --original {2} {3}", keyConfigFile, outputDir, outputPath, patchPath, NintendoContentMeta.GetRequiredSystemVersion() - 1));
                        Assert.IsTrue(errorMsg.IndexOf(string.Format("Specified required system version (= {0}) is older than that of patch", NintendoContentMeta.GetRequiredSystemVersion() - 1)) >= 0);
                    }
                }
                else
                {
                    Assert.IsTrue(errorMsg.Contains(config.MakePatchErrorMessage), errorMsg);
                }

                if (config.NeedsDeleteOutputFiles)
                {
                    File.Delete(outputPath);
                    File.Delete(outputPath2);
                    File.Delete(patchPath);
                }

                config.OriginalOutputPath = outputPath;
                config.CurrentOutputPath = outputPath2;
                config.PatchOutputPath = patchPath;
            }
        }

        [TestMethod]
        public void TestExecutionMakeNsp1stPatchAll()
        {
            var config = TestExecutionMakeNsp1stPatchConfiguration.All;
            config.TestName = MethodBase.GetCurrentMethod().Name;
            TestExecutionMakeNsp1stPatchImpl(config);
        }

        [TestMethod]
        public void TestExecutionMakeNsp1stPatchWithLegalWithoutHtmlToWithHtmlWithoutLegal()
        {
            var config = TestExecutionMakeNsp1stPatchConfiguration.All;
            config.TestName = MethodBase.GetCurrentMethod().Name;
            config.HasLegalInformationCurrent = false;
            config.HasHtmlDocumentOriginal = false;
            config.HasAccessibleUrlsOriginal = false;
            config.CheckSizeResult = false;
            TestExecutionMakeNsp1stPatchImpl(config);
        }

        [TestMethod]
        public void TestExecutionMakeNsp1stPatchBinaryRegion()
        {
            var config = TestExecutionMakeNsp1stPatchConfiguration.All;
            config.TestName = MethodBase.GetCurrentMethod().Name;
            config.FileBlockSize = 8 * 1024 * 1024;
            config.HasAccessibleUrlsOriginal = false;
            config.HasHtmlDocumentOriginal = false;
            config.HasLegalInformationOriginal = false;
            config.HasAccessibleUrlsCurrent = false;
            config.HasHtmlDocumentCurrent = false;
            config.HasLegalInformationCurrent = false;
            config.NeedsDeleteOutputFiles = false;
            TestExecutionMakeNsp1stPatchImpl(config);

            var testPath = new TestPath(TestContext);
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var outputDir = TestEnvironment.GetOutputPath(config.TestName);
            var metaFile = Path.Combine(outputDir, "describe_all_v1.nmeta");
            var outputPath = config.OriginalOutputPath;
            var outputPath2 = config.CurrentOutputPath;
            var patchPath = config.PatchOutputPath;

            string regionPath;
            {
                var files = Directory.GetFiles(Path.GetDirectoryName(config.OriginalOutputPath), "*.brf");
                Assert.AreEqual(1, files.Length);
                regionPath = files[0];
            }

            DateTime timestamp;

            // タイムスタンプを元に更新されることを確認
            {
                {
                    FileInfo originalInfo = new FileInfo(config.OriginalOutputPath);
                    FileInfo regionInfo = new FileInfo(regionPath);

                    timestamp = regionInfo.LastWriteTime;

                    regionInfo.LastWriteTime = originalInfo.LastWriteTime - new TimeSpan(1, 0, 0);
                }

                var stopwatch = Stopwatch.StartNew();
                string errorMsg = ExecuteProgram(string.Format(
                    "makepatch -o {0} --meta {1} --desc {2} --original {3} --current {4} --cache-directory {5}",
                    patchPath, metaFile, descFile, outputPath, outputPath2, Path.GetDirectoryName(outputPath)));
                stopwatch.Stop();
                Assert.AreEqual(string.Empty, errorMsg);
                {
                    var files = Directory.GetFiles(Path.GetDirectoryName(config.OriginalOutputPath), "*.brf");
                    Assert.AreEqual(1, files.Length);
                    Assert.AreEqual(regionPath, files[0]);

                    FileInfo regionInfo = new FileInfo(regionPath);
                    Assert.IsTrue(timestamp < regionInfo.LastWriteTime);

                    timestamp = regionInfo.LastWriteTime;
                }
                Console.WriteLine("Elapsed time without cache: {0}ms", stopwatch.ElapsedMilliseconds);
            }

            // 再度パッチの作成を行ってキャッシュが利用されている（ファイルの更新が無い）ことを確認
            {
                var stopwatch = Stopwatch.StartNew();
                string errorMsg = ExecuteProgram(string.Format(
                    "makepatch -o {0} --meta {1} --desc {2} --original {3} --current {4} --cache-directory {5}",
                    patchPath, metaFile, descFile, outputPath, outputPath2, Path.GetDirectoryName(outputPath)));
                stopwatch.Stop();
                Assert.AreEqual(string.Empty, errorMsg);
                {
                    var files = Directory.GetFiles(Path.GetDirectoryName(config.OriginalOutputPath), "*.brf");
                    Assert.AreEqual(1, files.Length);
                    Assert.AreEqual(regionPath, files[0]);

                    FileInfo regionInfo = new FileInfo(regionPath);
                    Assert.AreEqual(timestamp, regionInfo.LastWriteTime);
                }
                Console.WriteLine("Elapsed time with cache: {0}ms", stopwatch.ElapsedMilliseconds);
            }

            // キャッシュを書き換え
            {
                var bytes = File.ReadAllBytes(regionPath);
                for (int i = 50; i < 58; ++i)
                {
                    bytes[i] = 0;
                }
                File.WriteAllBytes(regionPath, bytes);

                timestamp = (new FileInfo(regionPath)).LastWriteTime;
            }

            // 不正なキャッシュは利用されない（ファイルが更新される）ことを確認
            {
                var stopwatch = Stopwatch.StartNew();
                string errorMsg = ExecuteProgram(string.Format(
                    "makepatch -o {0} --meta {1} --desc {2} --original {3} --current {4} --cache-directory {5}",
                    patchPath, metaFile, descFile, outputPath, outputPath2, Path.GetDirectoryName(outputPath)));
                stopwatch.Stop();
                Assert.AreEqual(string.Empty, errorMsg);
                {
                    var files = Directory.GetFiles(Path.GetDirectoryName(regionPath), "*.brf");
                    Assert.AreEqual(1, files.Length);
                    Assert.AreEqual(regionPath, files[0]);

                    FileInfo regionInfo = new FileInfo(regionPath);
                    Assert.IsTrue(timestamp < regionInfo.LastWriteTime);

                    timestamp = regionInfo.LastWriteTime;
                }
                Console.WriteLine("Elapsed time with invalid cache: {0}ms", stopwatch.ElapsedMilliseconds);
            }

            // リージョンサイズが違うキャッシュは利用されない（2 つ目のファイルが作られる）ことを確認
            {
                var stopwatch = Stopwatch.StartNew();
                string errorMsg = ExecuteProgram(string.Format(
                    "makepatch -o {0} --meta {1} --desc {2} --original {3} --current {4} --cache-directory {5} --minimum-matching-size 64",
                    patchPath, metaFile, descFile, outputPath, outputPath2, Path.GetDirectoryName(outputPath)));
                stopwatch.Stop();
                Assert.AreEqual(string.Empty, errorMsg);
                {
                    var files = Directory.GetFiles(Path.GetDirectoryName(regionPath), "*.brf");
                    Assert.AreEqual(2, files.Length);
                }
                Console.WriteLine("Elapsed time with other region size cache: {0}ms", stopwatch.ElapsedMilliseconds);
            }
        }

        [TestMethod]
        public void TestExecutionExcludeRange1stPatch()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var iconDir = testSourceDir + "\\Icon";
            var invalidIconDir = testSourceDir + "\\Icon\\Invalid";
            var mainVisualDir = testSourceDir + "\\MainVisual";
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var dataDir = outputDir + "\\program";
            var dataFile = dataDir + "\\data.dat";
            var excludeDir = dataDir + "\\.nrr";
            var excludeFile = excludeDir + "\\test.nrr";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var keyConfigFile = testSourceDir + "\\AuthoringToolsTest.keyconfig.xml";
            var testExtractDir = Path.Combine(outputDir, "extract");
            var testExtractEntryDir = Path.Combine(outputDir, "extractentry");
            var testExtractDir2 = Path.Combine(outputDir, "extract2");
            var testExtractEntryDir2 = Path.Combine(outputDir, "extractEntry2");
            var testReplaceDir = Path.Combine(outputDir, "replace");

            SafeDeleteDirectory(outputDir);

            Directory.CreateDirectory(testExtractDir);
            Directory.CreateDirectory(testExtractDir2);
            Directory.CreateDirectory(dataDir);
            Directory.CreateDirectory(excludeDir);
            {
                string metaFile = testSourceDir + "\\ApplicationMeta\\describe_all.nmeta";
                string patchMetaFile = outputDir + "\\describe_all_v1.nmeta";
                MakeSpecifiedVersionMetaFile(metaFile, patchMetaFile, 1);

                var randomSeed = Utils.GenerateRandomSeed();
                Utils.WriteLine("Test random seed is " + randomSeed);
                var random = new Random(randomSeed);

                using (FileStream stream = File.Create(dataFile))
                {
                    int fileSize = 256 * 1024;
                    byte[] data = new byte[fileSize];
                    for (int i = 0; i < fileSize; i++)
                    {
                        data[i] = (byte)i;
                    }
                    stream.Write(data, 0, fileSize);
                }
                using (FileStream stream = File.Create(excludeFile))
                {
                    int fileSize = 128 * 1024;
                    byte[] data = new byte[fileSize];
                    for (int i = 0; i < fileSize; i++)
                    {
                        data[i] = (byte)i;
                    }
                    // ハッシュ領域オフセット=0x00000000, 個数=0
                    for(int offs=0x340; offs < 0x348; ++offs)
                    {
                        data[offs] = 0x00;
                    }
                    stream.Write(data, 0, fileSize);
                }

                string testNpdm = dataDir + "\\main.npdm";
                MakeNpdm(testNpdm, metaFile, descFile);

                var outputPath = outputDir + "\\" + Path.GetFileName(metaFile) + ".nsp";
                var iconPath = iconDir + "\\" + Path.GetFileNameWithoutExtension(metaFile) + ".bmp";

                string errorMsg = null;
                errorMsg = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} {3} --icon AmericanEnglish {4} Japanese {4} --save-adf",
                    outputPath,
                    metaFile,
                    descFile,
                    dataDir,
                    iconPath));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                Assert.IsTrue(File.Exists(outputPath));
                VerifyNsp(outputPath, keyConfigFile);

                errorMsg = ExecuteProgram($"list {outputPath} --keyconfig {keyConfigFile}");
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);

                ExtractAndShowNsp(outputPath);

                using (FileStream stream = File.Create(excludeFile))
                {
                    int fileSize = 3 * 32 * 1024;
                    byte[] data = new byte[fileSize];
                    for (int i = 0; i < fileSize; i++)
                    {
                        data[i] = (byte)i;
                    }
                    // ハッシュ領域オフセット=0x00000000, 個数=0
                    for (int offs = 0x340; offs < 0x348; ++offs)
                    {
                        data[offs] = 0x00;
                    }
                    stream.Write(data, 0, fileSize);

                    fileSize = 32 * 1024;
                    for (int i = 0; i < fileSize; i++)
                    {
                        data[i] = (byte)random.Next(256);
                    }
                    stream.Write(data, 0, fileSize);
                }

                var outputPath2 = outputDir + "\\" + Path.GetFileName(patchMetaFile) + "2.nsp";
                errorMsg = ExecuteProgram(string.Format(
                    "creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} {3} --icon AmericanEnglish {4} Japanese {4} --save-adf",
                    outputPath2,
                    patchMetaFile,
                    descFile,
                    dataDir,
                    iconPath));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                Assert.IsTrue(File.Exists(outputPath2));
                VerifyNsp(outputPath2, keyConfigFile);

                errorMsg = ExecuteProgram($"list {outputPath2} --keyconfig {keyConfigFile}");
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                Assert.IsTrue((new FileInfo(outputPath)).Length == (new FileInfo(outputPath2)).Length);

                ExtractAndShowNsp(outputPath2);

                var patchPath = outputDir + "\\" + Path.GetFileName(metaFile) + "_patch.nsp";
                File.Delete(patchPath);
                errorMsg = ExecuteProgram(string.Format("makepatch -o {0} --desc {1} --original {2} --current {3}", patchPath, descFile, outputPath, outputPath2));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                Assert.IsTrue(File.Exists(patchPath));
                VerifyNsp(patchPath, keyConfigFile);

                CheckPatchIntegrity(patchPath, outputPath, keyConfigFile);
                Assert.IsTrue((new FileInfo(patchPath)).Length < (new FileInfo(outputPath2)).Length);

                ExtractAndShowNsp(patchPath);

                string[] entryRuleStringArray = new string[] {
                        @".*\.nca/fs0/\.nrr/test\.nrr",
                        @".*\.nca/fs0/data\.dat",
                        @".*\.nca/fs0/main\.npdm",
                        @".*\.nca/fs0/control\.nacp",
                        @".*\.nca/fs0/icon_AmericanEnglish\.dat",
                        @".*\.nca/fs0/icon_Japanese\.dat",
                        @".*\.nca/fs1/\.nrr/test\.nrr",
                        @".*\.nca/fs1/data\.dat",
                        @".*\.nca/fs1/main\.npdm",
                        @".*\.cnmt\.nca/fs0/Patch_0005000c10000801.cnmt",
                        @".*\.cnmt.xml",
                        @".*\.nacp.xml",
                        @".*\.nx\.AmericanEnglish\.jpg",
                        @".*\.nx\.Japanese\.jpg",
                        @".*\.raw\.AmericanEnglish\.jpg",
                        @".*\.raw\.Japanese\.jpg",
                        @".*\.programinfo\.xml",
                        @"cardspec\.xml",
                        @"[0-9a-fA-F]*\.tik",
                        @"[0-9a-fA-F]*\.cert",
                        @"authoringtoolinfo.xml",
                };
                TestExtractWithNsp(outputPath, patchPath, testExtractDir, testExtractEntryDir, entryRuleStringArray);

                // *.nrr が outputPath を参照していないことを確認
                // （outputPath の *.nrr を参照しているとハッシュの検証で落ちる）
                TestExtractWithNsp(outputPath2, patchPath, testExtractDir2, testExtractEntryDir2, entryRuleStringArray);

                // 製品化
                errorMsg = ExecuteProgram(string.Format("prodencryption-patch --no-check --keyconfig {0} --requiredSystemVersion {4} -o {1} --original {2} {3}", keyConfigFile, outputDir, outputPath, patchPath, NintendoContentMeta.GetRequiredSystemVersion() + 1));
                Assert.IsTrue(errorMsg == string.Empty, errorMsg);
                CheckResultXml(outputDir, patchPath);
                CheckRequiredSystemVersion(outputDir, patchPath, NintendoContentMeta.GetRequiredSystemVersion() + 1);

                File.Delete(outputPath);
                File.Delete(outputPath2);
                File.Delete(patchPath);
            }
        }

        [TestMethod]
        public void TestExecutionOptimizePatch()
        {
            var arg = new OptimizePatchTestArgument { TestName = MethodBase.GetCurrentMethod().Name };
            // 2回目の最適化も行う
            ExecutionOptimizePatch(arg, true);
        }

        [TestMethod]
        public void TestExecutionOptimizePatchAddToTail()
        {
            // バージョンアップにつれデータが末尾に追加されるだけなので
            // 最適化が望めない場合のテスト

            const int c_V0Seed = 0;
            const int c_V1Seed = 10;
            const int c_V2Seed = 20;
            const int c_V3Seed = 30;

            var helper = new OptimizePatchTestHelper(this);

            // V0 のデータ (ランダムなデータ列)
            var dataV0 = helper.GenerateRandomArray(1 * 1024 * 1024, c_V0Seed);

            // V1 のデータ (V0 の末尾にランダムなデータ列を追加)
            byte[] dataV1 = null;
            {
                var dataV1List = new List<byte>(dataV0);
                dataV1List.AddRange(helper.GenerateRandomArray(128 * 1024, c_V1Seed));

                dataV1 = dataV1List.ToArray();
            }

            // V2 のデータ (V1 の末尾にランダムなデータ列を追加)
            byte[] dataV2 = null;
            {
                var dataV2List = new List<byte>(dataV1);
                dataV2List.AddRange(helper.GenerateRandomArray(128 * 1024, c_V2Seed));

                dataV2 = dataV2List.ToArray();
            }

            // V3 のデータ (V2 の末尾にランダムなデータ列を追加)
            byte[] dataV3 = null;
            {
                var dataV3List = new List<byte>(dataV2);
                dataV3List.AddRange(helper.GenerateRandomArray(1023, c_V3Seed));
                dataV3 = dataV3List.ToArray();
            }

            var arg = new OptimizePatchTestArgument
            {
                TestName = MethodBase.GetCurrentMethod().Name,
                DataV0 = dataV0,
                DataV1 = dataV1,
                DataV2 = dataV2,
                DataV3 = dataV3,
                NeedsHtmlDocument = true,
            };
            ExecutionOptimizePatchCore(arg);
        }

        private void ExecutionOptimizePatchSame(string testName, bool isV1Same, bool isV2Same)
        {
            const int c_V0Seed = 0;
            const int c_V1Seed = 10;
            const int c_V2Seed = 20;

            var helper = new OptimizePatchTestHelper(this);

            // V0 のデータ
            var dataV0 = helper.GenerateRandomArray(10 * 1024 * 1024, c_V0Seed);
            // V1 のデータ (ランダムなデータ列)
            var dataV1 = helper.GenerateRandomArray(10 * 1024 * 1024, c_V1Seed);
            // V2 のデータ
            byte[] dataV2 = null;
            {
                //  V1 のデータの中央に新規のデータを挿入
                var dataV2List = new List<byte>(dataV1);
                dataV2List.InsertRange(dataV2List.Count / 2, helper.GenerateRandomArray(16 * 1024, c_V2Seed));
                dataV2 = dataV2List.ToArray();
            }

            // 入力されたフラグに応じて差分をなくす
            if (isV1Same)
            {
                dataV1 = dataV0;
            }

            if (isV2Same)
            {
                dataV2 = dataV0;
            }

            var arg = new OptimizePatchTestArgument
            {
                TestName = testName,
                DataV0 = dataV0,
                DataV1 = dataV1,
                DataV2 = dataV2,
            };
            ExecutionOptimizePatchCore(arg);
        }
        [TestMethod]
        public void TestExecutionOptimizePatchSameV0V1V2()
        {
            ExecutionOptimizePatchSame(MethodBase.GetCurrentMethod().Name, true, true);
        }
        [TestMethod]
        public void TestExecutionOptimizePatchSameV0V1()
        {
            ExecutionOptimizePatchSame(MethodBase.GetCurrentMethod().Name, true, false);
        }
        [TestMethod]
        public void TestExecutionOptimizePatchSameV0V2()
        {
            ExecutionOptimizePatchSame(MethodBase.GetCurrentMethod().Name, false, true);
        }
        [TestMethod]
        public void TestExecutionOptimizePatchAlreadyOptimizedError()
        {
            var helper = new OptimizePatchTestHelper(this);

            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var iconDir = testSourceDir + "\\Icon";
            var outputDirRoot = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var keyConfigFile = testSourceDir + "\\AuthoringToolsTest.keyconfig.xml";
            var metaFile = testSourceDir + "\\ApplicationMeta\\describe_all.nmeta";
            var metaFileV1 = outputDirRoot + "\\describe_all_v1.nmeta";
            var metaFileV2 = outputDirRoot + "\\describe_all_v2.nmeta";

            // 別のテストのlogoデータを流用
            var logoFile = testPath.GetSigloRoot() + "\\Tests\\Patch\\NactBuild\\logo.png";

            const int c_V1Seed = 10;
            const int c_V2Seed = 20;

            SafeDeleteDirectory(outputDirRoot);

            // V0 のデータ
            var dataV0 = new byte[] { 0, 1, 2 };
            // V1 のデータ (ランダムなデータ列)
            var dataV1 = helper.GenerateRandomArray(1 * 1024 * 1024, c_V1Seed);
            // V2 のデータ
            byte[] dataV2 = null;
            {
                //  V1 のデータの中央に新規のデータを挿入
                var dataV2List = new List<byte>(dataV1);
                dataV2List.InsertRange(dataV2List.Count / 2, helper.GenerateRandomArray(16 * 1024, c_V2Seed));
                dataV2 = dataV2List.ToArray();
            }

            var outputNspPathV0 = outputDirRoot + "\\v0.nsp";
            {
                var outputDir = outputDirRoot + "\\v0";
                var logoFileV0 = logoFile;
                // v0 ROM(元ROM)を作成
                helper.MakeNsp(outputNspPathV0, outputDir, iconDir, metaFile, descFile, logoFileV0, dataV0, false, keyConfigFile, keyIndex: 0, keyGeration: NintendoContentFileSystemMetaConstant.CurrentKeyGeneration, hasTicket: false);
            }

            var outputNspPathV1 = outputDirRoot + "\\v1.nsp";
            var outputNspPathV1Patch = outputDirRoot + "\\v1_patch.nsp";
            {
                var outputDir = outputDirRoot + "\\v1";
                var logoFileV1 = logoFile;
                MakeSpecifiedVersionMetaFile(metaFile, metaFileV1, 1);
                // v1 ROM、v0-v1パッチを作成
                helper.MakeNsp(outputNspPathV1, outputDir, iconDir, metaFile, descFile, logoFileV1, dataV1, false, keyConfigFile, keyIndex: 0, keyGeration: NintendoContentFileSystemMetaConstant.CurrentKeyGeneration, hasTicket: false);
                helper.MakePatch(outputNspPathV1Patch, outputNspPathV0, outputNspPathV1, metaFileV1, descFile, minimumMatchingSize: 0, stronglyOptimizeSize: 0, keyConfigFile: keyConfigFile, isOutputBuildLog: false);
            }

            var outputNspPathV2 = outputDirRoot + "\\v2.nsp";
            var outputNspPathV2Patch = outputDirRoot + "\\v2_patch.nsp";
            {
                var outputDir = outputDirRoot + "\\v2";
                var logoFileV2 = logoFile;
                MakeSpecifiedVersionMetaFile(metaFile, metaFileV2, 2);
                // v2 ROM、v0-v2パッチを作成
                helper.MakeNsp(outputNspPathV2, outputDir, iconDir, metaFile, descFile, logoFileV2, dataV2, false, keyConfigFile, keyIndex: 0, keyGeration: NintendoContentFileSystemMetaConstant.CurrentKeyGeneration, hasTicket: false);
                helper.MakePatch(outputNspPathV2Patch, outputNspPathV0, outputNspPathV2, metaFileV2, descFile, minimumMatchingSize: 0, stronglyOptimizeSize: 0, keyConfigFile: keyConfigFile, isOutputBuildLog: false);
            }

            // v0-v2 パッチを最適化（v0-v1パッチとの差分を減らす）
            var outputNspPathV2PatchIsc = outputDirRoot + "\\v2_patch_isc.nsp";
            helper.OptimizePatch(outputNspPathV2PatchIsc, outputNspPathV1Patch, outputNspPathV2Patch, descFile, defragmentSize: 0, stronglyOptimizeSize: 0, keyConfigFile: keyConfigFile, isOutputBuildLog: false);
            VerifyPatch(outputNspPathV2PatchIsc, outputNspPathV1Patch, keyConfigFile);

            try
            {
                // 最適化済みパッチを最適化しようとするとエラー
                var outputNspPathV2PatchIscFailed = outputDirRoot + "\\v2_patch_isc_failed.nsp";
                helper.OptimizePatch(outputNspPathV2PatchIscFailed, outputNspPathV1Patch, outputNspPathV2PatchIsc, descFile, defragmentSize: 0, stronglyOptimizeSize: 0, keyConfigFile: keyConfigFile, isOutputBuildLog: false);

                // エラーが起こるのでここには来ないはず
                throw new Exception("Already optimized error is not occured.");
            }
            catch (Exception e)
            {
                Assert.IsTrue(e.Message.Contains("Current patch is already optimized."));
            }
        }

        [TestMethod]
        public void TestExecutionOptimizePatchWithHtmlDocumentAndLegalInformation()
        {
            var arg = new OptimizePatchTestArgument
            {
                TestName = MethodBase.GetCurrentMethod().Name,
                NeedsHtmlDocument = true,
            };
            ExecutionOptimizePatch(arg, false);
        }
        private void ExecutionOptimizePatchNoData(string testName, bool[] isUsedData, bool[] isUsedHtmlDocument)
        {
            var helper = new OptimizePatchTestHelper(this);

            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var iconDir = testSourceDir + "\\Icon";
            var outputDirRoot = TestEnvironment.GetOutputPath(testName);
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";
            var keyConfigFile = testSourceDir + "\\AuthoringToolsTest.keyconfig.xml";
            var metaFile = testSourceDir + "\\ApplicationMeta\\describe_all.nmeta";
            var metaFileV1 = outputDirRoot + "\\describe_all_v1.nmeta";
            var metaFileV2 = outputDirRoot + "\\describe_all_v2.nmeta";
            var metaFileV3 = outputDirRoot + "\\describe_all_v3.nmeta";

            // 別のテストのlogoデータを流用
            var logoFile = testPath.GetSigloRoot() + "\\Tests\\Patch\\NactBuild\\logo.png";

            SafeDeleteDirectory(outputDirRoot);

            const int c_V1Seed = 10;
            const int c_V2Seed = 20;

            // V0 のデータ
            byte[] dataV0 = null;
            if (isUsedData[0])
            {
                dataV0 = new byte[] { 0, 1, 2 };
            }
            // V1 のデータ (ランダムなデータ列)
            byte[] dataV1 = null;
            if (isUsedData[1])
            {
                dataV1 = helper.GenerateRandomArray(1 * 1024 * 1024, c_V1Seed);
            }
            // V2 のデータ
            byte[] dataV2 = null;
            if (isUsedData[2])
            {
                dataV2 = helper.GenerateRandomArray(2 * 1024 * 1024, c_V2Seed);
            }

            var outputNspPathV0 = outputDirRoot + "\\v0.nsp";
            {
                var outputDir = outputDirRoot + "\\v0";
                string logoFileV0 = null;
                // v0 ROM(元ROM)を作成
                helper.MakeNsp(outputNspPathV0, outputDir, iconDir, metaFile, descFile, logoFileV0, dataV0, isUsedHtmlDocument[0], keyConfigFile, keyIndex: 0, keyGeration: NintendoContentFileSystemMetaConstant.CurrentKeyGeneration, hasTicket: false);
            }

            var outputNspPathV1 = outputDirRoot + "\\v1.nsp";
            var outputNspPathV1Patch = outputDirRoot + "\\v1_patch.nsp";
            {
                var outputDir = outputDirRoot + "\\v1";
                string logoFileV1 = null;
                MakeSpecifiedVersionMetaFile(metaFile, metaFileV1, 1);
                // v1 ROM、v0-v1パッチを作成
                helper.MakeNsp(outputNspPathV1, outputDir, iconDir, metaFile, descFile, logoFileV1, dataV1, isUsedHtmlDocument[1], keyConfigFile, keyIndex: 0, keyGeration: NintendoContentFileSystemMetaConstant.CurrentKeyGeneration, hasTicket: false);
                helper.MakePatch(outputNspPathV1Patch, outputNspPathV0, outputNspPathV1, metaFileV1, descFile, minimumMatchingSize: 0, stronglyOptimizeSize: 0, keyConfigFile: keyConfigFile, isOutputBuildLog: false);
            }

            var outputNspPathV2 = outputDirRoot + "\\v2.nsp";
            var outputNspPathV2Patch = outputDirRoot + "\\v2_patch.nsp";
            {
                var outputDir = outputDirRoot + "\\v2";
                string logoFileV2 = null;
                MakeSpecifiedVersionMetaFile(metaFile, metaFileV2, 2);
                // v2 ROM、v0-v2パッチを作成
                helper.MakeNsp(outputNspPathV2, outputDir, iconDir, metaFile, descFile, logoFileV2, dataV2, isUsedHtmlDocument[2], keyConfigFile, keyIndex: 0, keyGeration: NintendoContentFileSystemMetaConstant.CurrentKeyGeneration, hasTicket: false);
                helper.MakePatch(outputNspPathV2Patch, outputNspPathV0, outputNspPathV2, metaFileV2, descFile, minimumMatchingSize: 0, stronglyOptimizeSize: 0, keyConfigFile: keyConfigFile, isOutputBuildLog: false);
            }

            // v0-v2 パッチを最適化（v0-v1パッチとの差分を減らす）
            var outputNspPathV2PatchIsc = outputDirRoot + "\\v2_patch_isc.nsp";
            helper.OptimizePatch(outputNspPathV2PatchIsc, outputNspPathV1Patch, outputNspPathV2Patch, descFile, defragmentSize: 0, stronglyOptimizeSize: 0, keyConfigFile: keyConfigFile, isOutputBuildLog: false);
            CheckPatchData(keyConfigFile, outputNspPathV2PatchIsc, isUsedHtmlDocument[2], isUsedHtmlDocument[2], isUsedData[2]);
            VerifyPatch(outputNspPathV2PatchIsc, outputNspPathV1Patch, keyConfigFile);
        }
        [TestMethod]
        public void TestExecutionOptimizePatchDataAdditionAtV1()
        {
            bool[] isUsedData = { false, true, true };
            bool[] isUsedHtmlDocument = { true, true, true };
            ExecutionOptimizePatchNoData(MethodBase.GetCurrentMethod().Name, isUsedData, isUsedHtmlDocument);
        }
        [TestMethod]
        public void TestExecutionOptimizePatchDataAdditionAtV2()
        {
            bool[] isUsedData = { false, false, true };
            bool[] isUsedHtmlDocument = { true, true, true };
            ExecutionOptimizePatchNoData(MethodBase.GetCurrentMethod().Name, isUsedData, isUsedHtmlDocument);
        }
        [TestMethod]
        public void TestExecutionOptimizePatchDataRemovalAtV1()
        {
            bool[] isUsedData = { true, false, false };
            bool[] isUsedHtmlDocument = { true, true, true };
            ExecutionOptimizePatchNoData(MethodBase.GetCurrentMethod().Name, isUsedData, isUsedHtmlDocument);
        }
        [TestMethod]
        public void TestExecutionOptimizePatchDataRemovalAtV2()
        {
            bool[] isUsedData = { true, true, false };
            bool[] isUsedHtmlDocument = { true, true, true };
            ExecutionOptimizePatchNoData(MethodBase.GetCurrentMethod().Name, isUsedData, isUsedHtmlDocument);
        }
        [TestMethod]
        public void TestExecutionOptimizePatchHtmlDocumentAdditionAtV1()
        {
            bool[] isUsedData = { true, true, true };
            bool[] isUsedHtmlDocument = { false, true, true };
            ExecutionOptimizePatchNoData(MethodBase.GetCurrentMethod().Name, isUsedData, isUsedHtmlDocument);
        }
        [TestMethod]
        public void TestExecutionOptimizePatchHtmlDocumentAdditionAtV2()
        {
            bool[] isUsedData = { true, true, true };
            bool[] isUsedHtmlDocument = { false, false, true };
            ExecutionOptimizePatchNoData(MethodBase.GetCurrentMethod().Name, isUsedData, isUsedHtmlDocument);
        }
        [TestMethod]
        public void TestExecutionOptimizePatchHtmlDocumentRemovalAtV1()
        {
            bool[] isUsedData = { true, true, true };
            bool[] isUsedHtmlDocument = { true, false, false };
            ExecutionOptimizePatchNoData(MethodBase.GetCurrentMethod().Name, isUsedData, isUsedHtmlDocument);
        }
        [TestMethod]
        public void TestExecutionOptimizePatchHtmlDocumentRemovalAtV2()
        {
            bool[] isUsedData = { true, true, true };
            bool[] isUsedHtmlDocument = { true, true, false };
            ExecutionOptimizePatchNoData(MethodBase.GetCurrentMethod().Name, isUsedData, isUsedHtmlDocument);
        }

        [TestMethod]
        public void TestExecutionSavePatchBuildLog()
        {
            var arg = new OptimizePatchTestArgument
            {
                TestName = MethodBase.GetCurrentMethod().Name,
                IsOutputBuildLog = true,
            };
            ExecutionOptimizePatch(arg, true);

            var outputDirRoot = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);

            int fileCount;
            fileCount = Directory.EnumerateFiles(outputDirRoot, "v1_patch.*.txt").Count() +
                        Directory.EnumerateFiles(outputDirRoot, "v1_patch.*.csv").Count();
            Assert.AreEqual(2, fileCount);

            fileCount = Directory.EnumerateFiles(outputDirRoot, "v2_patch_isc.*.txt").Count() +
                        Directory.EnumerateFiles(outputDirRoot, "v2_patch_isc.*.csv").Count();
            Assert.AreEqual(3, fileCount);

            fileCount = Directory.EnumerateFiles(outputDirRoot, "v3_patch_isc.*.txt").Count() +
                        Directory.EnumerateFiles(outputDirRoot, "v3_patch_isc.*.csv").Count();
            Assert.AreEqual(3, fileCount);
        }

        [TestMethod]
        public void TestExecutionDefragmentPatch()
        {
            const int c_V1Seed = 10;
            const int c_V2Seed = 20;

            var helper = new OptimizePatchTestHelper(this);

            var dataV0 = new byte[] { 0, 1, 2 };
            var dataV1 = helper.GenerateRandomArray(1 * 1024 * 1024, c_V1Seed);

            byte[] dataV2 = new byte[dataV1.Length];
            dataV1.CopyTo(dataV2, 0);
            {
                var random = helper.GenerateRandomArray(512 * 1024, c_V2Seed);
                Array.Copy(random, 0, dataV2, 0, random.Length);
            }

            byte[] dataV3 = new byte[dataV2.Length - 512 * 1024];
            Array.Copy(dataV2, 0, dataV3, 0, dataV3.Length - 256 * 1024);
            Array.Copy(dataV2, dataV2.Length - 256 * 1024, dataV3, dataV3.Length - 256 * 1024, 256 * 1024);

            var arg = new OptimizePatchTestArgument
            {
                TestName = MethodBase.GetCurrentMethod().Name,
                DataV0 = dataV0,
                DataV1 = dataV1,
                DataV2 = dataV2,
                DataV3 = dataV3,
                IsOutputBuildLog = true,
                DefragmentSize = 512 * 1024,
            };
            ExecutionOptimizePatchCore(arg);
        }

        [TestMethod]
        public void TestExecutionMakePatchMinimumRegionSize()
        {
            var arg = new OptimizePatchTestArgument
            {
                TestName = MethodBase.GetCurrentMethod().Name,
                MinimumMatchingSize = IndirectStorageStream.RegionSizeMin * 2,
                DefragmentSize = IndirectStorageStream.RegionSizeMin * 2,
            };
            ExecutionOptimizePatch(arg, true);
        }

        [TestMethod]
        public void TestExecutionMakePatchSmallDeltaSize()
        {
            const int RegionSize = IndirectStorageSource.MinimumRegionSize;
            const int DataSize = 32 * RegionSize;
            const int CopyOffset = 8 * RegionSize + 1;
            const int CopySize = 8 * RegionSize;

            var helper = new OptimizePatchTestHelper(this);

            // V0 のデータ
            var dataV0 = new byte[DataSize];
            for(int i = 0; i < DataSize; )
            {
                for(int j = 0; j <= byte.MaxValue && i < DataSize; ++i, ++j)
                {
                    dataV0[i] = (byte)j;
                }
            }

            // V1 のデータ
            var dataV1 = new byte[DataSize];
            for (int i = 0; i < CopyOffset;)
            {
                for (int j = 0; j <= byte.MaxValue && i < CopyOffset; ++i, ++j)
                {
                    dataV1[i] = (byte)(byte.MaxValue - j);
                }
            }
            Array.Copy(dataV0, 4 * RegionSize, dataV1, CopyOffset, CopySize);
            for (int i = CopyOffset + CopySize; i < DataSize;)
            {
                for (int j = 0; j <= byte.MaxValue && i < DataSize; ++i, ++j)
                {
                    dataV1[i] = (byte)(byte.MaxValue - j);
                }
            }

            // V2 のデータ
            const int V2Seed = 20;
            byte[] dataV2 = null;
            {
                //  V1 のデータの中央に新規のデータを挿入

                var dataV2List = new List<byte>(dataV1);
                dataV2List.InsertRange(dataV2List.Count / 2 + 5, helper.GenerateRandomArray(4 * RegionSize, V2Seed));

                dataV2 = dataV2List.ToArray();
            }

            // DeltaSize = 16 で実行
            var arg = new OptimizePatchTestArgument
            {
                TestName = MethodBase.GetCurrentMethod().Name,
                MinimumMatchingSize = RegionSize * 2,
                DefragmentSize = RegionSize * 2,
                DataV0 = dataV0,
                DataV1 = dataV1,
                DataV2 = dataV2,
                IsOutputBuildLog = true,
            };
            ExecutionOptimizePatchCore(arg);

            string path1 = TestEnvironment.GetOutputPath(arg.TestName);

            // DeltaSize = 1 で実行
            arg.TestName += "2";
            arg.StronglyOptimizeSize = 1;
            ExecutionOptimizePatchCore(arg);

            string path2 = TestEnvironment.GetOutputPath(arg.TestName);

            var v1PatchPath1 = Directory.EnumerateFiles(path1, "*.nsp").Where(path => Path.GetFileName(path) == "v1_patch.nsp").Single();
            var v1PatchPath2 = Directory.EnumerateFiles(path2, "*.nsp").Where(path => Path.GetFileName(path) == "v1_patch.nsp").Single();
            var file1 = new FileInfo(v1PatchPath1);
            var file2 = new FileInfo(v1PatchPath2);
            Assert.IsTrue(file1.Length > file2.Length);
        }

        private void VerifyExtractDataForSparsifyNsp(string originalPath, string sparsePath, string patchPath, string basePatchPath, string keyConfigFile)
        {
            Assert.IsTrue(File.Exists(sparsePath));
            Assert.IsTrue((new FileInfo(sparsePath)).Length < (new FileInfo(originalPath)).Length);

            var extractBasePath = Path.Combine(Path.GetDirectoryName(originalPath), Path.GetFileNameWithoutExtension(sparsePath) + Path.GetFileNameWithoutExtension(patchPath));

            string errorMsg = string.Empty;

            // 元ロムと元パッチから extract
            var baseExtractDir = extractBasePath + "_extract_base";
            if (!string.IsNullOrEmpty(basePatchPath))
            {
                SafeDeleteDirectory(baseExtractDir);
                errorMsg = ExecuteProgram($"extract -o {baseExtractDir} --original {originalPath} {basePatchPath} --keyconfig {keyConfigFile}");
                Assert.IsTrue(string.IsNullOrEmpty(errorMsg), errorMsg);
            }

            // 元ロムと（スパース）パッチから extract
            var originalExtractDir = extractBasePath + "_extract_original";
            SafeDeleteDirectory(originalExtractDir);
            errorMsg = ExecuteProgram($"extract -o {originalExtractDir} --original {originalPath} {patchPath} --keyconfig {keyConfigFile}");
            Assert.IsTrue(string.IsNullOrEmpty(errorMsg), errorMsg);

            // スパースロムとパッチから extract
            var sparseExtractDir = extractBasePath + "_extract_sparse";
            SafeDeleteDirectory(sparseExtractDir);
            errorMsg = ExecuteProgram($"extract -o {sparseExtractDir} --original {sparsePath} {patchPath} --keyconfig {keyConfigFile}");
            Assert.IsTrue(string.IsNullOrEmpty(errorMsg), errorMsg);

            var sparseFiles = Directory.EnumerateFiles(sparseExtractDir, "*", SearchOption.AllDirectories);

            // extract_original と extract_sparse は同じパッチから extract しているのでファイル名が完全一致する
            foreach (var path1 in Directory.EnumerateFiles(originalExtractDir, "*", SearchOption.AllDirectories))
            {
                string path2 = sparseFiles.Where(value => Path.GetFileName(path1) == Path.GetFileName(value)).Single();
                Assert.IsTrue(File.ReadAllBytes(path1).SequenceEqual(File.ReadAllBytes(path2)));
            }

            // extract_base と extract_sparse はパッチが違うので 32 文字のハッシュ名を削除して比較してみる
            if (!string.IsNullOrEmpty(basePatchPath))
            {
                var hashName = new Regex("[0-9a-fA-F]{32}");

                foreach (var path1 in Directory.EnumerateFiles(baseExtractDir, "*", SearchOption.AllDirectories))
                {
                    // .cnmt は内容が異なるので確認しない
                    if (path1.IndexOf(".cnmt") < 0)
                    {
                        string path2 = null;
                        var list = sparseFiles.Where(value => Path.GetFileName(path1) == Path.GetFileName(value));
                        switch (list.Count())
                        {
                            case 0:
                                path2 = sparseFiles.Where(
                                    value => hashName.Replace(Path.GetFileName(path1), "") == hashName.Replace(Path.GetFileName(value), "")
                                ).Single();
                                break;

                            case 1:
                                path2 = list.Single();
                                break;

                            default:
                                throw new InvalidOperationException();
                        }
                        Assert.IsTrue(File.ReadAllBytes(path1).SequenceEqual(File.ReadAllBytes(path2)), string.Format("\"{0}\" not equals \"{1}\"", path1, path2));
                    }
                }
            }
        }

        private void VerifySparseGenerationForSparsifyNsp(string nspPath, string keyConfigPath, uint programCode, uint programData)
        {
            using (var nsp = new NintendoSubmissionPackageReader(nspPath))
            {
                var config = new AuthoringConfiguration();
                config.KeyConfigFilePath = keyConfigPath;
                var model = GetContentMetaModel(nsp);

                using (var reader = GetContentArchiveReader(nsp, model, "Program", config.GetKeyConfiguration()))
                {
                    Assert.IsTrue(reader.HasFsInfo(0));
                    {
                        var info = reader.GetFsHeaderInfo(0);
                        Assert.AreEqual(programCode << 16, info.GetSparseGeneration());
                    }
                    Assert.IsTrue(reader.HasFsInfo(1));
                    {
                        var info = reader.GetFsHeaderInfo(1);
                        Assert.AreEqual(programData << 16, info.GetSparseGeneration());
                    }
                }
            }
        }

        private void VerifyEncryptedKeyForSparsifyNsp(string originalNspPath, string sparseNspPath, string keyConfigPath)
        {
            using (var originalNsp = new NintendoSubmissionPackageReader(originalNspPath))
            using (var sparseNsp = new NintendoSubmissionPackageReader(sparseNspPath))
            {
                var config = new AuthoringConfiguration();
                config.KeyConfigFilePath = keyConfigPath;

                var originalModel = GetContentMetaModel(originalNsp);
                var sparseModel = GetContentMetaModel(sparseNsp);

                using (var originalReader = GetContentArchiveReader(originalNsp, originalModel, NintendoContentMetaConstant.ContentTypeProgram, config.GetKeyConfiguration()))
                using (var sparseReader = GetContentArchiveReader(sparseNsp, sparseModel, NintendoContentMetaConstant.ContentTypeProgram, config.GetKeyConfiguration()))
                {
                    Assert.AreEqual(originalReader.GetKeyIndex(), sparseReader.GetKeyIndex());
                    Assert.AreEqual(originalReader.GetKeyGeneration(), sparseReader.GetKeyGeneration());

                    var index = (int)NintendoContentArchiveEncryptionKeyIndex.AesCtr;
                    Assert.IsTrue(originalReader.GetInternalKey(index).SequenceEqual(sparseReader.GetInternalKey(index)));
                }
            }
        }

        private string MakeSparsePatch(int version, string outputPath, string originalPath, string currentPatchPath, string previousPatchPath, string descFilePath, int blockSize, int eraseSize, string keyConfigFile)
        {
            var inputDir = Path.GetDirectoryName(currentPatchPath);
            var metaFilePath = Path.Combine(inputDir, $"describe_all_v{version}.nmeta");
            var currentPath = Path.Combine(inputDir, $"v{version}.nsp");

            var args = $"makepatch -o {outputPath} --meta {metaFilePath} --desc {descFilePath} --original {originalPath} --current {currentPath} --previous {previousPatchPath} --keyconfig {keyConfigFile} --save-patch-build-log";
            if (0 < blockSize)
            {
                args += $" --do-application-compaction --compaction-block-size {blockSize} --compaction-erase-size {eraseSize}";
            }

            // パッチを作成する
            var errorMsg = ExecuteProgram(args);
            Assert.IsTrue(string.IsNullOrEmpty(errorMsg), errorMsg);
            Assert.IsTrue(File.Exists(outputPath));
            VerifyNsp(outputPath, keyConfigFile);

            return outputPath.Replace(".nsp", ".original.nsp");
        }

        private struct SparsifyNspRomData
        {
            public byte[] V0;
            public byte[] V1;
            public byte[] V2;

            public static SparsifyNspRomData Create(OptimizePatchTestHelper helper, bool needsV2)
            {
                const int c_V0Seed = 0;
                const int c_V2Seed = 20;

                // V0 のデータ (ランダムなデータ列)
                var dataV0 = helper.GenerateRandomArray(1024 * 1024, c_V0Seed);

                // Item1 = offset, Item2 = size
                var ranges = new Tuple<int, int>[4]
                {
                    Tuple.Create(  5 * 1024,  64 * 1024),
                    Tuple.Create(149 * 1024,  40 * 1024),
                    Tuple.Create(351 * 1024, 101 * 1024),
                    Tuple.Create(700 * 1024,  88 * 1024),
                };

                // V1 のデータ
                var dataV1 = new byte[dataV0.Length];
                dataV0.CopyTo(dataV1, 0);
                for (int i = 0; i < ranges.Length; ++i)
                {
                    var value = (byte)(i + 1);
                    var offset = ranges[i].Item1;
                    var size = ranges[i].Item2;
                    for (int j = 0; j < size; ++j)
                    {
                        dataV1[offset + j] = value;
                    }
                }

                // V2 のデータ
                byte[] dataV2 = null;
                if (needsV2)
                {
                    //  V1 のデータの中央に新規のデータを挿入
                    {
                        var dataV2List = new List<byte>(dataV1);
                        dataV2List.InsertRange(dataV2List.Count / 2, helper.GenerateRandomArray(16 * 1024, c_V2Seed));

                        dataV2 = dataV2List.ToArray();
                    }
                    // V1 で書き換えた部分を V0 のデータに戻す
                    {
                        for (int j = 0; j < ranges[0].Item2; ++j)
                        {
                            dataV2[ranges[0].Item1 + j] = dataV0[ranges[0].Item1 + j];
                        }
                    }
                }

                return new SparsifyNspRomData() { V0 = dataV0, V1 = dataV1, V2 = dataV2 };
            }
        }

        public void TestExecutionSparsifyNspImpl(bool hasTicket)
        {
            var helper = new OptimizePatchTestHelper(this);
            var data = SparsifyNspRomData.Create(helper, true);

            OptimizePatchTestArgument arg = new OptimizePatchTestArgument()
            {
                TestName = MethodBase.GetCurrentMethod().Name,
                MinimumMatchingSize = IndirectStorageSource.MinimumRegionSize * 2,
                DataV0 = data.V0,
                DataV1 = data.V1,
                DataV2 = data.V2,
                NeedsHtmlDocument = true,
                IsOutputBuildLog = true,
                KeyIndex = 2, // 鍵を乱数にする
                GetKeyGeneration = (v) => v >= 1 ? (v + 1) : v, // 鍵世代 1 は存在しない
                HasTicket = hasTicket,
            };

            ExecutionOptimizePatchCore(arg);

            var dir = Path.GetDirectoryName(arg.OriginalNspPath);
            var V0 = Path.Combine(dir, "_V0.nsp");
            File.Move(arg.OriginalNspPath, V0);
            var V1p = Path.Combine(dir, "_V1p.nsp");
            File.Move(arg.V1PatchNspPath, V1p);
            var V2p = Path.Combine(dir, "_V2p.nsp");
            File.Move(arg.V2PatchNspPath, V2p);

            // V0/V1p でスパース化
            var V0s1 = Path.Combine(dir, "_V0s1.nsp");
            var V1ps1 = Path.Combine(dir, "_V1ps1.nsp");
            helper.SparsifyPatch(V0s1, V1ps1, V0, V1p, arg.KeyConfigFilePath);
            VerifySparseGenerationForSparsifyNsp(V0s1, arg.KeyConfigFilePath, programCode: 1, programData: 1);
            VerifyEncryptedKeyForSparsifyNsp(V0, V0s1, arg.KeyConfigFilePath);
            VerifyExtractDataForSparsifyNsp(V0, V0s1, V1p, null, arg.KeyConfigFilePath);

            // v0/v2p でスパース化
            var V0s2 = Path.Combine(dir, "_V0s2.nsp");
            var V2ps2 = Path.Combine(dir, "_V2ps2.nsp");
            helper.SparsifyPatch(V0s2, V2ps2, V0, V2p, arg.KeyConfigFilePath);
            VerifySparseGenerationForSparsifyNsp(V0s2, arg.KeyConfigFilePath, programCode: 1, programData: 1);
            VerifyEncryptedKeyForSparsifyNsp(V0, V0s2, arg.KeyConfigFilePath);
            VerifyExtractDataForSparsifyNsp(V0, V0s2, V2p, null, arg.KeyConfigFilePath);

            // V0/V2/V1ps1 でパッチ作成
            {
                var V2ps12 = Path.Combine(dir, "_V2ps12.nsp");
                var V0s12Raw = MakeSparsePatch(2, V2ps12, V0, arg.V2PatchNspPath, V1ps1, arg.DescFilePath,  blockSize: 8, eraseSize: 8, keyConfigFile: arg.KeyConfigFilePath);
                Assert.IsTrue(File.Exists(V0s12Raw));
                var V0s12 = Path.Combine(Path.GetDirectoryName(V0s12Raw), "_V0s12.nsp");
                File.Move(V0s12Raw, V0s12);

                VerifySparseGenerationForSparsifyNsp(V0s12, arg.KeyConfigFilePath, programCode: 2, programData: 2);
                VerifyExtractDataForSparsifyNsp(V0, V0s1, V2ps12, V2p, arg.KeyConfigFilePath);
                VerifyExtractDataForSparsifyNsp(V0, V0s12, V2ps12, V2p, arg.KeyConfigFilePath);
                VerifyEncryptedKeyForSparsifyNsp(V0, V0s12, arg.KeyConfigFilePath);
            }
        }

        [TestMethod]
        public void TestExecutionSparsifyNspWithoutTicket()
        {
            TestExecutionSparsifyNspImpl(hasTicket: false);
        }

        [TestMethod]
        public void TestExecutionSparsifyNspWithTicket()
        {
            TestExecutionSparsifyNspImpl(hasTicket: true);
        }

        [TestMethod]
        public void TestExecutionSparsifyNspV0V1SameData()
        {
            var helper = new OptimizePatchTestHelper(this);
            var data = SparsifyNspRomData.Create(helper, false);

            OptimizePatchTestArgument arg = new OptimizePatchTestArgument()
            {
                TestName = MethodBase.GetCurrentMethod().Name,
                MinimumMatchingSize = IndirectStorageSource.MinimumRegionSize * 2,
                DataV0 = data.V0,
                DataV1 = data.V0,
                DataV2 = data.V1,
                NeedsHtmlDocument = true,
                IsOutputBuildLog = true
            };

            ExecutionOptimizePatchCore(arg);

            var dir = Path.GetDirectoryName(arg.OriginalNspPath);
            var V0 = Path.Combine(dir, "_V0.nsp");
            File.Move(arg.OriginalNspPath, V0);
            var V1p = Path.Combine(dir, "_V1p.nsp");
            File.Move(arg.V1PatchNspPath, V1p);
            var V2p = Path.Combine(dir, "_V2p.nsp");
            File.Move(arg.V2PatchNspPath, V2p);

            // V0/V1p でスパース化
            var V0s1 = Path.Combine(dir, "_V0s1.nsp");
            var V1ps1 = Path.Combine(dir, "_V1ps1.nsp");
            helper.SparsifyPatch(V0s1, V1ps1, V0, V1p, arg.KeyConfigFilePath);
            // rom に変化がないので data 領域はスパース化されない
            VerifySparseGenerationForSparsifyNsp(V0s1, arg.KeyConfigFilePath, programCode: 1, programData: 0);
            VerifyExtractDataForSparsifyNsp(V0, V0s1, V1p, null, arg.KeyConfigFilePath);

            // V0/V2/V1ps1 でパッチ作成
            {
                var V2ps12 = Path.Combine(dir, "_V2ps12.nsp");
                var V0s12Raw = MakeSparsePatch(2, V2ps12, V0, arg.V2PatchNspPath, V1ps1, arg.DescFilePath, blockSize: 8, eraseSize: 8, keyConfigFile: arg.KeyConfigFilePath);
                Assert.IsTrue(File.Exists(V0s12Raw));
                var V0s12 = Path.Combine(Path.GetDirectoryName(V0s12Raw), "_V0s12.nsp");
                File.Move(V0s12Raw, V0s12);

                VerifySparseGenerationForSparsifyNsp(V0s12, arg.KeyConfigFilePath, programCode: 2, programData: 1);
                VerifyExtractDataForSparsifyNsp(V0, V0s1, V2ps12, V2p, arg.KeyConfigFilePath);
                VerifyExtractDataForSparsifyNsp(V0, V0s12, V2ps12, V2p, arg.KeyConfigFilePath);
            }
        }

        [TestMethod]
        public void TestExecutionSparsifyNspV1V2SameData()
        {
            var helper = new OptimizePatchTestHelper(this);
            var data = SparsifyNspRomData.Create(helper, false);

            OptimizePatchTestArgument arg = new OptimizePatchTestArgument()
            {
                TestName = MethodBase.GetCurrentMethod().Name,
                MinimumMatchingSize = IndirectStorageSource.MinimumRegionSize * 2,
                DataV0 = data.V0,
                DataV1 = data.V1,
                DataV2 = data.V1,
                NeedsHtmlDocument = true,
                IsOutputBuildLog = true
            };

            ExecutionOptimizePatchCore(arg);

            var dir = Path.GetDirectoryName(arg.OriginalNspPath);
            var V0 = Path.Combine(dir, "_V0.nsp");
            File.Move(arg.OriginalNspPath, V0);
            var V1p = Path.Combine(dir, "_V1p.nsp");
            File.Move(arg.V1PatchNspPath, V1p);
            var V2p = Path.Combine(dir, "_V2p.nsp");
            File.Move(arg.V2PatchNspPath, V2p);

            // V0/V1p でスパース化
            var V0s1 = Path.Combine(dir, "_V0s1.nsp");
            var V1ps1 = Path.Combine(dir, "_V1ps1.nsp");
            helper.SparsifyPatch(V0s1, V1ps1, V0, V1p, arg.KeyConfigFilePath);
            VerifySparseGenerationForSparsifyNsp(V0s1, arg.KeyConfigFilePath, programCode: 1, programData: 1);
            VerifyExtractDataForSparsifyNsp(V0, V0s1, V1p, null, arg.KeyConfigFilePath);

            // V0/V2/V1ps1 でパッチ作成
            {
                var V2ps12 = Path.Combine(dir, "_V2ps12.nsp");
                var V0s12Raw = MakeSparsePatch(2, V2ps12, V0, arg.V2PatchNspPath, V1ps1, arg.DescFilePath, blockSize: 8, eraseSize: 8, keyConfigFile: arg.KeyConfigFilePath);
                Assert.IsTrue(File.Exists(V0s12Raw));
                var V0s12 = Path.Combine(Path.GetDirectoryName(V0s12Raw), "_V0s12.nsp");
                File.Move(V0s12Raw, V0s12);

                // rom に変化は無くても世代番号は更新される
                VerifySparseGenerationForSparsifyNsp(V0s12, arg.KeyConfigFilePath, programCode: 2, programData: 2);
                VerifyExtractDataForSparsifyNsp(V0, V0s1, V2ps12, V2p, arg.KeyConfigFilePath);
                VerifyExtractDataForSparsifyNsp(V0, V0s12, V2ps12, V2p, arg.KeyConfigFilePath);
            }
        }

        [TestMethod]
        public void TestExecutionSparsifyNspInsertNormalPatch()
        {
            var helper = new OptimizePatchTestHelper(this);
            var data = SparsifyNspRomData.Create(helper, true);

            OptimizePatchTestArgument arg = new OptimizePatchTestArgument()
            {
                TestName = MethodBase.GetCurrentMethod().Name,
                MinimumMatchingSize = IndirectStorageSource.MinimumRegionSize * 2,
                DataV0 = data.V0,
                DataV1 = data.V1,
                DataV2 = data.V1,
                DataV3 = data.V2,
                NeedsHtmlDocument = true,
                IsOutputBuildLog = true
            };

            ExecutionOptimizePatchCore(arg);

            var dir = Path.GetDirectoryName(arg.OriginalNspPath);
            var V0 = Path.Combine(dir, "_V0.nsp");
            File.Move(arg.OriginalNspPath, V0);
            var V1p = Path.Combine(dir, "_V1p.nsp");
            File.Move(arg.V1PatchNspPath, V1p);
            var V2p = Path.Combine(dir, "_V2p.nsp");
            File.Move(arg.V2PatchNspPath, V2p);
            var V3p = Path.Combine(dir, "_V3p.nsp");
            File.Move(arg.V3PatchNspPath, V3p);

            // V0/V1p でスパース化
            var V0s1 = Path.Combine(dir, "_V0s1.nsp");
            var V1ps1 = Path.Combine(dir, "_V1ps1.nsp");
            helper.SparsifyPatch(V0s1, V1ps1, V0, V1p, arg.KeyConfigFilePath);
            VerifySparseGenerationForSparsifyNsp(V0s1, arg.KeyConfigFilePath, programCode: 1, programData: 1);
            VerifyExtractDataForSparsifyNsp(V0, V0s1, V1p, null, arg.KeyConfigFilePath);

            // V0/V2/V1ps1 でパッチ作成
            var V2ps1 = Path.Combine(dir, "_V2ps1.nsp");
            {
                var V0s12Raw = MakeSparsePatch(2, V2ps1, V0, arg.V2PatchNspPath, V1ps1, arg.DescFilePath, blockSize: 0, eraseSize: 0, keyConfigFile: arg.KeyConfigFilePath);
                Assert.IsFalse(File.Exists(V0s12Raw)); // スパース化ではないのでファイルはない
            }

            // V0/V3/V2ps1 でスパースパッチ作成
            var V3ps13 = Path.Combine(dir, "_V3ps13.nsp");
            {
                var V0s13Raw = MakeSparsePatch(3, V3ps13, V0, arg.V3PatchNspPath, V2ps1, arg.DescFilePath, blockSize: 8, eraseSize: 8, keyConfigFile: arg.KeyConfigFilePath);
                Assert.IsTrue(File.Exists(V0s13Raw));
                var V0s13 = Path.Combine(Path.GetDirectoryName(V0s13Raw), "_V0s13.nsp");
                File.Move(V0s13Raw, V0s13);

                // 通常のパッチ生成でスパース情報が引き継がれているか確認
                VerifySparseGenerationForSparsifyNsp(V0s13, arg.KeyConfigFilePath, programCode: 2, programData: 2);
                VerifyExtractDataForSparsifyNsp(V0, V0s1, V3ps13, V3p, arg.KeyConfigFilePath);
                VerifyExtractDataForSparsifyNsp(V0, V0s13, V3ps13, V3p, arg.KeyConfigFilePath);
            }
        }

        [TestMethod]
        public void TestExecutionSparsifyNspIncreaseEraseSize()
        {
            var helper = new OptimizePatchTestHelper(this);
            var data = SparsifyNspRomData.Create(helper, true);

            OptimizePatchTestArgument arg = new OptimizePatchTestArgument()
            {
                TestName = MethodBase.GetCurrentMethod().Name,
                MinimumMatchingSize = IndirectStorageSource.MinimumRegionSize * 2,
                DataV0 = data.V0,
                DataV1 = data.V1,
                DataV2 = data.V2,
                NeedsHtmlDocument = true,
                IsOutputBuildLog = true
            };

            ExecutionOptimizePatchCore(arg);

            var dir = Path.GetDirectoryName(arg.OriginalNspPath);
            var V0 = Path.Combine(dir, "_V0.nsp");
            File.Move(arg.OriginalNspPath, V0);
            var V1p = Path.Combine(dir, "_V1p.nsp");
            File.Move(arg.V1PatchNspPath, V1p);
            var V2p = Path.Combine(dir, "_V2p.nsp");
            File.Move(arg.V2PatchNspPath, V2p);

            // V0/V1p でスパース化
            var V0s1 = Path.Combine(dir, "_V0s1.nsp");
            var V1ps1 = Path.Combine(dir, "_V1ps1.nsp");
            helper.SparsifyPatch(V0s1, V1ps1, V0, V1p, arg.KeyConfigFilePath);
            VerifySparseGenerationForSparsifyNsp(V0s1, arg.KeyConfigFilePath, programCode: 1, programData: 1);
            VerifyExtractDataForSparsifyNsp(V0, V0s1, V1p, null, arg.KeyConfigFilePath);

            // V0/V2/V1ps1 でパッチ作成
            {
                var V2ps12 = Path.Combine(dir, "_V2ps12.nsp");
                // 直前のスパース化で削除された領域が適切に処理されるか確認
                var V0s12Raw = MakeSparsePatch(2, V2ps12, V0, arg.V2PatchNspPath, V1ps1, arg.DescFilePath, blockSize: 8, eraseSize: 128, keyConfigFile: arg.KeyConfigFilePath);
                Assert.IsTrue(File.Exists(V0s12Raw));
                var V0s12 = Path.Combine(Path.GetDirectoryName(V0s12Raw), "_V0s12.nsp");
                File.Move(V0s12Raw, V0s12);

                VerifySparseGenerationForSparsifyNsp(V0s12, arg.KeyConfigFilePath, programCode: 2, programData: 2);
                VerifyExtractDataForSparsifyNsp(V0, V0s1, V2ps12, V2p, arg.KeyConfigFilePath);
                VerifyExtractDataForSparsifyNsp(V0, V0s12, V2ps12, V2p, arg.KeyConfigFilePath);
            }
        }

        [TestMethod]
        public void TestExecutionSparsifyNspModify()
        {
            var helper = new OptimizePatchTestHelper(this);
            var data = SparsifyNspRomData.Create(helper, false);

            OptimizePatchTestArgument arg = new OptimizePatchTestArgument()
            {
                TestName = MethodBase.GetCurrentMethod().Name,
                MinimumMatchingSize = IndirectStorageSource.MinimumRegionSize * 2,
                DataV0 = data.V0,
                DataV1 = data.V1,
                DataV2 = data.V1,
                NeedsHtmlDocument = true,
                IsOutputBuildLog = true
            };

            ExecutionOptimizePatchCore(arg);

            var dir = Path.GetDirectoryName(arg.OriginalNspPath);
            var V0 = arg.OriginalNspPath;
            var V1p = arg.V1PatchNspPath;
            var V0s1 = Path.Combine(dir, "_V0s1.nsp");
            var V1ps1 = Path.Combine(dir, "_V1ps1.nsp");
            helper.SparsifyPatch(V0s1, V1ps1, V0, V1p, arg.KeyConfigFilePath);

            Func<string, List<string>> parseListStdOutput = listOutput =>
            {
                var msg = listOutput.Replace("\r", "");
                msg = Regex.Replace(msg, @"\s+\(\d+ byte\)[\r\n]+", "\n", RegexOptions.Singleline) + "\n";    // byte 数を削除
                msg = Regex.Replace(msg, @"-+[\r\n]+", "", RegexOptions.Singleline).Replace(Environment.NewLine, ""); // 区切り文字列を削除
                msg = Regex.Replace(msg, @"^\s+$[\r\n]*", "", RegexOptions.Multiline);    // 空白行を削除
                return new List<string>(msg.TrimEnd('\n').Split('\n'));
            };

            var dummyPath = Path.Combine(dir, "dummy.txt");
            using (var file = File.Create(dummyPath)) { }

            Action<string> replaceCheck = inputPath =>
            {
                var stdMsg = ExecuteProgram($"list {inputPath} --targetregex \".\" --keyconfig {arg.KeyConfigFilePath}", true);
                var entryList = parseListStdOutput(stdMsg);
                var entryPath = entryList.Where(path => path.IndexOf(".nca/") >= 0).First();
                var addedEntry = entryPath + "2.dat";
                var errorMsg = ExecuteProgram($"replace {inputPath} -o {dir}\\ --desc {arg.DescFilePath} add:{addedEntry} {dummyPath}");
                Assert.IsTrue(errorMsg.Contains("Modification of compacted application nsp is not supported."), errorMsg);
            };

            replaceCheck(V0s1); // スパースアプリの replace
            replaceCheck(V1ps1); // スパースパッチの replace
        }

        class CreateNspApplicationForDeltaOption
        {
            public string Type { get; set; } = "Application";
            public int Version { get; set; } = 0;
            public string KeyConfigFilePath { get; set; } = string.Empty;
            public string CodeDirectoryPath { get; set; } = string.Empty;
            public string DataDirectoryPath { get; set; } = string.Empty;
            public string HtmlDocumentDirectoryPath { get; set; } = null;
            public string LegalInformationDirectoryPath { get; set; } = null;
            public bool WillCleanDirectory { get; set; } = true;
            public Action<string> GenerateData { get; set; } = (s) => { };
            public bool HasIcon { get; set; } = true;

            internal static CreateNspApplicationForDeltaOption All(
                int version,
                string nspBaseDir,
                string keyConfigFile,
                Action<Stream> generateData)
            {
                return new CreateNspApplicationForDeltaOption
                {
                    Type = "Application",
                    Version = version,
                    KeyConfigFilePath = keyConfigFile,
                    CodeDirectoryPath = nspBaseDir + @"\code" + version,
                    DataDirectoryPath = nspBaseDir + @"\data" + version,
                    HtmlDocumentDirectoryPath = nspBaseDir + @"\data" + version,
                    LegalInformationDirectoryPath = nspBaseDir + @"\legal" + version,
                    GenerateData = MakeGenerateFile(generateData)
                };
            }

            internal static CreateNspApplicationForDeltaOption Patch(
                int version,
                string nspBaseDir,
                string keyConfigFile,
                Action<Stream> generateData)
            {
                return new CreateNspApplicationForDeltaOption
                {
                    Type = "Patch",
                    Version = version,
                    KeyConfigFilePath = keyConfigFile,
                    CodeDirectoryPath = nspBaseDir + @"\code" + version,
                    DataDirectoryPath = nspBaseDir + @"\data" + version,
                    GenerateData = MakeGenerateFile(generateData)
                };
            }

            internal static CreateNspApplicationForDeltaOption NoManualForSizeCheck(
                int version,
                string keyConfigFile,
                string nspBaseDir,
                bool willClean,
                Action<string> generateData)
            {
                return new CreateNspApplicationForDeltaOption
                {
                    Type = "Application",
                    Version = version,
                    KeyConfigFilePath = keyConfigFile,
                    CodeDirectoryPath = nspBaseDir + @"\code",
                    DataDirectoryPath = nspBaseDir + @"\data",
                    WillCleanDirectory = willClean,
                    GenerateData = generateData
                };
            }

            internal static CreateNspApplicationForDeltaOption NoManual(
                int version,
                string keyConfigFile,
                string nspBaseDir,
                bool willClean,
                Action<Stream> generateData)
            {
                return NoManualForSizeCheck(
                    version,
                    keyConfigFile,
                    nspBaseDir,
                    willClean,
                    MakeGenerateFile(generateData));
            }

            private static Action<string> MakeGenerateFile(Action<Stream> generateData)
            {
                return (dataDirectoryPath) =>
                {
                    var testFile = dataDirectoryPath + @"\test.bin";
                    using (var f = File.OpenWrite(testFile))
                    {
                        generateData(f);
                    }
                };
            }
        }

        private void CreateNspApplicationForDelta(string outputPath, CreateNspApplicationForDeltaOption option)
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";

            var iconDir = testSourceDir + "\\Icon";
            var iconPath = iconDir + "\\describe_all.bmp";

            if (!Directory.Exists(Path.GetDirectoryName(outputPath)))
            {
                Directory.CreateDirectory(Path.GetDirectoryName(outputPath));
            }
            string metaFile = testSourceDir + "\\ApplicationMeta\\describe_all.nmeta";
            string patchMetaFile = Path.GetDirectoryName(outputPath) + "\\" + Path.GetFileNameWithoutExtension(outputPath) + ".nmeta";
            MakeSpecifiedVersionMetaFile(metaFile, patchMetaFile, option.Version);

            {
                if (option.WillCleanDirectory)
                {
                    SafeDeleteDirectory(option.CodeDirectoryPath);
                }
                if (!Directory.Exists(option.CodeDirectoryPath))
                {
                    Directory.CreateDirectory(option.CodeDirectoryPath);
                }

                var testNpdm = option.CodeDirectoryPath + "\\main.npdm";
                MakeNpdm(testNpdm, patchMetaFile, descFile);
            }

            {
                if (option.WillCleanDirectory)
                {
                    SafeDeleteDirectory(option.DataDirectoryPath);
                }
                if (!Directory.Exists(option.DataDirectoryPath))
                {
                    Directory.CreateDirectory(option.DataDirectoryPath);
                }

                option.GenerateData(option.DataDirectoryPath);
            }

            if (option.LegalInformationDirectoryPath != null)
            {
                if (option.WillCleanDirectory)
                {
                    SafeDeleteDirectory(option.LegalInformationDirectoryPath);
                }
                if (!Directory.Exists(option.LegalInformationDirectoryPath))
                {
                    Directory.CreateDirectory(option.LegalInformationDirectoryPath);
                }
                CreateLegalInfoXml(option.LegalInformationDirectoryPath);
            }

            var errorMesg = ExecuteProgram(string.Format(
                "creatensp -o {0} --type {1} --meta {2} --desc {3} {4} {5} {6} {7} --keyconfig {8} --save-adf",
                outputPath,
                option.Type,
                patchMetaFile,
                descFile,
                string.Format("--program {0} {1}", option.CodeDirectoryPath, option.DataDirectoryPath),
                option.HtmlDocumentDirectoryPath != null ? "--html-document " + option.HtmlDocumentDirectoryPath : string.Empty,
                option.HasIcon ? string.Format("--icon AmericanEnglish {0} Japanese {0}", iconPath) : string.Empty,
                option.LegalInformationDirectoryPath != null ? "--legal-information-dir " + option.LegalInformationDirectoryPath : string.Empty,
                option.KeyConfigFilePath));
            Assert.AreEqual(string.Empty, errorMesg);
            VerifyNsp(outputPath, option.KeyConfigFilePath);
        }

        private void Create1stPatchForDelta(
            string patchPath,
            string keyConfigFile,
            string originalNsp,
            string currentNsp)
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var descFile = testPath.GetSigloRoot() + "\\Programs\\Iris\\Resources\\SpecFiles\\Application.desc";

            if (File.Exists(patchPath))
            {
                File.Delete(patchPath);
            }

            var errorMesg = ExecuteProgram(string.Format(
                "makepatch -o {0} --desc {1} --original {2} --current {3} --keyconfig {4}",
                patchPath,
                descFile,
                originalNsp,
                currentNsp,
                keyConfigFile));
            Assert.IsTrue(errorMesg == string.Empty, errorMesg);
            VerifyNsp(patchPath, keyConfigFile);
        }

        private void CreatePatch(string outputFile, string metaFile, string keyConfigFile)
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var outputDir = Path.GetDirectoryName(outputFile);

            var dataDir = Path.Combine(outputDir, "program");
            var dataFile = Path.Combine(dataDir, "data.dat");
            var descFile = Path.Combine(testPath.GetSigloRoot(), "Programs\\Iris\\Resources\\SpecFiles\\Application.desc");

            Directory.CreateDirectory(dataDir);
            using (FileStream stream = File.Create(dataFile))
            {
                int fileSize = 1024;
                byte[] data = new byte[fileSize];
                for (int i = 0; i < fileSize; i++)
                {
                    data[i] = (byte)i;
                }
                stream.Write(data, 0, fileSize);
            }

            var testNpdm =  Path.Combine(dataDir, "main.npdm");
            MakeNpdm(testNpdm, metaFile, descFile); // W/A: MakeMeta が PatchId に対応していない

            var error = ExecuteProgram(string.Format(
                "creatensp -o {0} --type Patch --meta {1} --desc {2} --program {3} --keyconfig {4} --save-adf",
                outputFile,
                metaFile,
                descFile,
                dataDir,
                keyConfigFile));
            Assert.IsTrue(error == string.Empty, error);
            Assert.IsTrue(File.Exists(outputFile));
            VerifyNsp(outputFile, keyConfigFile);
        }
    }
}
