﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Security.Cryptography;
using System.Xml.Serialization;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.Authoring.AuthoringLibrary;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace AuthoringToolsTest
{
    [TestClass]
    public class NintendoContentMetaTest
    {
        public TestContext TestContext { get; set; }
        public KeyConfiguration KeyConfig { get; set; }

        private RNGCryptoServiceProvider m_Rng;
        private Random m_Random;

        public NintendoContentMetaTest()
        {
            KeyConfig = new KeyConfiguration();
            m_Rng = new RNGCryptoServiceProvider();

            var randomSeed = Utils.GenerateRandomSeed();
            var random = new Random(randomSeed);
            Utils.WriteLine("NintendoContentMetaTest random seed is " + randomSeed);

            m_Random = new Random(randomSeed);
        }

        /*
          - 各 MetaType で正しく作成できること
          - 指定したハッシュが正しく埋め込まれること
          - Digest が正しく埋め込まれること
          - cnmt.nca のパラメータが正しく作成されること
          - XML の内容がバイナリと一致し、漏れがないこと
          - 製品化時に各 MetaType で正しく作成できること
          - 製品化時に指定したハッシュが正しく埋め込まれること
          - 製品化時に XML の内容がバイナリと一致し、漏れがないこと
          - 製品化時に Digest が引き継がれること
          - 製品化時に cnmt.nca のパラメータが正しく作成されること
        */

        private List<Tuple<ISource, NintendoContentDescriptor>> CreateTestContentDescriptorList(List<string> contentTypes)
        {
            var contentInfoList = new List<Tuple<ISource, NintendoContentDescriptor>>();
            foreach (var contentType in contentTypes)
            {
                var hash = new byte[32];
                {
                    m_Rng.GetBytes(hash);
                }
                ISource hashSource = new MemorySource(hash, 0, hash.Length);
                long size = 0;
                {
                    size = m_Random.Next(1, 256 * 1024 * 1024);
                }
                var info = new NintendoContentInfo(contentType, size, 0, 0);
                contentInfoList.Add(Tuple.Create(hashSource, new NintendoContentDescriptor { ContentInfo = info }));
            }
            return contentInfoList;
        }

        private void GetCnmtData(out byte[] baseData, out byte[] ncaData, out byte[] xmlData, NintendoContentMetaBase cnmtBase, KeyConfiguration keyConfig, bool isProdEncryption)
        {
            var cnmtArchiveSource = new NintendoContentMetaArchiveSource(cnmtBase, 0, KeyConfig, false, false, false);
            var memorySink = new MemorySink((int)cnmtArchiveSource.Size);
            var cnmtXmlSource = new NintendoContentMetaXmlSource(cnmtBase, new SinkLinkedSource(memorySink, new Sha256StreamHashSource(memorySink.ToSource())), cnmtArchiveSource.Size);

            {
                var ssd = new SourceSinkDriver();
                ssd.Add(cnmtArchiveSource, memorySink);
                ssd.Run();
            }

            baseData = cnmtBase.GetSource().PullData(0, (int)cnmtBase.GetSource().Size).Buffer.Array;
            ncaData = memorySink.ToSource().PullData(0, (int)cnmtArchiveSource.Size).Buffer.Array;
            xmlData = cnmtXmlSource.PullData(0, (int)cnmtXmlSource.Size).Buffer.Array;
        }

        private void CheckCnmt(byte[] baseData, string metaType, List<Tuple<ISource, NintendoContentDescriptor>> contentInfoList, bool checkDigest, UInt32 defaultVersion = 0)
        {
            var reader = new NintendoContentMetaReader(baseData);
            UInt64 correctId = 0x0005000C10000001;
            UInt32 correctVersion = defaultVersion;
            if (metaType == "Patch")
            {
                correctId = 0x0005000C10000002;
            }
            if (metaType == "AddOnContent")
            {
                correctId = 0x0005000c10001004;
                correctVersion = 0x00010000;
            }
            if (metaType == "SystemUpdate")
            {
                correctId = 0x0000000000000001;
            }
            if (metaType == "Delta")
            {
                correctId = 0x0005000c10000005;
            }

            Assert.IsTrue(reader.GetId() == correctId);
            Assert.IsTrue(reader.GetVersion() == correctVersion);

            if (metaType != "SystemUpdate")
            {
                var checkedContent = new Dictionary<string, bool>();
                foreach (var contentType in contentInfoList.Select(x => x.Item2.ContentInfo.Type))
                {
                    checkedContent.Add(contentType, false);
                }

                foreach (var contentDescriptor in reader.GetContentDescriptorList())
                {
                    var info = contentInfoList.Where(x => x.Item2.ContentInfo.Type == contentDescriptor.ContentInfo.Type).Single();
                    Assert.IsTrue(contentDescriptor.ContentInfo.Size == info.Item2.ContentInfo.Size);
                    var hash = info.Item1.PullData(0, (int)info.Item1.Size).Buffer.Array;
                    Assert.IsTrue(contentDescriptor.ContentInfo.Id.SequenceEqual(hash.Take(16)));
                    Assert.IsTrue(contentDescriptor.ContentInfo.Hash.SequenceEqual(hash));
                    checkedContent[contentDescriptor.ContentInfo.Type] = true;
                }
                Assert.IsTrue(checkedContent.Select(x => x.Value).ToList().Aggregate((ret, x) => ret &= x));

                Assert.IsTrue(reader.GetContentMetaInfoList().Count == 0);
            }
            else
            {
                Assert.IsTrue(reader.GetContentDescriptorList().Count == 0);
                {
                    var contentMetaInfoList = reader.GetContentMetaInfoList();
                    Assert.IsTrue(contentMetaInfoList.Count == 2);
                    {
                        var info = contentMetaInfoList.Where(x => x.Type == "SystemProgram").Single();
                        Assert.IsTrue(info.Id == 0x0000000000000002);
                        Assert.IsTrue(info.Version == 0);
                    }
                    {
                        var info = contentMetaInfoList.Where(x => x.Type == "SystemData").Single();
                        Assert.IsTrue(info.Id == 0x0000000000000003);
                        Assert.IsTrue(info.Version == 0);
                    }
                }
            }

            if (checkDigest)
            {
                var digest = reader.GetDigest();
                var hashCalculator = new SHA256CryptoServiceProvider();
                var hash = hashCalculator.ComputeHash(baseData, 0, baseData.Length - digest.Length);
                Assert.IsTrue(digest.SequenceEqual(hash));
            }

            var exHeader = reader.GetExtendedHeader();
            switch (metaType)
            {
                case "Application" :
                    {
                        var applicationExHeader = exHeader as NintendoApplicationMetaExtendedHeader;
                        Assert.IsTrue(applicationExHeader.PatchId == 0x0005000C10000001 + 0x800);
                        Assert.IsTrue(applicationExHeader.RequiredSystemVersion == 1);
                        break;
                    }
                case "Patch" :
                    {
                        var patchExHeader = exHeader as NintendoPatchMetaExtendedHeader;
                        Assert.IsTrue(patchExHeader.ApplicationId == 0x0005000C10000001);
                        Assert.IsTrue(patchExHeader.RequiredSystemVersion == 1);
                        break;
                    }
                case "AddOnContent" :
                    {
                        var aocExHeader = exHeader as NintendoAddOnContentMetaExtendedHeader;
                        Assert.IsTrue(aocExHeader.ApplicationId == 0x0005000C10000000);
                        Assert.IsTrue(aocExHeader.RequiredApplicationVersion == (1 << 16));
                        Assert.IsTrue(aocExHeader.Tag == null); // バイナリには Tag は含まれない
                        break;
                    }
                case "Delta":
                    {
                        var deltaExHeader = exHeader as NintendoDeltaMetaExtendedHeader;
                        Assert.IsTrue(deltaExHeader.ExtendedDataSize == 0);
                        break;
                    }
                default:
                    Assert.IsTrue(exHeader == null);
                    break;
            }
        }

        private bool EqualDigest(byte[] baseData1, byte[] baseData2)
        {
            var reader1 = new NintendoContentMetaReader(baseData1);
            var reader2 = new NintendoContentMetaReader(baseData2);
            return reader1.GetDigest().SequenceEqual(reader2.GetDigest());
        }

        private void CheckCnmtNca(byte[] baseData, byte[] ncaData)
        {
            using (var stream = new MemoryStream(ncaData))
            {
                var reader = new NintendoContentArchiveReader(stream, new NcaKeyGenerator(KeyConfig));
                Assert.IsTrue(reader.GetContentType() == NintendoContentFileSystemMetaConstant.ContentTypeMeta);
                Assert.IsTrue(reader.GetDistributionType() == NintendoContentFileSystemMetaConstant.DistributionTypeDownload);
                Assert.IsTrue(reader.GetExistentFsIndices().Count == 1);
                {
                    var fsInfo = new NintendoContentArchiveFsHeaderInfo();
                    var fsReader = reader.OpenFileSystemArchiveReader(0, ref fsInfo);
                    Assert.IsTrue(fsInfo.GetEncryptionType() == NintendoContentFileSystemMetaConstant.EncryptionTypeAesCtr);
                    var fileInfo = fsReader.ListFileInfo().First();
                    var data = fsReader.ReadFile(fileInfo.Item1, 0, fileInfo.Item2);
                    Assert.IsTrue(baseData.SequenceEqual(data));
                }
            }
        }

        private ModelType GetXmlModel<ModelType>(byte[] xmlData)
        {
            var serializer = new XmlSerializer(typeof(ModelType));
            ModelType model;
            using (var stream = new MemoryStream(xmlData))
            {
                model = (ModelType)serializer.Deserialize(stream);
            }
            return model;
        }

        private ContentMetaModel GetXmlModel(byte[] xmlData, string metaType)
        {
            if (metaType == "Application")
            {
                return GetXmlModel<ApplicationContentMetaModel>(xmlData);
            }
            else if (metaType == "Patch")
            {
                return GetXmlModel<PatchContentMetaModel>(xmlData);
            }
            else if (metaType == "AddOnContent")
            {
                return GetXmlModel<AddOnContentContentMetaModel>(xmlData);
            }
            else if (metaType == "Delta")
            {
                return GetXmlModel<DeltaContentMetaModel>(xmlData);
            }
            else
            {
                return GetXmlModel<ContentMetaModel>(xmlData);
            }
        }

        private string ByteToString(byte[] bytes, int size)
        {
            return BitConverter.ToString(bytes, 0, size).Replace("-", string.Empty).ToLower();
        }

        private void CheckCnmtXml(byte[] baseData, byte[] ncaData, byte[] xmlData, string metaType)
        {
            if (metaType == "Application")
            {
                CheckCnmtXml(baseData, ncaData, GetXmlModel<ApplicationContentMetaModel>(xmlData));
            }
            else if (metaType == "Patch")
            {
                CheckCnmtXml(baseData, ncaData, GetXmlModel<PatchContentMetaModel>(xmlData));
            }
            else if (metaType == "AddOnContent")
            {
                CheckCnmtXml(baseData, ncaData, GetXmlModel<AddOnContentContentMetaModel>(xmlData));
            }
            else
            {
                CheckCnmtXml(baseData, ncaData, GetXmlModel<ContentMetaModel>(xmlData));
            }
        }

        private void CheckCnmtXml(byte[] baseData, byte[] ncaData, ApplicationContentMetaModel model)
        {
            var reader = new NintendoContentMetaReader(baseData);
            var exHeader = reader.GetExtendedHeader() as NintendoApplicationMetaExtendedHeader;
            Assert.IsTrue(exHeader.PatchId == Convert.ToUInt64(model.PatchId, 16));
            Assert.IsTrue(exHeader.RequiredSystemVersion == model.RequiredSystemVersion);
            CheckCnmtXml(reader, ncaData, (ContentMetaModel)model);
        }

        private void CheckCnmtXml(byte[] baseData, byte[] ncaData, AddOnContentContentMetaModel model)
        {
            var reader = new NintendoContentMetaReader(baseData);
            var exHeader = reader.GetExtendedHeader() as NintendoAddOnContentMetaExtendedHeader;
            Assert.IsTrue(exHeader.ApplicationId == Convert.ToUInt64(model.ApplicationId, 16));
            Assert.IsTrue(exHeader.RequiredApplicationVersion == model.RequiredApplicationVersion);
            Assert.IsTrue(model.Tag == "TestAoc");
            Assert.IsTrue(model.Index == reader.GetId() - (exHeader.ApplicationId + 0x1000));
            CheckCnmtXml(reader, ncaData, (ContentMetaModel)model);
        }

        private void CheckCnmtXml(byte[] baseData, byte[] ncaData, PatchContentMetaModel model)
        {
            var reader = new NintendoContentMetaReader(baseData);
            var exHeader = reader.GetExtendedHeader() as NintendoPatchMetaExtendedHeader;
            Assert.IsTrue(exHeader.ApplicationId == Convert.ToUInt64(model.ApplicationId, 16));
            Assert.IsTrue(exHeader.RequiredSystemVersion == model.RequiredSystemVersion);
            CheckCnmtXml(reader, ncaData, (ContentMetaModel)model);
        }
        private void CheckCnmtXml(byte[] baseData, byte[] ncaData, DeltaContentMetaModel model)
        {
            var reader = new NintendoContentMetaReader(baseData);
            var exHeader = reader.GetExtendedHeader() as NintendoDeltaMetaExtendedHeader;
            Assert.IsTrue(exHeader.ExtendedDataSize == 0);
            CheckCnmtXml(reader, ncaData, (ContentMetaModel)model);
        }

        private void CheckCnmtXml(byte[] baseData, byte[] ncaData, ContentMetaModel model)
        {
            var reader = new NintendoContentMetaReader(baseData);
            CheckCnmtXml(reader, ncaData, model);
        }

        private void CheckCnmtXml(NintendoContentMetaReader reader, byte[] ncaData, ContentMetaModel model)
        {
            Assert.IsTrue(reader.GetType() == model.Type);
            Assert.IsTrue(reader.GetId() == model.GetUInt64Id());
            Assert.IsTrue(reader.GetVersion() == model.Version);
            Assert.IsTrue(((ContentMetaAttribute)reader.GetAttributes()).ToStringList().Count == model.AttributeList.Count);
            Assert.IsTrue(ByteToString(reader.GetDigest(), NintendoContentMeta.GetDigestSize()) == model.Digest);
            Assert.IsTrue(reader.GetRequiredDownloadSystemVersion() == model.RequiredDownloadSystemVersion);
            {
                var contentDescriptorList = reader.GetContentDescriptorList();
                Assert.IsTrue(contentDescriptorList.Count + 1 == model.ContentList.Count);
                foreach (var descriptor in contentDescriptorList)
                {
                    var xmlContent = model.ContentList.Where(x => descriptor.ContentInfo.Type == x.Type).Single();
                    Assert.IsTrue(ByteToString(descriptor.ContentInfo.Id, NintendoContentMeta.GetContentInfoIdDataSize()) == xmlContent.Id);
                    Assert.IsTrue(ByteToString(descriptor.ContentInfo.Hash, NintendoContentMeta.GetContentInfoHashSize()) == xmlContent.Hash);
                    Assert.IsTrue(descriptor.ContentInfo.Size == xmlContent.Size);
                }
                {
                    var xmlMetaContent = model.ContentList.Where(x => x.Type == "Meta").Single();
                    var hashCalculator = new SHA256CryptoServiceProvider();
                    var hash = hashCalculator.ComputeHash(ncaData, 0, ncaData.Length);
                    Assert.IsTrue(ByteToString(hash, NintendoContentMeta.GetContentInfoIdDataSize()) == xmlMetaContent.Id);
                    Assert.IsTrue(ByteToString(hash, NintendoContentMeta.GetContentInfoHashSize()) == xmlMetaContent.Hash);
                    Assert.IsTrue(ncaData.Length == xmlMetaContent.Size);
                }
            }
            if (reader.GetType() == "SystemUpdate")
            {
                var contentMetaList = reader.GetContentMetaInfoList();
                Assert.IsTrue(contentMetaList.Count == model.ContentMetaList.Count);
                foreach (var contentMeta in contentMetaList)
                {
                    var xmlContent = model.ContentMetaList.Where(x => contentMeta.Id == Convert.ToUInt64(x.Id, 16)).Single();
                    Assert.IsTrue(contentMeta.Type == xmlContent.Type);
                }
            }
            else
            {
                Assert.IsTrue(model.ContentMetaList.Count == 0);
            }
        }

        private void CnmtTestCore(string metaFilePath, List<string> metaTypes, List<string> contentTypes, UInt32 defaultVersion = 0)
        {
            foreach (var metaType in metaTypes)
            {
                // 正解コンテンツ情報
                var contentDescriptorList = CreateTestContentDescriptorList(contentTypes);
                var cnmtBase = new NintendoContentMetaBase(contentDescriptorList, metaType, metaFilePath, null);

                // 各データ作成
                byte[] baseData;
                byte[] ncaData;
                byte[] xmlData;
                GetCnmtData(out baseData, out ncaData, out xmlData, cnmtBase, KeyConfig, false);

                CheckCnmt(baseData, metaType, contentDescriptorList, true, defaultVersion);
                CheckCnmtNca(baseData, ncaData);
                CheckCnmtXml(baseData, ncaData, xmlData, metaType);

                var prodContentInfoList = CreateTestContentDescriptorList(contentTypes);
                byte[] prodBaseData;
                byte[] prodNcaData;
                byte[] prodXmlData;

                // 製品化相当処理
                {
                    var prodCnmtBase = new NintendoContentMetaBase(prodContentInfoList, baseData, GetXmlModel(xmlData, metaType), true);

                    GetCnmtData(out prodBaseData, out prodNcaData, out prodXmlData, prodCnmtBase, KeyConfig, false); // テストのため暗号化は dev 相当に

                    CheckCnmt(prodBaseData, metaType, prodContentInfoList, false, defaultVersion);
                    CheckCnmtNca(prodBaseData, prodNcaData);
                    CheckCnmtXml(prodBaseData, prodNcaData, prodXmlData, metaType);

                    Assert.IsTrue(EqualDigest(baseData, prodBaseData));
                }

                // ファイル差し換え相当処理
                {
                    var reconstructedCnmtBase = new NintendoContentMetaBase(prodContentInfoList, baseData, GetXmlModel(xmlData, metaType), false);
                    GetCnmtData(out prodBaseData, out prodNcaData, out prodXmlData, reconstructedCnmtBase, KeyConfig, false);
                    if (metaType != "SystemUpdate")
                    {
                        Assert.IsFalse(EqualDigest(baseData, prodBaseData));
                    }
                    else
                    {
                        Assert.IsTrue(EqualDigest(baseData, prodBaseData));
                    }
                }
            }
        }

        [TestMethod]
        public void CreateProgramCnmtTest()
        {
            var metaTypes = new List<string>()
            {
                "Application",
                "SystemProgram",
            };

            var contentTypes = new List<string>()
            {
                "Program",
                "Control",
                // "Meta", Xml にのみ含まれる
                "HtmlDocument",
                "LegalInformation",
            };

            var testPath = new TestUtility.TestPath(this.TestContext);
            var metaFilePath = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources\\ApplicationMeta\\describe_all.nmeta";

            CnmtTestCore(metaFilePath, metaTypes, contentTypes);
        }

        [TestMethod]
        public void CreateAddOnContentCnmtTest()
        {
            var metaTypes = new List<string>()
            {
                "AddOnContent",
            };

            var contentTypes = new List<string>()
            {
                "Data",
            };

            var testPath = new TestUtility.TestPath(this.TestContext);
            var metaFilePath = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources\\AddOnContentMeta\\describe_addoncontent_id.nmeta";

            CnmtTestCore(metaFilePath, metaTypes, contentTypes);
        }

        [TestMethod]
        public void CreateDataCnmtTest()
        {
            var metaTypesSystemData = new List<string>()
            {
                "SystemData",
            };

            var metaTypesBootImagePackage = new List<string>()
            {
                "BootImagePackage",
                "BootImagePackageSafe",
            };

            var contentTypes = new List<string>()
            {
                "Data",
            };

            var testPath = new TestUtility.TestPath(this.TestContext);
            var metaFilePath = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources\\SystemDataMeta\\default.nmeta";

            CnmtTestCore(metaFilePath, metaTypesSystemData, contentTypes, NintendoContentMeta.GetDefaultVersion());
            CnmtTestCore(metaFilePath, metaTypesBootImagePackage, contentTypes);
        }

        [TestMethod]
        public void CreateSystemUpdateCnmtTest()
        {
            var metaTypes = new List<string>()
            {
                "SystemUpdate",
            };

            var contentTypes = new List<string>();

            var testPath = new TestUtility.TestPath(this.TestContext);
            var metaFilePath = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources\\SystemUpdateMeta\\basic.nmeta";

            CnmtTestCore(metaFilePath, metaTypes, contentTypes);
        }
        [TestMethod]
        public void CreateDeltaCnmtTest()
        {
            var metaTypes = new List<string>()
            {
                "Delta",
            };

            var contentTypes = new List<string>()
            {
                "DeltaFragment",
            };

            var testPath = new TestUtility.TestPath(this.TestContext);
            var metaFilePath = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources\\DeltaMeta\\default.nmeta";

            CnmtTestCore(metaFilePath, metaTypes, contentTypes);
        }
    }
}
