﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.Properties;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Test
{
    public class StringHelperTest : SharedTextContextBase
    {
        public StringHelperTest(TestContextFixture fixture) : base(fixture)
        {
        }

        [Fact]
        public void DefaultCtor()
        {
            // ReSharper disable once HeapView.ObjectAllocation.Evident
            // ReSharper disable once ObjectCreationAsStatement
            new StringHelper(_context.DiContainer);
        }

        public static IEnumerable<object[]> AllAppModeTypes =>
            Enum.GetValues(typeof(AppModeType)).Cast<AppModeType>().Select(t => new object[] {t});

        public static IEnumerable<object[]> AllResultTypes =>
            Enum.GetValues(typeof(NspHandleResult)).Cast<NspHandleResult>().Select(t => new object[] {t});

        [Theory]
        [MemberData(nameof(AllAppModeTypes))]
        public void AppModeName(AppModeType type)
        {
            var sh = new StringHelper(_context.DiContainer);

            _context.DiContainer.GetInstance<AppProfile>().AppMode = type;

            Assert.NotNull(sh.AppModeName);
        }

        [Theory]
        [MemberData(nameof(AllAppModeTypes))]
        public void CreateNew(AppModeType type)
        {
            var sh = new StringHelper(_context.DiContainer);

            _context.DiContainer.GetInstance<AppProfile>().AppMode = type;

            Assert.NotNull(sh.CreateNew);
        }

        [Theory]
        [MemberData(nameof(AllAppModeTypes))]
        public void Import(AppModeType type)
        {
            var sh = new StringHelper(_context.DiContainer);

            _context.DiContainer.GetInstance<AppProfile>().AppMode = type;

            Assert.NotNull(sh.Import);
        }

        public static IEnumerable<object[]> AllImportableFileType =>
            Enum.GetValues(typeof(ImportableFileType)).Cast<ImportableFileType>().Select(t => new object[] {t});
        [Theory]
        [MemberData(nameof(AllImportableFileType))]
        public void ImportFromType(ImportableFileType type)
        {
            var sh = new StringHelper(_context.DiContainer);

            Assert.NotNull(sh.ImportFromType(type));
        }

        public static IEnumerable<object[]> AllWorkingKind =>
            Enum.GetValues(typeof(WorkingKind)).Cast<WorkingKind>().Select(t => new object[] {t});

        [Theory]
        [MemberData(nameof(AllWorkingKind))]
        public void GetWorkingMessage_Waiting(WorkingKind kind)
        {
            Assert.Equal(string.Empty, StringHelper.GetWorkingMessage(kind, WorkingState.Waiting));
        }

        [Theory]
        [MemberData(nameof(AllWorkingKind))]
        public void GetWorkingMessage_Canceling(WorkingKind kind)
        {
            Assert.NotEqual(string.Empty, StringHelper.GetWorkingMessage(kind, WorkingState.Canceling));
        }

        [Theory]
        [MemberData(nameof(AllWorkingKind))]
        public void GetWorkingMessage_Working(WorkingKind kind)
        {
            Assert.NotEqual(string.Empty, StringHelper.GetWorkingMessage(kind, WorkingState.Working));
        }

        [Theory]
        [MemberData(nameof(AllWorkingKind))]
        public void GetWorkingMessage_Succeeded(WorkingKind kind)
        {
            Assert.NotEqual(string.Empty, StringHelper.GetWorkingMessage(kind, WorkingState.Succeeded));
        }

        [Theory]
        [MemberData(nameof(AllWorkingKind))]
        public void GetWorkingMessage_Failed(WorkingKind kind)
        {
            Assert.NotEqual(string.Empty, StringHelper.GetWorkingMessage(kind, WorkingState.Failed));
        }

        [Theory]
        [MemberData(nameof(AllWorkingKind))]
        public void GetWorkingMessage_Canceled(WorkingKind kind)
        {
            Assert.NotEqual(string.Empty, StringHelper.GetWorkingMessage(kind, WorkingState.Canceled));
        }

        [Theory]
        [MemberData(nameof(AllResultTypes))]
        public void ResultTypeName(NspHandleResult result)
        {
            var sh = new StringHelper(_context.DiContainer);

            Assert.NotNull(sh.GetNspHandleResultMessage(result));
        }

        [Theory]
        [InlineData(true, true)]
        [InlineData(false, false)]
        public void CardSpecReadableString(bool isContains, bool isAutomaticSetting)
        {
            var sh = new StringHelper(_context.DiContainer);
            using (var c = new CardSpec())
            {
                c.ClockRate = 25;
                c.Size = 1024 * 1024 * 1024; // 1 GiB

                c.IsAutomaticSettingSize = isAutomaticSetting;
                c.IsAutomaticSettingClockRate = isAutomaticSetting;
                Assert.Equal(isContains, sh.GetCardClockRateReadableString(c).Contains(Resources.CardSpec_AutoSet));
                Assert.Equal(isContains, sh.GetCardSizeReadableString(c).Contains(Resources.CardSpec_AutoSet));
            }
        }
    }
}
