﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Reactive.Linq;
using System.Threading;
using System.Threading.Tasks;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Core.Test.Fixtures;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.MakeNspPatchWindow;
using Reactive.Bindings.Extensions;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Test.MainWindow.SubPanel
{
    public class MakeNspPatchWindowVmTest : IDisposable, IClassFixture<BuildTestAppNspFixture>
    {
        private readonly TestContext _context = new TestContext();
        private readonly BuildTestAppNspFixture _fixture;

        public void Dispose()
        {
            _context.Dispose();
        }

        public MakeNspPatchWindowVmTest(BuildTestAppNspFixture fixture)
        {
            _fixture = fixture;
        }

        [Fact]
        public void DefaultCtor()
        {
            using (var vm = _context.DiContainer.GetInstance<MakeNspPatchWindowVm>())
            {
                Assert.NotNull(vm.CloseCommand);
                Assert.NotNull(vm.MakePatchCommand);
                Assert.NotNull(vm.OriginalNspPath);
                Assert.NotNull(vm.PreviousNspPath);
                Assert.NotNull(vm.ApplicationDescPath);
                Assert.NotNull(vm.Params);
                Assert.NotNull(vm.IsMakePatch);
                Assert.NotEmpty(vm.Params);
            }
        }

        [Fact]
        public void CloseCommand()
        {
            using (var vm = _context.DiContainer.GetInstance<MakeNspPatchWindowVm>())
            {
                vm.CloseCommand.Execute();
            }
        }

        [Fact]
        public void MakePatchCommand()
        {
            using (var vm = _context.DiContainer.GetInstance<MakeNspPatchWindowVm>())
            {
                vm.MakePatchCommand.Execute();
            }
        }

        [Fact]
        public async Task IsUsePreviousPatch()
        {
            if (_context.DiContainer.GetInstance<ApplicationCapability>().IsSupportMakingPatch == false)
                return;

            // テストフィクスチャの初期化 (アプリ nsp のビルドはこのテストでしか使用しない)
            _fixture.Initialize();

            var patchNspPath = Path.Combine(_context.TempDirPath, "PreviousPatch.nsp");

            // v1 パッチ
            var r = await new NspBuilder().MakePatchNspAsync(new NspBuilder.MakePatchOption()
            {
                OriginalNspFilePath = _fixture.NspFilePath,
                CurrentNspFilePath = _fixture.RevisedV1NspFilePath,
                OutputPath = patchNspPath
            }).ConfigureAwait(false);
            Assert.Equal(NspHandleResult.Ok, r.Result);

            using (var app = _context.DiContainer.GetInstance<App>())
            using (var vm = _context.DiContainer.GetInstance<MakeNspPatchWindowVm>())
            {
                Assert.False(app.NspPatch.IsUsePreviousNspPath);
                Assert.Equal(MakeNspPatchWindowVm.IncrementStatus.Ok, vm.NewPatchVersionIncrementStatus);

                app.NspPatch.IsUsePreviousNspPath = true;
                Assert.Equal(MakeNspPatchWindowVm.IncrementStatus.PreviousOrRevisedNotSet, vm.NewPatchVersionIncrementStatus);

                app.NspPatch.OriginalNspPath = _fixture.NspFilePath;
                app.NspPatch.CurrentNspPath = _fixture.RevisedV1NspFilePath;
                app.NspPatch.PreviousNspPath = patchNspPath;

                // 入力した nsp 情報の収集が完了するまで待機する
                await new[]
                    {
                        vm.OriginalAppContentMeta.Select(x => x != null),
                        vm.RevisedAppContentMeta.Select(x => x != null),
                        vm.PreviousPatchContentMeta.Select(x => x != null)
                    }
                    .CombineLatestValuesAreAllTrue()
                    .FirstAsync(x => x);

                Assert.Equal(MakeNspPatchWindowVm.IncrementStatus.NotIncremented,
                    vm.NewPatchVersionIncrementStatus);
            }
        }
    }
}
