﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Linq;
using System.Reactive.Disposables;
using System.Windows;
using BezelEditor.Foundation.Extentions;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params;
using Reactive.Bindings;
using Xunit;
using Application = Nintendo.Authoring.AuthoringEditor.Core.Application;

namespace Nintendo.Authoring.AuthoringEditor.Test.MainWindow.ProjectEditPanel.Params
{
    public class RatingParamVmTest : IDisposable
    {
        private readonly TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        [Fact]
        public void DefaultCtor()
        {
            using (var app = new Application())
            using (GetParamVm(app, out var vm))
            {
                Assert.NotEmpty(vm.AgeRatingSelections);
                Assert.NotEmpty(vm.TargetSalesRegions);
            }
        }

        [Fact]
        public void HasOnlySelectedRatings()
        {
            using (var app = new Application())
            using (GetParamVm(app, out var vm))
            {
                // 可視状態の販売地域がないなら選択可能なレーティング区分も空
                Assert.Empty(vm.AgeRatingSelections.Where(x => x.Visibility.Value == Visibility.Visible));
                // 販売地域の変更だけでは nmeta のレーティング情報に変更なし
                Assert.Empty(app.Ratings.Where(x => x.IsUse));

                // 日本を販売地域に加えると CERO レーティング区分のみ可視状態になる
                var regionJp = vm.TargetSalesRegions.FirstOrDefault(x => x.Region == TargetSalesRegion.Japan);
                Assert.NotNull(regionJp);
                regionJp.IsUse.Value = true;
                Assert.Equal(1, vm.AgeRatingSelections.Count(x => x.Visibility.Value == Visibility.Visible));

                // 全年齢 (0 歳以上) を選択状態とする
                var ratingCero = vm.AgeRatingSelections.First(x => x.Visibility.Value == Visibility.Visible);
                // 最初のエントリは null (空) なのでスキップ
                ratingCero.SelectedCategory.Value = ratingCero.Categories.Skip(1).First();
                // 全年齢が model に反映されていることを確認
                Assert.NotEmpty(app.Ratings.Where(x => x.IsUse));
                Assert.Equal(0, app.Ratings.First(x => x.IsUse).Age);

                // リージョンを非表示にすると、選択可能なレーティング区分も空
                regionJp.IsUse.Value = false;
                Assert.Empty(vm.AgeRatingSelections.Where(x => x.Visibility.Value == Visibility.Visible));
                // model にも選択済みレーティング区分情報だけが反映されていることを確認
                Assert.Empty(app.Ratings.Where(x => x.IsUse));

                // リージョンを再表示したときに model に反映されていることを確認
                regionJp.IsUse.Value = true;
                Assert.Equal(0, app.Ratings.First(x => x.IsUse).Age);
            }
        }

        [Fact]
        public void RatingModelChanges()
        {
            var app = new Application();

            var ceroRating = app.Ratings.First(x => x.Organization == "CERO");
            ceroRating.Age = 0;
            ceroRating.IsUse = true;

            using (app)
            using (GetParamVm(app, out var vm))
            {
                // 初期値が反映されている
                {
                    var selection = vm.AgeRatingSelections.First(x => x.Visibility.Value == Visibility.Visible);
                    Assert.Equal("CERO", selection.Organization);
                    Assert.Equal(0, selection.SelectedCategory.Value.Model.AgeValue);
                    Assert.Equal(TargetSalesRegion.Japan, selection.SelectedRegion.Region);
                }

                // モデルを追加・削除・更新して VM への反映を確認
                var esrbRating = app.Ratings.First(x => x.Organization == "ESRB");
                esrbRating.Age = 3;
                esrbRating.IsUse = true;
                Assert.True(vm.AgeRatingSelections.Any(x =>
                    x.Organization == "ESRB" &&
                    x.SelectedCategory.Value.Model.AgeValue == 3));

                ceroRating.IsUse = false;
                Assert.False(vm.AgeRatingSelections.All(x =>
                    x.Organization == "CERO" &&
                    x.SelectedCategory.Value.Model == null));

                esrbRating.Age = 6;
                Assert.True(vm.AgeRatingSelections.Any(x =>
                    x.Organization == "ESRB" &&
                    x.SelectedCategory.Value.Model.AgeValue == 6));
            }
        }

        [Fact]
        public void UnchangedModel()
        {
            var app = new Application();
            {
                var cero = app.Ratings.First(x => x.Organization == "CERO");
                cero.Age = 0;
                cero.IsUse = true;

                var usk = app.Ratings.First(x => x.Organization == "USK");
                usk.Age = 18;
                usk.IsUse = true;
            }
            var appDefaultYaml = YamlHelper.SaveToString(app);
            using (app)
            using (GetParamVm(app, out var vm))
            {
                // レーティング情報の変更前なので変更は検出されない
                Assert.Equal(appDefaultYaml, YamlHelper.SaveToString(app));

                // 可視状態にある販売地域の表示をトグルする
                var region = vm.AgeRatingSelections.First(x => x.SelectedRegion.Region == TargetSalesRegion.Japan);
                region.SelectedRegion.IsUse = false;
                region.SelectedRegion.IsUse = true;

                // 販売地域の表示・非表示のトグルでも変更は検出されない
                Assert.Equal(appDefaultYaml, YamlHelper.SaveToString(app));

                // 販売地域を非表示にした場合に対応するレーティング区分が出力されないことを確認
                region.SelectedRegion.IsUse = false;
                Assert.NotEqual(appDefaultYaml, YamlHelper.SaveToString(app));
                Assert.False( app.Ratings.First(x => x.Organization == "CERO").IsUse);
            }
        }

        [Fact]
        public void IsModelSynchronized()
        {
            var app = new Application();
            {
                var cero = app.Ratings.First(x => x.Organization == "CERO");
                cero.Age = 0;
                cero.IsUse = false;

                var usk = app.Ratings.First(x => x.Organization == "USK");
                usk.Age = 18;
                usk.IsUse = false;
            }
            using (app)
            using (GetParamVm(app, out var vm))
            {
                var ceroVm = vm.AgeRatingSelections.First(x => x.Organization == "CERO");
                ceroVm.SelectedCategory.Value = ceroVm.Categories.Skip(2).First();

                var ceroModel = app.Ratings.First(x => x.Organization == "CERO");
                Assert.Equal(12, ceroModel.Age); // 2 番目の要素は 12 歳 (CERO B)
                Assert.True(ceroModel.IsUse);
            }
        }

        private static IDisposable GetParamVm(Application app, out RatingParamVm vm)
        {
            var disposables = new CompositeDisposable();

            var readonlyRp = new ReactiveProperty<bool>(false).AddTo(disposables);
            vm = new RatingParamVm(string.Empty, string.Empty, app, readonlyRp).AddTo(disposables);

            return disposables;
        }
    }
}
