﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.MainWindow;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ComparisonPanel;
using Nintendo.Authoring.AuthoringEditor.SelectTwoNspFilesWindow;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Test.MainWindow
{
    public class MainWindowVmTest : IDisposable
    {
        private readonly TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        [Fact]
        public void DefaultCtor()
        {
            using (var vm = _context.DiContainer.GetInstance<MainWindowVm>())
            {
                Assert.NotNull(vm.SelectedPanel);
                Assert.NotNull(vm.Title);

                Assert.NotNull(vm.SaveAsNspCommand);
                Assert.NotNull(vm.ImportNspCommand);
                Assert.NotNull(vm.ShowNspEntriesCommand);
                Assert.NotNull(vm.SaveMetaCommand);
                Assert.NotNull(vm.ImportMetaCommand);
                Assert.NotNull(vm.MakeNspPatchCommand);
                Assert.NotNull(vm.OpenAllCommand);
                Assert.NotNull(vm.CloseAllCommand);
                Assert.NotNull(vm.ExtractNspFileCommand);

                Assert.NotNull(vm.CanClose);

                Assert.NotNull(vm.IsDisplayOnlyDifferences);
                Assert.NotNull(vm.IsExtractableFileType);
                Assert.NotNull(vm.ExtractionTarget0);
                Assert.NotNull(vm.ExtractionTarget1);

                Assert.NotNull(vm.CanExportProject);
                Assert.NotNull(vm.CanImportProject);
                Assert.NotNull(vm.CanSaveAsMeta);
                Assert.NotNull(vm.CanImportMeta);
                Assert.NotNull(vm.CanSaveAsNsp);
                Assert.NotNull(vm.CanImportNsp);
                Assert.NotNull(vm.CanImportPartialMeta);
                Assert.NotNull(vm.CanCompare);
                Assert.NotNull(vm.CanMakeNspPatch);
                Assert.NotNull(vm.CanShowNspEntries);

                Assert.NotNull(vm.WorkingPanel);
                Assert.NotNull(vm.IsWaiting);
            }
        }

        [Fact]
        public void ExportNspCommand()
        {
            var appProfile = _context.DiContainer.GetInstance<AppProfile>();
            appProfile.AppMode = AppModeType.ApplicationNsp;

            var app = _context.DiContainer.GetInstance<App>();
            app.Project = Project.Import(_context.DiContainer, ImportableFileType.Meta,
                NintendoSdkHelper.ApplicationMetaFilePath);
            foreach (var title in app.Project.Meta.Application.Titles)
            {
                title.IsReplaceIcon = false;
            }

            using (var vm = _context.DiContainer.GetInstance<MainWindowVm>())
            {
                Assert.True(vm.SaveAsNspCommand.CanExecute());
                app.Project.Meta.IsReadOnly = true;
                Assert.False(vm.SaveAsNspCommand.CanExecute());
            }
        }

        [Fact]
        public void ImportNspCommand()
        {
            using (var vm = _context.DiContainer.GetInstance<MainWindowVm>())
            {
                vm.ImportNspCommand.Execute();
            }
        }

        [Fact]
        public void SaveMetaCommand()
        {
            var profile = _context.DiContainer.GetInstance<AppProfile>();

            using (var app = _context.DiContainer.GetInstance<App>())
            using (var vm = _context.DiContainer.GetInstance<MainWindowVm>())
            {
                app.CreateNewProject();
                profile.AppMode = AppModeType.ApplicationMeta;

                Assert.True(app.HasErrors);
                Assert.False(vm.SaveAsMetaCommand.CanExecute());
                CreateApplicationMetaNoErrorDataSet(app);
                Assert.False(app.HasErrors);

                // 保存していないので実行不可
                Assert.False(vm.SaveMetaCommand.CanExecute());

                {
                    string tempFile = Path.GetRandomFileName();
                    _context.CreateFile(tempFile);
                    app.Project.TargetFilePath = Path.Combine(_context.TempDirPath, tempFile);
                }

                // 保存したことにすれば実行可
                Assert.True(vm.SaveMetaCommand.CanExecute());

                vm.SaveMetaCommand.Execute();
            }
        }

        [Fact]
        public void SaveAsMetaCommand()
        {
            using (var app = _context.DiContainer.GetInstance<App>())
            using (var vm = _context.DiContainer.GetInstance<MainWindowVm>())
            {
                app.CreateNewProject();
                CreateApplicationMetaNoErrorDataSet(app);
                Assert.True(vm.SaveAsMetaCommand.CanExecute());
                vm.SaveAsMetaCommand.Execute();
            }
        }

        [Fact]
        public void ImportMetaCommand()
        {
            using (var vm = _context.DiContainer.GetInstance<MainWindowVm>())
            {
                vm.ImportMetaCommand.Execute();
            }
        }

        [Fact]
        public void ImportPartialMetaCommand()
        {
            using (var app = _context.DiContainer.GetInstance<App>())
            using (var vm = _context.DiContainer.GetInstance<MainWindowVm>())
            {
                Assert.False(vm.ImportPartialMetaCommand.CanExecute());

                var profile = _context.DiContainer.GetInstance<AppProfile>();

                app.CreateNewProject();

                profile.AppMode = AppModeType.ApplicationMeta;
                Assert.True(vm.ImportPartialMetaCommand.CanExecute());

                profile.AppMode = AppModeType.AocMeta;
                Assert.False(vm.ImportPartialMetaCommand.CanExecute());

                profile.AppMode = AppModeType.ApplicationNsp;
                Assert.True(vm.ImportPartialMetaCommand.CanExecute());
                app.IsReadOnly = true;
                Assert.False(vm.ImportPartialMetaCommand.CanExecute());

                profile.AppMode = AppModeType.AocNsp;
                app.IsReadOnly = false;
                Assert.False(vm.ImportPartialMetaCommand.CanExecute());
            }
        }

        [Fact]
        public void MakeNspPatchCommand()
        {
            using (var vm = _context.DiContainer.GetInstance<MainWindowVm>())
            {
                vm.MakeNspPatchCommand.Execute();
            }
        }

        [Fact]
        public void OpenAllCommand()
        {
            using (var vm = _context.DiContainer.GetInstance<MainWindowVm>())
            {
                vm.CurrentPanelType = MainWindowVm.PanelType.Compare;
                vm.OpenAllCommand.Execute();
            }
        }

        [Fact]
        public void CloseAllCommand()
        {
            using (var vm = _context.DiContainer.GetInstance<MainWindowVm>())
            {
                vm.CurrentPanelType = MainWindowVm.PanelType.Compare;
                vm.CloseAllCommand.Execute();
            }
        }

        public static IEnumerable<object> AllPanelType =>
            Enum.GetValues(typeof(MainWindowVm.PanelType)).Cast<object>().Select(x => new[] {x});

        [Theory]
        [MemberData(nameof(AllPanelType))]
        public void DelaySearchWord(MainWindowVm.PanelType panelType)
        {
            using (var vm = _context.DiContainer.GetInstance<MainWindowVm>())
            {
                vm.CurrentPanelType = panelType;

                if (panelType != MainWindowVm.PanelType.ProjectEdit)
                    Assert.Null(vm.DelaySearchWord);
                else
                    Assert.NotNull(vm.DelaySearchWord);
            }
        }

        [Theory]
        [InlineData(false)]
        [InlineData(true)]
        public void IsDisplayOnlyDifferences(bool isDisplay)
        {
            using (var vm = _context.DiContainer.GetInstance<MainWindowVm>())
            {
                vm.CurrentPanelType = MainWindowVm.PanelType.Compare;
                vm.IsDisplayOnlyDifferences.Value = isDisplay;
            }
        }

        [Fact]
        public void ComparePanel()
        {
            using (var vm = _context.DiContainer.GetInstance<MainWindowVm>())
            {
                vm.CurrentPanelType = MainWindowVm.PanelType.Compare;

                var comparisonPanelVm = (ComparisonPanelVm) vm.SelectedPanel.Value;

                comparisonPanelVm.TargetFileType = ComparableFileType.AppMeta;
                comparisonPanelVm.Targets.Add(new ComparisonPanelVm.Target
                {
                    Project = _context.DiContainer.GetInstance<Project>()
                });
                comparisonPanelVm.Targets.Add(new ComparisonPanelVm.Target
                {
                    Project = _context.DiContainer.GetInstance<Project>()
                });
            }
        }

        private static void CreateApplicationMetaNoErrorDataSet(App app)
        {
            app.Project.Meta.Application.DisplayVersion = "1.0.0";
            app.Project.Meta.Application.SupportedLanguages
                .First(x => x.Language == LanguageType.AmericanEnglish)
                .IsSupported = true;
            app.Project.Meta.Application.Titles.Add(new Title
            {
                Language = LanguageType.AmericanEnglish,
                Name = "Test",
                Publisher = "Test",
                IsReplaceIcon = false
            });
        }
    }
}
