﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Foundation.Test
{
    public class FileSystemHelperTest : IDisposable
    {
        private readonly TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        [Fact]
        public void EnumerateDirectories()
        {
            var testPath = Path.Combine(_context.TestDataDirPath, "filesystemhelper_testdata");

            Assert.Equal(6, FileSystemHelper.EnumerateAllFiles(testPath).Count());
        }

        [Fact]
        public void EnumerateDirectoriesError()
        {
            const string testPath = @"aa:\";

            Assert.Equal(0, FileSystemHelper.EnumerateAllFiles(testPath).Count());
        }

        [Fact]
        public void EnumerateAllDirectories()
        {
            var testPath = Path.Combine(_context.TestDataDirPath, "filesystemhelper_testdata");

            Assert.Equal(8, FileSystemHelper.EnumerateAllDirectories(testPath).Count());
        }

        [Fact]
        public void EnumerateAllDirectoriesError()
        {
            const string testPath = @"aa:\";

            Assert.Equal(0, FileSystemHelper.EnumerateAllDirectories(testPath).Count());
        }

        [Fact]
        public void IsEqualDirectory_Null_Null()
        {
            Assert.Equal(true, FileSystemHelper.IsEqualDirectory(null, null));
        }

        [Fact]
        public void IsEqualDirectory_NotNull_Null()
        {
            _context.TempDir.Build()
                .CreateFolder("t0")
                .CreateFolder("t1");

            var t0 = Path.Combine(_context.TempDirPath, "t0");

            Assert.Equal(false, FileSystemHelper.IsEqualDirectory(t0, null));
        }

        [Fact]
        public void IsEqualDirectory_Null_NotNull()
        {
            _context.TempDir.Build()
                .CreateFolder("t0")
                .CreateFolder("t1");

            var t1 = Path.Combine(_context.TempDirPath, "t1");

            Assert.Equal(false, FileSystemHelper.IsEqualDirectory(null, t1));
        }

        [Fact]
        public void IsEqualDirectory_Empty_Empty()
        {
            _context.TempDir.Build()
                .CreateFolder("t0")
                .CreateFolder("t1");

            var t0 = Path.Combine(_context.TempDirPath, "t0");
            var t1 = Path.Combine(_context.TempDirPath, "t1");

            Assert.Equal(true, FileSystemHelper.IsEqualDirectory(t0, t1));
        }

        [Fact]
        public void IsEqualDirectory_OK()
        {
            _context.TempDir.Build()
                .CreateFolder("t0")
                .CreateFile("t0\\f0")
                .CreateFile("t0\\f1")
                .CreateFile("t0\\f2")
                .CreateFolder("t1")
                .CreateFile("t1\\f0")
                .CreateFile("t1\\f1")
                .CreateFile("t1\\f2");

            var t0 = Path.Combine(_context.TempDirPath, "t0");
            var t1 = Path.Combine(_context.TempDirPath, "t1");

            Assert.Equal(true, FileSystemHelper.IsEqualDirectory(t0, t1));
        }

        [Fact]
        public void IsEqualDirectory_Entry_NG1()
        {
            _context.TempDir.Build()
                .CreateFolder("t0")
                .CreateFile("t0\\f0")
                .CreateFile("t0\\f1")
                .CreateFolder("t1")
                .CreateFile("t1\\f0")
                .CreateFile("t1\\f1")
                .CreateFile("t1\\f2");

            var t0 = Path.Combine(_context.TempDirPath, "t0");
            var t1 = Path.Combine(_context.TempDirPath, "t1");

            Assert.Equal(false, FileSystemHelper.IsEqualDirectory(t0, t1));
        }

        [Fact]
        public void IsEqualDirectory_Entry_NG2()
        {
            _context.TempDir.Build()
                .CreateFolder("t0")
                .CreateFile("t0\\f0")
                .CreateFile("t0\\f1")
                .CreateFile("t0\\f2")
                .CreateFolder("t1")
                .CreateFile("t1\\f0")
                .CreateFile("t1\\f1");

            var t0 = Path.Combine(_context.TempDirPath, "t0");
            var t1 = Path.Combine(_context.TempDirPath, "t1");

            Assert.Equal(false, FileSystemHelper.IsEqualDirectory(t0, t1));
        }

        [Fact]
        public void IsEqualDirectory_Entry_NG3()
        {
            _context.TempDir.Build()
                .CreateFolder("t0")
                .CreateFile("t0\\f0")
                .CreateFile("t0\\f1")
                .CreateFile("t0\\XXXXXX")
                .CreateFile("t0\\f2")
                .CreateFolder("t1")
                .CreateFile("t1\\f0")
                .CreateFile("t1\\f1")
                .CreateFile("t1\\f2");

            var t0 = Path.Combine(_context.TempDirPath, "t0");
            var t1 = Path.Combine(_context.TempDirPath, "t1");

            Assert.Equal(false, FileSystemHelper.IsEqualDirectory(t0, t1));
        }

        [Fact]
        public void IsEqualDirectory_Entry_NG4()
        {
            _context.TempDir.Build()
                .CreateFolder("t0")
                .CreateFolder("t1")
                .CreateFile("t1\\f0")
                .CreateFile("t1\\f1")
                .CreateFile("t1\\f2");

            var t0 = Path.Combine(_context.TempDirPath, "t0");
            var t1 = Path.Combine(_context.TempDirPath, "t1");

            Assert.Equal(false, FileSystemHelper.IsEqualDirectory(t0, t1));
        }

        [Fact]
        public void IsEqualDirectory_Entry_NG5()
        {
            _context.TempDir.Build()
                .CreateFolder("t0")
                .CreateFile("t0\\f0")
                .CreateFile("t0\\f1")
                .CreateFile("t0\\XXXXXX")
                .CreateFile("t0\\f2")
                .CreateFolder("t1");

            var t0 = Path.Combine(_context.TempDirPath, "t0");
            var t1 = Path.Combine(_context.TempDirPath, "t1");

            Assert.Equal(false, FileSystemHelper.IsEqualDirectory(t0, t1));
        }

        [Fact]
        public void IsEqualDirectory_Entry_NG6()
        {
            _context.TempDir.Build()
                .CreateFolder("t0")
                .CreateFile("t0\\f0")
                .CreateFile("t0\\f2")
                .CreateFile("t0\\f3")
                .CreateFolder("t1")
                .CreateFile("t1\\f0")
                .CreateFile("t1\\f1")
                .CreateFile("t1\\f2");

            var t0 = Path.Combine(_context.TempDirPath, "t0");
            var t1 = Path.Combine(_context.TempDirPath, "t1");

            Assert.Equal(false, FileSystemHelper.IsEqualDirectory(t0, t1));
        }

        [Fact]
        public void IsEqualDirectory_File_OK()
        {
            _context.TempDir.Build()
                .CreateFolder("t0")
                .CreateFile("t0\\f0", new byte[] {1, 2, 3})
                .CreateFile("t0\\f1", new byte[] {1, 2, 3, 1, 2, 3})
                .CreateFile("t0\\f2", new byte[] {1, 2, 3, 1, 2, 3, 1, 2, 3})
                .CreateFolder("t1")
                .CreateFile("t1\\f0", new byte[] {1, 2, 3})
                .CreateFile("t1\\f1", new byte[] {1, 2, 3, 1, 2, 3})
                .CreateFile("t1\\f2", new byte[] {1, 2, 3, 1, 2, 3, 1, 2, 3});

            var t0 = Path.Combine(_context.TempDirPath, "t0");
            var t1 = Path.Combine(_context.TempDirPath, "t1");

            Assert.Equal(true, FileSystemHelper.IsEqualDirectory(t0, t1));
        }

        [Fact]
        public void IsEqualDirectory_File_NG1()
        {
            _context.TempDir.Build()
                .CreateFolder("t0")
                .CreateFile("t0\\f0", new byte[] {1, 2, 3})
                .CreateFile("t0\\f1")
                .CreateFile("t0\\f2")
                .CreateFolder("t1")
                .CreateFile("t1\\f0", new byte[] {1, 2, 3, 4})
                .CreateFile("t1\\f1")
                .CreateFile("t1\\f2");

            var t0 = Path.Combine(_context.TempDirPath, "t0");
            var t1 = Path.Combine(_context.TempDirPath, "t1");

            Assert.Equal(false, FileSystemHelper.IsEqualDirectory(t0, t1));
        }

        [Fact]
        public void IsEqualDirectory_File_NG2()
        {
            _context.TempDir.Build()
                .CreateFolder("t0")
                .CreateFile("t0\\f0", new byte[] {1, 2, 3})
                .CreateFile("t0\\f1")
                .CreateFile("t0\\f2")
                .CreateFolder("t1")
                .CreateFile("t1\\f0", new byte[] {4, 5, 6})
                .CreateFile("t1\\f1")
                .CreateFile("t1\\f2");

            var t0 = Path.Combine(_context.TempDirPath, "t0");
            var t1 = Path.Combine(_context.TempDirPath, "t1");

            Assert.Equal(false, FileSystemHelper.IsEqualDirectory(t0, t1));
        }
    }
}
