﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Core.Test
{
    public class SaveDataSizeCalculatorTest : IDisposable
    {
        private readonly TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        [Fact]
        public void DefaultCtor()
        {
            // ReSharper disable once ObjectCreationAsStatement
            new SaveDataSizeCalculator();
        }

        [Fact]
        public void AddFileEntry()
        {
            var c = new SaveDataSizeCalculator();

            c.AddFileEntry(100);
            c.AddFileEntry(10000);
        }

        [Fact]
        public void AddFileEntries()
        {
            var c = new SaveDataSizeCalculator();

            c.AddFileEntries(new[] {1, 10, 100, 1000});
        }

        [Fact]
        public void AddFileEntry_Exception()
        {
            var c = new SaveDataSizeCalculator();

            Assert.Throws<ArgumentOutOfRangeException>(() => c.AddFileEntry(-1));
        }

        [Fact]
        public void AddDirectoryEntry()
        {
            var c = new SaveDataSizeCalculator();

            c.AddDirectoryEntry();
        }

        [Fact]
        public void AddDirectoryEntries()
        {
            var c = new SaveDataSizeCalculator();

            c.AddDirectoryEntries(100);
        }

        // 1 ブロック当たりのバイトサイズ
        private static readonly int BlockByteSize = 16*1024;
        // ファイルエントリ、ディレクトリエントリ分として +2 ブロック分をマージンとして確保
        private static readonly int MarginBlockByteSize = BlockByteSize * (1+1);
        // 管理領域 1 エントリあたりのバイト数
        private static readonly int EntryByteSize = 96;
        // ファイル、ディレクトリそれぞれで必要になる管理エントリ数
        private static readonly int ManagementEntryCount = 2;

        private static int ManagementBlockCount(int v)
        {
            return ((ManagementEntryCount + v) * EntryByteSize + BlockByteSize - 1) / BlockByteSize;
        }

        private static object[] TestEntry(int expectedFileSize, int[] fileSizes, int dirCount)
        {
            return new object[]
            {
                // 期待されるシミュレーターによるセーブデータサイズの合計
                0
                    // 期待されるファイルサイズ
                    + expectedFileSize
                    // ファイルに必要な管理ブロックサイズ
                    + ManagementBlockCount(fileSizes.Length)*BlockByteSize
                    // ディレクトリに必要な管理ブロックサイズ
                    + ManagementBlockCount(dirCount)*BlockByteSize
                    // セーブデータサイズシミュレーターが保持するセーブデータのマージン
                    // （ファイル、ディレクトリそれぞれ 1 ブロックずつ)
                    + MarginBlockByteSize,
                fileSizes,
                dirCount,
            };
        }

        public static IEnumerable<object> SaveDataSizeTestData
        {
            get
            {
                // -------------------------------------------
                // 空
                yield return TestEntry(
                    0,
                    new int[0],
                    0);

                // -------------------------------------------
                // ファイル
                yield return TestEntry(
                    16 * 1024,
                    new[] {1},
                    0);

                yield return TestEntry(
                    16*1024 + 16*1024,
                    new[] {1, 1},
                    0);

                yield return TestEntry(
                    16*1024,
                    new[] {16*1024},
                    0);

                yield return TestEntry(
                    16 *1024*2,
                    new[] {16*1024 + 1},
                    0);

                yield return TestEntry(
                    16*1024*2 + 16*1024,
                    new[] {16*1024 + 1, 16*1024},
                    0);

                yield return TestEntry(
                    16*1024*2 + 16*1024,
                    new[] {16*1024 + 1, 16*1024 - 1},
                    0);

                yield return TestEntry(
                    16*1024*167,
                    Enumerable.Repeat(1, 167).ToArray(),
                    0);

                yield return TestEntry(
                    16*1024*168,
                    Enumerable.Repeat(1, 168).ToArray(),
                    0);

                yield return TestEntry(
                    16*1024*169,
                    Enumerable.Repeat(1, 169).ToArray(),
                    0);

                yield return TestEntry(
                    16*1024*(168*2 + 1),
                    Enumerable.Repeat(1, 168*2 + 1).ToArray(),
                    0);

                // -------------------------------------------
                // ディレクトリ
                yield return TestEntry(
                    0,
                    new int[0],
                    1
                );

                yield return TestEntry(
                    0,
                    new int[0],
                    167
                );

                yield return TestEntry(
                    0,
                    new int[0],
                    168
                );

                yield return TestEntry(
                    0,
                    new int[0],
                    169
                );

                yield return TestEntry(
                    0,
                    new int[0],
                    168*2 + 1
                );
            }
        }

        [Theory]
        [MemberData(nameof(SaveDataSizeTestData))]
        public void SaveDataSize(int expected, int[] fileSizes, int dirCount)
        {
            var c = new SaveDataSizeCalculator();

            foreach (var fileSize in fileSizes)
                c.AddFileEntry(fileSize);

            for (var i = 0; i != dirCount; ++i)
                c.AddDirectoryEntry();

            Assert.Equal(expected, c.SaveDataSize);
        }
    }
}
