﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.IO;
using System.Linq;
using System.Text;
using System.Xml;
using System.Xml.Linq;
using System.Xml.XPath;
using Livet.Converters;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Xunit;
using Xunit.Sdk;

namespace Nintendo.Authoring.AuthoringEditor.Core.Test
{
    public class ApplicationMetaTest : IDisposable
    {
        private readonly TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        [Fact]
        public void DefaultCtor()
        {
            using (var meta = new ApplicationMeta())
            {
                Assert.NotNull(meta.Core);
                Assert.NotNull(meta.Application);
                Assert.NotNull(meta.CardSpec);
            }
        }

        [Theory]
        [InlineData(true, ParentalControlType.FreeCommunication)]
        [InlineData(false, ParentalControlType.None)]
        public void MakeXmlForAuthoringTool_ParentalControl(bool contains, ParentalControlType t)
        {
            using (var meta = new ApplicationMeta {Application = {ParentalControl = t}})
            {
                var xml = _context.MakeAppMetaXml(meta);

                Assert.Equal(contains, xml.Contains("ParentalControl"));
            }
        }

        [Theory]
        [InlineData("Allow", true)]
        [InlineData("Deny", false)]
        public void MakeXmlForAuthoringTool_Screenshot(string expected, bool t)
        {
            using (var meta = new ApplicationMeta {Application = {IsAllowScreenshot = t}})
            {
                var xml = _context.MakeAppMetaXml(meta);

                var doc = XDocument.Parse(xml);

                var s = doc.Descendants("Screenshot").Select(x => x.Value).First();

                Assert.Equal(expected, s);
            }
        }

        [Fact]
        public void MakeXmlForAuthoringTool_SupportedLanguages()
        {
            using (var meta = new ApplicationMeta())
            {
                var xml = _context.MakeAppMetaXml(meta);

                Assert.DoesNotContain("SupportedLanguages", xml);

                var doc = XDocument.Parse(xml);

                var langs = doc.Descendants("SupportedLanguage").Any();

                Assert.False(langs);
            }
        }

        [Theory]
        [InlineData(null, false, false)]
        [InlineData("ABC", false, false)]
        [InlineData("", true, true)]
        [InlineData("ABC", true, true)]
        public void MakeXmlForAuthoringTool_ApplicationErrorCodeCategory(string category,
            bool isUseApplicationErrorCodeCategory,
            bool contains)
        {
            using (
                var meta = new ApplicationMeta
                {
                    Application =
                    {
                        IsUseApplicationErrorCode = isUseApplicationErrorCodeCategory,
                        ApplicationErrorCodeCategory = category,
                    }
                })
            {
                var xml = _context.MakeAppMetaXml(meta);

                Assert.Equal(contains, xml.Contains("ApplicationErrorCodeCategory"));
            }
        }

        [Fact]
        public void MakeXmlForAuthoringTool_ApplicationId()
        {
            using (
                var meta = new ApplicationMeta {Core = {ApplicationId = 0x0005000C10000000}})
            {
                var xml = _context.MakeAppMetaXml(meta);
                var doc = XDocument.Parse(xml);

                var id = doc.Descendants("ApplicationId").First().Value;

                Assert.Equal("0x0005000c10000000", id);
            }
        }

        [Fact]
        public void MakeXmlForAuthoringTool_Title()
        {
            using (var meta = new ApplicationMeta
            {
                Application =
                {
                    Titles = new ObservableCollection<Title>
                    {
                        new Title {Language = LanguageType.Japanese, Name = "タイトル", Publisher = "パブリッシャー"},
                        new Title {Language = LanguageType.AmericanEnglish, Name = "Title", Publisher = "Publisher"},
                    }
                }
            })
            {
                var xml = _context.MakeAppMetaXml(meta);
                var doc = XDocument.Parse(xml);

                var titles = doc.Descendants("Title").ToArray();
                Assert.Equal(2, titles.Length);

                Assert.Equal("Japanese", titles[0].Descendants("Language").First().Value);
                Assert.Equal("タイトル", titles[0].Descendants("Name").First().Value);
                Assert.Equal("パブリッシャー", titles[0].Descendants("Publisher").First().Value);

                Assert.Equal("AmericanEnglish", titles[1].Descendants("Language").First().Value);
                Assert.Equal("Title", titles[1].Descendants("Name").First().Value);
                Assert.Equal("Publisher", titles[1].Descendants("Publisher").First().Value);
            }
        }

        [Theory]
        [InlineData("True", ProcessAddressSpaceType.AddressSpace64Bit)]
        [InlineData("False", ProcessAddressSpaceType.AddressSpace32Bit)]
        [InlineData("False", ProcessAddressSpaceType.AddressSpace32BitNoReserved)]
        public void MakeXmlForAuthoringTool_Is64BitInstruction_ProcessAddressSpace(string is64BitInstruction,
            ProcessAddressSpaceType type)
        {
            using (var meta = new ApplicationMeta
            {
                Core = {ProcessAddressSpace = type}
            })
            {
                var xml = _context.MakeAppMetaXml(meta);
                var doc = XDocument.Parse(xml);

                var i = doc.Descendants("Is64BitInstruction").Select(x => x.Value).FirstOrDefault();

                Assert.Equal(is64BitInstruction, i);
            }
        }

        [Theory]
        [InlineData(ProcessAddressSpaceType.AddressSpace64Bit, "True", true)]
        [InlineData(ProcessAddressSpaceType.AddressSpace32Bit, "False", false)]
        public void MakeXmlForAuthoringTool_IsProcessAddressSpace64Bit(ProcessAddressSpaceType type,
            string is64BitInstruction, bool isProcessAddressSpace64Bit)
        {
            using (var meta = new ApplicationMeta
            {
                Core = {ProcessAddressSpace = type}
            })
            {
                var xml = _context.MakeAppMetaXml(meta);
                var doc = XDocument.Parse(xml);

                var i =
                    doc.Descendants("ProcessAddressSpace")
                        .Select(x => Enum.Parse(typeof(ProcessAddressSpaceType), x.Value))
                        .FirstOrDefault();
                Assert.Equal(type, i);

                var j = doc.Descendants("Is64BitInstruction").Select(x => x.Value).FirstOrDefault();
                Assert.Equal(is64BitInstruction, j);
            }
        }

        [Theory]
        [InlineData(null, true)]
        [InlineData("0x000000000000abcd", false)]
        public void MakeXmlForAuthoringTool_PresenceGroupId(string value, bool isUse)
        {
            using (
                var meta = new ApplicationMeta
                {
                    Core =
                    {
                        ApplicationId = 0x0000000000006789
                    },
                    Application =
                    {
                        PresenceGroupId = 0x000000000000ABCD,
                        IsUsePresenceGroupIdAppId = isUse
                    }
                })
            {
                var xml = _context.MakeAppMetaXml(meta);
                var doc = XDocument.Parse(xml);

                var id = doc.Descendants("PresenceGroupId").FirstOrDefault()?.Value;

                Assert.Equal(value, id);
            }
        }

        [Fact]
        public void MakeXmlForAuthoringTool_PresenceGroupId_Empty()
        {
            using (
                var meta = new ApplicationMeta
                {
                    Core =
                    {
                        ApplicationId = 0x0000000000006789
                    },
                })
            {
                var xml = _context.MakeAppMetaXml(meta);
                var doc = XDocument.Parse(xml);

                Assert.Equal(0, doc.Descendants("PresenceGroupId").Count());

                var inputMeta = _context.LoadAppMeta(xml);
                Assert.True(inputMeta.Application.IsUsePresenceGroupIdAppId);
            }
        }


        [Theory]
        [InlineData("0x0000000000000000", false)]
        [InlineData("0x0000000000001123", true)]
        public void MakeXmlForAuthoringTool_SaveDataSize(string value, bool isUse)
        {
            using (
                var meta = new ApplicationMeta
                {
                    Application =
                    {
                        SaveDataSize = 0x0000000000001123,
                        IsUseSaveData = isUse
                    }
                })
            {
                var xml = _context.MakeAppMetaXml(meta);
                var doc = XDocument.Parse(xml);

                var size = doc.Descendants("UserAccountSaveDataSize").First().Value;

                Assert.Equal(value, size);
            }
        }

        [Theory]
        [InlineData("0x0000000000001000", true, false)]
        [InlineData("0x0000000000002123", true, true)]
        [InlineData("0x0000000000002123", true, true)]
        [InlineData("0x0000000000002123", false, true)]
        [InlineData("0x0000000000000000", false, false)]
        public void MakeXmlForAuthoringTool_SaveDataJournalSize(string value, bool isUseSaveData,
            bool isJournalSpecified)
        {
            using (
                var meta = new ApplicationMeta
                {
                    Application =
                    {
                        SaveDataSize = 0x0000000000001000,
                        SaveDataJournalSize = 0x0000000000002123,
                        IsUseSaveData = isUseSaveData,
                        IsSpecifiedSaveDataJournal = isJournalSpecified
                    }
                })
            {
                var xml = _context.MakeAppMetaXml(meta);
                var doc = XDocument.Parse(xml);

                var size = doc.Descendants("UserAccountSaveDataJournalSize").First().Value;

                Assert.Equal(value, size);
            }
        }

        [Fact]
        public void MakeXmlForAuthoringTool_LocalCommunicationId()
        {
            using (var meta = new ApplicationMeta())
            {
                meta.Application.LocalCommunicationIds.Add(new NintendoApplicationId {Id = 0x000000000000AAAA});
                meta.Application.LocalCommunicationIds.Add(new NintendoApplicationId {Id = 0x000000000000BBBB});

                var xml = _context.MakeAppMetaXml(meta);
                var doc = XDocument.Parse(xml);

                var ids = doc.Descendants("LocalCommunicationId").Select(x => x.Value).ToArray();

                Assert.Equal(2, ids.Length);

                Assert.Equal("0x000000000000aaaa", ids[0]);
                Assert.Equal("0x000000000000bbbb", ids[1]);
            }
        }

        [Fact]
        public void MakeXmlForAuthoringTool_Ratings()
        {
            using (var meta = new ApplicationMeta())
            {
                var cero = meta.Application.Ratings.First(x => x.Organization == "CERO");
                cero.Age = 18;
                cero.IsUse = true;

                var classInd = meta.Application.Ratings.First(x => x.Organization ==  "ClassInd");
                classInd.Age = 10;
                classInd.IsUse = true;

                meta.Application.Ratings.Add(new Rating { Organization = "ZZZ", Age = 0, IsUse = true });

                var esrb = meta.Application.Ratings.First(x => x.Organization ==  "ESRB");
                esrb.Age = 0;
                esrb.IsUse = true;

                var xml = _context.MakeAppMetaXml(meta);
                var doc = XDocument.Parse(xml);

                var ratings = doc.Descendants("Rating").ToArray();

                Assert.Equal(3, ratings.Length);

                Assert.Equal("CERO", ratings[0].Descendants("Organization").First().Value);
                Assert.Equal("18", ratings[0].Descendants("Age").First().Value);
                Assert.Equal("ClassInd", ratings[1].Descendants("Organization").First().Value);
                Assert.Equal("10", ratings[1].Descendants("Age").First().Value);
                Assert.Equal("ESRB", ratings[2].Descendants("Organization").First().Value);
                Assert.Equal("0", ratings[2].Descendants("Age").First().Value);
            }
        }

        [Fact]
        public void MakeXmlForAuthoringTool_Bom()
        {
            using (var project = new Project())
            using (var meta = new ApplicationMeta())
            {
                var outputPath = Path.Combine(_context.TempDirPath, "test.xml");

                project.Meta = meta;
                project.OutputAppMetaXmlFileForAuthoringTool(outputPath);

                Assert.True(File.Exists(outputPath));

                var outputed = File.ReadAllBytes(outputPath);
                Assert.Equal(0xEF, outputed[0]);
                Assert.Equal(0xBB, outputed[1]);
                Assert.Equal(0xBF, outputed[2]);
            }
        }

        [Fact]
        public void MakeXmlForAuthoringTool_NoXmlNamespace()
        {
            using (var meta = new ApplicationMeta())
            {
                var xml = _context.MakeAppMetaXml(meta);

                Assert.False(xml.Contains("xmlns:xsi"));
                Assert.False(xml.Contains("xmlns:xsd"));
            }
        }

        [Fact]
        public void OutputXmlForAuthoringToolError()
        {
            using (var meta = new ApplicationMeta())
            {
                Assert.False(_context.OutputAppMetaXml(meta, @"aa:\"));
            }
        }

        [Theory]
        [InlineData(false, true)]
        [InlineData(true, false)]
        public void MakeXmlForAuthoringTool_CardSpecSize(bool isContains, bool isAutomaticSetting)
        {
            using (var meta = new ApplicationMeta())
            {
                meta.CardSpec.IsAutomaticSettingSize = isAutomaticSetting;

                var xml = _context.MakeAppMetaXml(meta);
                var doc = XDocument.Parse(xml);

                Assert.Equal(isContains, doc.Root.XPathSelectElements("CardSpec/Size").Any());
            }
        }

        [Theory]
        [InlineData(false, true)]
        [InlineData(true, false)]
        public void MakeXmlForAuthoringTool_CardSpecClockRate(bool isContains, bool isAutomaticSetting)
        {
            using (var meta = new ApplicationMeta())
            {
                meta.CardSpec.IsAutomaticSettingClockRate = isAutomaticSetting;

                var xml = _context.MakeAppMetaXml(meta);
                var doc = XDocument.Parse(xml);

                Assert.Equal(isContains, doc.Root.XPathSelectElements("CardSpec/ClockRate").Any());
            }
        }

        [Theory]
        [InlineData(true, null, 0)]
        [InlineData(true, "path/to/hoge", 1)]
        [InlineData(true, null, 0)]
        [InlineData(true, "path/to/hoge", 1)]
        public void MakeXmlForAuthoringTool_HtmlDocumentPath(bool isReplace, string path, int elementCount)
        {
            using (var meta = new ApplicationMeta())
            {
                meta.Application.IsReplaceHtmlDocumentPath = isReplace;
                meta.Application.HtmlDocumentPath = path;

                var xml = _context.MakeAppMetaXml(meta);
                var doc = XDocument.Parse(xml);

                Assert.Equal(elementCount, doc.Root.XPathSelectElements("Application/HtmlDocumentPath").Count());
            }
        }

        [Theory]
        [InlineData(true, null, 0)]
        [InlineData(true, "path/to/urls.txt", 1)]
        public void MakeXmlForAuthoringTool_AccessibleUrlsFile(bool isReplace, string path, int elementCount)
        {
            using (var meta = new ApplicationMeta())
            {
                meta.Application.IsReplaceAccessibleUrlsFilePath = isReplace;
                meta.Application.AccessibleUrlsFilePath = path;

                var xml = _context.MakeAppMetaXml(meta);
                var doc = XDocument.Parse(xml);

                Assert.Equal(elementCount, doc.Root.XPathSelectElements("Application/AccessibleUrlsFilePath").Count());
            }
        }

        [Theory]
        [InlineData(true, null, 0)]
        [InlineData(true, "", 0)]
        [InlineData(true, "a", 1)]
        [InlineData(true, "path/to/legal.zip", 1)]
        [InlineData(false, "path/to/legal.zip", 0)]
        public void MakeXmlForAuthoringTool_LegalInformation(bool isReplace, string path, int elementCount)
        {
            using (var meta = new ApplicationMeta())
            {
                meta.Application.IsReplaceLegalInformationFilePath = isReplace;
                meta.Application.LegalInformationFilePath = path;

                var xml = _context.MakeAppMetaXml(meta);
                var doc = XDocument.Parse(xml);

                Assert.Equal(elementCount,
                    doc.Root.XPathSelectElements("Application/LegalInformationFilePath").Count());
            }
        }

        [Theory]
        [InlineData(LogoHandlingType.Auto)]
        [InlineData(LogoHandlingType.Manual)]
        public void MakeXmlForAuthoringTool_LogoHandling(LogoHandlingType handlingType)
        {
            using (var meta = new ApplicationMeta())
            {
                meta.Application.LogoHandling = handlingType;

                var xml = _context.MakeAppMetaXml(meta);
                var doc = XDocument.Parse(xml);

                Assert.Equal(handlingType.ToString(),
                    doc.Root.XPathSelectElements("Application/LogoHandling").First().Value);

                var inputMeta = _context.LoadAppMeta(xml);
                Assert.Equal(handlingType, inputMeta.Application.LogoHandling);
            }
        }

        [Fact]
        public void MakeXmlForAuthoringTool_Attribute()
        {
            using (var meta = new ApplicationMeta())
            {
                meta.Application.Attribute = new[] {"Demo"};

                var xml = _context.MakeAppMetaXml(meta);
                var doc = XDocument.Parse(xml);

                Assert.Equal("Demo", doc.Root.XPathSelectElements("Application/Attribute").First().Value);

                var inputMeta = _context.LoadAppMeta(xml);
                Assert.Equal(new[] {"Demo"}, inputMeta.Application.Attribute);
            }
        }

        [Fact]
        public void MakeXmlForAuthoringTool_NoAttribute()
        {
            using (var meta = new ApplicationMeta())
            {
                meta.Application.Attribute = new string[] { };

                var xml = _context.MakeAppMetaXml(meta);
                var doc = XDocument.Parse(xml);

                Assert.Equal(0, doc.Root.XPathSelectElements("Application/Attribute").Count());

                var inputMeta = _context.LoadAppMeta(xml);
                Assert.Equal(new string[] { }, inputMeta.Application.Attribute);
            }
        }

        [Fact]
        public void MakeXmlForAuthoringTool_CardSpecOmitted()
        {
            using (var meta = new ApplicationMeta())
            {
                var xml = _context.MakeAppMetaXml(meta);
                var doc = XDocument.Parse(xml);

                Assert.Equal(0, doc.Root.XPathSelectElements("CardSpec").Count());

                var inputMeta = _context.LoadAppMeta(xml);
                Assert.True(inputMeta.CardSpec.IsAutomaticSettingClockRate);
                Assert.True(inputMeta.CardSpec.IsAutomaticSettingSize);
            }
        }

        [Fact]
        public void MakeXmlForAuthoringTool_ReleaseVersion()
        {
            using (var meta = new ApplicationMeta())
            {
                meta.Application.ReleaseVersion = 1024;

                var xml = _context.MakeAppMetaXml(meta);
                var doc = XDocument.Parse(xml);

                Assert.Equal("1024", doc.Root.XPathSelectElements("Application/ReleaseVersion").First().Value);
            }
        }

        [Fact]
        public void MakeXmlForAuthoringTool_PowerConsumptionRemoved()
        {
            var tempMetaXml = Path.Combine(_context.TempDirPath, "temp.nmeta");
            using (var meta = new ApplicationMeta())
            {
                var xml = _context.MakeAppMetaXml(meta);
                var doc = XDocument.Parse(xml);

                doc.Root?.Element("Application")?.Add(new XElement("PowerConsumption", "Normal"));
                Assert.Equal("Normal", doc.Root.XPathSelectElements("Application/PowerConsumption").First().Value);

                doc.Save(tempMetaXml);
            }
            using (var meta = _context.LoadAppMeta(File.ReadAllText(tempMetaXml)))
            {
                var xml = _context.MakeAppMetaXml(meta);
                var doc = XDocument.Parse(xml);
                // SIGLO-62682 の変更によって ApplicationMeta に存在しない要素は削除されず保持されるようになったため
                // PowerConsumption 要素は存在し続けるが、仕様
                Assert.NotEmpty(doc.Root.XPathSelectElements("Application/PowerConsumption"));
            }
        }

        [Fact]
        public void MakeXmlForAuthoringTool_SeedForPseudoDeviceId()
        {
            var isSupport = _context.DiContainer.GetInstance<Project>().AppCapability.IsSupportSeedForPseudoDeviceId;

            using (var meta = new ApplicationMeta())
            {
                {
                    var xml = _context.MakeAppMetaXml(meta);
                    var doc = XDocument.Parse(xml);
                    Assert.Equal(0, doc.Root.XPathSelectElements("Application/SeedForPseudoDeviceId").Count());
                }

                meta.Application.SeedForPseudoDeviceId = 1234ul;
                meta.Application.IsUseSeedForPseudoDeviceAppId = false;

                {
                    var xml = _context.MakeAppMetaXml(meta);
                    var doc = XDocument.Parse(xml);
                    var deviceId = doc.Root.XPathSelectElements("Application/SeedForPseudoDeviceId")?.FirstOrDefault()
                        ?.Value;
                    if (isSupport)
                        Assert.Equal(1234ul, deviceId.ToUlong());
                    else
                        Assert.Null(deviceId);

                    var appMeta = _context.LoadAppMeta(xml);
                    Assert.False(appMeta.Application.IsUseSeedForPseudoDeviceAppId);
                    Assert.Equal(1234ul, appMeta.Application.SeedForPseudoDeviceId);
                }

            }
        }

        [Theory]
        [InlineData(true)]
        [InlineData(false)]
        public void MakeXmlForAuthoringTool_SeedForPseudoDeviceId_SameAppId(bool isUseAppId)
        {
            using (var meta = new ApplicationMeta())
            {
                meta.Application.SeedForPseudoDeviceId = meta.Core.ApplicationId;
                meta.Application.IsUseSeedForPseudoDeviceAppId = isUseAppId;

                {
                    var xml = _context.MakeAppMetaXml(meta);
                    var doc = XDocument.Parse(xml);
                    Assert.Equal(0, doc.Root.XPathSelectElements("Application/SeedForPseudoDeviceId").Count());

                    var appMeta = _context.LoadAppMeta(xml);
                    Assert.True(appMeta.Application.IsUseSeedForPseudoDeviceAppId);
                    Assert.Equal(meta.Core.ApplicationId, appMeta.Application.SeedForPseudoDeviceId);
                }
            }
        }


        [Fact]
        public void MakeXmlForAuthoringTool_FsAccessControlData()
        {
            var isSupport = _context.DiContainer.GetInstance<Project>().AppCapability.IsSupportSaveDataOwnerIds;

            using (var meta = new ApplicationMeta())
            {
                meta.DiContainer = _context.DiContainer;
                {
                    var xml = _context.MakeAppMetaXml(meta);
                    var doc = XDocument.Parse(xml);
                    Assert.Equal(0, doc.Root.XPathSelectElements("Core/FsAccessControlData").Count());
                }

                meta.Core.FsAccessControlData = new FsAccessControlData();
                meta.Core.FsAccessControlData.SaveDataOwnerIds.Add(new SaveDataOwnerId()
                {
                    Accessibility = AccessibilityType.Read,
                    ApplicationId = 0x1000,
                });
                meta.Core.FsAccessControlData.SaveDataOwnerIds.Add(new SaveDataOwnerId()
                {
                    Accessibility = AccessibilityType.Read,
                    ApplicationId = 0xdeadbeef,
                });

                {
                    var xml = _context.MakeAppMetaXml(meta);
                    var doc = XDocument.Parse(xml);

                    if (isSupport == false)
                    {
                        Assert.Empty(doc.Root.XPathSelectElements("Core/SaveDataOwnerIds"));
                        Assert.Empty(doc.Root.XPathSelectElements("Core/FsAccessControlData"));
                        return;
                    }

                    Assert.Equal(1, doc.Root.XPathSelectElements("Core/FsAccessControlData").Count());
                    Assert.Equal("Debug",
                        doc.Root.XPathSelectElements("Core/FsAccessControlData/FlagPresets").FirstOrDefault()?.Value);

                    var saveDataOwnerIds = doc.Root.XPathSelectElements("Core/FsAccessControlData/SaveDataOwnerIds");

                    {
                        var saveDataOwnerId = saveDataOwnerIds.ElementAt(0);
                        Assert.NotNull(saveDataOwnerId);
                        Assert.Equal("Read", saveDataOwnerId.Element("Accessibility")?.Value);
                        Assert.Equal(0x1000ul.ToHex(), saveDataOwnerId.Element("Id")?.Value);
                    }

                    {
                        var saveDataOwnerId = saveDataOwnerIds.ElementAt(1);
                        Assert.NotNull(saveDataOwnerId);
                        Assert.Equal("Read", saveDataOwnerId.Element("Accessibility")?.Value);
                        Assert.Equal(0xdeadbeeful.ToHex(), saveDataOwnerId.Element("Id")?.Value);
                    }
                }
            }
        }

        [Fact]
        public void MakeXmlForAuthoringTool_FsAccessControlData_FlagPresets()
        {
            var isSupport = _context.DiContainer.GetInstance<Project>().AppCapability.IsSupportSaveDataOwnerIds;

            var defaultMetaXml = File.ReadAllText(NintendoSdkHelper.ApplicationMetaFilePath, Encoding.UTF8);
            using (var meta = _context.LoadAppMeta(defaultMetaXml))
            {
                Assert.False(meta.Core.FsAccessControlData.IsUseFlagPresets);
                Assert.Equal(FsAccessControlData.GetDefaultFlagPresets(), meta.Core.FsAccessControlData.FlagPresets);
            }

            if (isSupport == false)
                return;

            using (var meta = new ApplicationMeta())
            {
                meta.DiContainer = _context.DiContainer;

                Assert.False(meta.Core.FsAccessControlData.IsUseFlagPresets);
                Assert.Equal(FsAccessControlData.GetDefaultFlagPresets(), meta.Core.FsAccessControlData.FlagPresets);

                // 値が空の場合、デフォルト設定を上書きして「製品環境と同様」の動作をする
                meta.Core.FsAccessControlData.IsUseFlagPresets = true;
                meta.Core.FsAccessControlData.FlagPresets = string.Empty;
                {
                    var xml = _context.MakeAppMetaXml(meta);
                    var doc = XDocument.Parse(xml);
                    Assert.Equal(1, doc.Root.XPathSelectElements("Core/FsAccessControlData").Count());
                    Assert.Empty(doc.Root.XPathSelectElements("Core/FsAccessControlData/FlagPresets"));

                    // FlagPresets を使用する限り、空文字列であっても値は書き出される
                    var emptyFlagPresetsAppMeta = _context.LoadAppMeta(xml);
                    Assert.True(emptyFlagPresetsAppMeta.Core.FsAccessControlData.IsUseFlagPresets);
                    Assert.Equal(string.Empty, emptyFlagPresetsAppMeta.Core.FsAccessControlData.FlagPresets);
                }

                meta.Core.FsAccessControlData.FlagPresets = "Test";
                {
                    var xml = _context.MakeAppMetaXml(meta);
                    var doc = XDocument.Parse(xml);
                    Assert.Equal("Test", doc.Root.XPathSelectElement("Core/FsAccessControlData/FlagPresets").Value);
                }
                meta.Core.FsAccessControlData.IsUseFlagPresets = false;
                {
                    var xml = _context.MakeAppMetaXml(meta);
                    var doc = XDocument.Parse(xml);
                    Assert.Empty(doc.Root.XPathSelectElements("Core/FsAccessControlData"));
                    Assert.Empty(doc.Root.XPathSelectElements("Core/FsAccessControlData/FlagPresets"));
                }
            }
        }

        [Theory]
        [InlineData("", true)]
        [InlineData("a", true)]
        [InlineData("Test", true)]
        [InlineData("Debug", false)]
        public void MakeXmlForAuthoringTool_FsAccessControlData_FlagPresetsWithSaveDataOwnerIds(string flagPresets,
            bool isUseFlagPresets)
        {
            if (_context.GetInstance<ApplicationCapability>().IsSupportSaveDataOwnerIds == false)
                return;

            using (var meta = new ApplicationMeta())
            {
                meta.DiContainer = _context.DiContainer;

                meta.Core.FsAccessControlData.SaveDataOwnerIds.Add(new SaveDataOwnerId {ApplicationId = 0x1ul});
                meta.Core.FsAccessControlData.IsUseFlagPresets = isUseFlagPresets;
                meta.Core.FsAccessControlData.FlagPresets = flagPresets;
                {
                    var xml = _context.MakeAppMetaXml(meta);
                    var doc = XDocument.Parse(xml);

                    Assert.NotEmpty(doc.Root.XPathSelectElements("Core/FsAccessControlData/SaveDataOwnerIds"));

                    if (string.IsNullOrEmpty(flagPresets))
                        Assert.Empty(doc.Root.XPathSelectElements("Core/FsAccessControlData/FlagPresets"));
                    else
                        Assert.Equal(flagPresets,
                            doc.Root.XPathSelectElement("Core/FsAccessControlData/FlagPresets").Value);

                    var appMeta = _context.LoadAppMeta(xml);
                    Assert.Equal(isUseFlagPresets, appMeta.Core.FsAccessControlData.IsUseFlagPresets);
                    Assert.Equal(flagPresets, appMeta.Core.FsAccessControlData.FlagPresets);
                }
            }
        }

        public static IEnumerable<object> PlayLogPolicyTypes = Enum.GetValues(typeof(PlayLogPolicyType))
            .Cast<object>()
            .Select(x => new[] {x});

        [Theory]
        [MemberData(nameof(PlayLogPolicyTypes))]
        public void MakeXmlForAuthoringTool_PlayLogPolicy(PlayLogPolicyType t)
        {
            using (var meta = new ApplicationMeta())
            {
                Assert.False(meta.Application.PlayLogPolicySpecified);
                {
                    var xml = _context.MakeAppMetaXml(meta);
                    var doc = XDocument.Parse(xml);
                    Assert.Empty(doc.Root.XPathSelectElements("Application/PlayLogPolicy"));
                }

                meta.Application.PlayLogPolicy = t;
                Assert.True(meta.Application.PlayLogPolicySpecified);
                {
                    var xml = _context.MakeAppMetaXml(meta);
                    var doc = XDocument.Parse(xml);
                    Assert.Equal(t.ToString(), doc.Root.XPathSelectElement("Application/PlayLogPolicy")?.Value);
                }
            }
        }


        [Fact]
        public void MakeXmlForAuthoringTool_SystemResourceSize_Default()
        {
            using (var meta = new ApplicationMeta())
            {
                {
                    var xmlText = _context.MakeAppMetaXml(meta);
                    var xml = XDocument.Parse(xmlText);

                    Assert.Empty(xml.Root.XPathSelectElements("Core/SystemResourceSize"));
                }
            }
        }

        [Theory]
        [InlineData(0x200000, true, "0x00200000")]
        [InlineData(0x400000, true, "0x00400000")]
        [InlineData(0x200000, false, null)]
        [InlineData(0x400000, false, null)]
        [InlineData(0x0, true, null)]
        public void MakeXmlForAuthoringTool_SystemResourceSize(uint size, bool is64Bit, string expected)
        {
            if (_context.GetInstance<ApplicationCapability>().IsSupportSystemResourceSize == false)
                return;

            using (var meta = new ApplicationMeta())
            {
                meta.Core.ProcessAddressSpace = is64Bit
                    ? ProcessAddressSpaceType.AddressSpace64Bit
                    : ProcessAddressSpaceType.AddressSpace32Bit;
                meta.Core.SystemResourceSize = size;

                var xmlText = _context.MakeAppMetaXml(meta);
                var xml = XDocument.Parse(xmlText);

                Assert.Equal(expected, xml.Root.XPathSelectElement("Core/SystemResourceSize")?.Value);

                var newMeta = _context.LoadAppMeta(xmlText);
                if (expected == null)
                    Assert.Equal(0u, newMeta.Core.SystemResourceSize);
                else
                    Assert.Equal(size, newMeta.Core.SystemResourceSize);
            }
        }

        [Fact]
        public void SaveDataValidationSkippedTest()
        {
            using (var meta = new ApplicationMeta()
            {
                DiContainer = _context.DiContainer
            })
            {
                Assert.True(meta.HasErrors);

                meta.Application.DisplayVersion = "1.0.0";
                meta.Application.SupportedLanguages.First().IsSupported = true;
                meta.Application.Titles.Add(new Title {Name = "name", Publisher = "publisher", IsReplaceIcon = false});
                Assert.False(meta.HasErrors);

                meta.Application.IsUseSaveData = true;
                Assert.False(meta.HasErrors);

                meta.Application.SaveDataSize = 1024 * (1024 - 1);
                Assert.False(meta.HasErrors);

                meta.Application.IsSpecifiedSaveDataJournal = true;
                meta.Application.SaveDataJournalSize = 1024 * 2 * (1024 - 1);
                Assert.False(meta.HasErrors);

                meta.Application.IsSpecifiedDeviceSaveDataSize = true;
                meta.Application.DeviceSaveDataSize = 1024 * (1024 - 1);
                Assert.False(meta.HasErrors);

                meta.Application.IsSpecifiedDeviceSaveDataJournalSize = true;
                meta.Application.DeviceSaveDataJournalSize = 1024 * 2 * (1024 - 1);
                Assert.False(meta.HasErrors);
            }
        }

        [Fact]
        public void ApplicationMetaRootElement()
        {
            var isSupportNintendoSdkMeta = _context.GetInstance<ApplicationCapability>().IsSupportNintendoSdkMetaRoot;
            using (var meta = new ApplicationMeta()
            {
                DiContainer = _context.DiContainer
            })
            {
                var xml = _context.MakeAppMetaXml(meta);
                var doc = XDocument.Parse(xml);
                if (isSupportNintendoSdkMeta)
                    Assert.Equal("NintendoSdkMeta", doc.Root.Name);
                else
                    Assert.Equal("Meta", doc.Root.Name);
            }
        }

        [Theory]
        [InlineData(nameof(Application.SaveDataSize), nameof(Application.IsUseSaveData))]
        [InlineData(nameof(Application.DeviceSaveDataSize), nameof(Application.IsSpecifiedDeviceSaveDataSize))]
        public void SaveDataValidationIsSkipped(string propertyName, string isUsePropertyName)
        {
            using (var meta = new ApplicationMeta()
            {
                DiContainer = _context.DiContainer
            })
            {
                var saveDataSize = 1024 * 16 - 1L;
                var property = typeof(Application).GetProperty(propertyName);
                var isUseProperty = typeof(Application).GetProperty(isUsePropertyName);

                meta.Application.DisplayVersion = "1.0.0";
                meta.Application.SupportedLanguages.First().IsSupported = true;
                meta.Application.Titles.Add(new Title {Name = "name", Publisher = "publisher", IsReplaceIcon = false});

                property.SetValue(meta.Application, saveDataSize);
                isUseProperty.SetValue(meta.Application, true);

                Assert.False(meta.HasErrors);

                var xml = _context.MakeAppMetaXml(meta);

                var newMeta = _context.LoadAppMeta(xml);
                Assert.True((bool) isUseProperty.GetValue(newMeta.Application));
                Assert.Equal(saveDataSize, (long) property.GetValue(newMeta.Application));
                Assert.False(newMeta.HasErrors);
            }
        }

        [Fact]
        public void MakeXmlForAuthoringTool_Bcat()
        {
            if (_context.GetInstance<ApplicationCapability>().IsSupportBcat == false)
                return;

            using (var meta = new ApplicationMeta())
            {
                {
                    var xml = _context.MakeAppMetaXml(meta);
                    var doc = XDocument.Parse(xml);
                    Assert.Empty(doc.Root.XPathSelectElements("Application/BcatDeliveryCacheStorageSize"));
                    Assert.Empty(doc.Root.XPathSelectElements("Application/BcatPassphrase"));
                }

                var testPassphrase = "2be7b01684f98f8a668d9f474c6583a44c418bb72be7b01684f98f8a668d9f40";

                meta.Application.IsUseBcat = true;
                meta.Application.BcatDeliveryCacheStorageSize = 1024 * 1024 * 5;
                meta.Application.BcatPassphrase = testPassphrase;

                ApplicationMeta newMeta;
                {
                    var xml = _context.MakeAppMetaXml(meta);
                    var doc = XDocument.Parse(xml);
                    Assert.Equal((1024 * 1024 * 5L).ToHex(),
                        doc.Root.XPathSelectElement("Application/BcatDeliveryCacheStorageSize")?.Value);
                    Assert.Equal(testPassphrase, doc.Root.XPathSelectElement("Application/BcatPassphrase")?.Value);

                    newMeta = _context.LoadAppMeta(xml);
                }

                Assert.True(newMeta.Application.IsUseBcat);
                Assert.Equal(1024 * 1024 * 5L, newMeta.Application.BcatDeliveryCacheStorageSize);
                Assert.Equal(testPassphrase, newMeta.Application.BcatPassphrase);
            }
        }

        [Fact]
        public void MakeXmlForAuthoringTool_Hdcp()
        {
            if (_context.GetInstance<ApplicationCapability>().IsSupportHdcp == false)
                return;

            using (var meta = new ApplicationMeta())
            {
                {
                    var xml = _context.MakeAppMetaXml(meta);
                    var doc = XDocument.Parse(xml);
                    Assert.Empty(doc.Root.XPathSelectElements("Application/Hdcp"));
                }

                meta.Application.Hdcp = Application.HdcpType.Required;

                {
                    var xml = _context.MakeAppMetaXml(meta);
                    var doc = XDocument.Parse(xml);
                    Assert.Equal("Required", doc.Root.XPathSelectElement("Application/Hdcp")?.Value);
                }
            }
        }

        [Fact]
        public void MakeXmlForAuthoringTool_VideoCapture()
        {
            using (var project = Project.CreateNew(_context.DiContainer))
            {
                var xml = _context.MakeAppMetaXml(project.Meta);
                var doc = XDocument.Parse(xml);
                if (project.AppCapability.IsSupportVideoCapture == false)
                {
                    Assert.Null(doc.Root.XPathSelectElement("Application/VideoCapture"));
                    return;
                }

                var el = doc.Root.XPathSelectElement("Application/VideoCapture");
                Assert.NotNull(el);
                Assert.Equal("Enable", el.Value);

                {
                    // 設定値がそのまま ApplicationMeta に設定されるが、ファイルオープン時に例外が発生する
                    el.Value = "Allow";
                    Assert.Equal(VideoCaptureType.Allow, _context.LoadAppMeta(doc.ToString()).Application.VideoCapture);
                    el.Value = "Deny";
                    Assert.Equal(VideoCaptureType.Deny, _context.LoadAppMeta(doc.ToString()).Application.VideoCapture);
                    el.Remove();
                    Assert.Equal(VideoCaptureType.Enable,
                        _context.LoadAppMeta(doc.ToString()).Application.VideoCapture);
                }
            }
        }

        [Fact]
        public void MakeXmlForAuthoringTool_RuntimeAddOnContentInstall()
        {
            using (var project = Project.CreateNew(_context.DiContainer))
            {
                var xml = _context.MakeAppMetaXml(project.Meta);
                var doc = XDocument.Parse(xml);

                var el = doc.Root.XPathSelectElement("Application/RuntimeAddOnContentInstall");
                if (project.AppCapability.IsSupportRuntimeAddOnContentInstall == false)
                {
                    Assert.Null(el);
                    return;
                }

                Assert.NotNull(el);
                Assert.Equal("Deny", el.Value);
            }
        }

        [Theory]
        [InlineData(null, null)]
        [InlineData(CrashReportType.Allow, nameof(CrashReportType.Allow))]
        [InlineData(CrashReportType.Deny, nameof(CrashReportType.Deny))]
        public void MakeXmlForAuthoringTool_CrashReport(CrashReportType? reportType, string expected)
        {
            using (var project = Project.CreateNew(_context.DiContainer))
            {
                project.Meta.Application.CrashReport = reportType;

                var xml = _context.MakeAppMetaXml(project.Meta);
                var doc = XDocument.Parse(xml);

                var el = doc.Root.XPathSelectElement("Application/CrashReport");
                if (project.AppCapability.IsSupportCrashReport == false)
                {
                    Assert.Null(el);
                    return;
                }
                if (expected == null)
                {
                    Assert.Null(el);
                    return;
                }
                Assert.NotNull(el);
                Assert.Equal(expected, el.Value);
            }
        }

        [Fact]
        public void UnknownNmetaItems()
        {
            using (var meta = new ApplicationMeta())
            {
                var xml = _context.MakeAppMetaXml(meta);
                var doc = XDocument.Parse(xml);
                doc.Root.Element("Application").Add(new XElement("Foobar", "Hello, World"));

                var newMeta = _context.LoadAppMeta(doc.ToString());
                Assert.NotEmpty(newMeta.Application.UnknownXmlElements);

                {
                    var e = newMeta.Application.UnknownXmlElements[0];
                    Assert.Equal("Foobar", e.Name);
                    Assert.Equal("Hello, World", e.InnerText);
                }

                {
                    var newXml = _context.MakeAppMetaXml(newMeta);
                    XDocument.Parse(newXml);
                    var e = doc.Root.Element("Application").Element("Foobar");
                    Assert.NotNull(e);
                    Assert.Equal("Hello, World", e.Value);
                }
            }
        }

        [Theory]
        [InlineData(
            nameof(ApplicationCapability.IsSupportSaveDataSizeExtend),
            nameof(Application.UserAccountSaveDataSizeMax),
            nameof(Application.IsUseUserAccountSaveDataSizeMax),
            nameof(Application.UserAccountSaveDataJournalSizeMax),
            nameof(Application.IsUseUserAccountSaveDataJournalSizeMax)
            )]
        [InlineData(
            nameof(ApplicationCapability.IsSupportSaveDataSizeExtend),
            nameof(Application.DeviceSaveDataSizeMax),
            nameof(Application.IsUseDeviceSaveDataSizeMax),
            nameof(Application.DeviceSaveDataJournalSizeMax),
            nameof(Application.IsUseDeviceSaveDataJournalSizeMax)
        )]
        [InlineData(
            nameof(ApplicationCapability.IsSupportTempAndCacheStorage),
            nameof(Application.CacheStorageSize),
            nameof(Application.IsUseCacheStorageSize),
            nameof(Application.CacheStorageJournalSize),
            nameof(Application.IsUseCacheStorageJournalSize)
        )]
        [InlineData(
            nameof(ApplicationCapability.IsSupportTempAndCacheStorage),
            nameof(Application.TemporaryStorageSize),
            nameof(Application.IsUseTemporaryStorageSize),
            null,
            null
        )]
        public void MakeXmlForAuthoringTool_SaveData(string capability, string size, string isUse, string journalSize, string journalIsUse)
        {
            using (var project = Project.CreateNew(_context.DiContainer))
            {
                if (project.AppCapability.GetPropertyValue<bool>(capability) == false)
                    return;

                {
                    var doc = _context.MakeAppMetaXmlDocument(project.Meta);
                    Assert.Null(doc.Root.XPathSelectElement($"Application/{size}"));
                    if (!string.IsNullOrEmpty(journalSize))
                        Assert.Null(doc.Root.XPathSelectElement($"Application/{journalSize}"));
                }

                const long MiB = 1024 * 1024;

                var app = project.Meta.Application;

                app.SetPropertyValue(isUse, true);
                app.SetPropertyValue(size, MiB);

                {
                    var doc = _context.MakeAppMetaXmlDocument(project.Meta);
                    Assert.Equal(
                        MiB.ToHex(),
                        doc.Root.XPathSelectElement($"Application/{size}")?.Value);

                    var appMeta = _context.LoadAppMeta(doc.ToString());
                    Assert.Equal(MiB, appMeta.Application.GetPropertyValue<long>(size));
                    if (!string.IsNullOrEmpty(journalSize))
                        Assert.Equal(0L, appMeta.Application.GetPropertyValue<long>(journalSize));
                }

                if (!string.IsNullOrEmpty(journalSize))
                {
                    app.SetPropertyValue(journalIsUse, true);
                    app.SetPropertyValue(journalSize, 2 * MiB);

                    var doc = _context.MakeAppMetaXmlDocument(project.Meta);
                    Assert.Equal(
                        (2 * MiB).ToHex(),
                        doc.Root.XPathSelectElement($"Application/{journalSize}")?.Value);

                    var appMeta = _context.LoadAppMeta(doc.ToString());
                    Assert.Equal(MiB, appMeta.Application.GetPropertyValue<long>(size));
                    Assert.Equal(2 * MiB, appMeta.Application.GetPropertyValue<long>(journalSize));
                }

                app.SetPropertyValue(isUse, false);
                if (!string.IsNullOrEmpty(journalSize))
                    app.SetPropertyValue(journalIsUse, false);

                {
                    var doc = _context.MakeAppMetaXmlDocument(project.Meta);
                    Assert.Null(doc.Root.XPathSelectElement($"Application/{size}"));
                    if (!string.IsNullOrEmpty(journalSize))
                        Assert.Null(doc.Root.XPathSelectElement($"Application/{journalSize}"));
                }
            }
        }

        [Theory]
        [InlineData(false)]
        [InlineData(true)]
        public void MakeXmlForAuthoringTool_FilterDescriptionFilePath(bool isUse)
        {
            using (var project = Project.CreateNew(_context.DiContainer))
            {
                var app = project.Meta.Application;
                app.IsUseFilterDescriptionFilePath = isUse;
                app.FilterDescriptionFilePath = "filepath.fdf";

                var doc = _context.MakeAppMetaXmlDocument(project.Meta);
                var element = doc.Root.XPathSelectElement("Application/FilterDescriptionFilePath")?.Value;
                if (isUse)
                {
                    Assert.Equal(element, app.FilterDescriptionFilePath);
                }
                else
                {
                    Assert.Null(element);
                }
            }
        }

        [Fact]
        public void MakeXmlForAuthoringTool_PlayLogQuery()
        {
            using (var project = Project.CreateNew(_context.DiContainer))
            {
                if (project.AppCapability.IsSupportPlayLogs == false)
                    return;

                var app = project.Meta.Application;

                // None を指定した場合はその値が nmeta に残る、アプリ ID の一覧は空
                {
                    app.PlayLogQueryCapability = PlayLogQueryCapabilityType.None;
                    var root = _context.MakeAppMetaXmlDocument(project.Meta).Root;
                    var capability = root.XPathSelectElement("Application/PlayLogQueryCapability")?.Value;
                    Assert.Equal(nameof(PlayLogQueryCapabilityType.None), capability);
                    Assert.Empty(root.XPathSelectElements("Application/PlayLogQueryableApplicationId"));
                }

                // None 以外を指定してもアプリ ID が空の場合は None が格納される
                {
                    app.PlayLogQueryCapability = PlayLogQueryCapabilityType.All;
                    var root = _context.MakeAppMetaXmlDocument(project.Meta).Root;
                    var capability = root.XPathSelectElement("Application/PlayLogQueryCapability")?.Value;
                    Assert.Equal(nameof(PlayLogQueryCapabilityType.None), capability);
                    Assert.Empty(root.XPathSelectElements("Application/PlayLogQueryableApplicationId"));
                }
            }
        }

        [Theory]
        [InlineData(PlayLogQueryCapabilityType.All)]
        [InlineData(PlayLogQueryCapabilityType.WhiteList)]
        public void MakeXmlForAuthoringTool_PlayLogQuery_All(PlayLogQueryCapabilityType capabilityType)
        {
            using (var project = Project.CreateNew(_context.DiContainer))
            {
                if (project.AppCapability.IsSupportPlayLogs == false)
                    return;

                var app = project.Meta.Application;

                // None 以外を指定 && アプリ ID を指定しているときは、設定値と指定したアプリ ID が反映されていることを確認
                {
                    app.PlayLogQueryCapability = capabilityType;
                    app.PlayLogQueryableApplicationIds.Add(new NintendoApplicationId { Id = 0x1ul });
                    var root = _context.MakeAppMetaXmlDocument(project.Meta).Root;
                    var capability = root.XPathSelectElement("Application/PlayLogQueryCapability")?.Value;
                    var appId = root.XPathSelectElement("Application/PlayLogQueryableApplicationId")?.Value;
                    Assert.Equal(capabilityType.ToString(), capability);
                    Assert.Equal(0x1ul.ToHex(), appId);
                }
            }
        }

        [Fact]
        public void MakeXmlForAuthoringTool_PlayLogQuery_Empty()
        {
            using (var project = Project.CreateNew(_context.DiContainer))
            {
                if (project.AppCapability.IsSupportPlayLogs == false)
                    return;

                var app = project.Meta.Application;

                {
                    app.PlayLogQueryCapability = PlayLogQueryCapabilityType.WhiteList;
                    app.PlayLogQueryableApplicationIds.Add(new NintendoApplicationId { Id = 0x0ul });
                    app.PlayLogQueryableApplicationIds.Add(new NintendoApplicationId { Id = 0x1ul });
                    var doc = _context.MakeAppMetaXmlDocument(project.Meta);
                    var root = doc.Root;
                    var appIds = root.XPathSelectElements("Application/PlayLogQueryableApplicationId");
                    Assert.Equal(new[] {0x0ul.ToHex(), 0x1ul.ToHex()}, appIds.Select(x => x.Value));

                    var newApp = _context.LoadAppMeta(doc.ToString());
                    Assert.Equal(new[] {0x1ul}, newApp.Application.PlayLogQueryableApplicationIds.Select(x => x.Id));
                }
            }
        }

        [Fact]
        public void MakeXmlForAuthoringTool_RequiredNetworkServiceLicenseOnLaunch()
        {
            using (var project = Project.CreateNew(_context.DiContainer))
            {
                if (project.AppCapability.IsSupportRequiredNetworkServiceLicenseOnLaunch == false)
                    return;

                var app = project.Meta.Application;

                // 空の nmeta には要素は含まれない
                {
                    var doc = _context.MakeAppMetaXmlDocument(project.Meta);
                    Assert.Empty(doc.Root.XPathSelectElements("Application/RequiredNetworkServiceLicenseOnLaunch"));
                }

                // 値を設定すれば要素が含まれる
                {
                    app.RequiredNetworkServiceLicenseOnLaunch = RequiredNetworkServiceLicenseOnLaunchType.Common;
                    Assert.Equal(nameof(RequiredNetworkServiceLicenseOnLaunchType.Common),
                        _context.MakeAppMetaXmlDocument(project.Meta).Root?
                            .XPathSelectElement("Application/RequiredNetworkServiceLicenseOnLaunch")?.Value);

                    app.RequiredNetworkServiceLicenseOnLaunch = RequiredNetworkServiceLicenseOnLaunchType.None;
                    Assert.Equal(nameof(RequiredNetworkServiceLicenseOnLaunchType.None),
                        _context.MakeAppMetaXmlDocument(project.Meta).Root?
                        .XPathSelectElement("Application/RequiredNetworkServiceLicenseOnLaunch")?.Value);
                }

                // 無効な要素は null 値になり、XML に含まれなくなる
                {
                    var doc = _context.MakeAppMetaXmlDocument(project.Meta);
                    doc.Root.XPathSelectElement("Application/RequiredNetworkServiceLicenseOnLaunch").Value = "foobar";
                    var newApp = _context.LoadAppMeta(doc.ToString());
                    Assert.Null(newApp.Application.RequiredNetworkServiceLicenseOnLaunch);
                }
            }
        }
    }
}
