﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.IO;
using System.Linq;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Core.Test
{
    public class AocMetaTest : IDisposable
    {
        private readonly TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        [Fact]
        public void DefaultCtor()
        {
            using (var meta = new AocMeta())
            {
                Assert.NotNull(meta.Contents);
                Assert.Equal(0, meta.Contents.Count);
            }
        }

        [Fact]
        public void LoadFromString()
        {
            var metaFilePath = Path.Combine(TestContext.TestDataDirPath, @"metafile_test\AddOnContent.nmeta");
            var srcString = File.ReadAllText(metaFilePath);
            var meta = _context.LoadAocMeta(srcString);

            Assert.NotNull(meta);
            Assert.NotNull(meta.Contents);
            Assert.Equal(3, meta.Contents.Count);

            Assert.Equal(meta.ApplicationId, 0x0100cb3000062000ul);
        }

        [Fact]
        public void SaveToFile()
        {
            var metaFilePath = Path.Combine(TestContext.TestDataDirPath, @"metafile_test\AddOnContent.nmeta");
            var srcString = File.ReadAllText(metaFilePath);
            var project = _context.LoadProjectFromXml(srcString);

            var xml = project.MakeAocMetaXmlForAuthoringTool();

            var expected = TestContext.NormalizeXml(srcString);
            var actual = TestContext.NormalizeXml(xml);

            Assert.Equal(expected, actual);
        }

        [Fact]
        public void SaveToFile_RelativePath()
        {
            var metaFilePath = Path.Combine(TestContext.TestDataDirPath, @"metafile_test\AddOnContent.nmeta");
            var srcString = File.ReadAllText(metaFilePath);
            var project = _context.LoadProjectFromXml(srcString);

            var savedString = project.MakeAocMetaXmlForAuthoringTool();
            var saved = _context.LoadAocMeta(savedString);

            Assert.Equal("Index1Data", saved.Contents[0].DataPath.Path);
            Assert.Equal("Index2Data", saved.Contents[1].DataPath.Path);
            Assert.Equal("Index3Data", saved.Contents[2].DataPath.Path);

            foreach (var c in saved.Contents)
            {
                c.DataPath.Path = Path.Combine(_context.TempDirPath, c.DataPath.Path);
            }

            var tempMetaFile = Path.Combine(_context.TempDirPath, "AddOnContent.nmeta");

            {
                var app = _context.GetInstance<App>();
                app.Project = project;
                Assert.True(app.Save(ExportableFileType.AocMeta, tempMetaFile));
            }

            {
                var savedProject = Project.Import(_context.DiContainer, ImportableFileType.Meta, tempMetaFile).AocMeta;
                Assert.Equal("Index1Data", savedProject.Contents[0].DataPath.Path);
                Assert.Equal("Index2Data", savedProject.Contents[1].DataPath.Path);
                Assert.Equal("Index3Data", savedProject.Contents[2].DataPath.Path);
            }
        }

        [Fact]
        public void AddContent()
        {
            using (var aocMeta = new AocMeta { DiContainer = _context.DiContainer })
            {
                Assert.Equal(0, aocMeta.Contents.Count);

                aocMeta.AddContent();
                Assert.Equal(1, aocMeta.Contents.Count);
                Assert.Equal(1ul, aocMeta.Contents[0].Index);

                aocMeta.AddContent();
                Assert.Equal(2, aocMeta.Contents.Count);
                Assert.Equal(2ul, aocMeta.Contents[1].Index);

                aocMeta.Contents[1].Index = 1000;

                aocMeta.AddContent();
                Assert.Equal(3, aocMeta.Contents.Count);
                Assert.Equal(1001ul, aocMeta.Contents[2].Index);
            }
        }

        [Fact]
        public void DeleteContent()
        {
            using (var aocMeta = new AocMeta {DiContainer = _context.DiContainer})
            {
                aocMeta.AddContent();
                aocMeta.AddContent();
                aocMeta.AddContent();

                var c0 = aocMeta.Contents[0];
                var c1 = aocMeta.Contents[1];
                var c2 = aocMeta.Contents[2];

                Assert.Equal(3, aocMeta.Contents.Count);

                aocMeta.DeleteContent(c1);

                Assert.Equal(2, aocMeta.Contents.Count);
                Assert.Equal(1ul, aocMeta.Contents[0].Index);
                Assert.Equal(3ul, aocMeta.Contents[1].Index);

                aocMeta.DeleteContent(new AocContent());
                Assert.Equal(2, aocMeta.Contents.Count);

                aocMeta.DeleteContent(c0);
                aocMeta.DeleteContent(c2);
                Assert.Equal(0, aocMeta.Contents.Count);
            }
        }

        [Fact]
        public void ValidateContents()
        {
            Assert.Equal(AocMeta.ContentsValidationType.EmptyError, AocMeta.ValidateContents(null));

            var contents = new ObservableCollection<AocContent>();

            Assert.Equal(AocMeta.ContentsValidationType.EmptyError, AocMeta.ValidateContents(contents));

            contents.Add(new AocContent());

            Assert.Equal(AocMeta.ContentsValidationType.Ok, AocMeta.ValidateContents(contents));
        }

        public static IEnumerable<object> ValidationApplicationIdTestData
        {
            get
            {
                yield return
                    new object[] {AocMeta.ApplicationIdValidationType.RangeOver, Constants.ProgramIdMinimum - 1};
                yield return new object[] {AocMeta.ApplicationIdValidationType.Ok, Constants.ProgramIdMinimum};
                yield return new object[] {AocMeta.ApplicationIdValidationType.Ok, Constants.ProgramIdMinimum + 1};
                yield return new object[] {AocMeta.ApplicationIdValidationType.Ok, Constants.ProgramIdMaximum - 1};
                yield return new object[] {AocMeta.ApplicationIdValidationType.Ok, Constants.ProgramIdMaximum};
                yield return
                    new object[] {AocMeta.ApplicationIdValidationType.RangeOver, Constants.ProgramIdMaximum + 1};
            }
        }

        [Theory]
        [MemberData(nameof(ValidationApplicationIdTestData))]
        public void ValidateApplicationId(AocMeta.ApplicationIdValidationType actual, ulong id)
        {
            Assert.Equal(actual, AocMeta.ValidateApplicationId(id));
        }

        [Theory]
        [MemberData(nameof(ValidationApplicationIdTestData))]
        public void ValidationApplicationId(AocMeta.ApplicationIdValidationType actual, ulong id)
        {
            using (var aocMeta = new AocMeta())
            {
                aocMeta.ApplicationId = id;

                Assert.Equal(actual, aocMeta.ValidationApplicationId);
            }
        }

        [Fact]
        public void AocContent_Ctor()
        {
            using (new AocContent())
            {
            }
        }

        [Theory]
        [InlineData(AocContent.DataPathValidationType.Ok, ".")]
        [InlineData(AocContent.DataPathValidationType.DirectoryNotFound, @"zzz:fff")]
        [InlineData(AocContent.DataPathValidationType.PathIsEmpty, "")]
        [InlineData(AocContent.DataPathValidationType.PathIsEmpty, null)]
        public void AocContent_ValidateDataPath(AocContent.DataPathValidationType actual, string path)
        {
            Assert.Equal(actual, AocContent.ValidateDataPath(path));
        }

        [Theory]
        [InlineData(AocContent.DataPathValidationType.Ok, ".")]
        [InlineData(AocContent.DataPathValidationType.DirectoryNotFound, @"zzz:fff")]
        [InlineData(AocContent.DataPathValidationType.PathIsEmpty, "")]
        [InlineData(AocContent.DataPathValidationType.PathIsEmpty, null)]
        public void AocContent_ValidationDataPath(AocContent.DataPathValidationType actual, string path)
        {
            _context.GetInstance<AppProfile>().AppMode = AppModeType.AocMeta;

            using (var aocMeta = new AocMeta())
            using (var aocContent = new AocContent(_context.DiContainer, aocMeta))
            {
                aocContent.DataPath.Path = path;

                Assert.Equal(actual, aocContent.ValidationDataPath);
            }
        }

        [Theory]
        [InlineData(AocContent.IndexValidationType.Ok, 100)]
        [InlineData(AocContent.IndexValidationType.RangeOver, 0)]
        [InlineData(AocContent.IndexValidationType.Ok, 1)]
        [InlineData(AocContent.IndexValidationType.RangeOver, 2047)]
        [InlineData(AocContent.IndexValidationType.RangeOver, 2048)]
        [InlineData(AocContent.IndexValidationType.Ok, Constants.AocIndexMinimum)]
        [InlineData(AocContent.IndexValidationType.Ok, Constants.AocIndexMaximum)]
        public void AocContent_ValidateIndex(AocContent.IndexValidationType actual, ulong index)
        {
            var allContent = new ObservableCollection<AocContent>();

            Assert.Equal(actual, AocContent.ValidateIndex(index, null, allContent));
        }

        [Fact]
        public void AocContent_ValidateIndex_AllContentIsNull()
        {
            Assert.Equal(AocContent.IndexValidationType.Ok, AocContent.ValidateIndex(123, null, null));
        }

        [Fact]
        public void AocContent_ValidateIndex_AlreadyExist()
        {
            var allContent = new ObservableCollection<AocContent>
            {
                new AocContent {Index = 123},
                new AocContent {Index = 456}
            };

            Assert.Equal(AocContent.IndexValidationType.AlreadyExist, AocContent.ValidateIndex(123, null, allContent));
            Assert.Equal(AocContent.IndexValidationType.Ok, AocContent.ValidateIndex(123, allContent[0], allContent));
        }

        [Theory]
        [InlineData(AocContent.TagValidationType.Ok, "abc")]
        [InlineData(AocContent.TagValidationType.EmptyError, "")]
        [InlineData(AocContent.TagValidationType.EmptyError, null)]
        public void AocContent_ValidateTag(AocContent.TagValidationType actual, string tag)
        {
            var allContent = new ObservableCollection<AocContent>();

            Assert.Equal(actual, AocContent.ValidateTag(tag, null, allContent));
        }

        [Fact]
        public void AocContent_ValidateTag_AllContentIsNull()
        {
            Assert.Equal(AocContent.TagValidationType.Ok, AocContent.ValidateTag("abc", null, null));
        }

        [Fact]
        public void AocContent_ValidateTag_AlreadyExist()
        {
            var allContent = new ObservableCollection<AocContent>
            {
                new AocContent {Tag = "abc"},
                new AocContent {Tag = "def"},
            };

            Assert.Equal(AocContent.TagValidationType.AlreadyExist, AocContent.ValidateTag("abc", null, allContent));
            Assert.Equal(AocContent.TagValidationType.Ok, AocContent.ValidateTag("abc", allContent[0], allContent));
        }

        [Theory]
        [InlineData(1, AocContent.TagValidationType.Ok)]
        [InlineData(0, AocContent.TagValidationType.EmptyError)]
        [InlineData(AocContent.MaxTagLength, AocContent.TagValidationType.Ok)]
        [InlineData(AocContent.MaxTagLength + 1, AocContent.TagValidationType.LengthError)]
        public void AocContent_ValidateTag_Length(int tagLength, AocContent.TagValidationType expected)
        {
            var contents = new ObservableCollection<AocContent>();
            var tag = string.Join(string.Empty, Enumerable.Repeat('a', tagLength));
            Assert.Equal(expected, AocContent.ValidateTag(tag, null, contents));
        }
    }
}
