﻿using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Reflection;
using _3dToolsTestUtility;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.G3dTool.Entities;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace _3dIntermediateFileTextureUtilityTest
{
    [TestClass]
    public class _3DIntermediateFileTextureUtilityTest
    {
        [TestMethod]
        public void GenericConverterTest()
        {
            var testFilePath = Path.Combine(
                IoUtility.GetG3dDemoRootPath(),
                "Resources/human/textures/human_A.ftxb");

            string noOriginalImageFilePath = string.Empty;
            try
            {
                // オリジナルイメージを消す
                noOriginalImageFilePath = IoUtility.GetTempFilePath($"{MethodBase.GetCurrentMethod().Name}.ftxb");
                {
                    var file = IfReadUtility.ReadIntermediateFile(testFilePath, IoUtility.GetXsdBasePath());
                    var tex = file.GetRootEntity<Texture>();
                    foreach (var image in tex.OriginalImages)
                    {
                        tex.Streams.Remove(image.Stream);
                    }
                    tex.OriginalImages.Clear();
                    IfWriteUtility.WriteIntermediateFile(file, noOriginalImageFilePath, IoUtility.GetXsdBasePath());
                }

                var paths = new List<string>();
                paths.Add(Path.Combine(IoUtility.GetSdkRootPath(), @"Tools\Graphics\GraphicsTools\")); // TextureConverterTexUtils.dll のパス

                var converter = new nw.g3d.iftexutil.Converter();
                converter.Initialize(paths.ToArray());

                // Nintendo.G3dTool のインターフェース
                Bitmap[] g3dToolBitmaps;
                {
                    var file = IfReadUtility.ReadIntermediateFile(testFilePath, IoUtility.GetXsdBasePath());
                    g3dToolBitmaps = converter.ConvertTo1d2dBitmap(file.GetRootEntity<Texture>());
                }

                // 旧インターフェース
                Bitmap[] iflibBitmaps;
                {
                    var streams = new List<G3dStream>();
                    var file = IfBinaryReadUtility.Read(streams, testFilePath, IoUtility.GetXsdBasePath());
                    iflibBitmaps = converter.ConvertTo1d2dBitmap((textureType)file.Item, streams);
                }

                // 新旧で同じビットマップができるかをチェック
                Assert.AreEqual(iflibBitmaps.Length, g3dToolBitmaps.Length);
                for (int bitmapIndex = 0; bitmapIndex < g3dToolBitmaps.Length; ++bitmapIndex)
                {
                    Bitmap g3dToolBitmap = g3dToolBitmaps[bitmapIndex];
                    Bitmap iflibBitmap = iflibBitmaps[bitmapIndex];
                    Assert.AreEqual(iflibBitmap.Height, g3dToolBitmap.Height);
                    Assert.AreEqual(iflibBitmap.Width, g3dToolBitmap.Width);
                    for (int y = 0; y < g3dToolBitmap.Height; ++y)
                    {
                        for (int x = 0; x < g3dToolBitmap.Width; ++x)
                        {
                            Assert.AreEqual(iflibBitmap.GetPixel(x, y), g3dToolBitmap.GetPixel(x, y));
                        }
                    }
                }
            }
            finally
            {
                if (System.IO.File.Exists(noOriginalImageFilePath))
                {
                    System.IO.File.Delete(noOriginalImageFilePath);
                }
            }
        }

        [TestMethod]
        public void ConvertTest()
        {
            // サンプルファイルパス
            var fileName = Path.Combine(
                Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location),
                "../../../../../../Tools/Sources/Tests/3DEditorTest/Resources/Models/PrjDemActPierrot/textures/",
                "PierrotEye.0.ftxb");

            // コンバータを作成
            var converter = new nw.g3d.iftexutil.Converter();
            Assert.IsNotNull(converter);

            // ファイルを読み込む
            var streams= new List<G3dStream>();
            var nw4f_3dif = IfBinaryReadUtility.Read(streams, fileName, null);

            // テクスチャファイルとして読み込めたことを確認
            Assert.IsNotNull(nw4f_3dif);
            Assert.IsInstanceOfType(nw4f_3dif.Item, typeof(textureType));
            var texture = (textureType)nw4f_3dif.Item;
            Bitmap[] bitmaps = null;
            Bitmap[][] bitmapsArray = null;

            // テクスチャの種類に応じて読み込む
            switch (texture.texture_info.dimension)
            {
                case texture_info_dimensionType.Item1d:
                case texture_info_dimensionType.Item2d:
                {
                    bitmaps = converter.ConvertTo1d2dBitmap(texture, streams);
                    // テクスチャが1枚以上読み込めたことを確認
                    Assert.IsNotNull(bitmaps);
                    Assert.IsTrue(bitmaps.Any());
                    Assert.IsTrue(bitmaps[0].Width > 0 && bitmaps[0].Height> 0);
                    return;
                }
                case texture_info_dimensionType.Item1d_array:
                {
                    bitmapsArray = converter.ConvertTo1dArrayBitmap(texture, streams);
                    break;
                }
                case texture_info_dimensionType.Item2d_array:
                {
                    bitmapsArray = converter.ConvertTo2dArrayBitmap(texture, streams);
                    break;
                }
                case texture_info_dimensionType.Item3d:
                {
                    bitmapsArray = converter.ConvertTo3dBitmap(texture, streams);
                    break;
                }
                case texture_info_dimensionType.cube:
                {
                    bitmapsArray = converter.ConvertToCubeBitmap(texture, streams);
                    break;
                }
                case texture_info_dimensionType.cube_array:
                {
                    bitmapsArray = converter.ConvertToCubeArrayBitmap(texture, streams);
                    break;
                }
                default:
                    Assert.Fail();
                    break;
            }

            // テクスチャが1枚以上読み込めたことを確認
            Assert.IsNotNull(bitmapsArray);
            Assert.IsTrue(bitmapsArray.Any());
            Assert.IsTrue(bitmapsArray[0].Any());
            Assert.IsTrue(bitmapsArray[0][0].Width > 0 && bitmapsArray[0][0].Height > 0);
        }
    }
}
